/*
    This file is part of KolabAdmin.

    Copyright (C) 2006 Tobias Koenig <tobias.koenig@credativ.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include <QGridLayout>
#include <QLabel>
#include <QLineEdit>
#include <QListView>
#include <QPushButton>

#include "ldapsearchdialog.h"

class LdapModel : public QAbstractListModel
{
  public:
    LdapModel( QLdap *connection, QObject *parent = 0 )
      : QAbstractListModel( parent ),
        mConnection( connection )
    {
    }

    virtual int columnCount( const QModelIndex &parent = QModelIndex() ) const
    {
      return 1;
    }

    virtual int rowCount( const QModelIndex &parent = QModelIndex() ) const
    {
      return mList.count();
    }

    virtual QVariant data( const QModelIndex &index, int role = Qt::DisplayRole ) const
    {
      if ( !index.isValid() )
        return QVariant();

      if ( role == Qt::DisplayRole ) {
        if ( mDisplayFormat.isEmpty() )
          return mList[ index.row() ].dn();

        QLdapEntry entry = mList[ index.row() ];

        QString data = mDisplayFormat;
        const QStringList attributeNames = entry.attributeNames();

        for ( int i = 0; i < attributeNames.count(); ++i ) {
          data.replace( QString( "%" ) + attributeNames[ i ] + QString( "%" ), entry.value( attributeNames[ i ] ) );
        }

        return data;
      } else if ( role == Qt::UserRole ) {
        if ( mResultKey.isEmpty() || mResultKey.toLower() == "dn" )
          return mList[ index.row() ].dn();
        else
          return mList[ index.row() ].value( mResultKey );
      } else
        return QVariant();
    }

    void setSearchFilter( const QString &searchFilter )
    {
      mSearchFilter = searchFilter;
    }

    void setDisplayFormat( const QString &displayFormat )
    {
      mDisplayFormat = displayFormat;
    }

    void setBlackList( const QStringList &blacklist )
    {
      mBlackList = blacklist;
    }

    void setResultKey( const QString &key )
    {
      mResultKey = key;
    }

    QLdapEntry entry( const QModelIndex &index )
    {
      return mList[ index.row() ];
    }

    void query( const QString &input )
    {
      QString filter = mSearchFilter;
      filter.replace( "%i", (input.isEmpty() ? "*" : input) );

      // normalize wildcards
      while ( filter.contains( "**" ) )
        filter.replace( "**", "*" );

      QLdapResponse response = mConnection->search( mConnection->baseDn(), QLdap::Sub, filter,
                                                    QStringList(), QLatin1String( "sn" ) );
      if ( response.isValid() ) {
        mList.clear();
        QLdapEntry::List entries = response.entries();
        for ( int i = 0; i < entries.count(); ++i )
          if ( !mBlackList.contains( entries[ i ].dn() ) )
            mList.append( entries[ i ] );
      } else
        mList.clear();

      reset();
    }

  private:
    QLdapEntry::List mList;
    QString mSearchFilter;
    QString mDisplayFormat;
    QStringList mBlackList;
    QString mResultKey;
    QLdap* mConnection;
};


class LdapSearchDialog::Private
{
  public:
    Private( QLdap *connection )
    {
      model = new LdapModel( connection );
    }

    ~Private()
    {
      delete model;
      model = 0;
    }

    LdapModel* model;

    QLineEdit* searchText;
    QPushButton* button;
    QListView* view;
};

LdapSearchDialog::LdapSearchDialog( QLdap* connection, QWidget *parent )
  : QDialog( parent ), d( new Private( connection ) )
{
  QGridLayout *layout = new QGridLayout( this );

  QLabel *label = new QLabel( tr( "Filter:" ), this );
  layout->addWidget( label, 0, 0 );

  d->searchText = new QLineEdit( this );
  label->setBuddy( d->searchText );
  layout->addWidget( d->searchText, 0, 1 );

  d->button = new QPushButton( tr( "Search" ), this );
  layout->addWidget( d->button, 0, 2 );
  connect( d->button, SIGNAL( clicked() ), this, SLOT( search() ) );

  d->view = new QListView( this );
  d->view->setSelectionMode( QAbstractItemView::MultiSelection );
  d->view->setModel( d->model );
  layout->addWidget( d->view, 1, 0, 1, 3 );

  QHBoxLayout *buttonLayout = new QHBoxLayout;
  layout->addLayout( buttonLayout, 2, 0, 1, 3 );

  QPushButton *button = new QPushButton( tr( "&Ok" ), this );
  connect( button, SIGNAL( clicked() ), this, SLOT( accept() ) );
  buttonLayout->addWidget( button, 0, Qt::AlignRight );

  button = new QPushButton( tr( "&Cancel" ), this );
  connect( button, SIGNAL( clicked() ), this, SLOT( reject() ) );
  buttonLayout->addWidget( button, 0, Qt::AlignRight );
}

LdapSearchDialog::~LdapSearchDialog()
{
  delete d;
  d = 0;
}

void LdapSearchDialog::setSearchFilter( const QString &searchFilter )
{
  d->model->setSearchFilter( searchFilter );
}

void LdapSearchDialog::setDisplayFormat( const QString &displayFormat )
{
  d->model->setDisplayFormat( displayFormat );
}

void LdapSearchDialog::setBlackList( const QStringList &blacklist )
{
  d->model->setBlackList( blacklist );
}

QStringList LdapSearchDialog::selectedValues( const QString& key ) const
{
  QStringList values;

  QItemSelectionModel *selectionModel = d->view->selectionModel();
  QModelIndexList selectedIndexes = selectionModel->selectedIndexes();

  d->model->setResultKey( key );
  for ( int i = 0; i < selectedIndexes.count(); ++i ) {
    values.append( d->model->data( selectedIndexes[ i ], Qt::UserRole ).toString() );
  }

  return values;
}

QLdapEntry::List LdapSearchDialog::selectedEntries() const
{
  QItemSelectionModel *selectionModel = d->view->selectionModel();
  QModelIndexList selectedIndexes = selectionModel->selectedIndexes();

  QLdapEntry::List entries;
  for ( int i = 0; i < selectedIndexes.count(); ++i )
    entries.append( d->model->entry( selectedIndexes[ i ] ) );

  return entries;
}

void LdapSearchDialog::search()
{
  d->model->query( d->searchText->text() );
}
