/***************************************************************************
 *   Copyright (C) 2005 by Tommaso Frazzetto   *
 *   tommaso.frazzetto@gmail.com   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


#include "kompile.h"
#include <kapplication.h>
#include <kaboutdata.h>
#include <kcmdlineargs.h>
#include <klocale.h>
#include <kfiledialog.h>
#include <kstandarddirs.h>
#include <kmessagebox.h>
#include <kiconloader.h>
#include <qtooltip.h>

#include "kompileprofiledialog.h"
#include "kompileprofileselectionwidget.h"
#include "kompilewidget.h"

static const char description[] =
  I18N_NOOP("Kompile is a KDE frontend for compilation and installation of source tarball packages.");

static const char version[] = "0.3";

static KCmdLineOptions options[] =
  {
    { "i", 0, 0 },
    { "install", I18N_NOOP( "Perform install operation" ), 0 },
    { "u", 0, 0 },
    { "uninstall", I18N_NOOP( "Perform uninstall operation" ), 0 },
    { "t", 0, 0 },
    { "use-tarball <tarball>", I18N_NOOP( "Use specified tarball for requested operation [need -i or -u]" ), 0 },
    { "p", 0, 0 },
    { "use-profile <profile name>", I18N_NOOP( "Use specified profile for operation [need -i or -u. Exclude \"--use-tarball\"]" ), 0 },
    { "s", 0, 0 },
    { "search-profile", I18N_NOOP( "Use with \"--use-tarball <tarball>\".Kompile searchs a profile compatible with \"<tarball>\". If doesn't find one compatible use default profile. " ), 0 },

    KCmdLineLastOption
  };

int main(int argc, char **argv)
{
  KAboutData about("kompile", I18N_NOOP("Kompile"), version, description,
                   KAboutData::License_GPL, "(C) 2006 Tommaso Frazzetto", 0, 0, "tommaso.frazzetto@gmail.com");
  about.addAuthor( "Tommaso Frazzetto", 0, "tommaso.frazzetto@gmail.com" );
  KCmdLineArgs::init(argc, argv, &about);
  KCmdLineArgs::addCmdLineOptions( options );
  KApplication app;
  Kompile *mainWin = 0;

  if (app.isRestored())
  {
    RESTORE(Kompile);
  }
  else
  {
    KCmdLineArgs *args = KCmdLineArgs::parsedArgs();

    if (args->isSet( "install" ) )
    {
      bool profile_not_valid = FALSE;

      if (args->isSet( "use-profile"))
      {
        QString profile_name = args->getOption( "use-profile" );
        KConfig * profile = new KConfig(locateLocal("appdata", "") + "/" + profile_name + ".komprof");

        if (!profile->readEntry( "tarball", "" ).isEmpty())
        {
          KDialogBase * installdialog = new KDialogBase(NULL, "KompileInstall", TRUE, tr2i18n("Installing ") + profile->readEntry("tarball", ""), 0);
          KompileWidget * installwidget = new KompileWidget(profile, KompileWidget::OPERATION_INSTALL, installdialog);
          installdialog->setMainWidget( installwidget );
          installdialog->adjustSize();
          installdialog->exec();


          delete installdialog;
          delete installwidget;
          exit(0);
        }
        else
        {
          int result = KMessageBox::warningYesNoCancel(NULL, i18n("Specified sources tarball file in profile isn't a valid sources tarball archive!.\nDo you want manual select a source tarball and proceed with default installation?"), i18n("Sources tarball missing"), i18n("&Select tarball"), i18n("Load/Create &Profile"));

          if (result == KMessageBox::Cancel)
          {
            exit(-1);
          }
          else if (result == KMessageBox::No)
          {
            KompileProfileDialog * profiledialog = new KompileProfileDialog(NULL, "KompileInstallProfile");
            profiledialog->adjustSize();

            if (profiledialog->exec() != QDialog::Rejected)
            {
              KConfig * profile = profiledialog->getProfile();
              KDialogBase * installdialog = new KDialogBase(NULL, "KompileInstall", TRUE, tr2i18n("Installing ") + profile->readEntry("tarball", ""), 0);
              KompileWidget * installwidget = new KompileWidget(profile, KompileWidget::OPERATION_INSTALL, installdialog);
              installdialog->setMainWidget( installwidget );
              installdialog->adjustSize();
              installdialog->exec();


              delete installdialog;
              delete installwidget;
              delete profile;
            }

            delete profiledialog;
            exit(0);
          }
          else
          {
            profile_not_valid = TRUE;
          }
        }

        delete profile;
      }

      if (args->isSet( "use-tarball" ) || profile_not_valid)
      {
        QString tarball = 0;
        KConfig * profile;

        if (args->isSet( "use-tarball" ))
        {
          tarball = args->getOption( "use-tarball" );
        }
        else
        {
          tarball = KFileDialog::getOpenFileName("", "*.tar *.tar.gz *.tar.gz2 *.tgz *.tar.bz *.tar.bz2|" + tr2i18n("Source Tarball Archives"), NULL, tr2i18n("Select source tarball to install"));
          if (tarball.isEmpty())
            exit(-1);
        }

        KDialogBase * installdialog = new KDialogBase(NULL, "KompileInstall", TRUE, tr2i18n("Installing ") + tarball, 0);
        if (args->isSet( "search-profile" ))
        {
          QStringList tokens = QStringList::split("/", tarball);
          QStringList file = QStringList::split(".t", tokens[tokens.count() - 1]);
          QStringList prof = QStringList::split("-", file[0]);

          QDir * dir =  new QDir(locateLocal("appdata", ""));
          QStringList profiles = dir->entryList("*.komprof");
          delete dir;
          QStringList * p = new QStringList();

          while (!prof.isEmpty())
          {
            if (!profiles.grep(prof.join("-")).isEmpty())
            {
              profiles = profiles.grep(prof.join("-"));
              p->append( profiles[0] );
            }

            prof.pop_back();
          }

          if (!p->isEmpty())
          {
            KDialogBase * compatible_dialog = new KDialogBase( NULL, "KompileCompatibleProfiles", TRUE, tr2i18n( "Select Profile" ), KDialogBase::Ok|KDialogBase::Cancel, KDialogBase::Ok, TRUE );
            KListBox * profiles_list = new KListBox( compatible_dialog );
            for ( QStringList::Iterator p_entry = p->begin(); p_entry != p->end(); ++p_entry )
            {
              QString profile_name(*p_entry);
              profile_name.setLength(profile_name.length() - 8);
              profiles_list->insertItem( KGlobal::iconLoader()->loadIcon("make", KIcon::Toolbar, 22), profile_name, 0 );
            }

            compatible_dialog->setMainWidget( profiles_list );
            compatible_dialog->adjustSize();
            if (compatible_dialog->exec() != KDialogBase::Cancel)
            {
              profile = new KConfig(locateLocal("appdata", "") + profiles_list->currentText() + ".komprof");
              profile->writeEntry("tarball", tarball);
              profile->sync();
            }
            else
            {
              profile = new KConfig(locateLocal("appdata", "") + "/default-profile.komprof");
            }
          }
          else
          {
            profile = new KConfig(locateLocal("appdata", "") + "/default-profile.komprof");
          }

          delete p;
        }
        else
        {
          profile = new KConfig(locateLocal("appdata", "") + "/default-profile.komprof");
        }

        profile->writeEntry("tarball", tarball);

        KompileWidget * installwidget = new KompileWidget(profile, KompileWidget::OPERATION_INSTALL, installdialog);
        installdialog->setMainWidget( installwidget );
        installdialog->adjustSize();
        installdialog->exec();
        delete profile;
        delete installwidget;
        delete installdialog;
        exit(0);
      }
    }
    else if (args->isSet( "uninstall" ))
    {
      bool profile_not_valid = FALSE;

      if (args->isSet( "use-profile"))
      {
        QString profile_name = args->getOption( "use-profile" );
        KConfig * profile = new KConfig(locateLocal("appdata", "") + "/" + profile_name + ".komprof");

        if (!profile->readEntry( "tarball", "" ).isEmpty())
        {
          KDialogBase * installdialog = new KDialogBase(NULL, "KompileUninstall", TRUE, tr2i18n("Uninstalling ") + profile->readEntry("tarball", ""), 0);
          KompileWidget * installwidget = new KompileWidget(profile, KompileWidget::OPERATION_UNINSTALL, installdialog);

          installdialog->setMainWidget( installwidget );
          installdialog->adjustSize();
          installdialog->exec();


          delete installdialog;
          delete installwidget;
          exit(0);
        }
        else
        {
          int result = KMessageBox::warningYesNoCancel(NULL, i18n("Specified sources tarball file in profile isn't a valid sources tarball archive!.\nDo you want manual select a source tarball and proceed with default uninstallation?"), i18n("Sources tarball missing"), i18n("&Select tarball"), i18n("Load &Profile"));

          if (result == KMessageBox::Cancel)
          {
            exit(-1);
          }
          else if (result == KMessageBox::No)
          {
            KDialogBase * load_profile = new KDialogBase(NULL, "LoadProfile", TRUE, tr2i18n("Select existing profile"), KDialogBase::Ok|KDialogBase::Cancel);

            KompileProfileSelectionWidget * wgt = new KompileProfileSelectionWidget(load_profile);
            load_profile->setMainWidget(wgt);
            load_profile->adjustSize();

            if (load_profile->exec() == KDialogBase::Cancel)
              exit(-1);

            KConfig * profile = new KConfig(locateLocal("appdata", "") + "/" + wgt->selectedProfile() + ".komprof");
            KDialogBase * installdialog = new KDialogBase(NULL, "KompileUninstall", TRUE, tr2i18n("Uninstalling ") + profile->readEntry("tarball", ""), 0);
            KompileWidget * installwidget = new KompileWidget(profile,  KompileWidget::OPERATION_INSTALL, installdialog);

            installdialog->setMainWidget( installwidget );
            installdialog->adjustSize();
            installdialog->exec();

            delete profile;
            delete installwidget;
            delete installdialog;
            delete wgt;
            delete load_profile;
            exit(0);
          }
          else
          {
            profile_not_valid = TRUE;
          }
        }

        delete profile;
      }

      if (args->isSet( "use-tarball" ) || profile_not_valid)
      {
        QString tarball = 0;

        if (args->isSet( "use-tarball" ))
        {
          tarball = args->getOption( "use-tarball" );
        }
        else
        {
          tarball = KFileDialog::getOpenFileName("", "*.tar *.tar.gz *.tar.gz2 *.tgz *.tar.bz *.tar.bz2" + tr2i18n("Source Tarball Archives"), NULL, tr2i18n("Select source tarball to uninstall"));
          if (tarball.isEmpty())
            exit(-1);
        }

        KDialogBase * installdialog = new KDialogBase(NULL, "KompileUninstall", TRUE, tr2i18n("Uninstalling ") + tarball, 0);
        KConfig * profile = new KConfig(locateLocal("appdata", "") + "/default-profile.komprof");
        profile->writeEntry("tarball", tarball);
        KompileWidget * installwidget = new KompileWidget(profile, KompileWidget::OPERATION_UNINSTALL, installdialog);
        installdialog->setMainWidget( installwidget );
        installdialog->adjustSize();
        installdialog->exec();
        delete profile;
        delete installwidget;
        delete installdialog;
        exit(0);
      }
    }

    mainWin = new Kompile();
    app.setMainWidget( mainWin );
    mainWin->show();

    args->clear();
  }

  return app.exec();
}

