/*
   Copyright (C) 2006 by Stefan Taferner <taferner@kde.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

// case.cpp

#include "case.h"

#include <qdir.h>
#include <qxml.h>

#include <kmessagebox.h>
#include <kglobal.h>
#include <klocale.h>
#include <kstandarddirs.h>

#include <iostream>



namespace KoverArtist {

QValueVector<Case> Case::sCases;


//
//  CLASS  CaseLoader
//

class CaseLoader: public QXmlDefaultHandler
{
public:
   CaseLoader(Case* c, const QString& fn);
   virtual ~CaseLoader();

   virtual bool startDocument();
   virtual bool startElement(const QString&, const QString&, const QString&,
                             const QXmlAttributes&);
   virtual bool endElement(const QString&, const QString&, const QString&);
   virtual bool characters(const QString&);

protected:
   Case* mCase;
   int mVersion;
   QString mPath, mFileName;
};


CaseLoader::CaseLoader(Case* aCase, const QString& aFileName)
:QXmlDefaultHandler()
,mCase(aCase)
,mVersion(0)
,mPath()
,mFileName(aFileName)
{
}


CaseLoader::~CaseLoader()
{
}


bool CaseLoader::startDocument()
{
   mPath = "";
   mVersion = 0;
   mCase->clear();
   mCase->mName = "";
   mCase->mLabel = "";
   return true;
}


bool CaseLoader::startElement(const QString&, const QString&, const QString& aName,
                              const QXmlAttributes& aAtts)
{
   mPath += "/"+aName;
   //std::cout<<mPath<<std::endl;

   if (mPath=="/koverartistcase")
   {
      mVersion = aAtts.value("version").toInt();
      if (mVersion<1) mVersion = 1;
   }
   else if (mPath=="/koverartistcase/Label")
   {
   }
   else if (mPath=="/koverartistcase/FrontBackConnected")
   {
   }
   else if (mPath=="/koverartistcase/Front")
   {
      mCase->mFront.setWidth(aAtts.value("width").toInt());
      mCase->mFront.setHeight(aAtts.value("height").toInt());
   }
   else if (mPath=="/koverartistcase/Back")
   {
      mCase->mBack.setWidth(aAtts.value("width").toInt());
      mCase->mBack.setHeight(aAtts.value("height").toInt());
   }
   else if (mPath=="/koverartistcase/BackLeftSide")
   {
      mCase->mBackLeftSide = aAtts.value("width").toInt();
   }
   else if (mPath=="/koverartistcase/BackRightSide")
   {
      mCase->mBackRightSide = aAtts.value("width").toInt();
   }
   else if (mPath=="/koverartistcase/FrontLeftSide")
   {
      mCase->mFrontLeftSide = aAtts.value("width").toInt();
   }
   else if (mPath=="/koverartistcase/FrontRightSide")
   {
      mCase->mFrontRightSide = aAtts.value("width").toInt();
   }
   else
   {
      KMessageBox::error(0, i18n("Error reading %1:\nInvalid xml element: %2")
         .arg(mFileName).arg(mPath));
      return false;
   }

   return true;
}



bool CaseLoader::characters(const QString& aData)
{
   QString data = aData.stripWhiteSpace();
   if (mPath=="/koverartistcase/Label")
   {
      mCase->mLabel = data;
   }
   else if (mPath=="/koverartistcase/FrontBackConnected")
   {
      if (data=="left") mCase->mConnected = Case::ConnectLeft;
      else if (data=="right") mCase->mConnected = Case::ConnectRight;
      else if (data=="none") mCase->mConnected = Case::NotConnected;
      else
      {
	 KMessageBox::error(0, i18n("Error reading %1:\nInvalid 'Connected' value: %2")
	    .arg(mFileName).arg(data));
	 return false;
      }
   }
   return true;
}


bool CaseLoader::endElement(const QString&, const QString&, const QString&)
{
   int idx = mPath.findRev('/');
   if (idx>0) mPath = mPath.left(idx);
   else mPath = "";
   return true;
}



//
//  CLASS  Case
//

Case::Case()
:mName()
,mLabel()
,mConnected(Case::NotConnected)
,mFront(0, 0)
,mBack(0, 0)
,mBackLeftSide(0)
,mBackRightSide(0)
,mFrontLeftSide(0)
,mFrontRightSide(0)
{
}


Case::Case(const Case& o)
:mName(o.mName)
,mLabel(o.mLabel)
,mConnected(o.mConnected)
,mFront(o.mFront)
,mBack(o.mBack)
,mBackLeftSide(o.mBackLeftSide)
,mBackRightSide(o.mBackRightSide)
,mFrontLeftSide(o.mFrontLeftSide)
,mFrontRightSide(o.mFrontRightSide)
{
}


Case::~Case()
{
}


Case& Case::operator=(const Case& o)
{
   if (&o!=this)
   {
      mName = o.mName;
      mLabel = o.mLabel;
      mConnected = o.mConnected;
      mFront = o.mFront;
      mBack = o.mBack;
      mBackLeftSide = o.mBackLeftSide;
      mBackRightSide = o.mBackRightSide;
      mFrontLeftSide = o.mFrontLeftSide;
      mFrontRightSide = o.mFrontRightSide;
   }
   return *this;
}


void Case::clear()
{
   mName = "";
   mLabel = "";
   mConnected = NotConnected;
   mFront = QSize(0, 0);
   mBack = QSize(0, 0);
   mBackLeftSide = 0;
   mBackRightSide = 0;
   mFrontLeftSide = 0;
   mFrontRightSide = 0;
}


Case& Case::standardCase()
{
   static Case stdCase;
   if (stdCase.mFront.isNull())
   {
      stdCase.mName = "cd-standard";
      stdCase.mLabel = "Default Case";
      stdCase.mConnected = NotConnected;
      stdCase.mFront = QSize(1210, 1210);
      stdCase.mBack = QSize(1370, 1180);
      stdCase.mBackLeftSide = 60;
      stdCase.mBackRightSide = 60;
      stdCase.mFrontLeftSide = 0;
      stdCase.mFrontRightSide = 0;
   }
   return stdCase;
}


bool Case::loadFile(const QString& aFileName)
{
   clear();

   CaseLoader loader(this, aFileName);
   QFile file(aFileName);
   QXmlInputSource source(&file);
   QXmlSimpleReader reader;
   reader.setContentHandler(&loader);
   if (!reader.parse(&source, false)) return false;

   int idx = aFileName.findRev('.');
   if (idx>0) mName = aFileName.left(idx).lower();
   else mName = aFileName.lower();

   idx = mName.findRev('/');
   if (idx>=0) mName = mName.mid(idx+1);

   return true;
}


bool Case::loadCases()
{
   KStandardDirs* stdDirs = KGlobal::dirs();
   QStringList lst;
   QString nm;
   Case c;

   sCases.clear();

   lst = stdDirs->findAllResources("appdata", "cases/*.koac");
   QStringList::Iterator it;

   for (it=lst.begin(); it!=lst.end(); ++it)
   {
      nm = *it;
      if (c.loadFile(nm) && !findCase(nm))
         sCases.append(c);
   }

   qHeapSort(sCases);

   return !sCases.isEmpty();
}


Case* Case::findCase(const QString& aName)
{
   int i;
   for (i=sCases.count()-1; i>=0; --i)
      if (sCases[i].name()==aName) return &sCases[i];
   return 0;
}


int Case::indexOfCase(const QString& aName)
{
   int i;
   for (i=sCases.count()-1; i>=0; --i)
      if (sCases[i].name()==aName) break;
   return i;
}


bool Case::operator<(const Case& o) const
{
   return mLabel<o.mLabel;
}


} //namespace
