/*
   Copyright (C) 2006 by Stefan Taferner <taferner@kde.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

// textstyle.cpp

#include "textstyle.h"

#include <qdom.h>


namespace KoverArtist
{


TextStyle::TextStyle()
:mFont()
,mColor()
,mAlign(Qt::AlignCenter)
{
}


TextStyle::TextStyle(const QFont& aFont, const QColor& aColor,
                     Qt::AlignmentFlags aAlign)
:mFont(aFont)
,mColor(aColor)
,mAlign(aAlign)
{
}


TextStyle::TextStyle(const TextStyle& o)
:mFont(o.mFont)
,mColor(o.mColor)
,mAlign(o.mAlign)
{
}


TextStyle& TextStyle::operator=(const TextStyle& o)
{
   if (&o!=this)
   {
      mFont = o.mFont;
      mColor = o.mColor;
      mAlign = o.mAlign;
   }
   return *this;
}


void TextStyle::clear()
{
   *this = TextStyle();
}


void TextStyle::toDom(QDomDocument&, QDomElement& aElem) const
{
   aElem.setAttribute("font", mFont.toString());
   aElem.setAttribute("color", mColor.name());
   aElem.setAttribute("halign", alignAsString(mAlign & Qt::AlignHorizontal_Mask));
   aElem.setAttribute("valign", alignAsString(mAlign & Qt::AlignVertical_Mask));
}


bool TextStyle::fromDom(const QDomElement& aElem)
{
   mFont.fromString(aElem.attribute("font"));
   mColor = QColor(aElem.attribute("color"));

   int f = alignFromString(aElem.attribute("halign"), Qt::AlignHCenter) |
           alignFromString(aElem.attribute("valign"), Qt::AlignVCenter);
   mAlign = (Qt::AlignmentFlags)f;

   return true;
}


Qt::AlignmentFlags TextStyle::alignFromString(const QString& aStr,
                                            Qt::AlignmentFlags aDefault)
{
   if (aStr=="left") return Qt::AlignLeft;
   if (aStr=="hcenter") return Qt::AlignHCenter;
   if (aStr=="right") return Qt::AlignRight;
   if (aStr=="top") return Qt::AlignTop;
   if (aStr=="vcenter") return Qt::AlignVCenter;
   if (aStr=="bottom") return Qt::AlignBottom;

   return aDefault;
}


const char* TextStyle::alignAsString(int f)
{
   switch ((Qt::AlignmentFlags)f)
   {
   case Qt::AlignLeft: return "left";
   case Qt::AlignHCenter: return "hcenter";
   case Qt::AlignRight: return "right";
   case Qt::AlignTop: return "top";
   case Qt::AlignVCenter: return "vcenter";
   case Qt::AlignBottom: return "bottom";
   default: return "";
   }
}


} //namespace
