 /**************************************************************************
 *   Copyright (C) 2006 by Danny Kukawka                                   *
 *                            <dkukawka@suse.de>, <danny.kukawka@web.de>   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of version 2 of the GNU General Public License     *
 *   as published by the Free Software Foundation.                         *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.          *
 ***************************************************************************/

/*! 
 * \file 	dbusHAL.cpp
 * \brief 	In this file can be found the functionality to connect to 
 *		the HAL daemon via D-Bus, to handle D-Bus calls/events and to
 *		provide wrapper to HAL lib and functions
 * \author 	Danny Kukawka, <dkukawka@suse.de>, <danny.kukawka@web.de>
 * \date    	2006
 */
 
  // KDE Header
#include <klocale.h>

// DBUS - Header
#include "dbusHAL.h"

#include <iostream>

static void* myInstance = 0;

/*! The default constructor of the class dbusHAL. */
dbusHAL::dbusHAL(){
	myDebug ("dbusHAL::dbusHAL");
	dbus_is_connected = false;
	hal_is_connected = false;
	aquiredPolicyPower = false;
	hal_ctx = NULL;

	// add pointer to this for filter_function()
	myInstance=this;
	// init connection to dbus
	if(!initDBUS()) {
		myDebug("ERROR: Can't connect to D-Bus");
		m_dBusQtConnection = NULL;
	}
	if(!initHAL()) 
		myDebug("ERROR: Can't connect to HAL");

}

/*! This is the default destructor of class dbusPowersaveConnection. */
dbusHAL::~dbusHAL(){
	myDebug ("dbusHAL::~dbusHAL");
	close();
	myInstance = NULL;
}

/*! 
 * This function return information about connection status to the DBUS daemon.
 * \return boolean with the state of the connection to D-Bus
 * \retval true if connected
 * \retval false if disconnected
 */
bool dbusHAL::isConnectedToDBUS() {
	return dbus_is_connected;
}

/*! 
 * This function return information about connection status to the HAL daemon.
 * \return boolean with the state of the connection to HAL
 * \retval true if connected
 * \retval false if disconnected
 */
bool dbusHAL::isConnectedToHAL() {
	return hal_is_connected;
}

/*! 
 * This function return information if the org.freedesktop.Policy.Power 
 * interface was claimed.
 * \return boolean with the status of claim the interface
 * \retval true if aquired 
 * \retval false if not
 */
bool dbusHAL::aquiredPolicyPowerInterface() {
	return aquiredPolicyPower;
}

/*! 
 * This function try a reconnect to D-Bus and HAL  daemon.
 * \return boolean with the result of the operation
 * \retval true if successful reconnected to D-Bus and HAL
 * \retval false if unsuccessful
 */
bool dbusHAL::reconnect() {
	// free HAL context
	freeHAL();
	// close D-Bus connection
	close();
	// init D-Bus conntection and HAL context
	return (initDBUS() && initHAL());
}

/*! 
 * This function close the connection to powersave over the D-Bus daemon.
 * \return boolean with the result of the operation
 * \retval true if successful closed the connection
 * \retval false if any problems
 */
bool dbusHAL::close() {
	if ( m_dBusQtConnection != NULL ) {
		m_dBusQtConnection->close();
		m_dBusQtConnection = NULL;
	}
	dbus_is_connected = false;

	return true;
}

/* ----> D-Bus section :: START <---- */

/*! 
 * This function initialise the connection to the D-Bus daemon.
 * \return boolean with the result of the operation
 * \retval true if successful initialised D-Bus connection
 * \retval false if unsuccessful
 */
bool dbusHAL::initDBUS(){
	myDebug ("dbusHAL::initDBUS");

	dbus_is_connected = false;

        DBusError error;
        dbus_error_init(&error);
	
	dbus_connection = dbus_bus_get( DBUS_BUS_SYSTEM, &error );

	if (dbus_connection == NULL){
		myDebug ( "Failed to open connection to system message bus: %s\n", error.message);
		dbus_error_free (&error);
		return false;
	}

	if ( dbus_error_is_set( &error ) ) {
		myDebug ( "Failed to register connection with system message bus: %s\n", error.message);
		return false;
	}

	switch (dbus_bus_request_name(dbus_connection, "org.freedesktop.Policy.Power",
				      DBUS_NAME_FLAG_REPLACE_EXISTING, NULL)) {
		case DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER:
			myDebug("Acquired org.freedesktop.Policy.Power interface");
			aquiredPolicyPower = true;
			break;
		case DBUS_REQUEST_NAME_REPLY_IN_QUEUE:
			myDebug("Queued to aquire org.freedesktop.Policy.Power interface");
			aquiredPolicyPower = false;
			break;
		default:
			myDebug("Unknown error while aquire org.freedesktop.Policy.Power interface");
			aquiredPolicyPower = false;
			break;
	}

	dbus_connection_set_exit_on_disconnect( dbus_connection, false );

        /* add the filter function which should be executed on events on the bus */
        if ( ! dbus_connection_add_filter( dbus_connection, filterFunction, this, NULL) ) {
                myDebug ("Error: Not enough memory to add filter to dbus connection\n");
                exit(EXIT_FAILURE);
        }

        /* add a match rule to catch all signals going through the bus with D-Bus interface */
	dbus_bus_add_match( dbus_connection, "type='signal',"
			    "interface='org.freedesktop.DBus'," 
			    "member='NameOwnerChanged'", NULL);

	/* add a match rule to catch all signals going through the bus with HAL interface */
	dbus_bus_add_match( dbus_connection, "type='signal',"
			    "interface='org.freedesktop.Hal.Manager'," 
			    "member='DeviceAdded'", NULL);
	dbus_bus_add_match( dbus_connection, "type='signal',"
			    "interface='org.freedesktop.Hal.Manager'," 
			    "member='DeviceRemoved'", NULL);
	dbus_bus_add_match( dbus_connection, "type='signal',"
			    "interface='org.freedesktop.Hal.Device'," 
			    "member='PropertyModified'", NULL);
	dbus_bus_add_match( dbus_connection, "type='signal',"
			    "interface='org.freedesktop.Hal.Device'," 
			    "member='Condition'", NULL);
	
	m_dBusQtConnection = new DBusQt::Connection(this);
        m_dBusQtConnection->dbus_connection_setup_with_qt_main(dbus_connection);
	
	dbus_is_connected = true;
	return true;
}

/* ----> DBUS section :: END   <---- */
/* ----> HAL  section :: START <---- */

/*! 
 * This function initialise the connection to HAL over the D-Bus daemon.
 * \return boolean with the result of the operation
 * \retval true if successful initialised HAL connection and context
 * \retval false if unsuccessful
 */
bool dbusHAL::initHAL(){

	if ( !dbus_is_connected ) {
		freeHAL();
		return false;
	} else if ( hal_is_connected && (hal_ctx != NULL)) { 
		return true;
	}
	
	// could not connect to HAL, reset all and try again
	freeHAL();

	DBusError error;
	dbus_error_init(&error);

	dbus_connection = dbus_bus_get(DBUS_BUS_SYSTEM, &error);
	if (dbus_connection == NULL || dbus_error_is_set(&error)) {
		myDebug("could not open connection to system bus: %s", error.message);
		dbus_error_free(&error);
		return false;
	}

	bool hal_is_ready = dbus_bus_name_has_owner(dbus_connection, "org.freedesktop.Hal", &error);

	if (!hal_is_ready) {
		myDebug("HAL is not ready. We will try later... ");

		if ( dbus_error_is_set( &error ) ) {
			myDebug("Error checking if hal service exists: %s", error.message );
			dbus_error_free( &error );
		}

		freeHAL();
		return false;
	}

	if((hal_ctx = libhal_ctx_new()) == NULL) {
		myDebug("Could not init HAL context");
		return false;
	}

	/* setup dbus connection for hal */
	if (!libhal_ctx_set_dbus_connection(hal_ctx, dbus_connection)) {
		myDebug("Could not set up connection to dbus for hal");
		freeHAL();
		return false;
	}

	/* init the hal library */
	if (!libhal_ctx_init(hal_ctx, &error)) {
		myDebug("Could not init hal library: %s", error.message);
		freeHAL();
		return false;
	}

	hal_is_connected = true;
	return hal_is_connected;
}

/*! 
 * This function free the hal connection/context.
 */
void dbusHAL::freeHAL(){

	if ( hal_ctx != NULL ) {
		libhal_ctx_free( hal_ctx );
		hal_ctx = NULL;
	}
	hal_is_connected = false;
}

/*! 
 * This function try a reconnect to the HAL daemon only.
 * \return boolean with the result of the operation
 * \retval true if successful reconnected to HAL
 * \retval false if unsuccessful
 */
bool dbusHAL::reconnectHAL() {
	// free HAL context
	freeHAL();
	// init HAL context
	return (initHAL());
}

/*! 
 * This function query a integer property from HAL for a given device
 * \param udi		QString with the UDI of the device
 * \param property 	QString with the property
 * \param returnval 	pointer to the return value
 * \return 		If the query was successful or not
 */
bool dbusHAL::halGetPropertyInt(QString udi, QString property, int *returnval){

	if (!initHAL()) 
		return false;
	if (udi.isEmpty() || property.isEmpty())
		return false;

	DBusError error;
	dbus_error_init(&error);

	if (!libhal_device_property_exists(hal_ctx, udi, property, &error)) {
		myDebug("Property '%s' for device '%s' does not exist.", property.latin1(), udi.latin1());
		return false;
	}

	*returnval = libhal_device_get_property_int(hal_ctx, udi, property, &error);

	if (dbus_error_is_set(&error)) {
		myDebug("Error fetching property '%s' for device '%s': '%s'", property.latin1(),
									      udi.latin1(), error.message);
		dbus_error_free(&error);
		return false;
	}

	return true;
}

/*! 
 * This function query a boolean property from HAL for a given device
 * \param udi		QString with the UDI of the device
 * \param property 	QString with the property
 * \param returnval 	pointer to the return value
 * \return 		If the query was successful or not
 */
bool dbusHAL::halGetPropertyBool(QString udi, QString property, bool *returnval){

	if (!initHAL()) 
		return false;
	if (udi.isEmpty() || property.isEmpty())
		return false;

	DBusError error;
	dbus_error_init(&error);

	if (!libhal_device_property_exists(hal_ctx, udi, property, &error)) {
		myDebug("Property '%s' for device '%s' does not exist.", property.latin1(), udi.latin1());
		return false;
	}

	*returnval = libhal_device_get_property_bool(hal_ctx, udi, property, &error);

	if (dbus_error_is_set(&error)) {
		myDebug("Error fetching property '%s' for device '%s': '%s'", property.latin1(),
									      udi.latin1(), error.message);
		dbus_error_free(&error);
		return false;
	}

	return true;
}


/*! 
 * This function query a Sting property from HAL for a given device
 * \param udi		QString with the UDI of the device
 * \param property 	QString with the property
 * \param returnval 	pointer to the return value
 * \return 		If the query was successful or not
 */
bool dbusHAL::halGetPropertyString(QString udi, QString property, QString *returnval){

	if (!initHAL()) 
		return false;
	if (udi.isEmpty() || property.isEmpty())
		return false;

	DBusError error;
	dbus_error_init(&error);

	if (!libhal_device_property_exists(hal_ctx, udi, property, &error)) {
		myDebug("Property '%s' for device '%s' does not exist.", property.latin1(), udi.latin1());
		return false;
	}

	// TODO: check if we maybe need a cast to QString 
	*returnval = libhal_device_get_property_string(hal_ctx, udi, property, &error);

	if (dbus_error_is_set(&error)) {
		myDebug("Error fetching property '%s' for device '%s': '%s'", property.latin1(),
									      udi.latin1(), error.message);
		dbus_error_free(&error);
		return false;
	}

	return true;
}

/*!
 * This function query a String List property from HAL for a given device
 * \param udi		QString with the udi of the device
 * \param property	QString with the property to query 
 * \param devices 	QStringList to return the values
 * \return 		If the query was successful or not
 */
bool dbusHAL::halGetPropertyStringList (QString udi, QString property, QStringList *devices) {

	if (!initHAL()) 
		return false;
	if (udi.isEmpty() || property.isEmpty())
		return false;

	DBusError error;
	char ** found;

	dbus_error_init(&error);
	
	found = libhal_device_get_property_strlist (hal_ctx, udi, property, &error);

	if (dbus_error_is_set(&error)) {
                myDebug("Could not get list of devices with property '%s': %s", property.latin1(), error.message);
                dbus_error_free(&error);
                libhal_free_string_array(found);
                return false;
        } else {
 		for (int i = 0; found[i] != NULL ; ++i) {
			QString _to_add = found[i];
			if (!_to_add.isEmpty()) {
				// TODO: check if this really work, not complete sure atm
				*devices += _to_add;
			}
		}
		libhal_free_string_array(found);
		return true;
	}
}


/*! 
 * This function query a capability from HAL for a given device
 * \param udi		QString with the UDI of the device
 * \param capability	QString with the capability to query
 * \param returnval 	pointer to the return value as boolean
 * \return 		If the query was successful or not
 */
bool dbusHAL::halQueryCapability(QString udi, QString capability, bool *returnval) {

	if (!initHAL()) 
		return false;
	if (udi.isEmpty() || capability.isEmpty())
		return false;

	DBusError error;
	dbus_error_init(&error);

	*returnval = libhal_device_query_capability(hal_ctx, udi, capability, &error);

	if (dbus_error_is_set(&error)) {
		myDebug("Error fetching capability '%s' for device '%s': '%s'", capability.latin1(),
									        udi.latin1(), error.message);
		dbus_error_free(&error);
		return false;
	}

	return true;
}

/*! 
 * Use this function to check if a device has a specia property/key.
 * \param udi		QString with the UDI of the device
 * \param property 	QString with the property
 * \return 		If the query was successful or not
 */
bool dbusHAL::halDevicePropertyExist(QString udi, QString property ) {

	if (!initHAL()) 
		return false;
	if (udi.isEmpty() || property.isEmpty())
		return false;

	DBusError error;
	dbus_error_init(&error);

	if (! libhal_device_property_exists (hal_ctx, udi, property, &error)) {
		if (dbus_error_is_set(&error)) {
			myDebug("Error check for property '%s' for device '%s': '%s'", 
				 property.latin1(), udi.latin1(), error.message);
			dbus_error_free(&error);
		}
		return false;
	} else {
		return true;
	}
}

/*!
 * Use this function to search find devices with a give capability
 * \param capability	QString with the capability to query
 * \param devices 	QStringList to return the found devices
 * \return 		If the query was successful or not
 */
bool dbusHAL::halFindDeviceByCapability (QString capability, QStringList *devices) {

	if (!initHAL()) 
		return false;
	if (capability.isEmpty())
		return false;

	DBusError error;
	char ** found;
	int num = 0;

	dbus_error_init(&error);
	
	found = libhal_find_device_by_capability (hal_ctx, capability, &num, &error);

	if (dbus_error_is_set(&error)) {
                myDebug("Could not get list of devices with capability '%s': %s", capability.latin1(),
										  error.message);
                dbus_error_free(&error);
                libhal_free_string_array(found);
                return false;
        } else {
 		for (int i = 0; i < num; ++i) {
			QString _to_add = found[i];
			if (!_to_add.isEmpty()) {
				// TODO: check if this really work, not complete sure atm
				*devices += _to_add;
			}
		}
		libhal_free_string_array(found);
		return true;
	}
}

/*!
 * Use this function to search find devices with a special string property
 * \param property      QString with the name of the property 
 * \param keyval	QString with value of the string property
 * \param devices 	QStringList to return the found devices
 * \return 		If the query was successful or not
 */
bool dbusHAL::halFindDeviceByString (QString property, QString keyval, QStringList *devices) {

	if (!initHAL()) 
		return false;
	if (property.isEmpty() || keyval.isEmpty())
		return false;

	DBusError error;
	char ** found;
	int num = 0;
	
	dbus_error_init(&error);
	
	found = libhal_manager_find_device_string_match (hal_ctx, property, keyval, &num, &error);

	if (dbus_error_is_set(&error)) {
                myDebug("Could not get list of devices with key '%s' and string value '%s': %s", 
			 property.latin1(), keyval.latin1(), error.message);
                dbus_error_free(&error);
                libhal_free_string_array(found);
                return false;
        } else {
 		for (int i = 0; i < num; ++i) {
			QString _to_add = found[i];
			if (!_to_add.isEmpty()) {
				// TODO: check if this really work, not complete sure atm
				*devices += _to_add;
			}
		}
		libhal_free_string_array(found);
		return true;
	}
}

/* ----> HAL section :: END <---- */
/* ----> D-Bus methode calls functions :: START <---- */
/*! 
 * This function call a D-Bus method
 * \param interface	QString with te dbus interface
 * \param path		QString with the object path
 * \param object	QString with the object name
 * \param method	QString with the name of the methode
 * \param first_arg_type integer with the dbus type of the first argument
 * \param ... 		more arguments
 * \return 		If the query was successful or not
 */
bool dbusHAL::dbusSystemMethodCall( QString interface, QString path, QString object, QString method,
			   	    int first_arg_type, ... ) {
	myDebug("dbusHAL::dbusSystemMethodCall");
	
	bool _ret = false;
        va_list var_args;

        va_start(var_args, first_arg_type);
	_ret = dbusMethodCall( interface, path, object, method, DBUS_BUS_SYSTEM,
			       NULL, -1, first_arg_type, var_args);
	va_end(var_args);
	
	return _ret;
}


/*! 
 * This overloaded function call a D-Bus method on the D-Bus system bus with a return value
 * \param interface	 QString with the dbus interface
 * \param path		 QString with the object path
 * \param object	 QString with the object name
 * \param method	 QString with the name of the method
 * \param retvalue	 void pointer to arguments, if NULL we make a simple call
 * \param retval_type    Integer with the dbus type of the return value, set to -1 if retvalue is NULL
 * \param first_arg_type Integer with the dbus type of the first argument followed by the value
 * \return 		 If the query was successful or not
 */
bool dbusHAL::dbusSystemMethodCall( QString interface, QString path, QString object, QString method,
				    void *retvalue, int retval_type, int first_arg_type, ... ) {
	myDebug("dbusHAL::dbusSystemMethodCallWithReturn");
	
	bool _ret = false;
        va_list var_args;

        va_start(var_args, first_arg_type);
	_ret = dbusMethodCall( interface, path, object, method, DBUS_BUS_SYSTEM, 
			       retvalue, retval_type, first_arg_type, var_args);
	va_end(var_args);
	
	return _ret;
}


/*! 
 * This function call a D-Bus method with a return value
 * \param interface	 QString with the dbus interface
 * \param path		 QString with the object path
 * \param object	 QString with the object name
 * \param method	 QString with the name of the method
 * \param dbus_type	 DBusBusType with the D-Bus BUS Type
 * \param retvalue	 void pointer to arguments, if NULL we make a simple call
 * \param retval_type    Integer with the dbus type of the return value, set to -1 if retvalue is NULL
 * \param first_arg_type Integer with the dbus type of the first argument followed by the value
 * \param var_args	 va_list with more arguments
 * \return 		 If the query was successful or not
 */
bool dbusHAL::dbusMethodCall( QString interface, QString path, QString object, QString method,
			      DBusBusType dbus_type, void *retvalue, int retval_type, int first_arg_type,
			      va_list var_args ) {
	myDebug("dbusHAL::dbusMethodCall");

	DBusMessage *message;
	DBusMessage *reply;
	DBusError    error;
	
	dbus_error_init(&error); 

	dbus_connection = dbus_bus_get(dbus_type, &error);
	
	if (dbus_error_is_set(&error)) {
		myDebug("Could not get dbus connection: %s", error.message);
		dbus_error_free(&error);
		return false;
	}

	message = dbus_message_new_method_call( interface, path, object, method );
	dbus_message_append_args_valist(message, first_arg_type, var_args);

	if (retvalue == NULL) {
		if (!dbus_connection_send(dbus_connection, message, NULL)) {
			myDebug ("Could not send method call.");
			dbus_message_unref( message );
			return false;
		}
	} else {
		reply = dbus_connection_send_with_reply_and_block(dbus_connection, message, -1, &error);

		if (dbus_error_is_set(&error)) {
			myDebug("Could not send dbus message: %s", error.message);
			dbus_message_unref(message);
			dbus_error_free(&error);
			return false;
		}

		if (!dbus_message_get_args(reply, &error, retval_type, retvalue, DBUS_TYPE_INVALID)){
			if (dbus_error_is_set(&error)) {
				myDebug("Could not get argument from reply: %s", error.message);
				dbus_error_free(&error);
			}
			dbus_message_unref(reply);
			dbus_message_unref(message);
			return false;
        	}
	
	}
	dbus_message_unref(message);
	dbus_connection_flush(dbus_connection);
        return true;
}

/*! 
 * Function to call a suspend and call if resumed \ref callBackSuspend()
 * to emit a resume signal.
 * \param suspend 	a char pointer with the name of the suspend interface
 * \return 		If the query was successful or not
 */
bool dbusHAL::dbusMethodCallSuspend ( char *suspend ) {
	myDebug("dbusHAL::dbusMethodCallSuspend");

    	DBusMessage *message;
	DBusError    error;
	DBusPendingCall* pcall = NULL;

	dbus_error_init(&error);
	dbus_connection = dbus_bus_get(DBUS_BUS_SYSTEM, &error);
	
	if (dbus_error_is_set(&error)) {
		myDebug("Could not get dbus connection: %s", error.message);
		dbus_error_free(&error);
		return false;
	}
		
	message = dbus_message_new_method_call( HAL_SERVICE, HAL_COMPUTER_UDI, HAL_PM_IFACE, suspend);
	if (strcmp( suspend, "Suspend") == 0) {
		int wake_up = 0;
		dbus_message_append_args (message, DBUS_TYPE_INT32, &wake_up, DBUS_TYPE_INVALID);
	}
	
	if (message) {
		// need to set INT_MAX as default and not -1
		dbus_connection_send_with_reply (dbus_connection, message, &pcall, INT_MAX);
		if (pcall) {
			dbus_pending_call_ref (pcall); // really needed?
			dbus_pending_call_set_notify (pcall, dbusHAL::callBackSuspend, NULL, NULL);
		}
		dbus_message_unref (message);
		return true;
	}
	return false;
}

/*!
 * Slot called by D-Bus as set in \ref dbusMethodCallSuspend() 
 * Here we emit the resume signal.
 */
void dbusHAL::callBackSuspend (DBusPendingCall* pcall, void* /*data*/) {
	myDebug("dbusHAL::callBackSuspend");

	DBusMessage* reply = NULL;
	DBusError error;
	unsigned int result;

        if (!pcall) {
		myDebug("dbusHAL::callBackSuspend - DBusPendingCall not set, return");
		return;
        }

        reply = dbus_pending_call_steal_reply (pcall);
        if (reply == NULL) {
		myDebug("dbusHAL::callBackSuspend - Got no reply, return");
                goto out;
        }

	dbus_error_init(&error);

        if (dbus_message_get_args (reply, &error, DBUS_TYPE_UINT32, &result, DBUS_TYPE_INVALID)) {
                emit ((dbusHAL*) myInstance)->backFromSuspend( result );
        } else {
		if (dbus_error_is_set(&error)) {
			myDebug("Could not get argument from reply: %s", error.message);
			dbus_error_free(&error);
		}

		myDebug("dbusHAL::callBackSuspend - dbus_message_get_args failed, maybe timouted");
		emit ((dbusHAL*) myInstance)->backFromSuspend( -1 );
	}

        dbus_message_unref (reply);

out:
        dbus_pending_call_unref (pcall);

        return;
}


/* ----> D-Bus methode calls functions :: END  <---- */
/* ---> PolicyKit method call section :: START <--- */

/*!
 * Check if the user is privileged to a special privilege
 * \param privilege	QString with the name of the requested privilege
 * \param user		QString with the name of the user. If empty the current user is used.
 * \param ressource	QString with the name of the ressource.
 * \return int with info if the user is allowed or not.
 * \retval 0		if not allowed
 * \retval 1		if allowed
 * \retval -1		if a error occurs or we could not query the interface
 */
int dbusHAL::isUserPrivileged(QString privilege, QString user, QString ressource) {
	myDebug("dbusHAL::isUserAllowedTo");

	const char *_unique_name;
	const char *_user;
	const char *_privilege;	
	const char *_ressource;
	int retval = 0;

	if (user.isEmpty() || user.isNull()) user = getenv("USER"); 

	_unique_name = dbus_bus_get_unique_name(dbus_connection);
	_user = user.latin1();
	_privilege = privilege.latin1();
	_ressource = ressource.latin1();

	if (dbusSystemMethodCall( "org.freedesktop.PolicyKit",
				  "/org/freedesktop/PolicyKit/Manager",
				  "org.freedesktop.PolicyKit.Manager",
				  "IsUserPrivileged",
				  &retval, DBUS_TYPE_BOOLEAN,
				  DBUS_TYPE_STRING, &_unique_name,
				  DBUS_TYPE_STRING, &_user,
				  DBUS_TYPE_STRING, &_privilege,
				  DBUS_TYPE_STRING, &_ressource,
				  DBUS_TYPE_INVALID)) {
		return retval;
	}

	return -1;
}
/* ---> PolicyKit method call section :: END   <--- */

/*!
 * Use this SLOT to emit a reviced messages to the kpowersave.
 * NOTE: Because of the filter function this need to be a public function.
 *       Don't use this function in any other place than this class.
 * \param  type enum with the type of the message
 * \param  message String with the message
 * \param  string String with additional info
 */
void dbusHAL::emitMsgReceived( msg_type type, QString message, QString string ) {
	
	if (message.startsWith("dbus.terminate"))
		dbus_is_connected = false;
	
	emit msgReceived_withStringString( type, message, string );
}

/*! 
 * This function is needed filter function for the D-Bus connection to filter
 * all needed messages from the bus which are needful for KPowersave. 
 * \param connection	existing connection to the D-Bus daemon
 * \param message 	the recieved message from the D-Bus daemon
 * \param data		void pointer (see dbus bindings for more information)
 * \return DBusHandlerResult
 */
DBusHandlerResult 
filterFunction (DBusConnection *connection, DBusMessage *message, void *data) {
 	myDebug ("HAL - filterFunction");
	
	bool reply_wanted;
	char *value;
	QString ifaceType;

	if( connection == NULL || data == NULL ) ; // to prevent compiler warning

	DBusError error;
        dbus_error_init( &error );

	if (dbus_message_is_signal (message,
				    DBUS_INTERFACE_LOCAL,
				    "Disconnected")){
		((dbusHAL*) myInstance)->emitMsgReceived( DBUS_EVENT, "dbus.terminate", 0 );
		dbus_connection_unref(connection);
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	}

        if ( dbus_message_get_type( message ) != DBUS_MESSAGE_TYPE_SIGNAL ) {
                // TODO: add debug message and print type
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
        }

	ifaceType = dbus_message_get_interface( message );
        if (ifaceType == NULL) {
                myDebug ("Received message from invalid interface");
                return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
        }

	reply_wanted = !dbus_message_get_no_reply( message );

	if (ifaceType.startsWith(DBUS_INTERFACE_DBUS)) {
		 myDebug ("Received from DBUS_INTERFACE_DBUS");
		/* get the name of the signal */
		const char *signal = dbus_message_get_member( message );
		
		/* get the first argument. This must be a string at the moment */
		dbus_message_get_args( message, &error, DBUS_TYPE_STRING, &value, DBUS_TYPE_INVALID );
	
		if ( dbus_error_is_set( &error ) ) {
			myDebug ("Warning: Received signal %s, but no string argument\n", error.message );
			dbus_error_free( &error );
			return DBUS_HANDLER_RESULT_HANDLED;
		}
	
		myDebug("filter_function::SIGNAL=%s|VALUE=%s", signal, value);
	
		/* our name is... */
		if ( ! strcmp( signal, "NameAcquired" ) ) {
			return DBUS_HANDLER_RESULT_HANDLED;
		}
		
		else if ( ! strcmp( signal, "NameOwnerChanged" )) {
			char *service;
			char *old_owner;
			char *new_owner;
			
                	if (dbus_message_get_args (message, NULL, DBUS_TYPE_STRING, &service,
                                                   DBUS_TYPE_STRING, &old_owner,
                                                   DBUS_TYPE_STRING, &new_owner, DBUS_TYPE_INVALID)) {
				if (!strcmp(service, "org.freedesktop.Hal")) {
					if (!strcmp(new_owner, "") && strcmp(old_owner, "")) {
						// Hal service stopped.
						myDebug("=== org.freedesktop.Hal terminated ===");
						((dbusHAL*) myInstance)->emitMsgReceived( DBUS_EVENT,
											"hal.terminate", 
											NULL );
					}
					else if (!strcmp(old_owner, "") && strcmp(new_owner, "")) {
						// Hal service started.
						myDebug("=== org.freedesktop.Hal started ===");
						((dbusHAL*) myInstance)->emitMsgReceived( DBUS_EVENT,
											"hal.started", 
											NULL );
					}
				}
			}
		}
		return DBUS_HANDLER_RESULT_HANDLED;	
	} else if (ifaceType.startsWith("org.freedesktop.Hal.Manager")) {
		myDebug ("Received from org.freedesktop.Hal.Manager");
		char *udi;

		const char *signal = dbus_message_get_member( message );
		/* get the first argument. This must be a string at the moment */
		dbus_message_get_args( message, &error, DBUS_TYPE_STRING, &value, DBUS_TYPE_INVALID );
	
		if ( dbus_error_is_set( &error ) ) {
			myDebug ("Warning: Received signal %s, but no string argument\n", error.message );
			dbus_error_free( &error );
			return DBUS_HANDLER_RESULT_HANDLED;
		}

                if (dbus_message_get_args( message, &error, DBUS_TYPE_STRING, &udi, DBUS_TYPE_INVALID )) {
                        if (! strcmp(signal, "DeviceRemoved") || ! strcmp(signal, "DeviceAdded")) {
				((dbusHAL*) myInstance)->emitMsgReceived( HAL_DEVICE, signal, udi );
				myDebug ("org.freedesktop.Hal.Manager: uid: %s, signal: %s", udi, signal);
                        } else {
				myDebug ("Received unknown signal from org.freedesktop.Hal.Manager: %s", signal);
                        }
                }
		return DBUS_HANDLER_RESULT_HANDLED;
		
	} else if (ifaceType.startsWith("org.freedesktop.Hal.Device")) {		
		const char *udi = dbus_message_get_path (message);
		const char *signal = dbus_message_get_member( message );

		/* Code taken from libhal */
		 if (! strcmp(signal, "PropertyModified")) {
			myDebug ("PropertyModified --------------------");
			int i, num_modifications;
			DBusMessageIter iter;
			DBusMessageIter iter_array;

			dbus_message_iter_init (message, &iter);
			dbus_message_iter_get_basic (&iter, &num_modifications);
			dbus_message_iter_next (&iter);
			
			dbus_message_iter_recurse (&iter, &iter_array);
			
			for (i = 0; i < num_modifications; i++) {
				dbus_bool_t removed, added;
				char *key;
				DBusMessageIter iter_struct;
			
				dbus_message_iter_recurse (&iter_array, &iter_struct);
			
				dbus_message_iter_get_basic (&iter_struct, &key);
				dbus_message_iter_next (&iter_struct);
				dbus_message_iter_get_basic (&iter_struct, &removed);
				dbus_message_iter_next (&iter_struct);
				dbus_message_iter_get_basic (&iter_struct, &added);
			
				/* don't check if this is a device we really need ... check this in an other class */
				((dbusHAL*) myInstance)->emitMsgReceived( HAL_PROPERTY_CHANGED, udi, key);
				myDebug ("PropertyModified: uid: %s, key: %s", udi, key);

				dbus_message_iter_next (&iter_array);
			}
		} else if (! strcmp(signal, "Condition")) {
			char *name, *detail;

			dbus_message_get_args( message, &error, DBUS_TYPE_STRING, &value, DBUS_TYPE_INVALID );	

			if (dbus_message_get_args (message, &error, DBUS_TYPE_STRING, &name,
						   DBUS_TYPE_STRING, &detail, DBUS_TYPE_INVALID)) {
				((dbusHAL*) myInstance)->emitMsgReceived( HAL_CONDITION, name, detail );
			} else {
				if (dbus_error_is_set( &error )) dbus_error_free( &error );
			}
		} else {
			myDebug ("Received unknown signal from org.freedesktop.Hal.Device: %s", signal);
		}
		return DBUS_HANDLER_RESULT_HANDLED;
	} else {
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	}
}

// --> some functions to get private members

//! to get the current connection to D-Bus
DBusConnection * dbusHAL::get_DBUS_connection() {
	myDebug("dbusHAL::get_DBUS_connection");

	return dbus_connection;
}
