/***************************************************************************
 *
 * Copyright (C) 2007 Elad Lahav (elad_lahav@users.sourceforge.net)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ***************************************************************************/

#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/mman.h>
#include <qfileinfo.h>
#include "filesemaphore.h"

/**
 * Class constructor.
 * @param	szRefFile	Path to the file holding the reference counter
 * @param	szMutexName	The name of the global semaphore
 */
FileSemaphore::FileSemaphore(const char* szRefFile, const char* szMutexName) :
	m_sRefFile(szRefFile),
	m_nRefFd(-1),
	m_pRefCount(NULL),
	m_bAcquired(false)
{	
	// Open/create the global mutex
	m_pMutex = sem_open(szMutexName, O_CREAT, S_IRUSR | S_IWUSR, 1);
}

/**
 * Class destructor.
 */
FileSemaphore::~FileSemaphore()
{
	if (m_pRefCount != NULL)
		munmap(m_pRefCount, sizeof(uint));
		
	if (m_nRefFd >= 0)
		close(m_nRefFd);
		
	if (m_pMutex != (sem_t*)SEM_FAILED)
		sem_close(m_pMutex);
}

/**
 * Ensures that the reference counter can be accessed atomically.
 * @return	true if successful, false otherwise
 */
bool FileSemaphore::acquire()
{
	// Must lock the mutex first
	if (!lockMutex())
		return false;
		
	// Make sure the reference counter is memory mapped
	if (!openRefFile() || !mapRefFile()) {
		unlockMutex();
		return false;
	}

	m_bAcquired = true;
	return true;
}

/**
 * Releases the semaphore, so it can be accessed by other clients.
 */
void FileSemaphore::release()
{
	m_bAcquired = false;
	unlockMutex();
}

/**
 * Returns the current value of the reference counter.
 * To ensure atomic access, acquire() must be called first.
 * @return	The current value of the semaphore
 */
uint FileSemaphore::getValue() const
{
	if (m_pRefCount == NULL)
		return 0;
		
	return *m_pRefCount;
}

/**
 * Increments the current value of the reference counter.
 * To ensure atomic access, acquire() must be called first.
 */
void FileSemaphore::inc()
{
	if (m_bAcquired && (m_pRefCount != NULL))
		(*m_pRefCount)++;
}

/**
 * Decrements the current value of the reference counter.
 * To ensure atomic access, acquire() must be called first.
 */
void FileSemaphore::dec()
{
	if (m_bAcquired && (m_pRefCount != NULL) && ((*m_pRefCount) > 0))
		(*m_pRefCount)--;
}

/**
 * Opens the file holding the reference counter.
 * If the file does not exist, it is created and initialised.
 * Must be called with the mutex locked.
 * @return	true if successful, false otherwise
 */
bool FileSemaphore::openRefFile()
{
	// Do nothing if the file is already open
	if (m_nRefFd >= 0)
		return true;
		
	// Open the file
	m_nRefFd = open(m_sRefFile.latin1(), O_RDWR | O_CREAT, 0644);
	if (m_nRefFd < 0)
		return false;
			
	QFileInfo fi(m_sRefFile);
	
	// Make sure the file contains a reference counter
	if (fi.size() < sizeof(uint)) {
		uint nRef = 0;
		
		// Write an initial reference counter
		if (write(m_nRefFd, &nRef, sizeof(uint)) < (int)sizeof(uint)) {
			close(m_nRefFd);
			m_nRefFd = -1;
			return false;
		}
	}
	
	return true;
}

/**
 * Memory-maps the section of the file holding the reference counter.
 * @return	true if successful, false otherwise
 */
bool FileSemaphore::mapRefFile()
{
	void* pMap;
	
	// Do nothing if the reference counter is already mapped
	if (m_pRefCount != NULL)
		return true;
		
	// Map the reference counter
	pMap = mmap(0, sizeof(uint), PROT_READ | PROT_WRITE, MAP_SHARED, m_nRefFd,
		0);
	if (pMap == (uint*)MAP_FAILED)
		return false;
	
	m_pRefCount = (uint*)pMap;
	return true;
}

/**
 * Acquires the mutex.
 * This call may block until the semaphore can be decremented.
 * @return	true if successful, false otherwise
 */
inline bool FileSemaphore::lockMutex()
{
	if (m_pMutex == (sem_t*)SEM_FAILED)
		return false;
		
	return sem_wait(m_pMutex) == 0;
}

/**
 * Releases the mutex.
 * @return	true if successful, false otherwise
 */
inline bool FileSemaphore::unlockMutex()
{
	if (m_pMutex == (sem_t*)SEM_FAILED)
		return false;
		
	return sem_post(m_pMutex) == 0;
}
