/***************************************************************************
 *
 * Copyright (C) 2005 Elad Lahav (elad_lahav@users.sourceforge.net)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ***************************************************************************/

#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <kmessagebox.h>
#include <klocale.h>
#include "projectmanager.h"
#include "kscopeconfig.h"
#include "cscopefrontend.h"
#include "filesemaphore.h"

#define CONFIG_VER 2

#define DEF_IS_KERNEL		false
#define DEF_INV_INDEX		true
#define DEF_NO_COMPRESS		false
#define DEF_SLOW_PATH		false
#define DEF_AC_MIN_CHARS	3
#define DEF_AC_DELAY		500
#define DEF_AC_MAX_ENTRIES	100
#define DEF_TAB_WIDTH		0 /* Use editor's default */

#define PROJECT_MUTEX		"/kscope_proj_mutex"

/**
 * Class constructor.
 */
ProjectManager::ProjectManager() :
	m_pFiles(NULL),
	m_pConf(NULL),
	m_bOpen(false),
	m_bTempProj(false),
	m_nArgs(0),
	m_nAutoRebuildTime(-1),
	m_nTabWidth(0),
	m_pSem(NULL)
{
}

/**
 * Class destructor.
 */
ProjectManager::~ProjectManager()
{
	close();
}

/**
 * Creates a project's directory, and associates this directory with the
 * current object. This directory is created under the given path, and using
 * the project's name (which, thus, has to be a legal file name). 
 * Note: this function attempts to create a new directory, so the given path
 * and name must not lead to an existing one.
 * @param	sName		The project's name
 * @param	sPath		The parent directory under which to create the 
 *						project's directory
 * @param	opt			A structure containing project options
 * @return	true if successful, false otherwise
 */
bool ProjectManager::create(const QString& sName, const QString& sPath, 
	const Options& opt)
{
	QFile* pFilesFile;
	KConfig* pConf;
	
	// If the project is open, close it first
	if (m_bOpen)
		close();

	// Make sure the directory doesn't exist
	QDir dir(sPath);
	if (dir.exists(sName)) {
		KMessageBox::error(0, i18n("Cannot create a project inside an "
			"existing directory"));
		return false;
	}

	// Try to create the projcet's directory
	if (!dir.mkdir(sName, false) || !dir.cd(sName, false)) {
		KMessageBox::error(0, i18n("Failed to create the project's "
			"directory"));
		return false;
	}
	
	// Prepare the project's files
	initConfig(dir.absPath(), &pFilesFile, &pConf);

	// Write the configuration file version
	pConf->setGroup("");
	pConf->writeEntry("Version", CONFIG_VER);
	
	// Write project properties in the configuration file
	pConf->setGroup("Project");
	pConf->writeEntry("Name", sName);
	writeOptions(pConf, opt);
	
	// Flush the config file data, so the project is created even if KScope
	// crashes...
	pConf->sync();
	delete pConf;

	// Create the "files" file
	if (!pFilesFile->open(IO_WriteOnly)) {
		delete pFilesFile;
		KMessageBox::error(0, i18n("Failed to open the \"files\" file"));
		return false;
	}

	pFilesFile->close();
	delete pFilesFile;
	
	return true;
}

/**
 * Opens a project and makes it the current one.
 * @param	sPath	The directory containing the project's files
 * @return	true if successful, false otherwise
 */
bool ProjectManager::open(const QString& sPath)
{
	QString sConfFile;
	Options opt;
	FileSemaphore* pSem;
	
	// Associate the object with the project directory
	m_dir.setPath(sPath);
	if (!m_dir.exists()) {
		KMessageBox::error(0, i18n("Project directory does not exist"));
		return false;
	}

	// Open the configuration files
	initConfig(sPath, &m_pFiles, &m_pConf);

	// Verify the configuration file's version is compatible
	m_pConf->setGroup("");
	if (m_pConf->readUnsignedNumEntry("Version", 0) != CONFIG_VER) {
		KMessageBox::error(0, i18n("Your project is not compatible with this "
		"version of KScope.\nPlease re-create the project."));
		return false;
	}
	
	// Get the project name
	m_pConf->setGroup("Project");
	m_sName = m_pConf->readEntry("Name");
	if (m_sName == QString::null) {
		KMessageBox::error(0, i18n("Cannot read project name"));
		return false;
	}

	// Try to acquire a named semaphore for the project
	pSem = new FileSemaphore(sPath + "/cscope.sem", PROJECT_MUTEX);
	if (!pSem->acquire()) {
		QString sMsg;
		
		sMsg = QString(i18n("Cannot acquire the project's semaphore\n"
			"Error: %2\n").arg(errno));
		KMessageBox::error(0, sMsg);
		delete pSem;
		return false;
	}
	
	pSem->inc();
	pSem->release();
	
	// Indicate that a project has been opened
	m_bOpen = true;
	m_bTempProj = false;
	m_pSem = pSem;
	
	// Create the argument list for invoking Cscope
	getOptions(opt);
	if (opt.bKernel)
		m_nArgs |= CscopeFrontend::Kernel;
	if (opt.bInvIndex)
		m_nArgs |= CscopeFrontend::InvIndex;
	if (opt.bNoCompress)
		m_nArgs |= CscopeFrontend::NoCompression;
	if (opt.bSlowPathDef)
		m_nArgs |= CscopeFrontend::SlowPathDef;
	
	// Get the auto-rebuild	time value
	m_nAutoRebuildTime = opt.nAutoRebuildTime;
	
	m_nTabWidth = opt.nTabWidth;
	
	// Add to the list of recently opened projects
	Config().addRecentProject(sPath);
	
	return true;
}

/**
 * Opens a Cscope.out file as a temporary project.
 * @param	sFilePath	The full path of the Cscope.out file
 * @return	true if successful, false otherwise
 */
bool ProjectManager::openCscopeOut(const QString& sFilePath)
{
	QFileInfo fi(sFilePath);
	
	// Make sure the file exists, and that is is a cross-reference file
	if (!fi.exists() || !isCscopeOut(fi.absFilePath()))
		return false;
		
	// Set the project's directory
	m_dir = fi.dirPath(true);
	
	// Set the name of the project to be the full path of the file
	m_sName = fi.absFilePath();

	// Indicate that a project has been opened
	m_bOpen = true;
	m_bTempProj = true;
	
	return true;
}

/**
 * Performs clean-up on the project's variables, and detaches the associated
 * directory.
 */
void ProjectManager::close()
{
	if (!m_bTempProj) {
		m_dir = QDir();
		delete m_pFiles;
		m_pFiles = NULL;
		delete m_pConf;
		m_pConf = NULL;
	
		// Delete the project semaphore
		if (m_pSem) {
			// Decrement the reference counter
			if (m_pSem->acquire()) {
				m_pSem->dec();
				m_pSem->release();
			}
			
			delete m_pSem;
			m_pSem = NULL;
		}
	}
	
	m_nArgs = 0;
	m_slSymHistory.clear();
	m_nAutoRebuildTime = -1;
	m_nTabWidth = 0;
	m_bOpen = false;
}

/**
 * Adds a set of files to the current project.
 * @param	slFiles	A list of file names to add
 * @return	The total number of files added to the project
 */
int ProjectManager::addFiles(const QStringList& slFiles)
{
	QStringList::const_iterator itr;
	int nFiles = 0;

	// Cannot add files if no project is open, or the project is temporary
	if (!m_bOpen || m_bTempProj)
		return 0;

	// Open the source files file, to which the file paths will be added
	if (!m_pFiles->open(IO_WriteOnly | IO_Append))
		return 0;

	// Add each file to the project
	QTextStream str(m_pFiles);
	for (itr = slFiles.begin(); itr != slFiles.end(); ++itr) {
		str << *itr << "\n";
		nFiles++;
	}

	m_pFiles->close();
	emit filesAdded(slFiles);
	
	return nFiles;
}

/**
 * Fills a list object with all files in the project.
 * List items are created by reading and parsing all file name entries from
 * the project's 'cscope.files' file.
 * Note that the file may contain option lines, beginning with a dash. These
 * should be ignored.
 * @param	pList	Pointer to the object to fill
 */
void ProjectManager::fillList(FileListTarget* pList)
{
	QString sFilePath;
	
	// Must have an open persistent project
	if (!m_bOpen || m_bTempProj)
		return;
	
	// Open the 'cscope.files' file
	if (!m_pFiles->open(IO_ReadOnly))
		return;

	// Read all file names from the file
	QTextStream str(m_pFiles);
	while ((sFilePath = str.readLine()) != QString::null) {
		// Skip option lines
		if (sFilePath.at(0) == '-')
			continue;

		// Set the new list item
		pList->addItem(sFilePath);
	}

	m_pFiles->close();
}

/**
 * Writes all file entries in a list view widget to the project's
 * 'cscope.files' file (replacing current file contents.)
 * @param	pList	Pointer to the object from which to take the new entries
 */
void ProjectManager::writeList(FileListSource* pList)
{
	QString sFilePath;
	
	// Must have an open persistent project
	if (!m_bOpen || m_bTempProj)
		return;
	
	// Open the 'cscope.files' file
	if (!m_pFiles->open(IO_WriteOnly))
		return;

	QTextStream str(m_pFiles);

	// Write all file names
	if (pList->firstItem(sFilePath)) {
		do {
			str << sFilePath << "\n";
		} while (pList->nextItem(sFilePath));
	}

	m_pFiles->close();
	emit fileListChanged();
}

/**
 * Determines whether the project includes any files.
 * Reads the 'cscope.files' file and looks for any file names in it. If none
 * is present, then the project is considered empty.
 * Note that the file may contain option lines, beginning with a dash. These
 * should be ignored.
 * @return	true if no files are included in the project, false otherwise
 */
bool ProjectManager::isEmpty()
{
	QString sPath, sFileName;
	bool bResult = true;
	
	// Must have an active project
	if (!m_bOpen)
		return true;

	// Assume there are some files if using only a cscope.out file
	if (m_bTempProj)
		return false;
	
	// Open the 'cscope.files' file
	if (!m_pFiles->open(IO_ReadOnly))
		return true;

	// Find at least one file name entry in the file
	QTextStream str(m_pFiles);
	while ((sPath = str.readLine()) != QString::null) {
		if (sPath.at(0) != '-') {
			bResult = false;
			break;
		}
	}

	m_pFiles->close();
	return bResult;
}

/**
 * Returns a semi-colon separated list of the file types included in the
 * current project.
 */
QString ProjectManager::getFileTypes() const
{
	QString sTypes;
	
	// Must have an open persistent project
	if (!m_bOpen || m_bTempProj)
		return QString::null;

	m_pConf->setGroup("Project");
	return m_pConf->readEntry("FileTypes");
}

/**
 * Reads the project's options from the configuration file.
 * @param	opt	A structure to fill with the read options
 */
void ProjectManager::getOptions(Options& opt) const
{
	// Must have an open persistent project
	if (!m_bOpen || m_bTempProj)
		return;

	// Get project properties
	m_pConf->setGroup("Project");
	opt.slFileTypes = m_pConf->readListEntry("FileTypes", ' ');
	opt.bKernel = m_pConf->readBoolEntry("Kernel", DEF_IS_KERNEL);
	opt.bInvIndex = m_pConf->readBoolEntry("InvIndex", DEF_INV_INDEX);
	opt.bNoCompress = m_pConf->readBoolEntry("NoCompress", DEF_NO_COMPRESS);
	opt.bSlowPathDef = m_pConf->readBoolEntry("SlowPathDef", DEF_SLOW_PATH);
	opt.nAutoRebuildTime = m_pConf->readNumEntry("AutoRebuildTime");
	opt.nTabWidth = m_pConf->readUnsignedNumEntry("TabWidth");
		
	// Get auto-completion options
	m_pConf->setGroup("AutoCompletion");
	opt.bACEnabled = m_pConf->readBoolEntry("Enabled");
	opt.nACMinChars = m_pConf->readUnsignedNumEntry("MinChars",
		DEF_AC_MIN_CHARS);
	opt.nACDelay = m_pConf->readUnsignedNumEntry("Delay", DEF_AC_DELAY);
	opt.nACMaxEntries = m_pConf->readUnsignedNumEntry("MaxEntries",
		DEF_AC_MAX_ENTRIES);
}

/**
 * Sets project options.
 * @param	opt	A structure containing the new parameters to set
 */
void ProjectManager::setOptions(const Options& opt)
{
	// Must have an open persistent project
	if (!m_bOpen || m_bTempProj)
		return;

	// Write the options to the configuratio nfile
	writeOptions(m_pConf, opt);	
			
	// Update project parameters
	m_nAutoRebuildTime = opt.nAutoRebuildTime;
	m_nTabWidth = opt.nTabWidth;
	
	// Create the argument list for invoking Cscope
	m_nArgs = 0;
	if (opt.bKernel)
		m_nArgs |= CscopeFrontend::Kernel;
	if (opt.bInvIndex)
		m_nArgs |= CscopeFrontend::InvIndex;
	if (opt.bNoCompress)
		m_nArgs |= CscopeFrontend::NoCompression;
	if (opt.bSlowPathDef)
		m_nArgs |= CscopeFrontend::SlowPathDef;
}

/**
 * Reads the paths of the files that were open when the project was closed,
 * and fills a list used to restore the previous session.
 * The list is composed of the file paths and the current position of the
 * cursor inside each file (@see FileLocation).
 * @param	list	The list to fill
 */
void ProjectManager::getOpenFiles(FileLocationList& list) const
{
	QStringList slEntry;
	QStringList::Iterator itr;
	QString sPath;
	uint nLine, nCol;
	
	// Must have an open persistent project
	if (!m_bOpen || m_bTempProj)
		return;

	// Read the file paths
	m_pConf->setGroup("Session");
	slEntry = m_pConf->readListEntry("OpenFiles");
	
	// Transform the entry into a list of file locations
	for (itr = slEntry.begin(); itr != slEntry.end(); ++itr) {
		sPath = (*itr).section(':', 0, 0);
		nLine = (*itr).section(':', 1, 1).toUInt();
		nCol = (*itr).section(':', 2, 2).toUInt();
		list.append(new FileLocation(sPath, nLine, nCol));
	}
}

/**
 * Writes a list of open files to the project's configuration file.
 * The list is composed of the file paths and the current position of the
 * cursor inside each file.
 * The method is called before a project is closed, so these file paths can
 * be read when the project is restored.
 * @param	list	The list of file paths to write
 */
void ProjectManager::setOpenFiles(FileLocationList& list)
{
	FileLocation* pLoc;
	QString sLoc;
	QStringList slEntry;
	
	// Must have an open persistent project
	if (!m_bOpen || m_bTempProj)
		return;

	// Turn the object list into a string list, so that it can be written in
	// the configuration file
	for (pLoc = list.first(); pLoc != NULL; pLoc = list.next()) {
		sLoc = "";
		QTextOStream(&sLoc) << pLoc->m_sPath << ":" << pLoc->m_nLine << ":" 
			<< pLoc->m_nCol;
		slEntry.append(sLoc);
	}

	// Write the file paths
	m_pConf->setGroup("Session");
	m_pConf->writeEntry("OpenFiles", slEntry);
}

/**
 * Reads the stored session's bookmarks.
 * The list is composed of the file paths and the current position of the
 * cursor inside each file (@see FileLocation).
 * @param	list	The list to fill
 */
void ProjectManager::getBookmarks(FileLocationList& list) const
{
	QStringList slEntry;
	QStringList::Iterator itr;
	QString sPath;
	uint nLine, nCol;
	
	// Must have an open persistent project
	if (!m_bOpen || m_bTempProj)
		return;

	// Read the file paths
	m_pConf->setGroup("Session");
	slEntry = m_pConf->readListEntry("Bookmarks");
	
	// Transform the entry into a list of file locations
	for (itr = slEntry.begin(); itr != slEntry.end(); ++itr) {
		sPath = (*itr).section(':', 0, 0);
		nLine = (*itr).section(':', 1, 1).toUInt();
		nCol = (*itr).section(':', 2, 2).toUInt();
		list.append(new FileLocation(sPath, nLine, nCol));
	}
}

/**
 * Writes a list of bookmarks.
 * @param	list	A list of file locations, each defining a bookmark
 */
void ProjectManager::setBookmarks(FileLocationList& list)
{
	FileLocation* pLoc;
	QString sLoc;
	QStringList slEntry;
	
	// Must have an open persistent project
	if (!m_bOpen || m_bTempProj)
		return;

	// Turn the object list into a string list, so that it can be written in
	// the configuration file
	for (pLoc = list.first(); pLoc != NULL; pLoc = list.next()) {
		sLoc = "";
		QTextOStream(&sLoc) << pLoc->m_sPath << ":" << pLoc->m_nLine << ":" 
			<< pLoc->m_nCol;
		slEntry.append(sLoc);
	}

	// Write the file paths
	m_pConf->setGroup("Session");
	m_pConf->writeEntry("Bookmarks", slEntry);
}

/**
 * @return	The full path of the file that was active when the session was
 *			closed
 */
QString ProjectManager::getLastOpenFile() const
{
	// Must have an open persistent project
	if (!m_bOpen || m_bTempProj)
		return "";

	// Read the file paths
	m_pConf->setGroup("Session");
	return m_pConf->readEntry("LastOpenFile");
}

/**
 * @param	sPath	The full path of the currently active file
 */
void ProjectManager::setLastOpenFile(const QString& sPath)
{
	// Must have an open persistent project
	if (!m_bOpen || m_bTempProj)
		return;

	// Read the file paths
	m_pConf->setGroup("Session");
	m_pConf->writeEntry("LastOpenFile", sPath);
}

/**
 * Reads the names of the files corresponding to queries that were locked
 * when the project was closed, and fills a list used to restore the
 * previous session.
 * @param	slQueryFiles	The list to fill
 */
void ProjectManager::getQueryFiles(QStringList& slQueryFiles) const
{
	// Must have an open persistent project
	if (!m_bOpen || m_bTempProj)
		return;

	// Read the file paths
	m_pConf->setGroup("Session");
	slQueryFiles = m_pConf->readListEntry("QueryFiles");
}

/**
 * Writes a list of query file names to the project's configuration file.
 * The method is called before a project is closed, so these file names can
 * be read when the project is restored.
 * @param	slQueryFiles	The list of file names to write
 */
void ProjectManager::setQueryFiles(QStringList& slQueryFiles)
{
	// Must have an open persistent project
	if (!m_bOpen || m_bTempProj)
		return;

	// Write the file names
	m_pConf->setGroup("Session");
	m_pConf->writeEntry("QueryFiles", slQueryFiles);
}

/**
 * Reads the names of the files corresponding to call trees that were opened
 * when the project was closed, and fills a list used to restore the
 * previous session.
 * @param	slQueryFiles	The list to fill
 */
void ProjectManager::getCallTreeFiles(QStringList& slQueryFiles) const
{
	// Must have an open persistent project
	if (!m_bOpen || m_bTempProj)
		return;

	// Read the file paths
	m_pConf->setGroup("Session");
	slQueryFiles = m_pConf->readListEntry("CallTreeFiles");
}

/**
 * Writes a list of call tree file names to the project's configuration file.
 * The method is called before a project is closed, so these file names can
 * be read when the project is restored.
 * @param	slQueryFiles	The list of file names to write
 */
void ProjectManager::setCallTreeFiles(QStringList& slQueryFiles)
{
	// Must have an open persistent project
	if (!m_bOpen || m_bTempProj)
		return;

	// Write the file names
	m_pConf->setGroup("Session");
	m_pConf->writeEntry("CallTreeFiles", slQueryFiles);
}

/**
 * Reads the value of the project's root directory from the project's
 * configuration file.
 * @return	The root directory of the active project
 */
QString ProjectManager::getRoot() const
{
	// Empty root if no project is currently open
	if (!m_bOpen)
		return "";
	
	// Return the file-system root if using a temporary project
	if (m_bTempProj)
		return "/";

	m_pConf->setGroup("Project");
	return m_pConf->readEntry("RootPath", "/");
}

/**
 * Sets a new root directory for a project.
 * @param	sRoot	The full path of the new root directory
 */
void ProjectManager::setRoot(const QString& sRoot)
{
	// Do nothing if there is no open project, or using a temporary project
	if (!m_bOpen || m_bTempProj)
		return;
	
	m_pConf->setGroup("Project");
	m_pConf->writeEntry("RootPath", sRoot);
}
	
/**
 * Copies the list of previously queried symbols to the target object.
 * @param	slSymHistory	The list object to copy into
 */
void ProjectManager::getSymHistory(QStringList& slSymHistory) const
{
	slSymHistory = m_slSymHistory;
}

/**
 * Copies the list of previously queried symbols from the target object.
 * @param	slSymHistory	The list object to copy from
 */
void ProjectManager::setSymHistory(QStringList& slSymHistory)
{
	m_slSymHistory = slSymHistory;
}

/**
 * Retrieves parameters used to run a make-like command on this project.
 * @param	sCmd	Holds the make command
 * @param	sDir	Holds the directory in which to run the command
 */
void ProjectManager::getMakeParams(QString& sCmd, QString& sDir) const
{
	// Do nothing if there is no open project, or using a temporary project
	if (!m_bOpen || m_bTempProj)
		return;
	
	m_pConf->setGroup("Project");
	sCmd = m_pConf->readEntry("MakeCommand", "make");
	sDir = m_pConf->readEntry("MakeRoot", getRoot());
}

/**
 * Sets parameters used to run a make-like command on this project.
 * @param	sCmd	The make command
 * @param	sDir	The directory in which to run the command
 */
void ProjectManager::setMakeParams(const QString& sCmd, const QString& sDir)
{
	// Do nothing if there is no open project, or using a temporary project
	if (!m_bOpen || m_bTempProj)
		return;
	
	m_pConf->setGroup("Project");
	m_pConf->writeEntry("MakeCommand", sCmd);
	m_pConf->writeEntry("MakeRoot", sDir);
}

/**
 * Determines if the cscope.out file for this project exists.
 * @return	true if the database exists, false otherwise
 */
bool ProjectManager::dbExists()
{
	if (!m_bOpen)
		return false;
		
	return m_dir.exists("cscope.out");
}

/**
 * Fills a structure with default properties for a new project.
 * Default properties are partly based on the system profile.
 * @param	opt	The structure to fill
 */
void ProjectManager::getDefOptions(Options& opt)
{
	// Initialise MIME-type list
	opt.slFileTypes.append("*.c");
	opt.slFileTypes.append("*.h");

	// Set other options
	opt.bKernel = DEF_IS_KERNEL;
	opt.bInvIndex = DEF_INV_INDEX;
	opt.bNoCompress = DEF_NO_COMPRESS;
	opt.bSlowPathDef = DEF_SLOW_PATH;
	opt.nACMinChars = DEF_AC_MIN_CHARS;
	opt.nACDelay = DEF_AC_DELAY;
	opt.nACMaxEntries = DEF_AC_MAX_ENTRIES;
	opt.nTabWidth = DEF_TAB_WIDTH;
	
	// Set profile-dependant options
	if (Config().getSysProfile() == KScopeConfig::Fast) {
		opt.nAutoRebuildTime = 10;
		opt.bACEnabled = true;
	}
	else {
		opt.nAutoRebuildTime = -1;
		opt.bACEnabled = false;
	}
}

/**
 * Determines whether a project is currently open by any running instance of 
 * KScope.
 * @param	sPath	The path of the project to check
 * @return	true if the project is currently open, false otherwise
 */
bool ProjectManager::isGloballyOpen(const QString& sPath)
{
	FileSemaphore sem(sPath + "/cscope.sem", PROJECT_MUTEX);
	bool bResult;
	
	if (!sem.acquire())
		return false;
		
	bResult = (sem.getValue() > 0);
	sem.release();
	
	return bResult;
}

/**
 * Creates the "cscope.files" and "cscope.proj" file objects for the given
 * project.
 * @param	sPath		The project directory, in which the files reside
 * @param	ppFilesFile	Holds the new "cscope.files" object, upon return
 * @param	ppConf		Holds the new "cscope.proj" object, upon return
 */
void ProjectManager::initConfig(const QString& sPath, QFile** ppFilesFile,
	KConfig** ppConf)
{
	QString sFilesFile, sConfFile;

	// Attach to the source files file
	sFilesFile = sPath + "/cscope.files";
	*ppFilesFile = new QFile(sFilesFile);
	
	// Open the configuraton file
	sConfFile = sPath + "/cscope.proj";	
	*ppConf = new KConfig(sConfFile);
}

/**
 * Determines if the given file is a Cscope cross-reference database.
 * @param	sPath	The full path of the file to check
 * @return	true if the given file is a cscope.out file, false otherwise
 */
bool ProjectManager::isCscopeOut(const QString& sPath)
{
	QFile file(sPath);
	QString sLine;
	int nVer;
	char szDir[PATH_MAX];

	// Try to open the file
	if (!file.open(IO_ReadOnly))
		return false;
		
	// Check if the first line matches the expected format
	sLine = QTextStream(&file).readLine();
	return sscanf(sLine.latin1(), "cscope %d %s", &nVer, szDir) == 2;
}

/**
 * Writes a set of project options to the given configuration file.
 * @param	pConf	An initialised configuration file
 * @param	opt		The options to write
 */
void ProjectManager::writeOptions(KConfig* pConf, const Options& opt)
{
	// Set new project options
	pConf->setGroup("Project");
	pConf->writeEntry("FileTypes", opt.slFileTypes.join(" "));
	pConf->writeEntry("Kernel", opt.bKernel);
	pConf->writeEntry("InvIndex", opt.bInvIndex);		
	pConf->writeEntry("NoCompress", opt.bNoCompress);		
	pConf->writeEntry("SlowPathDef", opt.bSlowPathDef);		
	pConf->writeEntry("AutoRebuildTime", opt.nAutoRebuildTime);
	pConf->writeEntry("TabWidth", opt.nTabWidth);
	
	// Set auto-completion options
	pConf->setGroup("AutoCompletion");
	pConf->writeEntry("Enabled", opt.bACEnabled);
	pConf->writeEntry("MinChars", opt.nACMinChars);
	pConf->writeEntry("Delay", opt.nACDelay);
	pConf->writeEntry("MaxEntries", opt.nACMaxEntries);
}

#include "projectmanager.moc"
