/*
	progressbar.cpp - Progress Bar
	Copyright (C) 2006  Konrad Twardowski <kdtonline@poczta.onet.pl>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <qapplication.h>
#include <qdesktopwidget.h>
#include <qpainter.h>

#include <kapplication.h>
#include <kdebug.h>
#include <klocale.h>
#include <kpopupmenu.h>

#include "configuration.h"
#include "progressbar.h"

// static

ProgressBar *ProgressBar::_instance = 0;

// public

ProgressBar::~ProgressBar() { }

void ProgressBar::setPosition(const Position value) {
	// kdDebug() << "ProgressBar::setPosition: " << value << endl;
	kshutdownrc->progressBarPosition = value;

	_position = value;
	QDesktopWidget *desktop = QApplication::desktop();
	resize(desktop->width(), height());
	switch (_position) {
		case TOP:
			move(0, 0);
			break;
		case BOTTOM:
			move(0, desktop->height() - height());
			break;
		default:
			move(0, 0);
			break;
	}
}

void ProgressBar::setHeight(const int value) {
	int newHeight = (value < 2) ? 2 : value;
	resize(width(), newHeight);
}

void ProgressBar::setProgress(const int value) {
	// kdDebug() << "ProgressBar::setProgress: " << value << endl;

	int complete = _total - value;

	if (_complete == complete)
		return;

	_complete = complete;

	repaint();
}

void ProgressBar::setValues(const int complete, const int total) {
	// kdDebug() << "ProgressBar::setValues: " << complete << ", " << total << endl;

	if ((_complete == complete) && (_total == total))
		return;

	_complete = complete;
	_total = total;
	repaint();
}

// protected

void ProgressBar::mousePressEvent(QMouseEvent *e) {
	// kdDebug() << "ProgressBar::mousePressEvent" << endl;
	if (e->button() == RightButton) {
// TODO: 2.0: color configuration
// TODO: 2.0: size configuration
		// show popup menu
		KPopupMenu *popup = new KPopupMenu(this);
		popup->insertTitle(kapp->miniIcon(), "KShutDown");
		popup->insertItem(i18n("Hide"), this, SLOT(hide()));
		popup->insertTitle("Position");
		int topPositionItem = popup->insertItem(i18n("Top"), this, SLOT(slotSetTopPosition()));
		popup->setItemChecked(topPositionItem, _position == TOP);
		int bottomPositionItem = popup->insertItem(i18n("Bottom"), this, SLOT(slotSetBottomPosition()));
		popup->setItemChecked(bottomPositionItem, _position == BOTTOM);
		popup->popup(e->globalPos());
		e->accept();
	}
	QWidget::mousePressEvent(e);
}

void ProgressBar::paintEvent(QPaintEvent *) {
	// kdDebug() << "ProgressBar::paintEvent" << endl;

	QPainter g(this);
	int w = width();
	int h = height();
	g.fillRect(0, 0, w, h, QBrush(backgroundColor()));

	if ((_complete <= 0) || (_total <= 0))
		return;

	w = w * (_complete * 100 / _total) / 100;
	g.fillRect(0, 0, w, h, QBrush(foregroundColor()));
}

// private

ProgressBar::ProgressBar()
	: QWidget(
		0,
		"ProgressBar",
		WStyle_NoBorder | WStyle_StaysOnTop | WType_TopLevel | WX11BypassWM
	),
	_complete(0),
	_total(0) {

	// kdDebug() << "ProgressBar::ProgressBar()" << endl;

	setBackgroundColor(black);
	setPaletteForegroundColor(yellow);
	setHeight(3);
	setPosition((Position)kshutdownrc->progressBarPosition);
}

// private slots

void ProgressBar::slotSetBottomPosition() {
	setPosition(BOTTOM);
}

void ProgressBar::slotSetTopPosition() {
	setPosition(TOP);
}
