/***************************************************************************
 *   Copyright (C) 2004-2007 by Giovanni Venturi                           *
 *   giovanni@ksniffer.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02110-1301, USA.          *
 ***************************************************************************/

#include <unistd.h>

#include <stdio.h>
#include <signal.h>

#include <qstring.h>
#include <qpopupmenu.h>
#include <qdragobject.h>
#include <qpainter.h>
#include <qpaintdevicemetrics.h>
#include <qtooltip.h>
#include <qfile.h>
#include <qtextstream.h>
#include <qtimer.h>
#include <qdom.h>

#include <kuniqueapplication.h>
#include <kprinter.h>
#include <ksystemtray.h>
#include <kpopupmenu.h>
#include <kglobal.h>
#include <kstandarddirs.h>
#include <kmenubar.h>
#include <kstatusbar.h>
#include <kkeydialog.h>
#include <kaccel.h>
#include <kio/netaccess.h>
#include <kfiledialog.h>
#include <kconfig.h>
#include <kurl.h>
#include <kurldrag.h>
#include <kio/job.h>

#include <kedittoolbar.h>
#include <kmainwindow.h>
#include <kstdaccel.h>
#include <kstdaction.h>

#include <kmessagebox.h>
#include <kactionclasses.h>
#include <kaction.h>
#include <ktempfile.h>
#include <kprocess.h>

#include <klocale.h>
#include <kdebug.h>

#include "ksniffer.h"
#include "capturedialog.h"
#include "options/ksnifferoptiondialog.h"
#include "protocols/protocolname.h"
#include "sniffer.h"
#include "packetmanager.h"
#include "packet.h"
#include "ksniffermainwidget.h"
//#include "ksnifferiface.h"
#include "ksnifferpassivepopup.h"

#include "errorcode.h"

KSniffer::KSniffer()
  : /*KSnifferIface( "KSniffer-Interface" ),*/ KMainWindow( 0, "KSniffer", 0 )
{
  // accept dnd
  setAcceptDrops(true);

  m_mainWidgetUI = new KSnifferMainWidget( this );

  // tell the KMainWindow that this is indeed the main widget
  setCentralWidget( m_mainWidgetUI );

  // assign global configuration pointer
  m_config = kapp->config();

  // then, setup our actions
  setupActions();

  // reading application settings
  m_options = new CaptureOptions();

  // set into m_options the available options directly from the ksnifferrc configuration file
  readSettings();

  // set the view widget configuration
  m_mainWidgetUI->setOptions( m_options );

  // option just assigned
  m_optionChanged = false;

  // show a status bar
  statusBar()->show();

  setPlainCaption( i18n("KSniffer - The KDE Network Sniffer") );

  resize(780,520);  // default optimized for 800x600

  changeStatusbar(i18n( "Ready." ));

  // apply the saved mainwindow settings, if any, and ask the mainwindow
  // to automatically save settings if changed: window size, toolbar
  // position, icon size, etc.
  setAutoSaveSettings();

  // allow the view to change the statusbar and caption
  connect( m_mainWidgetUI, SIGNAL(signalChangeStatusbar(const QString&)),
          this,   SLOT(changeStatusbar(const QString&)) );
  connect( m_mainWidgetUI, SIGNAL(signalChangeCaption(const QString&)),
          this,   SLOT(changeCaption(const QString&)) );

  // we need a PacketManager to "store" packets and manage them for the View
  // and we had to specify the temporary directory in the constructor parameters
  m_packets = new PacketManager( this, "packetmanager" );
  m_packets->setFilePath( m_options->tmpFilePath() );

  m_sniffer = new Sniffer( m_options );

  // we define how to manage new captured packets:
    //  1 - store the packet in a PacketList (calling savePacket)
    connect( m_sniffer, SIGNAL(gotPacket(ptrPacketType, struct pcap_pkthdr, long, int, int)),
             m_packets, SLOT(savePacket(ptrPacketType, struct pcap_pkthdr, long, int, int)) );
    //  2 - display the packet into the View
    connect( m_packets, SIGNAL(savedPacket(long, PacketManager *)), m_mainWidgetUI, SLOT(displayPacket(long, PacketManager *)) );

  // Setting System tray icon if needed
  // prepare all the stuff for the show/hide tray bar...
    m_sysTray = new KSystemTray( this );
    m_sysTray->setTextFormat( Qt::RichText );
    m_sysTray->setPixmap( m_sysTray->loadIcon( "ksniffer" ) );
    QToolTip::add( m_sysTray, QString( "KSniffer " ) + APPVER );
    connect( m_sysTray, SIGNAL(quitSelected()), SLOT(slotQuit()) );

    // Sub menu: "Capture" item
    KPopupMenu *popupCapture = new KPopupMenu ( this );

    // the subitems...
    m_actNew->plug( popupCapture );
    m_actOpen->plug( popupCapture );
    m_actSave->plug( popupCapture );
    m_actSaveAs->plug( popupCapture );
    m_actPauseContinue->plug( popupCapture );
    m_actStop->plug( popupCapture );
    m_actPrint->plug( popupCapture );

    m_sysTray->contextMenu()->insertItem( i18n("&Capture"), popupCapture, 0, 1 );

    // Sub menu: "Help" item
    KPopupMenu *popupHelp = helpMenu();
    m_sysTray->contextMenu()->insertItem ( i18n("&Help"), popupHelp, 0, 2 );

    // KSniffer settings
    KAction *preferences = KStdAction::preferences(this, SLOT(optionsSettings()), actionCollection());
    m_sysTray->contextMenu()->insertSeparator( 3 );
    preferences->plug( m_sysTray->contextMenu(), 4 );

    // ...now show tray bar if needed
  if (m_showTrayBar)
    m_sysTray->show();

  m_action = ACT_DO_NOTHING;  // no action when saving job finished
  m_strFilename = QString::null;     // not defined file name to save, it has to be choosen by the user at first save
  m_strSavedFilename = QString::null;
  m_needToSave = false;   // no captured-packets available yet
  m_doingPause = false;   // when you start KSniffer you're not in pause status
  m_tmpFile = 0;

  m_loadFromFile = false;
  m_startedShowing = false;

  // need to remove temporary file just when I load a not local file and packets are available
  m_wroteTmpLocalFile = false;

  m_timer = new QTimer(this);
  connect( m_timer, SIGNAL(timeout()), this, SLOT(checkIfSniffing()));
}


KSniffer::~KSniffer()
{
  m_packets->clearList();

  delete m_sysTray;
  delete m_sniffer;
  delete m_options;
  delete m_packets;
  delete m_actStop;
  delete m_mainWidgetUI;
  delete m_actOpenRecent;
  delete m_timer;
}

void KSniffer::load(const KURL& url)
{
  QString target;
  // the below code is what you should normally do.  in this
  // example case, we want the url to our own.  you probably
  // want to use this code instead for your app

  #if 0
  // download the contents
  if (KIO::NetAccess::download(url, target))
  {
    // set our caption
    setCaption(url);

    // load in the file (target is always local)
    loadFile(target);

    // and remove the temp file
    KIO::NetAccess::removeTempFile(target);
  }
  #endif

  //m_mainWidgetUI->openURL( url );

  // setting the filename to load sniffed packets
  m_options->setSniffingFilename( url.path() );
  m_sniffer->setCaptureOptions( m_options );

  // maybe a pcap file want to be loaded
  if ( m_sniffer->checkFile( url.path() ) == Sniffer::RIGHT_FORMAT_FILE )
  {
    if (m_packets->packetAvailable())
    {
      // ask if continue loading if you have packets available
      if ( KMessageBox::warningContinueCancel( this,
             i18n("<p>Packet-capture session data available. If you want to drop the" \
                  " dragged file you are going to lose all captured data.</p>" \
                  "<p>Do you really want to continue?</p>" ) ) == KMessageBox::Cancel )
        return;
      else
      {
        // it appends the file captured data to the displayed data
        // without clean view and remove packets from the Packet Manager
        setCaption( url.prettyURL() );

        // before to insert packets into the view reset stored packets and clean the view...

        m_mainWidgetUI->clearView();

        // no need to save data: they're from file
        m_needToSave = false;

        // we have to remove the captured data
        m_packets->clearList();

        // we need to store packets again
        m_packets->acceptPacket( true );

        // adding URL to the recent opened files
        m_actOpenRecent->addURL( url );

        // to avoid crash disable sorting
        m_mainWidgetUI->setSortList( false );

        // getting packets
        startSniffingFromFile();
      }
    }
    else
    {
      // ok, no packets available
      setCaption( url.prettyURL() );

      // no need to save data: they're from file
      m_needToSave = false;

      // we need to store packets
      m_packets->acceptPacket( true );

      // adding URL to the recent opened files
      m_actOpenRecent->addURL( url );

      // to avoid crash disable sorting
      m_mainWidgetUI->setSortList( false );

      // getting packets
      startSniffingFromFile();
      changeStatusbar( i18n("Getting packets from file.") );
    }
  }
  else
    KMessageBox::information(
        this,
        i18n( "libpcap format is a standard format used by a lot of sniffer programs",
              "The file you choose to drag & drop is not in libpcap format!" ),
        i18n( "Format Error" ) );
}


void KSniffer::setupActions()
{
  // the action New Capture
  m_actNew = KStdAction::openNew( this, SLOT( slotNewCapture()), actionCollection(), "capt_new" );
  m_actNew->setWhatsThis( i18n( "<b>New Capture</b><br>Start a new packet-capture session." ) );
  m_actNew->setText( i18n( "New Capture..." ) );

  // the action Open Captured-Packet File
  m_actOpen = KStdAction::open( this, SLOT( slotOpen()), actionCollection(), "capt_open" );
  m_actOpen->setWhatsThis( i18n("<b>Open</b><br>Open a previous packet-capture session file.") );

  // the action Open Recent Captured-Packet File
  m_actOpenRecent = KStdAction::openRecent(this, SLOT( slotOpen( const KURL& ) ), actionCollection(), "capt_open_recent" );

  // the action Save Captured-Packet File
  m_actSave = KStdAction::save( this, SLOT( slotSave()), actionCollection(), "capt_save" );
  m_actSave->setWhatsThis( i18n( "<b>Save</b><br>Save the current packet-capture session." ) );
  m_actSave->setEnabled( false );

  // the action Save Captured-Packet File As
  m_actSaveAs = KStdAction::saveAs( this, SLOT( slotSaveAs()), actionCollection(), "capt_save_as" );
  m_actSaveAs->setWhatsThis( i18n( "<b>Save As...</b><br>Save into a file the current packet-capture session." ) );
  m_actSaveAs->setEnabled( false );

  // the action Pause/Continue Capture
  m_actPauseContinue =  new KAction( i18n("&Pause Capture"), "capt_pause_continue", 0, this,
    SLOT(slotPauseContinue()), actionCollection(), "capt_pause_continue" );
  m_actPauseContinue->setIcon( "player_pause" );
  m_whatsthisPause = i18n("<b>Pause Capture</b><br>Pause the packet-capture. With "\
      "this command you will not capture any packet till you don't decide to continue to capture them "\
      "again.");
  m_actPauseContinue->setWhatsThis( m_whatsthisPause );
  m_actPauseContinue->setEnabled( false );

  // the action Stop Capture
  m_actStop = new KAction(i18n("S&top"), "stop",
                    KStdAccel::shortcut(KStdAccel::End),
                    this, SLOT(slotStop()),
                    actionCollection(), "capt_stop");
  m_actStop->setWhatsThis( i18n("<b>Stop</b><br>Stop the packet-capture session.") );
  m_actStop->setEnabled( false );

  // the action Print Captured-Packet
  m_actPrint = KStdAction::print(this, SLOT(slotPrint()), actionCollection(), "capt_print");
  m_actPrint->setWhatsThis( i18n("<b>Print...</b><br>Print the packet-capture session.") );
  m_actPrint->setEnabled( false );

  // the action Quit
  KAction *m_actQuit = KStdAction::quit(this, SLOT(slotQuit()), actionCollection(), "capt_quit");
  m_actQuit->setWhatsThis( i18n("<b>Quit</b><br>Stop the packet-capture session and quit from KSniffer.") );
  m_actQuit->setText( i18n("Quit") );

  KStdAction::keyBindings(this, SLOT(optionsConfigureKeys()), actionCollection());
  KStdAction::configureToolbars(this, SLOT(optionsConfigureToolbars()), actionCollection());
  KStdAction::preferences(this, SLOT(optionsSettings()), actionCollection());

  // make sure the tooltips are used for the menu
  //actionCollection()->setHighlightingEnabled( true );

  // setup Settings menu
  m_statusbarAction = KStdAction::showStatusbar( this, SLOT(optionsShowStatusbar()), actionCollection() );

  createGUI("ksnifferui.rc");
}


void KSniffer::saveProperties( KConfig *config )
{
  // the 'config' object points to the session managed
  // config file. Anything you write here will be available
  // later when this app is restored

  //config->setGroup("General Options");

  m_actOpenRecent->saveEntries( config );
  config->sync();
}


void KSniffer::readProperties( KConfig *config )
{
  // the 'config' object points to the session managed
  // config file. This function is automatically called whenever
  // the app is being restored. Read in here whatever you wrote
  // in 'saveProperties'

  m_actOpenRecent->loadEntries( config );
  m_actOpenRecent->setEnabled( true ); // force enabling
  m_actOpenRecent->setToolTip( i18n("Click to open a file\nClick and hold to open a recent file") );

  m_snifferOption->getOptions( m_options );

  m_mustShowPacketsNow = !m_options->showAfter();
}


void KSniffer::readSettings()
{
  m_actOpenRecent->loadEntries( m_config );
  m_actOpenRecent->setEnabled( true ); // force enabling
  m_actOpenRecent->setToolTip( i18n("Click to open a file\nClick and hold to open a recent file") );

  // set into m_options the sniffer options (ShowAfter, TrayBar, TmpDirectory, Warn, PassivePopUp)
  m_snifferOption->getOptions( m_options );

  m_mustShowPacketsNow = !m_options->showAfter();
  m_showTrayBar = m_options->showTrayBar();
}


void KSniffer::saveSettings()
{
  // save recent opened files
  m_actOpenRecent->saveEntries( m_config );
  m_config->sync();
}


void KSniffer::dragEnterEvent( QDragEnterEvent *event )
{
  // accept uri drops only
  event->accept( KURLDrag::canDecode(event) );
}


void KSniffer::dropEvent( QDropEvent *event )
{
  // this is a very simplistic implementation of a drop event.  we
  // will only accept a dropped URL.  the Qt dnd code can do *much*
  // much more, so please read the docs there
  KURL::List urls;

  // see if we can decode a URI.. if not, just ignore it
  if (KURLDrag::decode( event, urls ) && !urls.isEmpty())
  {
    // okay, we have a URI... process it
    const KURL &url = urls.first();

    // load in the file
    load(url);
  }
}


// let close the application just from the system tray icon
bool KSniffer::queryClose()
{
  if (!m_showTrayBar)
  {
    slotQuit();
    if ((m_action & ACT_QUIT_APPL) == ACT_QUIT_APPL)
      return true;
    else
      return false;
  }

  if( kapp->sessionSaving())
    return true;
  else
  {
    if ( isShown() && ((m_action & ACT_QUIT_APPL) == 0) )
    {
      KMessageBox::information( this,
         i18n( "<qt>Closing the main window will keep KSniffer running in the system " \
               "tray. Use 'Quit' from the system tray menu to quit the application.</qt>" ),
         i18n( "Docking in System Tray" ), "hideOnCloseInfo");
      hide();
    }
    return false;
  }
}


// start a new Capture section if no available
void KSniffer::slotNewCapture()
{
  // this slot is called whenever the Capture->New menu is selected,
  // the New shortcut is pressed (usually CTRL+N) or the New toolbar
  // button is clicked

  // create a new window
  //(new KSniffer)->show();

  //FIXME: always to get the "show after" parameter because if not then the new capture forgot
  // don't show packet if required. First time ok, but later the m_options->showAfter() is not
  // correct than bug... Try to fix it better: m_optionChanged not useful anymore

  /*if (m_optionChanged)
  {
    // option changed during sniffing
    m_mainWidgetUI->setOptions( m_options );
    m_optionChanged = false;
  }*/
  m_action = ACT_DO_NOTHING;

  // set into m_options the available options directly from the ksnifferrc configuration file
  // just for the Show After option
  m_snifferOption->getShowAfterOptions( m_options );
  m_mainWidgetUI->setOptions( m_options );

  // remember if the packets has to be displayed or not when stop sniffing
  // so if you change sniffing options during capture they will not take effect
  m_mustShowPacketsNow = !m_options->showAfter();

  changeStatusbar( i18n("New Capture: select the network interface to start sniffing packets.") );

  CaptureDialog *cod = new CaptureDialog( this );

  if (cod->exec())
  {
    if (m_packets->packetAvailable() && m_needToSave)
      if ( KMessageBox::warningContinueCancel( this,
              i18n("<p>Packet-capture session data available. If you want to start a " \
                   "new capture session you are going to lose all captured data.</p>" \
                   "<p>Do you really want to continue?</p>" ) ) == KMessageBox::Cancel )
        return;
      else
      {
        // need to save data when available
        m_needToSave = true;

        // we have to remove the captured data
        m_packets->clearList();

        // we need to store packets again
        m_packets->acceptPacket( true );

        // reset main widget title
        setPlainCaption( i18n("KSniffer - The KDE Network Sniffer") );
      }
    else
    {
      // need to save data when available
      m_needToSave = true;

      // we need to store packets: no packets available and capture requested
      m_packets->acceptPacket( true );

      // reset main widget title
      setPlainCaption( i18n("KSniffer - The KDE Network Sniffer") );
    }
    changeStatusbar( i18n("Sniffing network by \"%1\" interface.").arg(cod->selectedInterface()) );

    m_wroteTmpLocalFile = true;
    startLiveSniffing( cod->selectedInterface() );
  }
  else
    changeStatusbar( i18n("Ready.") );

  delete cod;
}


/**
 * this slot is called whenever the File->Open menu is selected,
 * the Open shortcut is pressed (usually CTRL+O) or the Open toolbar
 * button is clicked
 */
void KSniffer::slotOpen( const KURL& urlParam )
{
  bool isLocal;              // true if the file to open is local
  KURL url0, url;            // URL to open: can be network resource
  KTempFile *tmpFile;
  int result = Sniffer::NO_RESULT_AVAILABLE;

  do
  {
    if (urlParam.isEmpty())
    {
      changeStatusbar( i18n("libpcap format is a standard format used by a lot of " \
                      "sniffer programs", "Select the libpcap format file to open." ) );
      url0 = KFileDialog::getOpenURL(QString::null, QString::null, this, i18n("Open File"));
    }
    else
      url0 = urlParam;
    url = url0;  // real URL on which operate: to have always a local file
    kdDebug() << "File to open: " << url.prettyURL() << endl;
    QFile file( url.path() );
    m_filesize = file.size();
    kdDebug() << "++++++++++ file: " << m_filesize << " bytes" << endl;
    m_packets->setFilename( url.path() );

    if ( !url.isEmpty() )
    {
      if ( !(isLocal = url.isLocalFile()) )
      {
        // create a new temporary file for network transparency
        tmpFile = new KTempFile;
        tmpFile->setAutoDelete( true );
        if ( tmpFile->status() == 0 )
        {
          // temporary file created succesfully
          QFile file( tmpFile->name() );
          if ( file.open( IO_ReadWrite ) )
          {
            file.close();
            QString name = tmpFile->name();
            if ( !KIO::NetAccess::download( url, name, 0 ) )
            {
              KMessageBox::sorry( this, i18n(
                  "Unable to access to the temporary file '%1': %2.").arg( tmpFile->name() ).arg( KIO::NetAccess::lastErrorString() ),
                  i18n("Load Failed") );
              delete tmpFile;
            }
          }
          else
          {
            KMessageBox::error( this,
                               i18n("Error during loading. Could not open '%1' temporary file for writing.").arg ( m_tmpFile->name() ),
                               i18n("Load Failed") );
            delete tmpFile;
          }
        }
        else
        {
          KMessageBox::sorry( this, i18n(
              "Loading error. Could not create temporary file: %1.").arg( strerror( m_tmpFile->status() ) ),
              i18n("Load Failed") );
          delete tmpFile;
        }
        url = tmpFile->name();
      }
      switch ( result = m_sniffer->checkFile( url.path() ) )
      {
        case Sniffer::BAD_FORMAT_FILE:
          KMessageBox::sorry(this,
                         i18n( "libpcap format is a standard format used by a "
                               "lot of sniffer programs",
                               "<b>%1</b> is not a libpcap format file." ).arg(url0.prettyURL()) );
          if (!isLocal)
            // temporary file has to be deleted
            delete tmpFile;
          break;

        case Sniffer::RIGHT_FORMAT_FILE:
          // standard filedialog
          m_actOpenRecent->addURL( url );
          if ( m_packets->packetAvailable() )
            if ( m_needToSave )
              // just if we get some packets we advice the user about
              // the possibilty to lost data is not yet saved
              if ( KMessageBox::warningContinueCancel( this,
                      i18n("<p>Packet-capture session data available. If you want to load a" \
                           "file captured session you are going to lose all captured data.</p>" \
                           "<p>Do you really want to continue?</p>" ) ) == KMessageBox::Cancel)
                return;
              else
              {
                // Packets available, need to save them, continue losting data

                // no need to save already saved file: data gets from a file
                m_needToSave = false;

                // we have to remove the captured data
                m_packets->clearList();

                // clearing the view before to start sniffing
                m_mainWidgetUI->clearView();

               // we need to store packets again
               m_packets->acceptPacket( true );
              }
            else
            {
              // Packets available, don't need to save them

              // we have to remove the captured data
              m_packets->clearList();

              // clearing the view before to start sniffing
              m_mainWidgetUI->clearView();

              // we need to store packets again
              m_packets->acceptPacket( true );
            }
          else
          {
            // No packets available

            // we need to store new packets: no previous packets available and capture requested
            m_packets->acceptPacket( true );
          }

          // set the title as the filename: the original choosed, for network trasparency to the user
          setCaption( url0.prettyURL() );
          break;
      }
    }
    else
      result = Sniffer::NO_FILE_SELECTED;
  } while ((result != Sniffer::RIGHT_FORMAT_FILE) && (result != Sniffer::NO_FILE_SELECTED));
  changeStatusbar( i18n("Ready.") );

  if ( (result != Sniffer::NO_FILE_SELECTED) && (result == Sniffer::RIGHT_FORMAT_FILE) )
  {
    // to avoid crash disable sorting
    m_mainWidgetUI->setSortList( false );

    // set capture options
    if (isLocal)
      m_options->setSniffingFilename( url.path() );
    else
    {
      m_options->setSniffingFilename( tmpFile->name() );
      if (m_tmpFile)
        delete m_tmpFile;
      m_tmpFile = tmpFile;

      // tell the view what file use to get packets
      m_packets->setFilename( m_tmpFile->name() );
    }
    m_sniffer->setCaptureOptions( m_options );

    if (m_wroteTmpLocalFile && !isLocal)
    {
      // need to remove old temporary file
      unlink( m_options->tmpFilePath() );
      kdDebug() << "++++++++++++++++++ removed old temporary file: " << m_options->tmpFilePath() << endl;
      m_wroteTmpLocalFile = false;
    }

    // start sniffing getting packets from a libpcap selected file
    startSniffingFromFile();

    KSnifferPassivePopup::message( i18n("Capturing"), i18n("<p>Now starting to get the packets from the file.</p>"), this );
    changeStatusbar( i18n("Getting packets from file.") );
  }
}


void KSniffer::slotSave()
{
  // this slot is called whenever the File->Save menu is selected,
  // the Save shortcut is pressed (usually CTRL+S) or the Save toolbar
  // button is clicked

  // save the current file
  if ( m_strFilename.isEmpty() )
    slotSaveAs();
  else
  {
    // copy file from temporary file directory
    QString tmp = m_options->tmpFilePath();
    kdDebug() << "saving file: " << tmp << endl;
    m_action |= ACT_SAVING_PACKETS;
    KURL src( tmp );
    KURL dest(m_strFilename);
    KIO::Job *job = KIO::copy( src, dest );
    connect( job, SIGNAL( result(KIO::Job *) ),
             this, SLOT( copyingJobFinished(KIO::Job *) ) );

    // set the title for the saved filename
    m_strSavedFilename = dest;
  }
}


// this slot is called whenever the File->Save As menu is selected,
void KSniffer::slotSaveAs()
{
  changeStatusbar( i18n("Save captured-packets as...") );

  KURL dest = KFileDialog::getSaveURL();
  kdDebug() << "saving file: " << m_options->tmpFilePath() << endl;
  KURL src( m_options->tmpFilePath() );
  if (!dest.isEmpty())
  {
    if (dest.isValid())
    {
      m_strFilename = dest.path();

      // copy file from temporary directory
      m_action |= ACT_SAVING_PACKETS;
      KIO::Job *job = KIO::copy( src, dest );
      connect( job, SIGNAL( result(KIO::Job *) ),
               this, SLOT( copyingJobFinished(KIO::Job *) ) );

      // set the title for the saved filename
      m_strSavedFilename = dest;
      m_actOpenRecent->addURL( dest );
    }
    else
      KMessageBox::information( this, i18n( "%1 not saved!" ).arg(m_strFilename), i18n( "Save Failed" ) );
  }
  else
    // user aborted saving... it was quitting, so remove temporary file
    if ((m_action & ACT_QUIT_APPL) == ACT_QUIT_APPL)
    {
      //kdDebug() << "+++++++++ I'm going to delete " << m_options->tmpFilePath() << endl;
      unlink( m_options->tmpFilePath() );
      //kdDebug() << "+++++++++ I'm going to delete " << m_options->tmpFileConfigPath() << endl;
      unlink( m_options->tmpFileConfigPath() );
    }
}


void KSniffer::slotPrint()
{
  // this slot is called whenever the File->Print menu is selected,
  // the Print shortcut is pressed (usually CTRL+P) or the Print toolbar
  // button is clicked
  /*KPrinter printer;
  if (printer.setup(this))
  {
    // setup the printer.  with Qt, you always "print" to a
    // QPainter.. whether the output medium is a pixmap, a screen,
    // or paper
    QPainter p;
    p.begin(&printer);

    // we let our view do the actual printing
    QPaintDeviceMetrics metrics(&printer);
    m_mainWidgetUI->print(&p, metrics.height(), metrics.width());

    // and send the result to the printer
    p.end();
  }*/
}


void KSniffer::slotPauseContinue()
{
  if ( m_doingPause )
  {
    // it's in pause we have to let continue capture again
    m_packets->acceptPacket( true );
    m_sniffer->continueSniffing();
    m_doingPause = false;

    // change icon and text menu and hint
    m_actPauseContinue->setIcon( "player_pause" );
    m_actPauseContinue->setWhatsThis( m_whatsthisPause );
    m_actPauseContinue->setText( i18n("&Pause Capture") );
    m_sniffProcess->kill(SIGUSR1);
  }
  else
  {
    m_sniffer->pauseSniffing();
    m_packets->acceptPacket( false );
    m_doingPause = true;

    // change icon and text menu and hint
    m_actPauseContinue->setIcon( "player_play" );
    m_actPauseContinue->setWhatsThis( i18n("<b>Continue Capture</b><br>Continue the paused packet-capture.") );
    m_actPauseContinue->setText( i18n("&Continue Capture") );
    m_sniffProcess->kill(SIGUSR1);
  }
}


// with this member we stop packets capture
void KSniffer::slotStop()
{
  // it doesn't need to store packets anymore
  m_packets->acceptPacket( false );

  if (m_sniffProcess->normalExit())
  {
    if (m_sniffProcess->exitStatus() == SNIFF_NETWORK_ERROR)
    {
      KMessageBox::error( this, i18n("<p>A network error occurred.<br>Cannot continue capturing packets.</p>"));
    }
    else
      kdDebug() << " m_sniffProcess->exitStatus(): " << m_sniffProcess->exitStatus() << endl;
  }
  else
    kdDebug() << "m_sniffProcess->normalExit() is FALSE" << endl;
  stopSniffing();

  // reset continue sniffing variable
  m_sniffer->continueSniffing();

  m_doingPause = false;
  if (!m_loadFromFile)
  {
    if (m_mustShowPacketsNow || m_startedShowing)
    {
      // if you want you can load a new file or start a new capture session
      m_actNew->setEnabled(true);
      m_actOpen->setEnabled(true);
      m_actOpenRecent->setEnabled( true );
      // print is not implemented better set it not enabled
      //m_actPrint->setEnabled(true);

      // and you don't need to stop or continue any capture anymore
      m_actStop->setEnabled(false);
      m_actPauseContinue->setEnabled(false);

      // set again Pause stuff on Pause/Continue Capture icon/menu
      m_actPauseContinue->setIcon( "player_pause" );
      m_actPauseContinue->setWhatsThis( m_whatsthisPause );
      m_actPauseContinue->setText( i18n("&Pause Capture") );

      m_needToSave = true;
      changeStatusbar( i18n("Stopped capturing. Now, ready again.") );

      if (!m_packets->packetAvailable())
      {
        setPlainCaption( i18n("KSniffer - The KDE Network Sniffer") );
        if (m_options->showWarning())
        {
          if (m_options->showPassiveWarn())
            KSnifferPassivePopup::message( i18n("Capturing"), i18n("<p>You got no packets.</p>"), this );
          else
            KMessageBox::information(this, i18n("You got no packets."));
        }
        m_actSave->setEnabled( false );
        m_actSaveAs->setEnabled( false );
      }
      else
      {
        setPlainCaption( i18n("KSniffer - Unsaved Packets") );
        m_actSave->setEnabled( true );
        m_actSaveAs->setEnabled( true );

        // ok you can enable sorting now
        //FIXME: this sorting will freeze GUI
        m_mainWidgetUI->setSortList(true);
      }
    }
    else if (!m_startedShowing && !m_mustShowPacketsNow && !m_packets->packetAvailable())
    {
      setPlainCaption( i18n("KSniffer - The KDE Network Sniffer") );
      if (m_options->showWarning())
      {
        if (m_options->showPassiveWarn())
          KSnifferPassivePopup::message( i18n("Capturing"), i18n("<p>You got no packets.</p>"), this );
        else
          KMessageBox::information(this, i18n("You got no packets."));
        changeStatusbar(i18n("Ready."));
      }
      m_actSave->setEnabled( false );
      m_actSaveAs->setEnabled( false );

      // if you want you can load a new file or start a new capture session
      m_actNew->setEnabled(true);
      m_actOpen->setEnabled(true);
      m_actOpenRecent->setEnabled( true );
      // print is not implemented better set it not enabled
      //m_actPrint->setEnabled(true);

      // and you don't need to stop or continue any capture anymore
      m_actStop->setEnabled(false);
      m_actPauseContinue->setEnabled(false);

      // set again Pause stuff on Pause/Continue Capture icon/menu
      m_actPauseContinue->setIcon( "player_pause" );
      m_actPauseContinue->setWhatsThis( m_whatsthisPause );
      m_actPauseContinue->setText( i18n("&Pause Capture") );

      m_needToSave = false;
    }
    else
    {
      // need to enable accepting packet for Packet Manager again
      m_packets->acceptPacket( true );
      kdDebug() << "void KSniffer::slotStop(): destFile = " << m_options->tmpFilePath() << endl;
      showAfterDisplay( m_options->tmpFilePath() );
    }
  }

  if (m_sniffer->countPackets() > 0)
  {
    KSnifferPassivePopup::message( i18n("Capturing"), i18n("<p>Captured <b>%1</b> packets for a total of <b>%2</b> bytes.</p>")
    .arg(m_sniffer->countPackets()).arg(m_sniffer->packetsTotalSize()), this );
  }
}


void KSniffer::stopSniffing()
{
  if (m_sniffProcess)
  {
    m_sniffProcess->kill();
    delete m_sniffProcess;
    m_sniffProcess = NULL;
  }
  m_sniffer->stopSniffing();

  // the thread has a little delay to stop, so before
  // to start a new thread the last has to be finished
  if ( (!m_mustShowPacketsNow) && (m_sniffer->countPackets() > 0) )
    // check just if you have to display packets after you stopped the capture
    // and if you captured at least 1 packet (just in this case the thread starts)
    while (m_sniffer->running())
      usleep(1000);
}


void KSniffer::showAfterDisplay( const KURL& url )
{
  m_startedShowing = true;
  //kdDebug() << "File to open: " << url.prettyURL() << endl;
  QFile file( url.path() );
  m_filesize = file.size();
  //kdDebug() << "++++++++++ file: " << m_filesize << " bytes" << endl;
  m_packets->setFilename( url.path() );

  // Packets available, need to save them, continue losting data
  m_needToSave = true;

  // we have to remove the captured data
  m_packets->clearList();

  // clearing the view before to start sniffing
  m_mainWidgetUI->clearView();

  // we need to store packets again
  m_packets->acceptPacket( true );

  // to avoid crash disable sorting
  m_mainWidgetUI->setSortList( false );

  KSnifferPassivePopup::message( i18n("Capturing"), i18n("<p>Now starting to display the captured packets.</p>"), this );
  changeStatusbar( i18n("Displaying captured packets.") );

  // tell the view what file use to get packets
  m_packets->setFilename( url.path() );

  m_options->setSniffingFilename( url.path() );

  // 'show after' has to be disabled to display packets
  m_options->setShowAfter( false );
  m_sniffer->setCaptureOptions( m_options );

  //FIXME: to let "display after stopping capture" should "m_options->setShowAfter( true );" when thread stopped
  startSniffingFromFile();

  setPlainCaption( i18n("KSniffer - Unsaved Packets") );
}


void KSniffer::checkIfSniffing()
{
  if (!m_sniffer->isSniffing())
  {
    // if you want you can load a new file or start a new capture session
    m_actNew->setEnabled(true);
    m_actOpen->setEnabled(true);
    m_actOpenRecent->setEnabled( true );
    // print is not implemented better set it not enabled
    //m_actPrint->setEnabled(true);

    // and you don't need to stop or continue any capture anymore
    m_actStop->setEnabled(false);
    m_actPauseContinue->setEnabled(false);

    // set again Pause stuff on Pause/Continue Capture icon/menu
    m_actPauseContinue->setIcon( "player_pause" );
    m_actPauseContinue->setWhatsThis( m_whatsthisPause );
    m_actPauseContinue->setText( i18n("&Pause Capture") );

    // ok you can enable sorting now
    m_mainWidgetUI->setSortList(true);

    m_timer->stop();

    kdDebug() << "ok the file was loaded" << endl;

    // no need to save a loaded file...
    m_actSave->setEnabled( false );
    m_actSaveAs->setEnabled( false );
    m_loadFromFile = false;

    // need to save if packets are displayed cause of the 'Show After' option
    m_needToSave = m_startedShowing;

    if (!m_startedShowing)
    {
      KSnifferPassivePopup::message( i18n("Loading file"), i18n("<p>Ok, the packets has gotten from the file.</p>"), this );
      changeStatusbar( i18n("Packets captured from file. Now, ready again.") );
    }
    else
    {
      KSnifferPassivePopup::message( i18n("Displaying captured packets"),
        i18n("<p>Ok, the captured packets are all in the list.</p>"), this );
      changeStatusbar( i18n("Ready.") );

      // enable it when packets have been read
      m_actSave->setEnabled( true );
      m_actSaveAs->setEnabled( true );

      // ok all packets was displayed
      m_startedShowing = false;
    }
    m_packets->acceptPacket( false );
  }
  else
  {
    if (m_loadFromFile)
      if (!m_startedShowing)
        changeStatusbar( i18n("Getting packets from file: %1%").arg( (double)m_sniffer->packetsTotalSize() / m_filesize * 100) );
      else
        changeStatusbar( i18n("Displaying captured packets: %1%").arg( (double)m_sniffer->packetsTotalSize() / m_filesize * 100) );
  }
}


void KSniffer::slotQuit()
{
  // user wants to close KSniffer
  m_action = ACT_QUIT_APPL;

  // pause the Sniffing Thread: are we going to quit?
  if ( m_sniffer->isSniffing() )
    m_sniffer->pauseSniffing();

  // we don't need to store packets anymore
  m_packets->acceptPacket( false );

  if ( m_packets->packetAvailable() && m_needToSave )
  {
    changeStatusbar( i18n("You have got data, maybe you want to save them.") );

    int button = KMessageBox::questionYesNoCancel( this,
         i18n( "<p>Quitting KSniffer will cause the current packets to be lost.<br>Do you want to save them now?</p>" ),
         i18n( "Unsaved Captured Packets" ), KStdGuiItem::saveAs(), KStdGuiItem::discard() );

    // we have some data that maybe we want to save
    if ( button == KMessageBox::Yes )
      slotSave();
    else if ( button == KMessageBox::Cancel )
    {
      // ok the user don't want to quit KSniffer anymore
      m_action = ACT_DO_NOTHING;
      m_sniffer->continueSniffing();
      m_packets->acceptPacket( true );
      return;
    }
    else
      // quitting without saving data, so...
      // ...stop really sniffing and...
      stopSniffing();
  }

  // now we can remove the captured data
  m_packets->clearList();

  // saving the application settings
  saveSettings();

  if ( m_tmpFile )
    delete m_tmpFile;

  // delete temporary files if you have not to save the sniffed packets
  if ( m_wroteTmpLocalFile && ((m_action & ACT_SAVING_PACKETS) == 0) )
  {
    // ...remove the pcap temporary file
    //kdDebug() << "+++++++++ I'm going to delete " << m_options->tmpFilePath() << endl;
    unlink( m_options->tmpFilePath() );
    //kdDebug() << "+++++++++ I'm going to delete " << m_options->tmpFileConfigPath() << endl;
    unlink( m_options->tmpFileConfigPath() );
  }

  // delay exit from KSniffer if Job's not finished
  kapp->deref();
}


// decide what to do when the Job is finished
void KSniffer::copyingJobFinished( KIO::Job *job )
{
  if (job->error() == 0)
  {
    // no error happened
    changeStatusbar( i18n("Saved captured-packets.") );

    // set the main widget new title
    setCaption( m_strSavedFilename.prettyURL() );

    // no need to save: we just completed save
    m_needToSave = false;

    // no need to save it again: just done
    m_actSave->setEnabled(false);
    m_actSaveAs->setEnabled(false);
    m_strFilename = QString::null;
    if ((m_action & ACT_QUIT_APPL) == ACT_QUIT_APPL)
    {
      // deleting temporary files
      //kdDebug() << "+++++++++ I'm going to delete " << m_options->tmpFilePath() << endl;
      unlink( m_options->tmpFilePath() );
      //kdDebug() << "+++++++++ I'm going to delete " << m_options->tmpFileConfigPath() << endl;
      unlink( m_options->tmpFileConfigPath() );
    }
  }
}


void KSniffer::optionsShowStatusbar()
{
  // this is all very cut and paste code for showing/hiding the
  // statusbar
  if (m_statusbarAction->isChecked())
    statusBar()->show();
  else
    statusBar()->hide();
}


void KSniffer::optionsConfigureKeys()
{
  KKeyDialog::configure(actionCollection());
}


void KSniffer::optionsConfigureToolbars()
{
  // use the standard toolbar editor
  saveMainWindowSettings(KGlobal::config(), autoSaveGroup());

  // use the standard toolbar editor
  KEditToolbar dlg(factory());
  connect(&dlg, SIGNAL(newToolbarConfig()), SLOT(applyNewToolbarConfig()));
  dlg.exec();
}


void KSniffer::applyNewToolbarConfig()
{
  // this slot is called when user clicks "Ok" or "Apply" in the toolbar editor.
  // recreate our GUI, and re-apply the settings (e.g. "text under icons", etc.)
  createGUI();

  applyMainWindowSettings(KGlobal::config(), autoSaveGroup());
}


void KSniffer::optionsSettings()
{
  // popup some sort of preference dialog, here
  KSnifferOptionDialog *dialogSettings = new KSnifferOptionDialog(this, "SettingsDialog", true);
  connect( dialogSettings, SIGNAL( trayBarShown( bool ) ), this, SLOT( showTrayBar( bool ) ) );
  if (dialogSettings->exec())
  {
    // redo your settings
    resetOptions();
    m_sniffer->setCaptureOptions( m_options );
  }

  delete dialogSettings;
}


void KSniffer::changeStatusbar( const QString& text )
{
  // display the text on the statusbar
  statusBar()->message(text);
}


void KSniffer::changeCaption( const QString& text )
{
  // display the text on the caption
  setCaption(text);
}


void KSniffer::showTrayBar( bool show)
{
  if ( show )
  {
    // show the tray bar
    m_sysTray->show();
  }
  else
  {
    // hide the tray bar
    m_sysTray->hide();
  }
}


void KSniffer::prepareXMLConfigFile()
{
  QDomDocument doc;
  QDomElement root = doc.createElement( "options" );
  doc.appendChild( root );

  QDomElement subroot = doc.createElement( "standard" );
  root.appendChild( subroot );

  QDomElement tag = doc.createElement( "interface" );
  subroot.appendChild( tag );

  QDomAttr dev = doc.createAttribute("dev");
  dev.setValue(m_options->sniffingInterface());
  tag.setAttributeNode(dev);

  QDomAttr captFile = doc.createAttribute("capturefile");
  captFile.setValue(m_options->tmpFilePath());
  tag.setAttributeNode(captFile);

  subroot = doc.createElement( "extras" );
  root.appendChild( subroot );

  tag = doc.createElement( "capture" );
  subroot.appendChild( tag );

  QDomAttr type = doc.createAttribute("type");
  QDomAttr quantity = doc.createAttribute("quantity");
  type.setValue(m_options->sniffingType());
  tag.setAttributeNode(type);

  if (m_options->sniffingType() != "manually")
  {
    QDomAttr unit = doc.createAttribute("unit");

    if (m_options->sniffingType() == "packets")
    {
      quantity.setValue(QString::number(m_options->packetsNumber()));
      unit.setValue("1");
    }
    else if (m_options->sniffingType() == "size")
    {
      long val;
      quantity.setValue(QString::number(m_options->packetsSize()));
      switch (m_options->packetsInfoSizeUnits())
      {
        case 0:
          // byte
          unit.setValue("1");
          break;
        case 1:
          // KByte
          unit.setValue("1024");
          break;
        case 2:
          // MByte
          val = 1024 * 1024;
          unit.setValue(QString::number(val));
          break;
        case 3:
          // GByte
          val = 1024 * 1024 * 1024;
          unit.setValue(QString::number(val));
          break;
      }
    }
    else if (m_options->sniffingType() == "time")
    {
      long days = 3600 * 24;
      quantity.setValue(QString::number(m_options->packetsTime()));
      switch (m_options->packetsInfoTimeUnits())
      {
        case 0:
          // seconds
          unit.setValue("1");
          break;
        case 1:
          // minutes
          unit.setValue("60");
          break;
        case 2:
          // hours
          unit.setValue("3600");
          break;
        case 3:
          // days
          unit.setValue(QString::number(days));
          break;
      }
    }
    tag.setAttributeNode(unit);
    tag.setAttributeNode(quantity);
  }

  QString xml = "<?xml version=\"1.0\"?>\n";
  xml.append( doc.toString() );

  QFile file( m_options->tmpFileConfigPath() );
  if ( file.open( IO_WriteOnly ) )
  {
    QTextStream stream( &file );
    stream << xml << "\n";
    file.close();
  }
  else
    KMessageBox::sorry(0, i18n("Could not open temporary file '%1' for writing").arg(m_options->tmpFileConfigPath()),
      i18n("Writing Error") );
}


void KSniffer::startLiveSniffing( const QString& interface )
{
  // the temporary file (network transparency) can be removed
  if (m_tmpFile)
    delete m_tmpFile;

  // for the live capture we need the default temporary pcap file
  m_packets->setFilePath( m_options->tmpFilePath() );

  m_actNew->setEnabled( false );
  m_actOpen->setEnabled( false );
  m_actOpenRecent->setEnabled( false );
  m_actSave->setEnabled( false );
  m_actSaveAs->setEnabled( false );
  m_actPrint->setEnabled( false );

  m_actPauseContinue->setEnabled( true );
  m_actStop->setEnabled( true );

  // starting capture variable
  m_doingCapture = true;

  KSnifferPassivePopup::message( i18n("Capturing"),
                          i18n("<p>You selected <b>%1</b> interface.<br>Now starting to capture packets.</p>")
                              .arg(interface), this );

  setPlainCaption( i18n("KSniffer - Capturing on %1 interface").arg(interface) );

  // create CaptureOptions object to give it to the Sniffer's constuctor
  m_options->setSniffingInterface( interface );

  m_sniffer->setCaptureOptions( m_options );

  // clearing the view before to start sniffing
  m_mainWidgetUI->clearView();

  // to avoid crash disable sorting
  m_mainWidgetUI->setSortList( false );

  // tell the view we are going to start sniffing
  m_mainWidgetUI->loadFromFile( false );

  m_loadFromFile = false;

  if (m_mustShowPacketsNow)
  {
    kdDebug() << "ok, packets will be displayed" << endl;
    m_packets->acceptPacket( true );
    m_mainWidgetUI->displayPacket( true );
  }
  else
  {
    // if you have to display packet after capture is useful don't accept packets to display
    kdDebug() << "ok, NO packets will be displayed" << endl;
    m_mainWidgetUI->displayPacket( false );
  }

  prepareXMLConfigFile();
  m_sniffProcess = new KProcess;
  *m_sniffProcess << "sniff";
  *m_sniffProcess << m_options->tmpFileConfigPath();
  connect(m_sniffProcess, SIGNAL(processExited(KProcess *)), this, SLOT(slotStop()));
  m_sniffProcess->start();
  m_sniffer->startSniffing();
}


void KSniffer::startSniffingFromFile()
{
  m_actNew->setEnabled( false );
  m_actOpen->setEnabled( false );
  m_actOpenRecent->setEnabled( false );
  m_actSave->setEnabled( false );
  m_actSaveAs->setEnabled( false );
  m_actPrint->setEnabled( false );

  m_actPauseContinue->setEnabled( false );
  m_actStop->setEnabled( true );

  m_mainWidgetUI->loadFromFile();

  m_loadFromFile = true;

  m_sniffer->startSniffing();

  //FIXME: timer still useful with the new sniffing method?
  m_timer->start(700);
}


void KSniffer::resetOptions()
{
  m_snifferOption->getOptions( m_options );

  if ( !m_sniffer->isSniffing() )
  {
    //FIXME: the setting for the view is deprecated
    m_mainWidgetUI->setOptions( m_options );
  }
  else
    // during capture the options don't take effect
    m_optionChanged = true;
}

#include "ksniffer.moc"
