/***************************************************************************
 *   Copyright (C) 2003 by Tom Deblauwe                                    *
 *   tomNOSPAM.deblauwe@pandora.be                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/
#include "ksubtileplayer.h"

// include files for Qt
#include <qprinter.h>
#include <qpainter.h>
#include <qtabwidget.h>
#include <qlabel.h>
#include <qregexp.h>
#include <qdatetime.h>
#include <qdatetimeedit.h>
#include <qspinbox.h>
#include <qcheckbox.h>
#include <qpixmap.h>
#include <qradiobutton.h>
#include <qrect.h>
#include <qslider.h>
#include <qpopupmenu.h>

// application specific includes
#include "ksubtileview.h"
#include "ksubtiledoc.h"
#include "ksubtile.h"
#include "ksubtiledlg.h"
#include "ksqueezedtextlabel.h"
#include "kmessagebox.h"
#include "kfiledialog.h"
#include "klocale.h"
#include "csubtitles.h"
#include "kprocess.h"
#include "ktempfile.h"
#include "kwin.h"
#include "kwinmodule.h"
#include "kiconloader.h"
#include <kpushbutton.h>
#include <klineedit.h>
#include "kdockwidget.h"
#include <kmenubar.h>

#include <math.h>

#include <X11/Xutil.h>

#include "subtitlenavigator.h"
#include "selectfromlistdialog.h"
#include "pref.h"

KSubtilePlayer::KSubtilePlayer(QWidget *parent, const char *name) : KSubtilePlayerDlg(parent, name)
{
    //  the processes
    proc = 0;
    procLength = 0;

    //enable valuechanged action
    b_enable_player_startTime_valueChanged = true;

    b_slider_follow_mplayer = true;

    //load icons onto buttons

    KIconLoader* loader = KGlobal::iconLoader();

    //player
    pb_player_play->setPixmap(loader->loadIcon("player_play", KIcon::Toolbar));
    pb_player_pause->setPixmap(loader->loadIcon("player_pause", KIcon::Toolbar));
    pb_player_stop->setPixmap(loader->loadIcon("player_stop", KIcon::Toolbar));
    pb_player_forward->setPixmap(loader->loadIcon("player_fwd", KIcon::Toolbar));
    pb_player_backward->setPixmap(loader->loadIcon("player_rew", KIcon::Toolbar));

    pb_player_open->setPixmap(loader->loadIcon("fileopen", KIcon::Toolbar));


    //// make some actions for the player
    //(void) new KAction(i18n("Play"), 0,
}


KSubtilePlayer::~KSubtilePlayer()
{
    stopPlayingMovie();
}

KSubtileView* KSubtilePlayer::getEditor()
{
    KDockManager* manager = ((KDockWidget*)parentWidget())->dockManager();
    KDockWidget* editorDock = manager->getDockWidgetFromName("editor");
    return (KSubtileView* ) editorDock->getWidget();
}

long KSubtilePlayer::getPlayerCurrentPos()
{
    return playerCurrentPos;
}

long KSubtilePlayer::getPlayerMovieLength()
{
    if(l_player_fileName->text() == "")
    {
        return 0;
    }
    else
    {
        return playerMovieLength;
    }
}

void KSubtilePlayer::pb_player_open_clicked()
{
    KURL url=KFileDialog::getOpenURL(0, i18n("*.avi *.mpg *.mpeg *.mov *.wmv|Movie Files\n*|All Files"), this, i18n("Open Movie File..."));
    if(!url.isEmpty())
    {
        openMovie(url.path());
    }

    //KMessageBox::sorry((KSubtileApp *) parent(), "open movie!", "Sorry");
}

void KSubtilePlayer::openMovie(const QString& filename)
{
    fileNameOfCheckedMovie = filename;

    //get length of movie

    if(procLength != 0)
    {
        delete procLength;
    }

    procLength = new KProcess;

    //start mplayer in slave mode
    *procLength << "mplayer" << "-slave" << "-vo" << "null" << "-ao" << "null";

    //add filename
    *procLength << filename;

    //setup slots
    connect(procLength, SIGNAL(receivedStdout(KProcess *, char*,int)), this, SLOT(slotLengthReceivedStdout(KProcess *, char*, int)));
    connect(procLength, SIGNAL(processExited(KProcess *)), this, SLOT(slotLengthPlayingFinished(KProcess *)));

    //set the movie length back to zero first
    playerMovieLength = 0;

    playerMovieWidth = 0;
    playerMovieHeight = 0;
    //start mplayer and wait on this line until finished
    procLength->start(KProcess::NotifyOnExit, KProcess::All);

    // 	procLength.kill();

    //playerMovieLength will contain the length in msecs and width and height will be filled in too

    //we enable the player when the process is finished
}

void KSubtilePlayer::slotLengthReceivedStdout(KProcess* /*myProc*/, char* buf, int buflen)
{
    //   printf("ontvangen");
    QString myBuf = QString::fromLatin1(buf, buflen);
    // printf("%s\n",myBuf.latin1());
    if(myBuf.contains(QRegExp( "^VO: " , false, false)) >= 1)
    {
        QStringList myList = QStringList::split(QRegExp("=>"), myBuf);
        QStringList sizeList = QStringList::split(QRegExp(" "), myList[1] );
        QStringList widthAndHeightList = QStringList::split(QRegExp("x"), sizeList[0]);

                // printf("rcvwidth: %s\n", widthAndHeightList[0].latin1());
                // printf("rcvheight: %s\n", widthAndHeightList[1].latin1());
        playerMovieWidth = widthAndHeightList[0].toInt();
        playerMovieHeight = widthAndHeightList[1].toInt();
        return;
    }

    if(myBuf.contains(QRegExp( " *V: *[0-9]*[0-9].[0-9]" , false, false)) >= 1 || myBuf.contains(QRegExp( " *V: *[0-9]*[0-9],[0-9]" , false, false)) >= 1)
    {
        if(playerMovieLength == 0)
        {
            //we received the first input so the movie is playing
            //tell it to go to 100%
            QString cmd = QString("get_time_length\n");
            procLength->writeStdin(cmd.latin1(), cmd.length());

            //this is to prevent this function from being executed again when we are at 100%
            playerMovieLength = -1;

            //leave this function
            return;
        }
        //with the get_time_length we don't get super accurate information, it is rounded at 1 sec so the movie may be a little bit longer
        //therefore we seek to the time we got with get_time_length and let it play.  This will at most take 999 msecs but will give
        //the right movie length
        QStringList list = QStringList::split(QRegExp(":"), myBuf);
        int i = 0;
        for ( QStringList::Iterator it = list.begin(); it != list.end(); ++it )
        {
            if((*it).contains(QRegExp( "V" , false, false)) >= 1)
            {
                break;
            }
            i++;
        }
        i++;

        // printf("%s - %f\n", list[i].latin1(), list[i].toDouble() * 1000.0);
		
		// Make sure parsed value contains only a number
		// In my case it contained something like 5122.2 A-V
		// by Dimitris Michail
		QStringList list1 = QStringList::split(QString(" "), list[i]);
		QStringList::iterator si = list1.begin();
		if ( si != list1.end() )
		{
			long newvalue = (long) ( (*si).replace( ",", "." ).toDouble() * 1000.0 );
			playerMovieLength = ( newvalue > playerMovieLength )? newvalue : playerMovieLength;
		}

    }
    if(myBuf.contains(QRegExp( "ANS_LENGTH" , false, false)) >= 1)
    {
        //we received the answer
        QStringList myList = QStringList::split(QRegExp("="), myBuf);
        //please mplayer folks, don't round at 1 second :-)
        playerMovieLength = (long) (myList[1].replace( ",", "." ).toDouble() * 1000.0);
        //seek to last second of movie, and let it play to determine the milliseconds
        QString mycmd = QString("seek " + QString::number(myList[1].replace( ",", "." ).toDouble() - 1.0) + " 2\n");
        //printf("%s", mycmd.latin1());
	procLength->writeStdin(mycmd.latin1(), mycmd.length());
    }
}

void KSubtilePlayer::slotLengthPlayingFinished(KProcess *)
{
    if(playerMovieLength != 0 && playerMovieLength != -1)
    {
        //ok, we got the length of a movie so load the movie
        //enable player
        player_enable();

        //enable and reset slider
        sl_player_currentPosition->setEnabled(true);
        sl_player_currentPosition->setValue(0);
        //reset timeedit
        te_player_startTime->setTime(QTime(0,0,0,0));

        //fill in filename
        l_player_fileName->setText( fileNameOfCheckedMovie );

        //let people know we loaded a movie hooray!!
        emit movieLoaded( fileNameOfCheckedMovie );
        //printf("movielength: %d\n", playerMovieLength);
    }
    else
    {
        //no movie length, probably a bad movie, don't load the movie
        KMessageBox::sorry(0, i18n("The selected movie appears not to be readable by MPlayer!"), i18n("Sorry"));
        //no movie loaded
        emit movieLoaded( "" );
    }
}

void KSubtilePlayer::stopPlayingMovie()
{
    //try to exit the possibly playing movie.
    if(proc != 0)
    {
        QString cmd = QString("quit\nquit\n");
        proc->writeStdin(cmd.latin1(), cmd.length());
    }
}

void KSubtilePlayer::sl_player_currentPosition_sliderPressed()
{
    b_slider_follow_mplayer = false;
}

void KSubtilePlayer::sl_player_currentPosition_sliderReleased()
{    
    b_slider_follow_mplayer = true;
}

void KSubtilePlayer::sl_player_currentPosition_sliderMoved(int newValue)
{
    //detect playing or stopped
    if(pb_player_play->isEnabled())
    {
        //stopped
        // 		printf("slider:%d\n", newValue);
        long msecs = (long) ((newValue / 100.0)*playerMovieLength);
        b_enable_player_startTime_valueChanged = false;
        te_player_startTime->setTime(CSubtitles::getQTimeFromMSecs(msecs));
        b_enable_player_startTime_valueChanged = true;
    }
    else
    {
        //playing
        //seek to new time
        long secs = long( ((newValue / 100.0)*(double)playerMovieLength) / 1000.0);
        QString cmd = QString("seek ") + QString::number(secs) + QString(" 2\n");
        proc->writeStdin(cmd.latin1(), cmd.length());

    }
}

void KSubtilePlayer::te_player_startTime_valueChanged(const QTime & newTime)
{

    if(b_enable_player_startTime_valueChanged)
    {
        long newTimeMSecs = CSubtitles::getMSecs(newTime);
        //limit this so the user can't select a time after the endtime of the movie
        if(newTimeMSecs >= playerMovieLength)
        {
            newTimeMSecs = playerMovieLength - 1;
            //fill in this time instead in the timeedit
            te_player_startTime->setTime(CSubtitles::getQTimeFromMSecs(newTimeMSecs));
        }
        sl_player_currentPosition->setValue((int) (((double) newTimeMSecs / (double) playerMovieLength)*100.0));
    }
}

void KSubtilePlayer::player_enable()
{
    pb_player_play->setEnabled(true);
    pb_player_pause->setEnabled(false);
    pb_player_stop->setEnabled(false);
    pb_player_backward->setEnabled(false);
    pb_player_forward->setEnabled(false);
    te_player_startTime->setEnabled(true);

    //little play buttons in the tabpages
    getEditor()->pb_stretch_play_startTime->setEnabled(true);
    getEditor()->pb_move_play_startTime->setEnabled(true);
    getEditor()->pb_editor_play_startTime->setEnabled(true);

    checkEnableWithSubtitles();
}

/** Enable/Disable the play with subtitles checkbox.  If we don't
    have subtitles the box should disable.  Only enables if mplayer
    is available.
*/
void KSubtilePlayer::checkEnableWithSubtitles()
{
    //if open button is enabled, it means mplayer was found
    if(pb_player_open->isEnabled())
    {
        if(getEditor()->getDocument()->subtitles->getNrOfSubtitles() == 0)
        {
            //no subs!
            cb_player_withSubtitles->setEnabled(false);
        }
        else
        {
            cb_player_withSubtitles->setEnabled(true);
        }
    }
}

void KSubtilePlayer::pb_player_play_clicked()
{
    if(proc != 0)
    {
        delete proc;
    }
    //play the file
    proc = new KProcess;

    *proc << "mplayer";
    *proc << fileNameOfCheckedMovie.latin1();
    *proc << "-osdlevel" << "2";

    //use the specfied audio and video outputs
    kapp->config()->setGroup("MPlayer Options");
    QString audioOutput = kapp->config()->readEntry("Audio Output Driver", ksubtilePreferences::pref_AudioOutputDriver() );
    QString videoOutput = kapp->config()->readEntry("Video Output Driver", ksubtilePreferences::pref_VideoOutputDriver() );

    *proc << "-ao";
    *proc << audioOutput;
    *proc << "-vo";
    *proc << videoOutput;

    bool automaticResize = kapp->config()->readBoolEntry("Automatic Resize", ksubtilePreferences::pref_MPlayerAutomaticResize);

    //if editor is not visible we don't do the automaticResize
    KDockManager* manager = ((KDockWidget*)parentWidget())->dockManager();
    KDockWidget* editorDock = manager->getDockWidgetFromName("editor");
    
    if(!editorDock->isVisible())
    {
		automaticResize = false;
    }
    
	if(playerMovieHeight == 0 || playerMovieWidth == 0)
	{
		automaticResize = false;
	}
	
    if(automaticResize)
    {

        int xsize = 0;
        //calculate maximum space we have in the X-coordinate
        KWinModule kwinmod;
        QRect desktop = kwinmod.workArea();

        //get the bounds of the editor

        QRect editorBounds = getEditor()->geometry();
        //         printf("editor geometry: top: %d, left: %d\n", editorBounds.top(), editorBounds.left());
        QPoint editorPos = getEditor()->mapToGlobal(QPoint(0,0));
        // 		printf("editor pos: top: %d, left: %d\n", editorPos.y(), editorPos.x());

        int position = kapp->config()->readNumEntry("MPlayer Position", ksubtilePreferences::pref_MPlayerPosition );

        double aspectRatio = (double) playerMovieWidth / (double) playerMovieHeight;

        int spaceLeft = 0;
        int spaceTop = 0;
        int spaceBottom = 0;
        int spaceRight = 0;

        int testSize = 0;
		
        switch( position )
        {
        case ksubtilePreferences::posTopLeft:

            spaceLeft = (int) fabs(desktop.left() - editorPos.x());
            spaceTop = (int) fabs(desktop.top() - editorPos.y());			
            if(spaceLeft > spaceTop)
            {

                //testSize contains the new height(in the case we use spaceLeft as width)
                testSize = (int) ((double) spaceLeft / aspectRatio);

                if(testSize < spaceTop)
                {
                    //we have still have space left above, so position using the top
                    xsize = (int) ((double)spaceTop * aspectRatio);
                }
                else
                {
                    //no space left above
                    xsize = spaceLeft;
                }
            }
            else
            {
                testSize = (int) ((double) spaceTop * aspectRatio);

                if(testSize < spaceLeft)
                {
                    xsize = spaceLeft;
                }
                else
                {
                    xsize = (int) ((double)spaceTop * aspectRatio);
                }
            }
            break;

        case ksubtilePreferences::posBottomLeft:

            spaceLeft = (int) fabs(desktop.left() - editorPos.x());
            spaceBottom = (int) fabs(desktop.bottom() - (editorPos.y() + editorBounds.height()) );

            if(spaceLeft > spaceBottom)
            {

                //testSize contains the new height(in the case we use spaceLeft as width)
                testSize = (int) ((double) spaceLeft / aspectRatio);

                if(testSize < spaceBottom)
                {
                    //we have still have space left below, so position using the bottom
                    xsize = (int) ((double)spaceBottom * aspectRatio);
                }
                else
                {
                    //no space left above
                    xsize = spaceLeft;
                }
            }
            else
            {
                testSize = (int) ((double) spaceBottom * aspectRatio);

                if(testSize < spaceLeft)
                {
                    xsize = spaceLeft;
                }
                else
                {
                    xsize = (int) ((double)spaceBottom * aspectRatio);
                }
            }

            break;

        case ksubtilePreferences::posTopRight:

            spaceRight = (int) fabs(desktop.right() - (editorPos.x() + editorBounds.width()) );
            spaceTop = (int) fabs(desktop.top() - editorPos.y());
            if(spaceRight > spaceTop)
            {

                //testSize contains the new height(in the case we use spaceLeft as width)
                testSize = (int) ((double) spaceRight / aspectRatio);

                if(testSize < spaceTop)
                {
                    //we have still have space left above, so position using the top
                    xsize = (int) ((double)spaceTop * aspectRatio);
                }
                else
                {
                    //no space left above
                    xsize = spaceRight;
                }
            }
            else
            {
                testSize = (int) ((double) spaceTop * aspectRatio);

                if(testSize < spaceRight)
                {
                    xsize = spaceRight;
                }
                else
                {
                    xsize = (int) ((double)spaceTop * aspectRatio);
                }
            }
            break;

        case ksubtilePreferences::posBottomRight:

            spaceRight = (int) fabs(desktop.right() - (editorPos.x() + editorBounds.width()) );
            spaceBottom = (int) fabs(desktop.bottom() - (editorPos.y() + editorBounds.height()) );

            if(spaceRight > spaceBottom)
            {

                //testSize contains the new height(in the case we use spaceRight as width)
                testSize = (int) ((double) spaceRight / aspectRatio);

                if(testSize < spaceBottom)
                {
                    //we have still have space left below, so position using the bottom
                    xsize = (int) ((double)spaceBottom * aspectRatio);
                }
                else
                {
                    //no space left above
                    xsize = spaceRight;
                }
            }
            else
            {
                testSize = (int) ((double) spaceBottom * aspectRatio);

                if(testSize < spaceRight)
                {
                    xsize = spaceRight;
                }
                else
                {
                    xsize = (int) ((double)spaceBottom * aspectRatio);
                }
            }

            break;
        }

        *proc << "-zoom";
         		printf("xsize:%d\n", xsize);
        *proc << "-xy" << QString::number(xsize);
    }


    *proc << "-slave";
    *proc << "-noautosub";
    *proc << "-ss";
    *proc << te_player_startTime->time().toString( QString ( "hh:mm:ss" )).latin1();
    if(cb_player_withSubtitles->isChecked())
    {
        if(getEditor()->getDocument()->isModified())
        {
            KTempFile* tmp = new KTempFile();
            tmp->setAutoDelete(true);

            if( tmp->status() == 0 )
            {
                QTextStream stream(tmp->fstream(), 2);		 				
				
				if(getEditor()->getDocument()->text_codec != "")
				{
					stream.setCodec( QTextCodec::codecForName(getEditor()->getDocument()->text_codec) );
				}
				else
				{
					stream.setCodec( QTextCodec::codecForLocale() );
				}								
                
				getEditor()->getDocument()->subtitles->subDump(stream, CSubtitles::subFormatSRT);
                tmp->close();
                if( tmp->status() == 0 )
                {
                    //          printf("subnaam:%s",tmp->name().latin1());
                    *proc << "-sub";
                    *proc << tmp->name().latin1();
                }
            }
            //delete tmp;
        }
        else
        {
            //not modified use filename
            //      printf("notmodified: %s",getDocument()->URL().path().latin1());
            *proc << "-sub";
            *proc << getEditor()->getDocument()->URL().path().latin1();
        }
    }

    connect(proc, SIGNAL(processExited(KProcess *)), this, SLOT(slotPlayerPlayingFinished(KProcess *)));
    connect(proc, SIGNAL(receivedStdout(KProcess *, char*,int)), this, SLOT(slotReceivedStdout(KProcess *, char*, int)));
	//connect(proc, SIGNAL(receivedStderr(KProcess *, char*,int)), this, SLOT(slotReceivedStderr(KProcess *, char*, int)));

    bool ok = proc->start(KProcess::NotifyOnExit, KProcess::All);
    if(ok)
    {
        //disable play button
        pb_player_play->setEnabled(false);

	//disable open button
	pb_player_open->setEnabled(false);
	
        //enable pause and stop button
        pb_player_pause->setEnabled(true);
        pb_player_stop->setEnabled(true);
        pb_player_backward->setEnabled(true);
        pb_player_forward->setEnabled(true);

        checkEnableWithSubtitles();

        //disable time edit
        te_player_startTime->setEnabled(false);

        //enable curpos buttons
        if(getEditor()->te_stretch_startTime->isEnabled())
        {
            getEditor()->pb_stretch_curpos_startTime->setEnabled(true);
        }
        if(getEditor()->te_move_startTime->isEnabled())
        {
            getEditor()->pb_move_curpos_startTime->setEnabled(true);
        }
        if(getEditor()->te_editor_startTime->isEnabled())
        {
            getEditor()->pb_editor_curpos_startTime->setEnabled(true);
        }

        getEditor()->pb_stretch_play_startTime->setEnabled(false);
        getEditor()->pb_move_play_startTime->setEnabled(false);
        getEditor()->pb_editor_play_startTime->setEnabled(false);

        kwm = new KWinModule();
        connect( kwm, SIGNAL(windowAdded(WId)), this, SLOT(slotMPlayerWid(WId)) );
    }

}

void KSubtilePlayer::slotMPlayerWid( WId wid )
{
    KWin::WindowInfo info = KWin::windowInfo(wid, NET::WMVisibleName | NET::WMKDEFrameStrut | NET::WMGeometry);

    if (info.valid() && info.visibleName().contains("MPlayer"))
    {
        KWin::setState(wid, NET::StaysOnTop);

        //position mplayer window (and resize)

        //read config to know where
        kapp->config()->setGroup("MPlayer Options");
        int position = kapp->config()->readNumEntry("MPlayer Position", ksubtilePreferences::pref_MPlayerPosition );


        QRect outerBounds = info.frameGeometry();
        QRect innerBounds = info.geometry();
        QRect desktop = kwm->workArea();

        int topHeight = 0;
        int leftWidth = 0;
        int rightWidth = 0;
        int bottomHeight = 0;

        int newX = 0;
        int newY = 0;
        switch( position )
        {
        case ksubtilePreferences::posTopLeft:

            //calculate width of frame on the left side

            leftWidth = innerBounds.left() - outerBounds.left();
            topHeight = innerBounds.top() - outerBounds.top();

            //move topleft
            newX = leftWidth;
            newY = topHeight;
            break;

        case ksubtilePreferences::posTopRight:

            topHeight = innerBounds.top() - outerBounds.top();
            rightWidth = outerBounds.right() - innerBounds.right();

            //move topright
            newX = desktop.right() - innerBounds.width() - rightWidth;
            newY = topHeight;
            break;

        case ksubtilePreferences::posBottomLeft:

            bottomHeight = outerBounds.bottom() - innerBounds.bottom();
            leftWidth = innerBounds.left() - outerBounds.left();

            //move bottomleft
            newX = leftWidth;
            newY = desktop.bottom() - innerBounds.height() - bottomHeight;
            break;

        case ksubtilePreferences::posBottomRight:

            bottomHeight = outerBounds.bottom() - innerBounds.bottom();
            rightWidth = outerBounds.right() - innerBounds.right();

            //move bottomright
            newX = desktop.right() - innerBounds.width() - rightWidth;
            newY = desktop.bottom() - innerBounds.height() - bottomHeight;
            break;
        }
        //move the window effectively
        XMoveWindow(qt_xdisplay(), wid, newX, newY);

    }
    disconnect( kwm, SIGNAL(windowAdded(WId)), this, SLOT(slotMPlayerWid(WId)) );
}

void KSubtilePlayer::slotReceivedStderr(KProcess* /*proc*/, char* buf, int buflen)
{
            //printf("ontvangen\n");
    QString myBuf = QString::fromLatin1(buf, buflen);
     printf("ERROR:%s\n",myBuf.latin1());
	 
}

void KSubtilePlayer::slotReceivedStdout(KProcess* /*proc*/, char* buf, int buflen)
{
            //printf("ontvangen\n");
    QString myBuf = QString::fromLatin1(buf, buflen);
     //printf("%s\n",myBuf.latin1());
    if(myBuf.contains(QRegExp( " *V: *[0-9]*[0-9].[0-9]" , false, false)) >= 1 || myBuf.contains(QRegExp( " *V: *[0-9]*[0-9],[0-9]" , false, false)) >= 1)
    {
        //printf("%s",myBuf.latin1());
        QStringList list = QStringList::split(QRegExp(":"), myBuf);
        int i = 0;
        for ( QStringList::Iterator it = list.begin(); it != list.end(); ++it )
        {
            if((*it).contains(QRegExp( "V" , false, false)) >= 1)
            {
                break;
            }
            i++;
        }
        i++;
        // printf("%s - %f\n", list[i].latin1(), list[i].toDouble() * 1000.0);
         
        // Make sure parsed value contains only a number
        QStringList list1 = QStringList::split(QString(" "), list[i]);
        QStringList::iterator si = list1.begin();
        if ( si != list1.end() )
                playerCurrentPos = (long) ( (*si).replace( ",", "." ).toDouble() * 1000.0 );

        //     printf("stdout:%s\n", list[i].latin1() );

        //update slider position
        if(b_slider_follow_mplayer)
        {
            sl_player_currentPosition->setValue( (int) (((double) playerCurrentPos / (double) playerMovieLength)*100.0) );
        }
    }
}

void KSubtilePlayer::slotPlayerPlayingFinished(KProcess *)
{
    //enable play button
    pb_player_play->setEnabled(true);
    
    //enable open button
    pb_player_open->setEnabled(true);
    
    //disable pause and stop button
    pb_player_pause->setEnabled(false);
    pb_player_stop->setEnabled(false);
    pb_player_backward->setEnabled(false);
    pb_player_forward->setEnabled(false);

    checkEnableWithSubtitles();

    //reset slider to chosen play starttime
    sl_player_currentPosition->setValue((int) (((double) CSubtitles::getMSecs(te_player_startTime->time()) / (double) playerMovieLength)*100.0));
    //enable changing of starttime again
    te_player_startTime->setEnabled(true);

    b_slider_follow_mplayer = true;

    //disable curpos buttons
    getEditor()->pb_stretch_curpos_startTime->setEnabled(false);
    getEditor()->pb_move_curpos_startTime->setEnabled(false);
    getEditor()->pb_editor_curpos_startTime->setEnabled(false);

    //enable play buttons
    getEditor()->pb_stretch_play_startTime->setEnabled(true);
    getEditor()->pb_move_play_startTime->setEnabled(true);
    getEditor()->pb_editor_play_startTime->setEnabled(true);
}

void KSubtilePlayer::pb_player_pause_clicked()
{
    //   printf("toggled");
    QString cmd = QString("pause\n");
    proc->writeStdin(cmd.latin1(), cmd.length());
}

void KSubtilePlayer::pb_player_stop_clicked()
{
    QString cmd = QString("quit\nquit\n");
    b_slider_follow_mplayer = false;
    proc->writeStdin(cmd.latin1(), cmd.length());
    //   slotPlayerPlayingFinished(proc);
}

void KSubtilePlayer::pb_player_backward_clicked()
{
    QString cmd = QString("seek -5 type=0\n");
    proc->writeStdin(cmd.latin1(), cmd.length());
}

void KSubtilePlayer::pb_player_forward_clicked()
{
    QString cmd = QString("seek 5 type=0\n");
    proc->writeStdin(cmd.latin1(), cmd.length());
}


#include "ksubtileplayer.moc"
