/* This file is part of the KDE project
   Copyright (C) 2004 Nadeem Hasan <nhasan@kde.org>
   and Stefan Kombrink <katakombi@web.de>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#include "synconfig.h"
#include "syndock.h"
#include "syndaemon.h"

#include <kactioncollection.h>
#include <kapplication.h>
#include <kdebug.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kpassivepopup.h>
#include <kpopupmenu.h>
#include <kshortcut.h>
#include <kstdaction.h>

#include <qpixmap.h>
#include <qtooltip.h>

#include <synaptics.h>
#include <synparams.h>

using namespace Synaptics;

SynDock::SynDock( QWidget *parent, const char *name )
    : KSystemTray( parent, name )
{
    KLocale::setMainCatalogue( "ksynaptics" );

    m_dockPixmapEnabled = KSystemTray::loadIcon( "ksynaptics" );
    m_dockPixmapDisabled = KSystemTray::loadIcon( "syndockdisabled" );

    setPixmap( m_dockPixmapEnabled );

    QToolTip::add( this, i18n( "KSynaptics: Enable and disable your touch pad temporarily" ) );

    m_enableDisableAction = new KToggleAction(
        i18n( "Touch Pad Enabled" ),
        KKeySequence( i18n( "Ctrl+Alt+P" ) ),
        this,
        SLOT( toggleEnableDisable() ),
        actionCollection(),
        "enable_disable_touchpad"
    );
    m_enableDisableAction->setChecked(true);

    m_disableCompleteAction = new KRadioAction(
        i18n("Disable Touch Pad"),
        0,
        this,
        "disablemode_disable_complete"
    );
    m_disableCompleteAction->setExclusiveGroup("smartmode");

    m_disableTappingAction = new KRadioAction(
        i18n("Disable Tapping"),
        0,
        this,
        "disablemode_disable_tapping"
    );
    m_disableTappingAction->setExclusiveGroup("smartmode");
    m_disableTappingAction->setChecked(true);

    KAction *configureAction = KStdAction::preferences(
        this,
        SLOT(slotConfigure()),
        actionCollection() );

    // populate context-menu
    m_enableDisableAction->plug( contextMenu() );
    contextMenu()->insertTitle( i18n("Smart Mode") );
    m_disableCompleteAction->plug( contextMenu() );
    m_disableTappingAction->plug( contextMenu() );
    contextMenu()->insertSeparator();
    configureAction->plug(contextMenu());

    // disable if pad is not accessible
    bool isAccessible = Pad::hasShm() && Pad::hasDriver();
    
    m_enableDisableAction->setEnabled( isAccessible );
    m_disableCompleteAction->setEnabled( isAccessible );
    m_disableTappingAction->setEnabled( isAccessible );


    if ( isAccessible )
    {
        // install global accel for disabling/enabling the touch pad
        KGlobalAccel *globalAccel = new KGlobalAccel( this );
        globalAccel->insert(
            "toggle_touch_pad",
            i18n( "Toggle Touch Pad" ),
            i18n( "Switch touch pad on and off temporarily" ),
            m_enableDisableAction->shortcut(), // take shortcut from context-menu action
            0,
            m_enableDisableAction,  // let global accel toggle our action
            SLOT( activate() )
        );
        globalAccel->updateConnections();
    

        // initialize synaptics daemon thread
        mSynDaemon = new SynDaemon( (int)SynConfig::smartModeDelay() );
        mSynDaemon->start();

        connect( mSynDaemon, SIGNAL( startTyping() ), this, SLOT( disableDueToTyping() ) );
        connect( mSynDaemon, SIGNAL( stopTyping() ), this, SLOT( enableAfterTyping() ) );
    }
}


SynDock::~SynDock()
{
    kdDebug() << k_funcinfo << endl;

    delete mSynDaemon;
    mSynDaemon = 0;

    Pad::setParam( TOUCHPADOFF, SynConfig::enableTouchPad() );
}


void SynDock::toggleEnableDisable()
{
    kdDebug() << k_funcinfo << endl;

    bool off = padDisabled();

    // update tray-icon
    setPixmap( off ? m_dockPixmapDisabled : m_dockPixmapEnabled);

    QString accelStr = m_enableDisableAction->shortcut().toString();
    if ( off )
    {
        KPassivePopup::message(
            i18n( "Touch Pad Disabled" ),
            i18n( "Press %1 to enable touch pad again." ).arg( accelStr ),
            m_dockPixmapDisabled,
            this
            );
    }
    else
    {
        KPassivePopup::message(
            i18n( "Touch Pad Enabled" ),
            i18n( "Press %1 to disable touch pad again." ).arg( accelStr ),
            m_dockPixmapEnabled,
            this
            );
    }

    Pad::setParam( TOUCHPADOFF, off ? 1 : SynConfig::enableTouchPad() );
}


void SynDock::disableDueToTyping()
{
    if (!padDisabled())
    {
        kdDebug() << k_funcinfo << endl;
        Pad::setParam( TOUCHPADOFF, smartModeType() );
    }
}


void SynDock::enableAfterTyping()
{
    if (!padDisabled())
    {
        kdDebug() << k_funcinfo << endl;
        Pad::setParam( TOUCHPADOFF, SynConfig::enableTouchPad() );
    }
}


SynDock::DisableType SynDock::smartModeType() const
{
    if ( m_disableCompleteAction->isChecked() )
        return DT_FULL;
    return DT_TAPPING;
}


bool SynDock::padDisabled() const
{
    return m_enableDisableAction->isChecked() == false;
}


void SynDock::slotConfigure()
{
    QString errStr;
    kdDebug() << k_funcinfo << endl;
    if (KApplication::startServiceByDesktopName("ksynaptics", QString::null, &errStr) > 0)
    {
        // opening ksynaptics kcm failed
        KMessageBox::detailedError(this,
        i18n("The touch pad configuration module could not be loaded"),
        errStr
        /*, caption, options */);
    }
}

#include "syndock.moc"
