/* KwlanInterface defines a class for configuring and monitoring an interface
        Parts of it's implemtation are based on KNemo by Percy Leonhardt <percy@eris23.de>
   
        (c) 2006 by Thomas Michel <tom.michel@arcor.de>

        Kwlan is free software; you can redistribute it and/or modify
        it under the terms of the GNU Library General Public License as
        published by the Free Software Foundation; either version 2 of
        the License, or (at your option) any later version.

        Kwlan is distributed in the hope that it will be useful,
        but WITHOUT ANY WARRANTY; without even the implied warranty of
        MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
        GNU Library General Public License for more details.

        You should have received a copy of the GNU Library General Public License
        along with this library; see the file COPYING.LIB.  If not, write to
        the Free Software Foundation, Inc., 51 Franklin Steet, Fifth Floor,
        Boston, MA 02110-1301, USA.
*/

#include <stdlib.h>

#include "kwlaninterface.h"        

#include "wlanlib.h"
#include "configuration.h"
#include "kstartsupplicantdlg.h"
#include "kwlansuprocess.h"
#include "keventhistorydlg.h"
#include "kprofilesdlg.h"
#include "kscandlg.h"
#include "kprofileconfigdlg.h"
#include "kwlansettingsdlg.h"
#include "globals.h"


#include <kmessagebox.h>
#include <ktempfile.h>
#include <kstandarddirs.h>
#include <klocale.h>
#include <dirent.h>
#include <qtimer.h>
#include <kdebug.h>
#include <kprocess.h>
#include <qregexp.h>
#include <kio/global.h>
#include <kstandarddirs.h>
#include <qdir.h>
#include <qcheckbox.h>

#include <linux/if.h>
#include <sys/ioctl.h>

KwlanInterface::KwlanInterface()
{
    // initialise variables
    //Notifier = NULL;
    m_controlConnection = NULL;
    m_monitorConnection = NULL;
    m_controlInterfaceDir = strdup ("/var/run/wpa_supplicant");
    m_IpAddress = FALSE;
    m_associated = FALSE;
    m_wpaStarted = FALSE;
    m_userDataRequest = NULL;
    m_useDhclient = FALSE;
    m_useDhcpcd = FALSE;
    m_dhcpEnabled = TRUE;
    m_ifconfigProcess = 0L;
    m_iwconfigProcess = 0L;
    m_dhcpProcess = 0L;
    m_releaseProcess=0L;
    m_wpaWarned = FALSE;
    m_ipConfigured = FALSE;
    m_ssid = "";
    m_type = ETHERNET;
    m_configuration=0L;
    m_eventHistory = 0L;
    m_scanDlg = 0L;
    m_trayWin = 0L;
    m_fdSock = -1;
    KMessageBox::setDontShowAskAgainConfig(mainConfiguration().m_configuration);
    //get path to wpa_supplicant
    m_wpaPath = getPath("wpa_supplicant");
    if (m_wpaPath.isEmpty()){
        KMessageBox::error(0,i18n("Could not find wpa_supplicant binary. Without it Kwlan can't connect to encrypted networks.\n"
                "Please check your wpa_supplicant installation."));
    }
    //Check for dhcp client to use
    m_dhcpcdPath = getPath("dhcpcd");
    if (!m_dhcpcdPath.isEmpty()){
        m_useDhcpcd = TRUE;
    } else {
        m_dhclientPath = getPath("dhclient");
        if (!m_dhclientPath.isEmpty())
            m_useDhclient = TRUE;
    }
    if (! m_useDhclient && ! m_useDhcpcd)
        KMessageBox::sorry(0,i18n("Neither dhclient nor dhcpcd found. Dynamic IP configuration will not work."));
    // check for path to needed applications
    m_ifconfigPath = getPath("ifconfig");
    if (m_ifconfigPath.isEmpty())
        kdDebug() << i18n("ifconfig not found. Network statistics and static ip addresses will not work.") << endl;
    m_iwconfigPath = getPath("iwconfig");
    if (m_iwconfigPath.isEmpty())
        kdDebug() << i18n("iwconfig not found. Wireless statistics might not work.") << endl;

}

void KwlanInterface::init(QString interface)
{
    //char buf[10];
    //size_t len;
    if (!interface) return;
    m_sysClassPath="/sys/class/net/"+interface+"/";
    memset(&m_devInfo, 0, sizeof(m_devInfo));
    strcpy(m_devInfo.ifr_name, interface.latin1());
    m_confFile = QString("kwlanrc.") + interface;
    m_configureInterface = mainConfiguration().readConfigureInterface( interface);
    // find out if we have a wireless interface
    m_controlInterface = strdup(interface.ascii());
    QStringList wirelessInterfaces = getInterfaces();
    if (wirelessInterfaces.findIndex(m_controlInterface)!=-1)
        m_interfaceData.wirelessDevice = TRUE;
    else m_interfaceData.wirelessDevice = FALSE;
    // check for File existance to see wether interface is already configured
    QString tmpConf = KStandardDirs().findResource("config",m_confFile);
    m_configuration = new Configuration(m_confFile);
    // Check for interface type
    checkType();
    if ((tmpConf.isEmpty() || !hasProfiles() && m_type != PPP) && KMessageBox::shouldBeShownContinue("newinterfacewizard"))
    {
        // show wizard
        newInterfaceWizard(interface);
    }
    
    QStringList binDirs;
    if (m_type == ETHERNET)
    {
        // check if we have a custom wpa config file
        if (m_configuration && m_configuration->m_useCustomWpaConf)
        {
            m_wpaConf = m_configuration->m_customWpaConf;
        }
        else 
        {
            // check if wpa_config file for interface exists
            m_wpaConf = QString("kwlan.")+ interface; 
            if (locate("config",m_wpaConf).isEmpty())
            {
            //create a new config file containig the important global settings
                KConfig *conf= new KConfig(m_wpaConf,false,false,"config");
                conf->writeEntry("update_config","1");
                conf->writeEntry("ctrl_interface",m_controlInterfaceDir);
                conf->writeEntry("ctrl_interface_group","20");
                delete conf;
            }
            m_wpaConf = locate("config",m_wpaConf);
        }
    }
    // To avoid Getting an IP Address if interface already has one, check ip address
    // We set m_ipConfigured to true to avoid checkConfig getting an ip address
    // It's a little ugly and needs some rework
    m_ipConfigured = TRUE;
    checkConfig();
    if (m_interfaceData.ipAddress.isEmpty() ||  m_interfaceData.ipAddress=="0.0.0.0") m_ipConfigured = FALSE;
    else m_ipConfigured = TRUE;
    if (::debugOutput) kdDebug() << "IP Address " << m_interfaceData.ipAddress  << "M_ipconfigured " << m_ipConfigured << endl;

    //check for existing  WPA connection
    checkConnection();
    //Start WPA Supplicant if configured and no connection is established
    if (m_configuration)
    {
        if (m_interfaceData.wirelessDevice)
        {
            if (!m_wpaStarted && m_configuration->m_startWpa && m_configureInterface){
                KStartSupplicantDlg *startSupplicant = new KStartSupplicantDlg();
                QString driver;
                m_configuration->readDriver(driver);
                if (startSupplicant ){
                    startSupplicant->setData(&interface,&driver);
                    if (m_configuration->m_startWpaNoQuestion&& !driver.isEmpty()){
                        startSupplicant->startSupplicant();
                    }
                    else {
                        startSupplicant->show();
                        startSupplicant->exec();
                    }
                    QString wpa_supplicant;
                    if (startSupplicant->m_start){
                        KwlanSuProcess *startWpa= new KwlanSuProcess(NULL,"Start WPA Supplicant");
                        *startWpa << m_wpaPath << QString("-i%1").arg(QString(m_controlInterface)) << QString ("-D%1").arg(startSupplicant->getDriver()) << QString("-c")+ m_wpaConf << "-B";
                        startWpa->setDescription(i18n("Start WPA Supplicant"));
                        startWpa->start();
                        m_configuration->writeDriver(startSupplicant->getDriver());
                    }
                    delete startSupplicant;
                    m_networkChange = true;
                }
            }
        }
        else {
            if ((!m_wpaStarted && m_configuration->m_startWpa && m_configureInterface)&&(m_type == ETHERNET )){
                //wired device, so drvier is wired
                KwlanSuProcess *startWpa= new KwlanSuProcess(NULL,"Start WPA Supplicant");
                *startWpa << m_wpaPath << QString("-i%1").arg(QString(m_controlInterface)) << QString ("-D%1").arg(QString("wired")) << QString("-c" ) + m_wpaConf <<"-B";
                startWpa->setDescription(i18n("Start WPA Supplicant"));
                startWpa->start();
            }
        }
    }
    // Set Status update timer
    if (m_type ==ETHERNET) 
    {
        if (m_configuration) m_configuration->readLastSsid(m_ssid);
    }
    getIpData();
    // Now check if we have to use static ip addressing 
    // If so, scheck whether interface address is equal to configured address
    if (!m_dhcpEnabled)
    {
        if (m_staticIpAddress!=m_interfaceData.ipAddress)
        {
            m_ipConfigured = FALSE;
            checkConfig();
        }
    }
    emit (networksChanged());
    connect(this, SIGNAL(wpaStatusChanged( bool )), this, SLOT(slotWpaStatusChanged(bool)));
    m_timer = new QTimer(this);
    int timerInterval;
    timerInterval = 100;
    if (m_timer){
        connect (m_timer, SIGNAL(timeout()),SLOT(checkConnection()));
        m_timer->start(timerInterval,FALSE);
    }

}
        
KwlanInterface::~KwlanInterface()
{
    if (m_timer){
        m_timer->stop();
        delete (m_timer);
    } 
    if (m_ifconfigProcess) 
    {
        m_ifconfigProcess->kill();
        delete (m_ifconfigProcess);
    }
    if (m_iwconfigProcess) 
    {
        m_iwconfigProcess->kill();
        delete (m_iwconfigProcess);
    }
}
         
void KwlanInterface::slotWpaStatusChanged(bool status)
{
    if (status)
        profileActivate(m_ssid);
}

void KwlanInterface::cleanTemporaryFiles()
{
    resolvconf->unlink();
    if (resolvconf) {
        delete resolvconf;
        resolvconf = 0;
    }
}
        
int KwlanInterface::openControlConnection()
{
    bool tmpWpaStarted = m_wpaStarted;
    char *cfile;
    int flen;

    if ((m_controlInterface == NULL) || (m_type==PPP))
        return -1;
    //kdDebug() << "Trying to open control connection" << endl;
    flen = strlen(m_controlInterfaceDir) + strlen(m_controlInterface) + 2;
    cfile = (char *) malloc(flen);
    if (cfile == NULL)
        return -1;
    snprintf(cfile, flen, "%s/%s", m_controlInterfaceDir, m_controlInterface);
    

    if (m_controlConnection) {
        wpa_ctrl_close(m_controlConnection);
        m_controlConnection = NULL;
    }

    if (m_monitorConnection) {
        delete m_messageNotifier;
        m_messageNotifier = NULL;
        wpa_ctrl_detach(m_monitorConnection);
        wpa_ctrl_close(m_monitorConnection);
        m_monitorConnection = NULL;
    }

    //kdDebug() << "Trying to connect to " <<  cfile << endl;
    m_controlConnection = wpa_ctrl_open(cfile);
    if (m_controlConnection == NULL) {
        //kdDebug() << "Opening control connection failed" << endl;
        delete cfile;
        return -1;
    }
    m_wpaStarted = TRUE;
    //kdDebug() << "Control connection established" << endl;
    if (!tmpWpaStarted)
    {
        emit (wpaStatusChanged(m_wpaStarted));
        emit (networksChanged());
    }
    m_monitorConnection = wpa_ctrl_open(cfile);
    delete cfile;
    if (m_monitorConnection == NULL) {
        wpa_ctrl_close(m_controlConnection);
        //kdDebug() << "Could not establish monitor connection." << endl;
        return -1;
    }
    if (wpa_ctrl_attach(m_monitorConnection)) {
        printf("Failed to attach to wpa_supplicant\n");
        wpa_ctrl_close(m_monitorConnection);
        m_monitorConnection = NULL;
        wpa_ctrl_close(m_controlConnection);
        m_controlConnection = NULL;
        return -1;
    }

    m_messageNotifier = new QSocketNotifier(wpa_ctrl_get_fd(m_monitorConnection),QSocketNotifier::Read, this);
    if (m_messageNotifier)
        connect(m_messageNotifier, SIGNAL(activated(int)), SLOT(receiveMsgs()));
    else kdDebug() << "Could not create Message Notifier" << endl;
    return 0;
    
}

void KwlanInterface::checkConnection()
{
    char buf[10];
    size_t len;
    QString status;
    len = sizeof(buf)-1;
    bool tmpWpaStarted = m_wpaStarted;
    if (m_type != PPP)
    {
        if (ctrlRequest("PING",buf,&len) <0)
        {
            m_wpaStarted=FALSE;
            if (tmpWpaStarted)
            {
                m_controlConnection=0L;
                m_monitorConnection=0L;
                emit wpaStatusChanged (FALSE);
            }
            // try to open the connection
            openControlConnection();
        }
        else {
            m_wpaStarted = TRUE;
            if (!tmpWpaStarted)
                emit wpaStatusChanged (TRUE);
        }
    }
    checkConfig();
}

int KwlanInterface::ctrlRequest(const char *command, char *buf, size_t *buflen)
{
    if (m_controlConnection == NULL)
        return -1;
    return wpa_ctrl_request(m_controlConnection, command, strlen(command), buf, buflen, kwlan_msg_cb);
}


QStringList KwlanInterface::listProfiles()
{
    QStringList wpaProfiles;
    if (m_type == PPP) return wpaProfiles;
    
    wpaProfiles = listWpaProfiles();
    //kdDebug() << "WPA PRofiles: " << wpaProfiles << endl;
    return wpaProfiles + listNonWpaProfiles(wpaProfiles);
}
        
QStringList KwlanInterface::listWpaProfiles()
{
    char buf[2048], *start, *end, *id, *ssid, *bssid, *flags;
    size_t len =sizeof(buf);
    QStringList networks;
    // Are we connected to wpa?
    if (!m_controlConnection) {
        //kdDebug() << "No control connection to list wpa profiles! "<< m_controlInterface << endl;
        return networks;
    }
    if (ctrlRequest("LIST_NETWORKS", buf, &len) < 0)
    {
        kdDebug() << "Could not list wpa profiles" << endl;
        return networks;
    }
    buf[len] = '\0';
    start = strchr(buf, '\n');
    if (start == NULL)
        return networks;
    start++;
    while (*start) {
        bool last = false;
        end = strchr(start, '\n');
        if (end == NULL) {
            last = true;
            end = start;
            while (end[0] && end[1])
                end++;
        }
        *end = '\0';
	
        id = start;
        ssid = strchr(id, '\t');
        if (ssid == NULL)
            break;
        *ssid++ = '\0';
        bssid = strchr(ssid, '\t');
        if (bssid == NULL)
            break;
        *bssid++ = '\0';
        flags = strchr(bssid, '\t');
        if (flags == NULL)
            break;
        *flags++ = '\0';
	
        QString tmpNetwork(ssid);
        networks.append (tmpNetwork);
        if (last)
            break;
        start = end + 1;
    }

    return networks;
}        
        
QStringList KwlanInterface::listNonWpaProfiles(QStringList wpaProfiles)
{
    bool encrypted;
    QStringList profiles;
    if (!m_configuration) return profiles;
    profiles = m_configuration->getNetworks();
    // Now test if networks already esist from wpa config
    QStringList::Iterator it;
    it = profiles.begin();
    while (it !=profiles.end())
    {
        QStringList::Iterator index = wpaProfiles.find( *it );
        if ( !(*index).isEmpty() )
        {
            it = profiles.remove(it);
            continue;
        }
            // Check whether the Network can be used without encryption
        m_configuration->readEncryption(*it,encrypted);
        if (encrypted) {
            it = profiles.remove(it);
            continue;
        }
        it++;
    }
    kdDebug() << profiles << endl;
    return profiles;
}

bool KwlanInterface::hasProfiles()
{
    if (!m_configuration) return FALSE;
    QStringList profiles;
    profiles = m_configuration->getNetworks();
    if (profiles.isEmpty()) return FALSE;
    return TRUE;
}

int KwlanInterface::getWpaId(QString findSsid)
{
    char buf[2048], *start, *end, *id, *ssid, *bssid, *flags;
    size_t len = sizeof(buf);
    QStringList networks;
    // Are we connected to wpa?
    if (!m_controlConnection) {
        //kdDebug() << "No control connection! " <<  endl;
        return  -1;
    }
    if (ctrlRequest("LIST_NETWORKS", buf, &len) < 0)
    {
        kdDebug() << "Getting list of networks failed! " <<  endl;
        return -1;
    }
    
    buf[len] = '\0';
    start = strchr(buf, '\n');
    if (start == NULL)
        return -1;
    start++;
    while (*start) {
        bool last = false;
        end = strchr(start, '\n');
        if (end == NULL) {
            last = true;
            end = start;
            while (end[0] && end[1])
                end++;
        }
        *end = '\0';
	
        id = start;
        ssid = strchr(id, '\t');
        if (ssid == NULL)
            break;
        *ssid++ = '\0';
        bssid = strchr(ssid, '\t');
        if (bssid == NULL)
            break;
        *bssid++ = '\0';
        flags = strchr(bssid, '\t');
        if (flags == NULL)
            break;
        *flags++ = '\0';
        QString tmpSsid =QString(ssid); 
        if (tmpSsid==findSsid)
        {
            return QString(id).toInt();
        }
        if (last)
            break;
        start = end + 1;
    }
    return -1;
}

bool KwlanInterface::profileActivateNonWpa(QString ssid)
{
    if (m_interfaceData.wirelessDevice)
    {
        KwlanSuProcess *iwconfig = new KwlanSuProcess(this);
        *iwconfig << m_iwconfigPath << m_controlInterface << QString("essid") << ssid;
        iwconfig->setDescription(i18n("Configure wlan interface using iwconfig"));
        iwconfig->start();
    }
   // store in config
    if (m_configuration) m_configuration->writeLastSsid( ssid);
    m_ssid = ssid;
    // set ip data
    getIpData();

    return TRUE;
    
}

bool KwlanInterface::profileActivateWpa(QString ssid)
{
        // select network using wpa_supplicant
    int ret;
    char reply[40], cmd[256];
    size_t reply_len = sizeof(reply);
    int id = getWpaId(ssid);
    if (id == -1 ){
        KMessageBox::sorry(0, i18n("Could not find network in wpa_supplicant\nconfiguration."));
        return FALSE;
    }
    snprintf(cmd, sizeof(cmd), "DISCONNECT");
    reply_len = sizeof(reply);
    ctrlRequest(cmd, reply, &reply_len);
 
    // Enable Network
    snprintf(cmd, sizeof(cmd), "SELECT_NETWORK %d", id);
    reply_len = sizeof(reply);
    ret=ctrlRequest(cmd, reply, &reply_len);
    if (ret==-1) {
        KMessageBox::sorry(0,"No control connection!");
        return FALSE;
    }
    if (strncmp(reply, "OK", 2) != 0) {
        KMessageBox::sorry(0, i18n("Failed to enable network in wpa_supplicant\nconfiguration."));
        return FALSE;
    } 
    // Connect
    snprintf(cmd, sizeof(cmd), "REASSOCIATE");
    reply_len = sizeof(reply);
    ctrlRequest(cmd, reply, &reply_len);
    // store in config
    if (m_configuration) m_configuration->writeLastSsid( ssid);
    // set ip data
    m_ssid = ssid;
    getIpData();
    
    checkConnection();
    return TRUE;   
}

bool KwlanInterface::profileActivate(QString ssid)
{
    if (!m_configureInterface) return FALSE;
    // release any existing ip address
    if (m_interfaceData.connected) releaseIpAddress();
    //m_interfaceData.available = FALSE;
    //m_interfaceData.connected = FALSE;
    if (m_controlConnection){
        return profileActivateWpa(ssid);
    }
    else {
        // select network using iwconfig
        return profileActivateNonWpa(ssid);
    }
}

void KwlanInterface::releaseIpAddress()
{
    if (!m_controlInterface || (m_type == PPP) || !m_configureInterface)  return;
    if (m_releaseProcess) return;
    // Now start script before disconnect if configured
    QString script;
    bool useRoot;
    m_configuration->readScriptBeforeDisconnect( m_ssid,script, useRoot);
    if (::debugOutput) kdDebug() << "Now starting script before disconnect "<< script << endl;
    if (!script.isEmpty())
    {
        if (useRoot) {
            KwlanSuProcess *proc = new KwlanSuProcess();
            *proc << script;
            proc->setDescription( i18n("start script before disconnect"));
            proc->start( );
        }
        else {
            KProcess *proc = new KProcess();
            *proc << script;
            proc->start();
        }
    }

    m_releaseProcess = new KwlanSuProcess(this);    
    if  (!m_dhcpEnabled) {
        *m_releaseProcess << "ifconfig" << m_controlInterface << "0.0.0.0";
    } else {
        if (m_useDhclient)
        {
            *m_releaseProcess << m_dhclientPath << "-r" << m_controlInterface;
        } else if (m_useDhcpcd){
            *m_releaseProcess << m_dhcpcdPath << "-k" << m_controlInterface;
        }
    }
    m_releaseProcess->setDescription(i18n("Release IP address"));
    connect (m_releaseProcess, SIGNAL(processExited(KwlanSuProcess*)), this, SLOT(releaseProcessExited(KwlanSuProcess*)));
    m_releaseProcess->start();
    //m_interfaceData.ipAddress="";
    //emit ipChanged( QString(""));
}

void KwlanInterface::setNetworkSettings()
{
    if (m_type == PPP) return;
    if (m_staticDns1.isEmpty() && m_staticDns2.isEmpty() && m_domain.isEmpty() && m_dnsSearchlist.isEmpty())
        return;
    if (!m_dontOverrideDns || !hasNameserver()) {
         // Create temporary file containing DNS configuration	    
        resolvconf = new KTempFile(locateLocal("tmp", "resolv"), ".conf", 0644);
        if (!m_staticDns1.isEmpty())
            *(resolvconf->textStream()) << QString("nameserver %1").arg(m_staticDns1) << endl;
        if (!m_staticDns2.isEmpty())
            *(resolvconf->textStream()) << QString("nameserver %1").arg(m_staticDns2) << endl;
        if (!m_domain.isEmpty())
            *(resolvconf->textStream()) << QString("domain %1").arg(m_domain) << endl;
        if (!m_dnsSearchlist.isEmpty())
        {
            *(resolvconf->textStream()) << QString("search ");
            for (QStringList::iterator it = m_dnsSearchlist.begin(); it != m_dnsSearchlist.end(); it++)
                *(resolvconf->textStream()) << *it << " ";
            *(resolvconf->textStream()) << endl;
        }
        resolvconf->close();
        KwlanSuProcess * setNetwork = new KwlanSuProcess();
        connect(setNetwork, SIGNAL( destroyed() ), this, SLOT( cleanTemporaryFiles() ));

	// And copy it to /etc fixing permissions (moving wont work if /etc/resolv.conf is a symlink)
        *setNetwork << getPath("cp") << resolvconf->name() << "/etc/resolv.conf";	    
        *setNetwork << ";";
        *setNetwork << getPath("chown") << "root:root" << "/etc/resolv.conf";
        setNetwork->setDescription(i18n("Set network settings"));
        setNetwork->start();
    }
}

void KwlanInterface::checkType()
{
    // check Interface type
    int type = readInterfaceNumValue( "type");
    if ( type==1)
    {
        m_type = ETHERNET;
    }
    else
    {
        m_type = PPP;
    }
}

void KwlanInterface::getIpAddress()
{
    if ((m_type == PPP)||(m_ssid.isEmpty()) || !m_configureInterface) return;
    if (m_dhcpProcess) 
    {
        // We are already running
        return;
    }
    m_dhcpProcess = new KwlanSuProcess(this);
    if (!m_dhcpEnabled){
        if (m_staticIpAddress.isEmpty()) return;
	
        m_dhcpProcess->setDescription(i18n("Configure the network"));	
        *m_dhcpProcess << m_ifconfigPath << m_controlInterface << m_staticIpAddress.ascii();	
	
        if (!m_staticNetmask.isEmpty())
            *m_dhcpProcess << "netmask" << m_staticNetmask.ascii();

	// We can to set up our new route if there is no route (hasGateway()==false) or we have but m_ovrgw is true ()
        bool m_hasGateway = hasGateway();
        if (!m_staticGateway.isEmpty() && (!m_dontOverrideGw || !m_hasGateway)) {
            *m_dhcpProcess << ";";
	    
	    // Delete default route if we already have one
            if (m_hasGateway)
                *m_dhcpProcess << getPath("route") << "del" << "default" << ";";
		
            *m_dhcpProcess << getPath("route") << "add" << "default" << "gw" << m_staticGateway.ascii();
        }    
    }
    else {
	
        if (m_useDhclient)
        {
            *m_dhcpProcess << m_dhclientPath <<  m_controlInterface;
        } else if (m_useDhcpcd){
            *m_dhcpProcess << m_dhcpcdPath << "-nd" << m_controlInterface;
        }
	
        m_dhcpProcess->setDescription(i18n("Obtain IP address dynamically"));
        m_dhcpProcess->setRestart(true);
    }
    connect (m_dhcpProcess,SIGNAL(processExited(KwlanSuProcess*)), this, SLOT(dhcpProcessExited(KwlanSuProcess*)));
    m_dhcpProcess->start();

}

void KwlanInterface::processMessage(char *message)
{
    char *pos = message, *pos2;
    int priority = 2;
    //kdDebug() << message << endl;
    if (*pos == '<') {
        /* skip priority */
        pos++;
        priority = atoi(pos);
        pos = strchr(pos, '>');
        if (pos)
            pos++;
        else
            pos = message;
    }

    WpaMsg wm(pos, priority);
    emit wpaEvent(wm);
    m_messages.append(wm);
    while (m_messages.count() > 100)
        m_messages.pop_front();
    
    /* Update last message with truncated version of the event */
    if (strncmp(pos, "CTRL-", 5) == 0) {
        pos2 = strchr(pos, str_match(pos, WPA_CTRL_REQ) ? ':' : ' ');
        if (pos2)
            pos2++;
        else
            pos2 = pos;
    } else
        pos2 = pos;
        QString lastmsg = pos2;
        lastmsg.truncate(40);
    //textLastMessage->setText(lastmsg);
    //pingsToStatusUpdate = 0;
        m_networkChange = true;
    
        if (str_match(pos, WPA_CTRL_REQ))
            processCtrlReq(pos + strlen(WPA_CTRL_REQ));
        if (str_match(pos,WPA_EVENT_CONNECTED)){
        // getIpAddress
        //    if (!m_associated) {
        //        getIpAddress();
        //    }
            m_associated = true;
            //kdDebug() << "Connection established." << endl;
            //emit connectionStatusChanged( TRUE);
        }
        if (str_match(pos,WPA_EVENT_DISCONNECTED)){
            m_associated = false;
        }
}

void KwlanInterface::receiveMsgs()
{
    char buf[256];
    size_t len;
    if (!m_monitorConnection) return;
    while (wpa_ctrl_pending(m_monitorConnection)) {
        len = sizeof(buf) - 1;
        if (wpa_ctrl_recv(m_monitorConnection, buf, &len) == 0) {
            buf[len] = '\0';
            processMessage(buf);
        }
    }
}

void KwlanInterface::processCtrlReq( const char * request )
{
    // processe control request, for example ask for password for private key file.
    if (m_userDataRequest) {
        m_userDataRequest->close();
        delete m_userDataRequest;
    }
    m_userDataRequest = new KUserDataRequestDlg();
    if (m_userDataRequest == NULL)
        return;
    if (m_userDataRequest->setParams(this, request) < 0) {
        delete  m_userDataRequest;
        m_userDataRequest = NULL;
        return;
    }
    m_userDataRequest->show();
    m_userDataRequest->exec();
}

void KwlanInterface::startWpa(QString driver)
{
    if (m_wpaStarted) {
        kdDebug() << "wpa_supplicant is already started!" << endl;
        return;
    }
    if (m_configuration) 
    {
        if (m_configuration->m_useCustomWpaConf)
            m_wpaConf = m_configuration->m_customWpaConf;
        else {
            m_wpaConf = QString("kwlan.")+ m_controlInterface; 
            m_wpaConf = locate("config",m_wpaConf);
        }
    }
    KwlanSuProcess *startWpa = new KwlanSuProcess(this);
    *startWpa << m_wpaPath << QString("-i%1").arg(QString(m_controlInterface)) << QString ("-D%1").arg(driver) << QString("-c") +m_wpaConf << "-B";
    startWpa->setDescription(i18n("Start WPA Supplicant"));
    startWpa->start();
	//remember last interface driver
    if (m_configuration) m_configuration->writeDriver(driver);
    //emit (wpaStatusChanged(TRUE));
    emit (networksChanged());
}

void KwlanInterface::stopWpa()
{
    char reply[100];
    size_t reply_len;
    if (!m_wpaStarted) {
        kdDebug() << "wpa_supplicant is not started!" << endl;
        return;
    }
    if (::debugOutput) kdDebug() << "Releasing IP address" << endl;
    releaseIpAddress();
    reply_len = sizeof(reply);
    if (::debugOutput) kdDebug() << "Stopping wpa_supplicant" << endl;
    ctrlRequest("TERMINATE", reply, &reply_len);
    if (strncmp(reply,"OK",2) != 0) {
        KMessageBox::sorry(0,  i18n("Could not terminate WPA_Supplicant!"));
        return;
    }
    if (::debugOutput) kdDebug() << "wpa_supplicant stopped" << endl;
    //m_wpaStarted = FALSE;
    //m_controlConnection = 0L;
    //emit (wpaStatusChanged(FALSE));
    emit (networksChanged());
}

QString KwlanInterface::getInterfaceName()
{
    return QString(m_controlInterface);
}


void KwlanInterface::profileAdded()
{
    emit (networksChanged());
}

void KwlanInterface::checkConfig()
{
    bool tmpAvailable = m_interfaceData.available;
    bool tmpUp = m_interfaceData.up;
    bool tmpConnected = m_interfaceData.connected;
    FILE* flags_fp = fopen((m_sysClassPath+"flags").latin1(), "r");
    bool currentStatus;
    if (!flags_fp)
    {
        // Interface seems to have gone away
        emit (interfaceGone(this));
        deleteLater();
        return;
    }
    else {
        int flags;
        fscanf(flags_fp, "%Xu", &flags);
        fclose(flags_fp);
        currentStatus = IFF_UP & flags;
    }
    if (!currentStatus && m_interfaceData.up) { // interface down...
        // Interface is down and was up before -> set status variables
        m_interfaceData.up = FALSE;
        m_interfaceData.connected = FALSE;
    } else if (currentStatus) {
        // Interface is up, so check for connection status
        m_interfaceData.up = TRUE;
        // Now get connection status
        FILE* carrier_fp = fopen((m_sysClassPath+"carrier").latin1(), "r");
        char carrierFlag = '1';
        if (carrier_fp) {
            carrierFlag = fgetc(carrier_fp);
            fclose(carrier_fp);
        }
        if (carrierFlag == '0') { // carrier down
            // Interface is not connected, so return;
            if (m_interfaceData.connected) {
                m_interfaceData.connected = FALSE;
                say(i18n("%1 is disconnected").arg(m_controlInterface));
            }
        } else if (!m_interfaceData.connected) { 
            // Interface is connected
            m_interfaceData.connected = TRUE;
            say(i18n("%1 is connected").arg(m_controlInterface));
        }
        if (m_interfaceData.wirelessDevice)
        {
            long retval = readInterfaceNumValue("device/power/state");
            if (retval==0)
                m_wirelessData.radioOff = FALSE;
            else m_wirelessData.radioOff = TRUE;
        }
        if (m_interfaceData.connected)
        {
        // Now get interface data
            unsigned long currentRxBytes =  readInterfaceNumValue("statistics/rx_bytes");
            if ( currentRxBytes < m_interfaceData.prevRxBytes )
            {
            // there was an overflow
                m_interfaceData.rxBytes = currentRxBytes;
                m_interfaceData.prevRxBytes = 0L;
            }
            if ( m_interfaceData.rxBytes == 0L )
            {
            // on startup set to currently received bytes
                m_interfaceData.rxBytes = currentRxBytes;
                m_interfaceData.prevRxBytes = currentRxBytes;
            }
            else
            // afterwards only add difference to previous number of bytes
            {
                m_interfaceData.prevRxBytes = m_interfaceData.rxBytes;
                m_interfaceData.rxBytes = currentRxBytes;
            }

            //m_interfaceData.incomingBytes = currentRxBytes - m_interfaceData.prevRxBytes;
            //m_interfaceData.prevRxBytes = currentRxBytes;
            m_interfaceData.rxString = KIO::convertSize( m_interfaceData.rxBytes );
        
            unsigned long currentTxBytes = readInterfaceNumValue("statistics/tx_bytes");
            if ( currentTxBytes < m_interfaceData.prevTxBytes )
            {
            // there was an overflow
                m_interfaceData.txBytes = currentTxBytes;
                m_interfaceData.prevTxBytes = 0L;
            }
            if ( m_interfaceData.txBytes == 0L )
            {
            // on startup set to currently transmitted bytes
                m_interfaceData.txBytes = currentTxBytes;
                m_interfaceData.prevTxBytes = currentTxBytes;
            }
            else
            {
                m_interfaceData.prevTxBytes = m_interfaceData.txBytes;
                m_interfaceData.txBytes = currentTxBytes;
            }

            //m_interfaceData.outgoingBytes = currentTxBytes - m_interfaceData.prevTxBytes;
            //m_interfaceData.prevTxBytes = currentTxBytes;
            m_interfaceData.txString = KIO::convertSize( m_interfaceData.txBytes );

            m_interfaceData.rxPackets = readInterfaceNumValue("statistics/rx_packets");
            m_interfaceData.txPackets = readInterfaceNumValue("statistics/tx_packets");
            // Now get IP Address
            if (!((m_fdSock == -1) && !openFdSocket()))
            {
                memset(&m_devInfo, 0, sizeof(m_devInfo));
                strcpy(m_devInfo.ifr_name, m_controlInterface);
                ioctl(m_fdSock, SIOCGIFADDR, &m_devInfo);
                sockaddr_in sin = ((sockaddr_in&)m_devInfo.ifr_addr);
                m_interfaceData.ipAddress = inet_ntoa(sin.sin_addr);
                ioctl(m_fdSock, SIOCGIFNETMASK, &m_devInfo);
                sockaddr_in mask = ((sockaddr_in&)m_devInfo.ifr_netmask);
                m_interfaceData.subnetMask = inet_ntoa(mask.sin_addr);
                ioctl(m_fdSock, SIOCGIFBRDADDR, &m_devInfo);
                sockaddr_in broadcast = ((sockaddr_in&)m_devInfo.ifr_broadaddr);
                m_interfaceData.broadcastAddress = inet_ntoa(broadcast.sin_addr);

            }
            else {
                m_interfaceData.ipAddress = "";
                m_interfaceData.subnetMask="";
                m_interfaceData.broadcastAddress="";
            }
       //Now get wireless data
            if (m_interfaceData.wirelessDevice)
            {
                QString filename = m_sysClassPath+"device/bssinfo";
                QFile bssFile (filename);
                if (bssFile.open(IO_ReadOnly))
                {
                    QTextStream stream (&bssFile);
                    QString line="";
                    while (!stream.atEnd() && !line.isNull())
                    {
                        line = stream.readLine();
                        if (!line.isNull())
                        {
                            QRegExp regExp;
                            regExp.setPattern( "ESSID: (\\w*)" );
                            if ( regExp.search( line ) > -1 )
                                m_wirelessData.essid= regExp.cap( 1 );
                            regExp.setPattern( "Channel: (\\d*)" );
                            if ( regExp.search( line ) > -1 )
                                m_wirelessData.channel = regExp.cap( 1 );
                        }
                    }
                    bssFile.close();
                }
                unsigned long link = readInterfaceNumValue( "wireless/link");
                m_wirelessData.linkQuality = QString("%1/100").arg(link);
                m_wirelessData.linkQualityNum = link;
            }
        }
    }
    //if (tmpAvailable != m_interfaceData.available) 
    //    emit (connectionStatusChanged (m_interfaceData.available));
    if (tmpUp != m_interfaceData.up)
        emit (interfaceUp(m_interfaceData.up ));
    if (m_interfaceData.up && m_interfaceData.connected && !m_ipConfigured)
        getIpAddress();
}


void KwlanInterface::dhcpProcessExited(KwlanSuProcess *process)
{
    if (process == m_dhcpProcess)
    {
        delete m_dhcpProcess;    
        m_ipConfigured = TRUE;
        m_dhcpProcess =0L;
        if (::debugOutput) kdDebug() << "Now starting script after connect" << endl;
        // set Network settings like DNS etc
        setNetworkSettings();
        //start script after connect
        startScriptAfterConnect();
    }
}

void KwlanInterface::releaseProcessExited(KwlanSuProcess *process)
{
    if (process == m_releaseProcess)
    {
        delete m_releaseProcess;    
        m_ipConfigured = FALSE;
        m_releaseProcess =0L;
    }
}

void KwlanInterface::iwconfigProcessExited( KProcess* process )
{
    if ( process == m_iwconfigProcess )
    {
        delete m_iwconfigProcess;
        m_iwconfigProcess = 0L;
        parseIwconfigOutput();
    }
}

void KwlanInterface::iwconfigProcessStdout( KProcess*, char* buffer, int buflen )
{
    m_iwconfigStdout += QString::fromLatin1( buffer, buflen );
}

void KwlanInterface::parseIwconfigOutput()
{
    /* mIwconfigStdout contains the complete output of 'iwconfig' which we
    * are going to parse here.
    */
    QMap<QString, QString> configs;
    QStringList ifList = QStringList::split( "\n\n", m_iwconfigStdout );
    QStringList::Iterator it;
    QString key;
    for ( it = ifList.begin(); it != ifList.end(); ++it )
    {
        int index = ( *it ).find( ' ' );
        if ( index == -1 )
            continue;
        key = ( *it ).left( index );
        configs[key] = ( *it ).mid( index );
    }
    if ( configs.find( m_controlInterface) == configs.end() )
    {
        // The interface does not exist. Meaning the driver
        // isn't loaded and/or the interface has not been created.
        m_interfaceData.existing = false;
        m_interfaceData.available = false;
    }

    if ( configs[m_controlInterface].contains( "no wireless extensions" ) )
    {
        // The interface isn't a wireless device.
        m_interfaceData.wirelessDevice = false;
    }
    else
    {
        // Update the wireless data of the interface.
        m_interfaceData.wirelessDevice = true;
        updateWirelessData( configs[m_controlInterface] );
    }
}

void KwlanInterface::updateWirelessData( QString& config )
{
    QRegExp regExp( "radio off" );
    if ( regExp.search( config ) > -1 ) {
        m_wirelessData.radioOff = TRUE;
        return; //The rest of the information isn't needed when radio is off.
    } else
        m_wirelessData.radioOff = FALSE;

    regExp.setPattern( "ESSID:\"?([^\"]*)\"?" );
    if ( regExp.search( config ) > -1 )
        m_wirelessData.essid = regExp.cap( 1 );

    regExp.setPattern( "Mode:(\\w*)" );
    if ( regExp.search( config ) > -1 )
        m_wirelessData.mode = regExp.cap( 1 );

    regExp.setPattern( "Frequency:([\\w|\\.]*)" );
    if ( regExp.search( config ) > -1 )
        m_wirelessData.frequency = regExp.cap( 1 );
    else
    {
        regExp.setPattern( "Channel:(\\d*)" );
        if ( regExp.search( config ) > -1 )
            m_wirelessData.channel = regExp.cap( 1 );
    }

    regExp.setPattern( "Bit Rate[=:]([\\w/]*)" );
    if ( regExp.search( config ) > -1 )
        m_wirelessData.bitRate = regExp.cap( 1 );

    regExp.setPattern( "Signal level.(-?\\d+\\s*\\w+)" );
    if ( regExp.search( config ) > -1 )
        m_wirelessData.signal = regExp.cap( 1 );

    regExp.setPattern( "Noise level.(-?\\d+\\s*\\w+)" );
    if ( regExp.search( config ) > -1 )
        m_wirelessData.noise = regExp.cap( 1 );

    regExp.setPattern( "Link Quality[=:]([\\d/]*)" );
    if ( regExp.search( config ) > -1 )
        m_wirelessData.linkQuality = regExp.cap( 1 );
}

void KwlanInterface::updateInterfaceData( QString& config, int type )
{
    bool hasIpAddress = TRUE;
    if (m_interfaceData.ipAddress.isEmpty()) hasIpAddress = FALSE;
    QRegExp regExp( ".*RX.*:(\\d+).*:\\d+.*:\\d+.*:\\d+" );
    if ( regExp.search( config ) > -1 )
        m_interfaceData.rxPackets = regExp.cap( 1 ).toULong();

    regExp.setPattern( ".*TX.*:(\\d+).*:\\d+.*:\\d+.*:\\d+" );
    if ( regExp.search( config ) > -1 )
        m_interfaceData.txPackets = regExp.cap( 1 ).toULong();

    regExp.setPattern( "RX bytes:(\\d+)\\s*\\(\\d+\\.\\d+\\s*\\w+\\)" );
    if ( regExp.search( config ) > -1 )
    {
        // We count the traffic on ourself to avoid an overflow after
        // 4GB of traffic.
        unsigned long currentRxBytes = regExp.cap( 1 ).toULong();
        if ( currentRxBytes < m_interfaceData.prevRxBytes )
        {
            // there was an overflow
            m_interfaceData.rxBytes = currentRxBytes;
            m_interfaceData.prevRxBytes = 0L;
        }
        if ( m_interfaceData.rxBytes == 0L )
        {
            // on startup set to currently received bytes
            m_interfaceData.rxBytes = currentRxBytes;
            m_interfaceData.prevRxBytes = currentRxBytes;
        }
        else
            // afterwards only add difference to previous number of bytes
        {
            m_interfaceData.prevRxBytes = m_interfaceData.rxBytes;
            m_interfaceData.rxBytes = currentRxBytes;
        }

            //m_interfaceData.incomingBytes = currentRxBytes - m_interfaceData.prevRxBytes;
            //m_interfaceData.prevRxBytes = currentRxBytes;
        m_interfaceData.rxString = KIO::convertSize( m_interfaceData.rxBytes );
    }


    regExp.setPattern( "TX bytes:(\\d+)\\s*\\(\\d+\\.\\d+\\s*\\w+\\)" );
    if ( regExp.search( config ) > -1 )
    {
        // We count the traffic on ourself to avoid an overflow after
        // 4GB of traffic.
        unsigned long currentTxBytes = regExp.cap( 1 ).toULong();
        if ( currentTxBytes < m_interfaceData.prevTxBytes )
        {
            // there was an overflow
            m_interfaceData.txBytes = currentTxBytes;
            m_interfaceData.prevTxBytes = 0L;
        }
        if ( m_interfaceData.txBytes == 0L )
        {
            // on startup set to currently transmitted bytes
            m_interfaceData.txBytes = currentTxBytes;
            m_interfaceData.prevTxBytes = currentTxBytes;
        }
        else
        {
            m_interfaceData.prevTxBytes = m_interfaceData.txBytes;
            m_interfaceData.txBytes = currentTxBytes;
        }

            //m_interfaceData.outgoingBytes = currentTxBytes - m_interfaceData.prevTxBytes;
            //m_interfaceData.prevTxBytes = currentTxBytes;
        m_interfaceData.txString = KIO::convertSize( m_interfaceData.txBytes );
    }
  

    regExp.setPattern( "inet\\s+\\w+:(\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}\\.\\d{1,3})" );
    if ( regExp.search( config ) > -1 )
    {
        m_interfaceData.ipAddress = regExp.cap( 1 );
        //if (!hasIpAddress) startScriptAfterConnect();
    }
    else m_interfaceData.ipAddress = "";
    regExp.setPattern( "(\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}).*(\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}).*(\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}\\.\\d{1,3})" );
    if ( regExp.search( config ) > -1 )
    {
        m_interfaceData.broadcastAddress = regExp.cap( 2 );
        m_interfaceData.subnetMask = regExp.cap( 3 );
    }
    else {
        m_interfaceData.broadcastAddress = "";
        m_interfaceData.subnetMask = "";
    }
    

    if ( type == ETHERNET )
    {
        regExp.setPattern( "(.{2}:.{2}:.{2}:.{2}:.{2}:.{2})" );
        if ( regExp.search( config ) > -1 )
            m_interfaceData.hwAddress = regExp.cap( 1 );
    }
    else if (  type == PPP )
    {
        regExp.setPattern( "(\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}).*(\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}).*(\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}\\.\\d{1,3})" );
        if ( regExp.search( config ) > -1 )
            m_interfaceData.ptpAddress = regExp.cap( 2 );
    }
}

void KwlanInterface::logViewer()
{
    //kdDebug()<<"Log Viewer called" << endl;
    if (m_eventHistory) {
        m_eventHistory->close();
        delete m_eventHistory;
    }

    m_eventHistory = new KEventHistoryDlg();
    if (m_eventHistory == NULL)
        return;
    connect (this, SIGNAL (wpaEvent(WpaMsg)), m_eventHistory,SLOT (addEvent(WpaMsg)));
    connect (this, SIGNAL (wpaEvents(WpaMsgList)), m_eventHistory,SLOT (addEvents(WpaMsgList)));
    emit (wpaEvents( m_messages));
    m_eventHistory->show();
    m_eventHistory->exec();
}

void KwlanInterface::getIpData()
{
    if (!m_configuration) return;
    QString ipAddress,netmask,dns1,dns2,gateway, domain;
    QStringList dnsSearchlist;
    bool dontOverrideGw,dontOverrideDns, dhcpEnabled;
    int ret;
    ret = m_configuration->readIpSettings(m_ssid,dhcpEnabled, ipAddress, netmask, gateway, dns1, dns2,domain,dontOverrideGw,dontOverrideDns,dnsSearchlist);
    if (ret != 0){
        m_dhcpEnabled = true; //default will be dhcp
        m_staticIpAddress="";
        m_staticNetmask = "";
        m_staticGateway="";
        m_staticDns1 = "";
        m_staticDns2="";
        m_dontOverrideDns=TRUE;
        m_dontOverrideGw=TRUE;
        m_domain = "";
        m_dnsSearchlist.clear();
    }
    else {
        m_dhcpEnabled = dhcpEnabled;
        m_staticIpAddress= ipAddress;
        m_staticNetmask = netmask;
        m_staticGateway=gateway;
        m_staticDns1 = dns1;
        m_staticDns2=dns2;
        m_dontOverrideGw=dontOverrideGw;
        m_dontOverrideDns=dontOverrideDns;
        m_domain=domain;
        m_dnsSearchlist = dnsSearchlist;
    }
}

void KwlanInterface::enableInterface( bool enable)
{
    if (!m_controlInterface || !m_configureInterface) return;
    KwlanSuProcess *ifup = new KwlanSuProcess();
    if (enable) {
        *ifup << m_ifconfigPath << m_controlInterface  << "up";
        ifup->setDescription( i18n("Enable interface"));
        ifup->start( );
    }
    else if (m_type != PPP){
        releaseIpAddress(); // Don't release IP Address as it will reenable the interface if 
        // it is run after setting the interface to down state.
        *ifup << m_ifconfigPath << m_controlInterface << "down";
        ifup->setDescription( i18n("Disable interface"));
        ifup->start( );
    }
}


void KwlanInterface::editProfiles()
{
    KProfilesDlg *dlg = new KProfilesDlg();
    dlg->setInterface( this);
    dlg->show();
    dlg->exec();
}

void KwlanInterface::slotScan()
{
    if (m_scanDlg) {
        m_scanDlg->close();
        delete m_scanDlg;
    }

    m_scanDlg = new KScanDlg();
    if (m_scanDlg == NULL)
        return;
    m_scanDlg->setInterface(this);
    m_scanDlg->show();
    m_scanDlg->exec();
}

void KwlanInterface::slotNewProfile()
{
    KProfileConfigDlg *profileDlg = new KProfileConfigDlg();
    if (!profileDlg) return;
    profileDlg->setInterface(this);
    profileDlg->newNetwork();
    profileDlg->show();
    profileDlg->exec();
    emit (networksChanged());
    //updateNetworks();
}

void KwlanInterface::slotEditProfile(QString profile)
{
    
    //int wpaId = getWpaId( profile);    
    
    // Open Config Dialog
    KProfileConfigDlg *profileDlg = new KProfileConfigDlg();
    if (!profileDlg) return;
    profileDlg->setInterface(this);
    profileDlg->paramsFromConfig(this, profile);
    //profileDlg->show();
    profileDlg->exec();
}

void KwlanInterface::slotDeleteProfile(QString profile)
{
    char cmd[256],reply[100];
    size_t reply_len = sizeof(reply);
    if (!m_controlConnection && !m_wpaStarted) {
        int result = KMessageBox::questionYesNo(0,"wpa_supplicant is not started. Only the non-wpa related network settings will be deleted.\n"
                "Do you want to delete these settings?");
        if (result == KMessageBox::Yes){
            m_configuration->deleteNetwork( profile);
        }
        return; 
    }
    int id = getWpaId(profile);
    snprintf(cmd, sizeof(cmd), "REMOVE_NETWORK %i", id);
    reply_len = sizeof(reply);
    ctrlRequest(cmd, reply, &reply_len);
    if (strncmp(reply, "OK", 2) != 0) {
        KMessageBox::sorry(0, i18n("Failed to remove network from wpa_supplicant\nconfiguration."));
    } else {
        reply_len = sizeof(reply);
        ctrlRequest("SAVE_CONFIG", reply, &reply_len);
        if (strncmp(reply,"OK",2) != 0) {
            KMessageBox::sorry(0, "Failed to save the  wpa_supplicant configuration.\nIs update_config=1 defined in wpa_supplicant.conf?");
        }
    }
    m_configuration->deleteNetwork( profile);
    emit (networksChanged());
}

void KwlanInterface::startScriptAfterConnect()
{
    // Now start script after connect if configured
    QString script;
    bool useRoot;
    m_configuration->readScriptAfterConnect( m_ssid,script, useRoot);
    if (::debugOutput) kdDebug() << "Now starting script after connect "<< script << endl;
    if (!script.isEmpty())
    {
        if (useRoot) {
            KwlanSuProcess *proc = new KwlanSuProcess();
            *proc << script;
            proc->setDescription( i18n("start script after connect"));
            proc->start( );
        }
        else {
            KProcess *proc = new KProcess();
            *proc << script;
            proc->start();
        }
    }
}

unsigned long KwlanInterface::readInterfaceNumValue(const char* name) 
{
    // stdio functions appear to be more fast than QFile?
    FILE* fp = fopen((m_sysClassPath+name).latin1(), "r");
    if (!fp) return 0;
    long retval;
    fscanf(fp, "%lu", &retval);
    fclose(fp);
    return retval;
}

bool KwlanInterface::openFdSocket() {
    if (m_fdSock > 0)
        return TRUE;
    if ((m_fdSock = socket(AF_INET, SOCK_DGRAM, 0)) > 0)
        return TRUE;
    return FALSE;
}

void KwlanInterface::say(QString message) {
    emit sigMessage(message);
}


void KwlanInterface::newInterfaceWizard(QString interface)
{
    int res = KMessageBox::warningContinueCancel(0,QString(i18n("New interface found!\n%1\n Click Continue to configure it now.")).arg(interface),"Kwlan",
            KStdGuiItem::cont(),"newinterfacewizard");
    KMessageBox::saveDontShowAgainContinue("newinterfacewizard");
    if (res == KMessageBox::Continue)
    {
        //User wants to configure interface
        int result = KMessageBox::questionYesNo(0, i18n("Do you want to start wpa_supplicant for this interface?"));
        if (result == KMessageBox::Yes)
        {
            KStartSupplicantDlg *startSupplicant = new KStartSupplicantDlg();
            QString driver;
            if (m_interfaceData.wirelessDevice) driver = "wext";
            else driver = "wired";
            if (startSupplicant ){
                startSupplicant->setData(&interface,&driver);
//                    startSupplicant->show();
                startSupplicant->exec();
            }
            QString wpa_supplicant;
            if (startSupplicant->m_start){
                KwlanSuProcess *startWpa= new KwlanSuProcess(NULL,"Start WPA Supplicant");
                *startWpa << m_wpaPath << QString("-i%1").arg(QString(m_controlInterface)) << QString ("-D%1").arg(startSupplicant->getDriver()) << QString("-c")+ m_wpaConf << "-B";
                startWpa->setDescription(i18n("Start WPA Supplicant"));
                startWpa->start();
                m_configuration->writeDriver(startSupplicant->getDriver());
            }
            delete startSupplicant;
            // show wpa settings
            KwlanSettingsDlg *settings = new KwlanSettingsDlg();
            settings->setConfiguraton( m_configuration);

            if (!settings) return;
            settings->setCaption(QString(i18n("Settings for %1")).arg(interface));
            //settings->show();
            settings->exec();
        }
            // Now add a default profile
        if (!hasProfiles())
            slotNewProfile();
        m_networkChange = true;
    }
}
void KwlanInterface::slotEnableConfigureInterface(QString interface, bool enable)
{
    if (interface != QString(m_controlInterface)) return;
    m_configureInterface = enable;
}