/***************************************************************************
 *   Copyright (C) 2006 by Stephen Allewell                                *
 *   stephen@mirramar.fsnet.co.uk                                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#include <kdebug.h>
#include <qpushbutton.h>
#include <qfont.h>
#include <qfontmetrics.h>
#include <qlistbox.h>
#include <qcombobox.h>
#include <kapplication.h>
#include "charselectordialog.h"
#include "palettemanagerdialog.h"
#include "flosspalette.h"
#include "configuration.h"
#include "flosslistboxitem.h"
#include "flossscheme.h"
#include "kxstitch.h"
#include "calibratedialog.h"
#include "newcolordialog.h"

PaletteManagerDialog::PaletteManagerDialog(QWidget* parent, FlossPalette* palette, QValueList<int> usedFlosses)
  : PaletteManagerDlg(parent,"PaletteManagerDlg",true),
    m_scheme(palette->getScheme()),
    m_currentPalette(palette),
    m_usedFlosses(usedFlosses),
    m_charTable(0)
{
  m_font = KXSConfig().Editor_SymbolFont;
  FlossSymbol->setFont(QFont(m_font, 24));
  m_flosses = palette->m_palette;
  m_symbols = palette->m_symbols;
  m_stitchStrands = palette->m_stitchStrands;
  m_backstitchStrands = palette->m_backstitchStrands;

  connect(ColorList,SIGNAL(selectionChanged(QListBoxItem*)),this,SLOT(schemeSelectionChanged(QListBoxItem*)));
  connect(CurrentList,SIGNAL(selectionChanged(QListBoxItem*)),this,SLOT(currentSelectionChanged(QListBoxItem*)));
  connect(AddFloss,SIGNAL(clicked()),this,SLOT(addFloss()));
  connect(RemoveFloss,SIGNAL(clicked()),this,SLOT(removeFloss()));
  connect(SwapFlosses,SIGNAL(clicked()),this,SLOT(swapFlosses()));
  connect(StitchStrands,SIGNAL(activated(int)),this,SLOT(stitchStrandsChanged(int)));
  connect(BackstitchStrands,SIGNAL(activated(int)),this,SLOT(backstitchStrandsChanged(int)));
  connect(FlossSymbol,SIGNAL(clicked()),this,SLOT(selectSymbol()));
  connect(HelpButton,SIGNAL(clicked()),this,SLOT(contextHelp()));
  connect(NewColor,SIGNAL(clicked()),this,SLOT(newColor()));
  connect(ClearUnused,SIGNAL(clicked()),this,SLOT(removeUnused()));
  connect(CalibrateColors,SIGNAL(clicked()),this,SLOT(calibrateColors()));

  fillColorList();
  fillCurrentList();
}

PaletteManagerDialog::~PaletteManagerDialog()
{
  delete m_charTable;
}

void PaletteManagerDialog::newColor()
{
  Floss* floss = NewColorDialog::getFloss(m_scheme);
  if (floss)
    new FlossListBoxItem(ColorList, floss);
}

void PaletteManagerDialog::removeUnused()
{
  QMapIterator<int,Floss*> it;

  for (it = m_flosses.begin() ; it != m_flosses.end() ; ++it)
  {
    int mi = it.key();
    if (!m_usedFlosses.contains(mi))
    {                                           // if the color has not been used in the pattern
      if (mi == m_currentPalette->m_current) m_currentPalette->m_current = -1;
      m_flosses.remove(mi);
      m_symbols.remove(mi);
      m_stitchStrands.remove(mi);
      m_backstitchStrands.remove(mi);
      // need to remove item from the list box and add it to the other one
    }
  }
}

void PaletteManagerDialog::calibrateColors()
{
  CalibrateDialog *dlg = new CalibrateDialog(this);
  if (dlg->exec() == QDialog::Accepted)
  {
    // request the document to refresh its colors based on the calibrated ones
  }
}

void PaletteManagerDialog::accept()
{
  // copy the data back to the palette
  m_currentPalette->m_palette = m_flosses;
  m_currentPalette->m_symbols = m_symbols;
  m_currentPalette->m_stitchStrands = m_stitchStrands;
  m_currentPalette->m_backstitchStrands = m_backstitchStrands;
  PaletteManagerDlg::accept();
}

void PaletteManagerDialog::contextHelp()
{
  kapp->invokeHelp("PaletteManagerDialog");
}

void PaletteManagerDialog::addFloss()
{
  int i = ColorList->currentItem();
  FlossListBoxItem* item = (FlossListBoxItem*)ColorList->item(i);
  ColorList->takeItem(item);
  if (ColorList->count() == 0)
  {
    AddFloss->setEnabled(false);
    SwapFlosses->setEnabled(false);
  }
  else
  {
    ColorList->setSelected(i<?ColorList->count()-1,true);
  }
  insertFlossListBoxItem(CurrentList,item);
  CurrentList->setCurrentItem(item);
  i = freeIndex();
  m_flosses[i] = item->floss;
  m_symbols[i] = freeSymbol();
  m_stitchStrands[i] = KXSConfig().Palette_StitchStrands;
  m_backstitchStrands[i] = KXSConfig().Palette_BackstitchStrands;
  RemoveFloss->setEnabled(true);
  StitchStrands->setEnabled(true);
  StitchStrands->setCurrentItem(m_stitchStrands[i]-1);
  BackstitchStrands->setEnabled(true);
  BackstitchStrands->setCurrentItem(m_backstitchStrands[i]-1);
  FlossSymbol->setEnabled(true);
  FlossSymbol->setText(m_symbols[i]);
  if (ColorList->count())
    SwapFlosses->setEnabled(true);
}

void PaletteManagerDialog::removeFloss()
{
  int i = CurrentList->currentItem();
  FlossListBoxItem* item = (FlossListBoxItem*)CurrentList->item(i);
  CurrentList->takeItem(item);
  if (CurrentList->count() == 0)
  {
    RemoveFloss->setEnabled(false);
    SwapFlosses->setEnabled(false);
    StitchStrands->setEnabled(false);
    BackstitchStrands->setEnabled(false);
    FlossSymbol->setEnabled(false);
  }
  else
  {
    CurrentList->setSelected(i<?CurrentList->count()-1,true);
  }
  i = index(item->floss);
  m_flosses.remove(i);
  m_symbols.remove(i);
  m_stitchStrands.remove(i);
  m_backstitchStrands.remove(i);
  insertFlossListBoxItem(ColorList, item);
  ColorList->setCurrentItem(item);
}

void PaletteManagerDialog::insertFlossListBoxItem(QListBox* listBox, FlossListBoxItem* newItem)
{
  if (listBox->count() == 0)
  {
    listBox->insertItem(newItem);
  }
  else
  {
    for (int i = 0 ; i < listBox->count() ; i++)
    {
      FlossListBoxItem* item = (FlossListBoxItem*)listBox->item(i);
      if (newItem->floss->name < item->floss->name)
      {
        listBox->insertItem(newItem,i);
        return;
      }
    }
    listBox->insertItem(newItem);
  }
}

void PaletteManagerDialog::swapFlosses()
{
  FlossListBoxItem* from = (FlossListBoxItem*)ColorList->selectedItem();
  FlossListBoxItem* to   = (FlossListBoxItem*)CurrentList->selectedItem();
  ColorList->takeItem(from);
  CurrentList->takeItem(to);
  insertFlossListBoxItem(ColorList, to);
  ColorList->setCurrentItem(to);
  insertFlossListBoxItem(CurrentList, from);
  CurrentList->setCurrentItem(from);
  int i = index(to->floss);
  m_flosses[i] = from->floss;
}

void PaletteManagerDialog::selectSymbol()
{
  if (m_charTable == 0)
    m_charTable = new CharSelectorDialog(this, &m_symbols, &m_usedFlosses);
  int i = index(((FlossListBoxItem*)CurrentList->selectedItem())->floss);
  m_charTable->setSelectedChar(m_symbols[i]);
  if (m_charTable->exec() == QDialog::Accepted)
  {
    m_symbols[i] = m_charTable->selectedChar();
    FlossSymbol->setText(m_symbols[i]);
  }
}

void PaletteManagerDialog::schemeSelectionChanged(QListBoxItem* item)
{
  AddFloss->setEnabled(true);
  if (CurrentList->currentItem() != -1)
    SwapFlosses->setEnabled(true);
}

void PaletteManagerDialog::currentSelectionChanged(QListBoxItem* item)
{
  int stitchStrands = KXSConfig().Palette_StitchStrands;
  int backstitchStrands = KXSConfig().Palette_BackstitchStrands;

  int i = index(((FlossListBoxItem*)item)->floss->name);
  if (i != -1)
  {
    stitchStrands = m_stitchStrands[i];
    backstitchStrands = m_backstitchStrands[i];
  }
  StitchStrands->setCurrentItem(stitchStrands-1);
  BackstitchStrands->setCurrentItem(backstitchStrands-1);
  FlossSymbol->setText(m_symbols[i]);
  if (ColorList->currentItem() != -1)
    SwapFlosses->setEnabled(true);
  RemoveFloss->setEnabled(!m_usedFlosses.contains(i));
}

void PaletteManagerDialog::fillColorList()
{
  ColorList->clear();
  FlossScheme *scheme = ((KXStitchApplication *)kapp)->scheme(m_scheme);
  QPtrListIterator<Floss> it = scheme->flossIterator();
  Floss *f;
  for ( ; (f = it.current()) ; ++it)
  {
    if (!contains(f->name))
      new FlossListBoxItem(ColorList, f);
  }
  if (ColorList->count())
    ColorList->setCurrentItem(0);
  else
  {
    AddFloss->setEnabled(false);
    SwapFlosses->setEnabled(false);
  }
}

void PaletteManagerDialog::fillCurrentList()
{
  CurrentList->clear();
  FlossScheme *scheme = ((KXStitchApplication *)kapp)->scheme(m_scheme);
  QPtrListIterator<Floss> it = scheme->flossIterator();
  Floss *f;
  for ( ; (f = it.current()) ; ++it)
  {
    if (contains(f->name))
      new FlossListBoxItem(CurrentList, f);
  }
  if (CurrentList->count())
    CurrentList->setCurrentItem(0);
  else
  {
    RemoveFloss->setEnabled(false);
    SwapFlosses->setEnabled(false);
    StitchStrands->setEnabled(false);
    BackstitchStrands->setEnabled(false);
    FlossSymbol->setEnabled(false);
  }
}

bool PaletteManagerDialog::contains(QString name)
{
  QMapIterator<int,Floss*> it;
  for (it = m_flosses.begin() ; it != m_flosses.end() ; ++it )
  {
    if (it.data()->name == name)
      return true;
  }
  return false;
}

void PaletteManagerDialog::stitchStrandsChanged(int item)
{
  int i = index(((FlossListBoxItem*)CurrentList->selectedItem())->floss);
  m_stitchStrands[i] = item+1;
}

void PaletteManagerDialog::backstitchStrandsChanged(int item)
{
  int i = index(((FlossListBoxItem*)CurrentList->selectedItem())->floss);
  m_backstitchStrands[i] = item+1;
}

int PaletteManagerDialog::index(QString name)
{
  QMapIterator<int,Floss*> it;
  for (it = m_flosses.begin() ; it != m_flosses.end() ; ++it )
  {
    if (it.data()->name == name)
      return it.key();
  }
  return -1;
}

int PaletteManagerDialog::index(Floss* floss)
{
  QMapIterator<int,Floss*> it;
  for (it = m_flosses.begin() ; it != m_flosses.end() ; ++it )
  {
    if (it.data() == floss)
      return it.key();
  }
}

QChar PaletteManagerDialog::freeSymbol()
{
  int c = -1;
  bool found = false;
  QMapIterator<int,QChar> it;

  while (!found)
  {
    QChar rc(++c);
    if (rc.isPrint() && !rc.isSpace() && !rc.isPunct())
    {
      found = true;
      for (it = m_symbols.begin() ; it != m_symbols.end() && found ; ++it)
        if (it.data() == rc)
          found = false;
    }
  }
  return QChar(c);
}

int PaletteManagerDialog::freeIndex()
{
  int i = 0;
  while (m_flosses.contains(i)) i++;
  return i;
}
