/*
 * This file is part of lcrash, an analysis tool for Linux memory dumps.
 *
 * Created by Silicon Graphics, Inc.
 * Contributions by IBM, and others
 *
 * Copyright (C) 1999 - 2005 Silicon Graphics, Inc. All rights reserved.
 * Copyright (C) 2001, 2002 IBM Deutschland Entwicklung GmbH, IBM Corporation
 * Copyright (C) 2003 NEC Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */

#include <lcrash.h>

extern int defcpu_ia64;

#define _RD_USAGE "[-w outfile]"

void
rd_ia64_usage(command_t *cmd)
{
	CMD_USAGE(cmd, _RD_USAGE);
}

void
rd_ia64_help(command_t *cmd)
{
	CMD_HELP(cmd, _RD_USAGE,
			"Display the register contents of the default cpu."
			"This command can't be used on a live system ");
}

int
rd_ia64_parse(command_t *cmd)
{
	if (set_cmd_flags(cmd, (C_WRITE), 0)) {
		return(1);
	}
	return 0;
}

extern void *G_dump_header_asm;

int
rd_ia64_cmd(command_t *cmd)
{
	struct kl_pt_regs_ia64 * regs;

	if (cmd->nargs != 0) {
		rd_ia64_usage(cmd);
		return(1);
	}	

	if (CORE_IS_KMEM) {
		fprintf(cmd->efp, "Can't use this command on live system\n");
		return(1);
	}
	
	if (defcpu_ia64 == -1)
		defcpu_ia64 = KL_INT(G_dump_header_asm, 
			dha_typename, "dha_dumping_cpu");
	
	regs = G_dump_header_asm + kl_member_offset(dha_typename,
		"dha_smp_regs") + (defcpu_ia64 * PT_REGS_SZ);

	fprintf(cmd->ofp, "CPU:    %d\n", defcpu_ia64);

	fprintf(cmd->ofp, "psr : %016lx ifs : %016lx ip  : [<%016lx>]\n",
		regs->cr_ipsr, regs->cr_ifs, regs->cr_iip);
	fprintf(cmd->ofp, "unat: %016lx pfs : %016lx rsc : %016lx\n",
		regs->ar_unat, regs->ar_pfs, regs->ar_rsc);
	fprintf(cmd->ofp, "rnat: %016lx bsps: %016lx pr  : %016lx\n",
		regs->ar_rnat, regs->ar_bspstore, regs->pr);
	fprintf(cmd->ofp, "ldrs: %016lx ccv : %016lx fpsr: %016lx\n",
		regs->loadrs, regs->ar_ccv, regs->ar_fpsr);
	fprintf(cmd->ofp,"csd : %016lx ssd : %016lx\n", regs->ar_csd, regs->ar_ssd);
	fprintf(cmd->ofp, "b0  : %016lx b6  : %016lx b7  : %016lx\n",
		regs->b0, regs->b6, regs->b7);
	fprintf(cmd->ofp, "f6  : %05lx%016lx f7  : %05lx%016lx\n",
		regs->f6.u.bits[1], regs->f6.u.bits[0],
		regs->f7.u.bits[1], regs->f7.u.bits[0]);
	fprintf(cmd->ofp, "f8  : %05lx%016lx f9  : %05lx%016lx\n",
		regs->f8.u.bits[1], regs->f8.u.bits[0],
		regs->f9.u.bits[1], regs->f9.u.bits[0]);

	fprintf(cmd->ofp, "r1  : %016lx r2  : %016lx r3  : %016lx\n",
		regs->r1, regs->r2, regs->r3);
	fprintf(cmd->ofp, "r8  : %016lx r9  : %016lx r10 : %016lx\n",
		regs->r8, regs->r9, regs->r10);
	fprintf(cmd->ofp, "r11 : %016lx r12 : %016lx r13 : %016lx\n",
		regs->r11, regs->r12, regs->r13);
	fprintf(cmd->ofp, "r14 : %016lx r15 : %016lx r16 : %016lx\n",
		regs->r14, regs->r15, regs->r16);
	fprintf(cmd->ofp, "r17 : %016lx r18 : %016lx r19 : %016lx\n",
		regs->r17, regs->r18, regs->r19);
	fprintf(cmd->ofp, "r20 : %016lx r21 : %016lx r22 : %016lx\n",
		regs->r20, regs->r21, regs->r22);
	fprintf(cmd->ofp, "r23 : %016lx r24 : %016lx r25 : %016lx\n",
		regs->r23, regs->r24, regs->r25);
	fprintf(cmd->ofp, "r26 : %016lx r27 : %016lx r28 : %016lx\n",
		regs->r26, regs->r27, regs->r28);
	fprintf(cmd->ofp, "r29 : %016lx r30 : %016lx r31 : %016lx\n",
		regs->r29, regs->r30, regs->r31);

	return(0);
}
