/*
 * Copyright (C) 1999-2000, 2003 Hewlett-Packard Co
 *      David Mosberger-Tang <davidm@hpl.hp.com>
 * Copyright (C) 2004 Silicon Graphics, Inc. All rights reserved.
 *
 * A simple API for unwinding kernel stacks from user space. Based
 * on the kernel implementation of unwind and modified for use
 * from user space.
 */
#ifndef _LC_UNWIND_H
#define _LC_UNWIND_H

#define GATE_ADDR              (0xa000000000000000 + KL_PAGE_SIZE)

#define KL_IA64_RBS_OFFSET ((TASK_STRUCT_SZ + 15) & ~15)

typedef struct switch_stack_s {
	kaddr_t		 addr;
	void 		*sw;
} switch_stack_t;

typedef uint64_t u64;
typedef uint8_t u8;

/* XXX -- These hard coded values need to be derived from the kernel
 *        type information. Currently, lcrash does not know about
 *        the kdb_running_process and kdba_running_process data types.
 */             
#define KDB_RUNNING_PROCESS_SZ 32
#define KDB_REGS_OFFSET 8 
#define KDB_SEQNO_OFFSET 16 
#define KDB_SW_OFFSET 24 
                        
#define NEXT_KDB_CPU(p)         (void *)((uaddr_t)(p) + KDB_RUNNING_PROCESS_SZ)
#define KDB_CPU_TASK(p)         *(kaddr_t*)((uaddr_t)(p))
#define KDB_CPU_REGS(p)         *(kaddr_t*)((uaddr_t)(p) + KDB_REGS_OFFSET)
#define KDB_CPU_SEQNO(p)        *(int*)((uaddr_t)(p) + KDB_SEQNO_OFFSET)
#define KDB_CPU_SW(p)           *(kaddr_t*)((uaddr_t)(p) + KDB_SW_OFFSET)

extern int unw_debug_level;
#define UNW_DPRINT(n, ...) \
        if (n <= unw_debug_level) fprintf(stderr, __VA_ARGS__);

typedef uint64_t unw_word;
typedef unsigned char unw_hash_index_t;

#define TASK_STRUCT_T void
#define SPINLOCK_T uint32_t

enum unw_application_register {
	UNW_AR_BSP,
	UNW_AR_BSPSTORE,
	UNW_AR_PFS,
	UNW_AR_RNAT,
	UNW_AR_UNAT,
	UNW_AR_LC,
	UNW_AR_EC,
	UNW_AR_FPSR,
	UNW_AR_RSC,
	UNW_AR_CCV
};

/*
 * The following declarations are private to the unwind
 * implementation:
 */

struct unw_stack {
	uint64_t limit;
	uint64_t top;
};

#define UNW_FLAG_INTERRUPT_FRAME        (1UL << 0)

/*
 * This is a copy of the unw_frame_info struct defined in the kernel 
 * source. All types have been normalized so that they will be the
 * same on 32-bit and 64-bit targest. All pointers have been 
 * converted to uint64_t. Special handling will need to be 
 * performed to handle this.
 *
 * NOTE: Any changes to linux/linux/include/asm-ia64/unwind.h must
 *       also be made to this struct.
 */
struct unw_frame_info {
	struct unw_stack regstk;
	struct unw_stack memstk;
	uint32_t flags;
	int16_t  hint;
	int16_t  prev_script;

	/* current frame info: 
	 */
	uint64_t bsp;       /* backing store pointer value */
	uint64_t sp;        /* stack pointer value */
	uint64_t psp;       /* previous sp value */
	uint64_t ip;        /* instruction pointer value */
	uint64_t pr;        /* current predicate values */
	uint64_t cfm_loc;   /* cfm save location (or NULL) */
	uint64_t pt;        /* struct pt_regs location */

	kaddr_t task; 	    /* access via LKCD routines */
	kaddr_t sw;	    /* access via LKCD routines */

	/* preserved state: 
	 * 
	 * Note that these addresses can be either application 
	 * (user) or kernel addresses. It's necessary to have a 
	 * function do determine which type an address is to 
	 * avoid memory address errors.
	 */
	uint64_t bsp_loc;         /* previous bsp save location */
	uint64_t bspstore_loc;
	uint64_t pfs_loc;
	uint64_t rnat_loc;
	uint64_t rp_loc;
	uint64_t pri_unat_loc;
	uint64_t unat_loc;
	uint64_t pr_loc;
	uint64_t lc_loc;
	uint64_t fpsr_loc;
	struct unw_ireg {
		uint64_t loc;
		struct unw_ireg_nat {
			int64_t type :  3;        /* enum unw_nat_type */
			uint64_t off : 61; /* NaT word is at loc+nat.off */
		} nat;
	} r4, r5, r6, r7;
	uint64_t b1_loc, b2_loc, b3_loc, b4_loc, b5_loc;
	uint64_t f2_loc, f3_loc, f4_loc, f5_loc, fr_loc[16];
};

#define unw_get_gr(i,n,v,nat)   unw_access_gr(i,n,v,nat,0)
#define unw_get_br(i,n,v)       unw_access_br(i,n,v,0)
#define unw_get_fr(i,n,v)       unw_access_fr(i,n,v,0)
#define unw_get_ar(i,n,v)       unw_access_ar(i,n,v,0)
#define unw_get_pr(i,v)         unw_access_pr(i,v,0)

/*
 * Function prototypes
 */
void * kdb_cpu_ptr(uint64_t);
void ia64_init_frame_info(struct unw_frame_info *, kaddr_t, switch_stack_t *);
int ia64_unwind(struct unw_frame_info *);

#endif /* _LC_UNWIND_H */
