/*
 * kl_page.c
 *
 * This file is part of libklib. 
 * A library which provides access to Linux system kernel dumps.
 * This file handles the architecture-dependent parts of KLIB for
 * ARM/XScale based systems.
 *
 * Ported from kl_page_i386.c
 * by Fleming Feng(fleming.feng@intel.com)
 *
 * Copyright 1999 Silicon Graphics, Inc. All rights reserved.
 * Copyright 2000 Junichi Nomura, NEC Solutions <j-nomura@ce.jp.nec.com>
 * Copyright (C) 2003, Intel Corp. All rights reserved.
 *
 * This code is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */
#include <klib.h>

#define _DBG_COMPONENT KL_DBGCOMP_MEMMAP

extern int kl_vtop_arm(kaddr_t, kaddr_t*);

/*
 * declarations of static functions
 */
static kaddr_t kl_pgd_offset_arm(
	kaddr_t         /* kernel virtual address of page directory */,
	kaddr_t         /* kernel virtual address */);

static kaddr_t kl_pmd_offset_arm(
	kaddr_t         /* kernel virtual address of page middle directory */,
	kaddr_t         /* kernel virtual address */);

static kaddr_t kl_pte_offset_arm(
	kaddr_t         /* kernel virtual address of page table */,
	kaddr_t         /* kernel virtual address */);

/*
 * function definitions
 */

/* page table traversal functions */
static kaddr_t kl_pgd_offset_arm(kaddr_t pgd_base, kaddr_t vaddr)
{
	kaddr_t pgd_off, pgd_entry, pmd_base;

	pgd_off = ((vaddr >> KL_PGDIR_SHIFT_ARM) &
		   (KL_PTRS_PER_PGD_ARM-1)) * KL_NBPW;
	kl_trace4(0, "pgd_off: %"FMTPTR"x\n", pgd_off);
	kl_vtop_arm((pgd_base + pgd_off), &pgd_entry);
	pmd_base = KL_READ_PTR(pgd_entry);

	return pmd_base;
}

static kaddr_t kl_pmd_offset_arm(kaddr_t pmd_base, kaddr_t vaddr)
{
	return pmd_base;
}

static kaddr_t kl_pte_offset_arm(kaddr_t pte_base, kaddr_t vaddr)
{
	kaddr_t pte_off, pte, pte_val;

	pte_off = ((vaddr >> KL_PAGE_SHIFT_ARM) & 
		   (KL_PTRS_PER_PTE_ARM-1)) * KL_NBPW;
	kl_trace4(0, "pte_off: %"FMTPTR"x\n", pte_off);
	kl_vtop_arm((pte_base + pte_off), &pte);
	pte_val = KL_READ_PTR(pte);

	return pte_val;
}

/* lookup virtual address in page table */
kaddr_t kl_mmap_virtop_arm(kaddr_t vaddr, void *mmp)
{
	kaddr_t pgd_base, pmd_base;
	kaddr_t pte_base, pte_val, paddr = 0;

	kl_trace4(0, "vaddr: %"FMTPTR"x\n", vaddr);
	/* get the pgd entry */
	pgd_base = kl_kaddr(mmp, "mm_struct", "pgd"); /* c0004000*/
	kl_trace4(0, "pgd_base: %"FMTPTR"x\n", pgd_base);
	pmd_base = kl_pgd_offset_arm(pgd_base,vaddr);
	kl_trace4(0, "pmd_base: %"FMTPTR"x\n", pmd_base);
	if (KL_ERROR) {
		KL_ERROR = KLE_INVALID_MAPPING;
		kl_trace3(0, "Could not read pdb_entry "
			     "for vaddr=%"FMTPTR"x\n", vaddr);
		return(0);
	}

	/* add KL_PAGE_OFFSET to make pmd_base as virtual address */
	pmd_base = pmd_base & KL_PMD_BASE_MASK_ARM;
	pte_base = kl_pmd_offset_arm(pmd_base,vaddr);
	if (KL_ERROR) {
		return(0);
	}

	/* add KL_PAGE_OFFSET to make pte_base as virtual address */
	pte_base = pte_base & KL_PT_BASE_MASK_ARM;
	pte_val = kl_pte_offset_arm(pte_base,vaddr);
	kl_trace4(0, "ptr_val: %"FMTPTR"x\n", pte_val);
	if (KL_ERROR) {
		KL_ERROR = KLE_INVALID_MAPPING;
		kl_trace3(0, "Could not read page the table entry "
			     "for vaddr=%"FMTPTR"x\n", vaddr);
		return(0);
	}

	paddr = (pte_val & KL_PAGE_BASE_MASK_ARM) | 
		(vaddr & (~(KL_PAGE_MASK_ARM)));

	if(paddr < kl_start_physaddr){
		KL_ERROR = KLE_INVALID_PADDR;
		return 0;
	}

	return(paddr);
}
