/*
 * $Id: kl_savedump.c,v 1.2 2005/02/23 01:09:12 tjm Exp $
 *
 * This file is part of libklib.
 * A library which provides access to Linux system kernel dumps.
 *
 * Created by Silicon Graphics, Inc.
 * Contributions by IBM, NEC, and others
 *
 * Copyright (C) 1999 - 2005 Silicon Graphics, Inc. All rights reserved.
 * Copyright (C) 2001, 2002 IBM Deutschland Entwicklung GmbH, IBM Corporation
 * Copyright 2000 Junichi Nomura, NEC Solutions <j-nomura@ce.jp.nec.com>
 *
 * This code is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */

#include <klib.h>
#include <zlib.h>
#include <sys/stat.h>
#include <unistd.h>
#include <linux/stddef.h>

static int kl_check_type_dev(char *);
static int kl_check_type_dir(char *);
static int __kl_dump_retrieve_reg(int,int,char*,int,int);
static int __kl_dump_retrieve_s390sa(int,int,char*,int);
static int progress_tick, progress_last_percent, progress_pos;

/*
 * Name: __kl_get_bounds()
 * Func: Get the bounds number from the bounds file located in <dumpdir>
 *       Returns the bounds number, or 0 if no bounds file exists.
 */
static int
__kl_bounds_retrieve(char *dumpdir)
{
	int dirf, bounds;
	char *tfile;

	tfile = (char *)malloc(strlen(dumpdir) + 10);
	if (!tfile) {
		return (0);
	}

	sprintf(tfile, "%s/bounds", dumpdir);
	if ((dirf = open(tfile, O_RDONLY, 0)) < 0) {
		return (0);
	}

	(void)read(dirf, tfile, strlen(tfile));
	bounds = atoi(tfile);
	close(dirf);
	return (bounds);
}

/*
 * Name: __kl_bounds_update()
 * Func: Update the bounds number with a new value.  The function takes
 *       the <dumpdir> along with the new bounds number.  Returns 0 for
 *       failure, 1 for success.
 */
static void
__kl_bounds_update(char *dumpdir, int bounds)
{
	int dirf;
	char *tfile;

	tfile = (char *)malloc(strlen(dumpdir) + 10);
	if (!tfile) {
		return;
	}

	sprintf(tfile, "%s/bounds", dumpdir);
	if ((dirf = open(tfile, O_RDWR|O_CREAT,
		(S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH))) < 0) {
			fprintf(KL_ERRORFP, "Warning: could not "
				"create new bounds file in %s!\n", dumpdir);
			return;
	}

	sprintf(tfile, "%d\n", bounds);
	if (write(dirf, tfile, strlen(tfile)) != strlen(tfile)) {
		fprintf(KL_ERRORFP, "Warning: could not "
			"write to bounds file in %s!\n", dumpdir);
		close(dirf);
		return;
	}
	close(dirf);
	return;
}

static void
init_progress(void)
{
	progress_tick = progress_pos = 0;
	progress_last_percent = -1;
}

static void
update_progress(int cur, int max)
{
	static const char spinner[] = "\\|/-";
	int	tick;
	float	percent;
	struct timeval	tv;

	/*
	 * Calculate the new progress position.  If the
	 * percentage hasn't changed, then we skip out right
	 * away. 
	 */
	percent = ((float) cur / (float) max) * 100.0;
	if (progress_last_percent == (int) 10 * percent)
		return;
	progress_last_percent = (int) 10 * percent;
	
	/*
	 * If we've already updated the spinner once within
	 * the last 1/8th of a second, no point doing it
	 * again.
	 */
	gettimeofday(&tv, NULL);
	tick = (tv.tv_sec << 3) + (tv.tv_usec / (1000000 / 8));
	if ((tick == progress_tick) &&
	    (cur != max) && (cur != 0))
		return;
	progress_tick = tick;

	/*
	 * Advance the spinner and display the progress
	 */
	progress_pos = (progress_pos+1) & 3;
	printf("   ");
	if (percent == 100.0)
		fputc('|', stdout);
	else
		fputc(spinner[progress_pos & 3], stdout);
	printf(" %4.1f%%   \r", percent);
	fflush(stdout);
}


/*
 * Name: __kl_dump_retrieve()
 * Func: Gather a dump from a given device and save it into the dump
 *       directory.  The dump directory points to the possibility of
 *       a "bounds" file, which would be read, and be used to save the
 *       index number of the dump.  Returns 0 on failure, 1 if the
 *       core dump is short or if there was a problem finishing the
 *       read/write of the dump, or 2 on success.
 */
int
__kl_dump_retrieve(char *dumpdev, char *dumpdir, int progress, int local_debug)
{
	int bounds, devf = 0, outf = 0;
	char file_name[1024];
	uint64_t magic_nr;
	int rc;

	/* check the types */
	if (kl_check_type_dev(dumpdev) <= 0) {
		fprintf(KL_ERRORFP, "Error: %s not a block device!\n", dumpdev);
		rc = 0; goto out;
	}

	/* check the types */
	if (kl_check_type_dir(dumpdir) <= 0) {
		fprintf(KL_ERRORFP, "Error: %s not a directory!\n", dumpdir);
		DUMP_BP();
		rc = 0; goto out;
	}
	
	/* initialize progress count */
	if (progress)
		init_progress();

	/* try to read the bounds file */
	bounds = __kl_bounds_retrieve(dumpdir);

	/* try to open the dump device */
	if ((devf = open(dumpdev, O_RDONLY, 0)) < 0) {
		fprintf(KL_ERRORFP, "Error: open() on dumpdev failed!\n");
		DUMP_BP();
		rc = 0; goto out;
	}

	/* Check dump type */

	if(read(devf, &magic_nr, sizeof(magic_nr)) != sizeof(magic_nr)) {
		fprintf(KL_ERRORFP, "Error: read() on dumpdev failed!\n");
                DUMP_BP();
		rc = 0; goto out;
        }

        /* make the new filename */
        sprintf(file_name, "%s/dump.%d", dumpdir, bounds);
        if ((outf = open(file_name, O_CREAT|O_RDWR|O_TRUNC,
                (S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH))) < 0) {
                        fprintf(KL_ERRORFP,
                                "Error: open() of dump file \"%s\" failed!\n",
                                file_name);
                        DUMP_BP();
			rc = 0; goto out;
        }

	if(magic_nr == KL_DUMP_MAGIC_S390SA){
		rc = __kl_dump_retrieve_s390sa(devf, outf, dumpdir, progress);
	} else {

		rc = __kl_dump_retrieve_reg(devf, outf, dumpdir, progress, local_debug);
	}
out:
	if(rc != 0)
		__kl_bounds_update(dumpdir, bounds + 1);
	if(devf)
		close(devf);
	if(outf)
		close(outf);
	if(rc == 0)
		unlink(file_name);
	return rc;
}	

/*
 * Name: __kl_dump_retrieve()
 * Func: retrieve a regular dump
 *       Returns 0 on failure, 1 if the
 *       core dump is short or if there was a problem finishing the
 *       read/write of the dump, or 2 on success.
 */
static int
__kl_dump_retrieve_reg(int devf, int outf, char* dumpdir, int progress, int local_debug)
{
	kl_dump_page_t dp;
        kl_dump_header_t dh;
        kl_dump_header_asm_t *dha;
	char *asm_hdr_buf;
	u_char* tfile;
	uint64_t byte_offset, magic_number;
        uint32_t page_index;
        int page_nbr = 0;
	uint32_t dump_header_size, arch_header_size;

	if (lseek(devf, KL_DUMP_HEADER_OFFSET + 
		offsetof(struct kl_dump_header_s, magic_number), 
		SEEK_SET) < 0) {
		fprintf(KL_ERRORFP, "Error: lseek() on dumpdev failed!\n");
		DUMP_BP();
		return (0);
	}
	if(read(devf, &magic_number, sizeof(magic_number)) != 
		sizeof(magic_number)) {
		fprintf(KL_ERRORFP, "Error: read() on dumpdev failed!\n");
		DUMP_BP();
		return (0);
	}
	/* validate the dump header */
	if (magic_number != KL_DUMP_MAGIC_NUMBER) {
		if (local_debug) {
			fprintf(KL_ERRORFP, "Error: KL_DUMP_MAGIC_NUMBER in "
				"dump header invalid!\n");
		}
		DUMP_BP();
		return (0);
	}

	if (lseek(devf, KL_DUMP_HEADER_OFFSET + offsetof(struct kl_dump_header_s, header_size), SEEK_SET) < 0) {
		fprintf(KL_ERRORFP, "Error: lseek() on dumpdev failed!\n");
		DUMP_BP();
		return (0);
	}

	if(read(devf, &dump_header_size, sizeof(dump_header_size)) 
			!= sizeof(dump_header_size)) {
		fprintf(KL_ERRORFP, "Error: read() on dumpdev failed!\n");
                DUMP_BP();
		return (0);
        }
	/* 
	 * move beyond the swap header 
	 */
	if (lseek(devf, KL_DUMP_HEADER_OFFSET, SEEK_SET) < 0) {
		fprintf(KL_ERRORFP, "Error: lseek() on dumpdev failed!\n");
		DUMP_BP();
		return (0);
	}

	/* try to read the dump header */
	if (read(devf, (char *)&dh,
		dump_header_size) < dump_header_size) {
			fprintf(KL_ERRORFP,
				"Error: read() of dump header failed!\n");
			DUMP_BP();
			return (0);
	}

	KL_DUMP_PAGE_SIZE = dh.page_size;
	KL_DUMP_PAGE_MASK = ~((uint64_t)KL_DUMP_PAGE_SIZE-1);
	KL_DUMP_PAGE_SHIFT = 12;
	while ((KL_DUMP_PAGE_SIZE >> KL_DUMP_PAGE_SHIFT) != 1) {
		KL_DUMP_PAGE_SHIFT++;
	}

	if (dh.version >= 9) {
		tfile = (char *)malloc(dh.dump_buffer_size);
		KL_DUMP_HEADER_SIZE = dh.dump_buffer_size;
	} else {
		/* KL_DUMP_BUFFER_SIZE and KL_DUMP_HEADER_SIZE depend on 
		 * page size 
		 *  = 64k for pagesize < 64k (default)
		 *  = 256k for pagesize >=64k 
		 */
		if (KL_DUMP_PAGE_SHIFT >= 16)
			KL_DUMP_BUFFER_SIZE = 
				KL_DUMP_HEADER_SIZE = (256 * 1024);
		
		/* allocate the buffer for file data */
		tfile = (char *)malloc(KL_DUMP_BUFFER_SIZE);
	}
	

	/* 
	 * Try to read the arch specific dump header - Not sure now how many
	 * CPUS are configured in the kernel. So, get the arch specific
	 * header size first. 
	 */
	if (lseek(devf, KL_DUMP_HEADER_OFFSET + dump_header_size + offsetof(kl_dump_header_asm_t, header_size), SEEK_SET) < 0) {
		fprintf(KL_ERRORFP, "Error: lseek() on dumpdev failed!\n");
		DUMP_BP();
		return (0);
	}
	if(read(devf, &arch_header_size, sizeof(arch_header_size)) 
		!= sizeof(arch_header_size)) {
		fprintf(KL_ERRORFP, "Error: read() on dumpdev failed!\n");
		DUMP_BP();
		return (0);
        }
	/*
	 * Now seek back to arch header location.
	 */
	if (lseek(devf, KL_DUMP_HEADER_OFFSET + dump_header_size, SEEK_SET) < 0) {
		fprintf(KL_ERRORFP, "Error: lseek() on dumpdev failed!\n");
		DUMP_BP();
		return (0);
	}
	asm_hdr_buf = (char *)malloc(arch_header_size);
	if (read(devf, (char *)asm_hdr_buf, arch_header_size) 
		!= arch_header_size) {
		fprintf(KL_ERRORFP,
			"Error: read() of dump header failed!\n");
		DUMP_BP();
		return (0);
	}
	dha = (kl_dump_header_asm_t *)asm_hdr_buf;
	/* validate the architecture-specific dump header */
	if (dha->magic_number != KL_DUMP_ASM_MAGIC_NUMBER) {
		if (local_debug) {
			fprintf(KL_ERRORFP,
				"Error: KL_DUMP_ASM_MAGIC_NUMBER in "
				"dump header invalid!\n");
		}
		DUMP_BP();
		return (0);
	}

	/* write out the dump header */
	if (write(outf, (char *)&dh,
		dump_header_size) != dump_header_size) {
			fprintf(KL_ERRORFP,
				"Error: write() of dump header to dump file "  
				"failed! (Perhaps Dumpdir File System"
				" is Full?)\n");

			DUMP_BP();
			return (0);
	}

	/* write out the architecture-specific dump header */
	if (write(outf, (char *)asm_hdr_buf, arch_header_size) 
		!= arch_header_size) {
		fprintf(KL_ERRORFP,
			"Error: write() of architecture-specific dump"
			" header to dump file failed!"
			" (Perhaps Dumpdir File System is Full)\n");

		DUMP_BP();
		return (0);
	}

	/* now make sure we have more than just a header -- if not, leave */
	if (dh.dump_level == KL_DUMP_LEVEL_HEADER) {
		return (1);
	}

	/* 
	 * move to the right offset in the output and device file 
	 */
	(void)lseek(devf, KL_DUMP_HEADER_SIZE + KL_DUMP_HEADER_OFFSET,
		    SEEK_SET);
	(void)lseek(outf, KL_DUMP_HEADER_SIZE, SEEK_SET);

	/* now start rolling through the blocks */
	for (page_index = 0, byte_offset = KL_DUMP_HEADER_OFFSET +
		     KL_DUMP_PAGE_SIZE; 1; 
	      page_index++, byte_offset += (sizeof(dp) + dp.size) ) {

		/* try to read in the dump page header */
		if (read(devf, (char *)&dp, sizeof(dp)) < sizeof(dp)) {
			/* if this fails, we may have run out of space */
			fprintf(KL_ERRORFP,
				"Warning: short dump in dumpdev!\n");
			DUMP_BP();
			return (1);
		}

		if (local_debug > 1) {
			fprintf(KL_ERRORFP,
				"dp.address = 0x%"FMT64"x, "
				"dp.flags = 0x%x, dp.size = %d\n",
				dp.address, dp.flags, dp.size);
		}

#if DUMP_DEBUG >= 6
		/* DEBUG sanity check */
		if( dp.page_index != page_index ) {
			DUMP_BP();
		}

		/* DEBUG sanity check */
		if( dp.byte_offset != byte_offset ) {
			DUMP_BP();
		}
#endif
		/* sanity check */
		if ((!dp.flags) || (dp.flags >= KL_DUMP_DH_NOT_USED) ||
			((!dp.size) && (!(dp.flags & KL_DUMP_DH_END))) ||
			(dp.size > KL_DUMP_PAGE_SIZE)) {
/* 			(dp.size > dh.page_size)) { */
				fprintf(KL_ERRORFP,
					"Error: dump flags in dumpdev "
					"page index invalid!\n");

				fprintf(KL_ERRORFP, "NOTE: Likely a Dirty Truncated Dump\n");

				DUMP_BP();
				return (1);
		}

#if DUMP_DEBUG >= 6
		/*
		 * The dump file doesn't include the page we skipped over
		 * that has the swap info in it.
		 *
		 * Check the debuging information and adjust for no swap header.
		 * This will make it possible to matchup the byte_offset with 
		 * the cur_addr in __cmppindexcreate() while building the index file.
		 */
		if( dp.byte_offset != byte_offset ) {
			DUMP_BP();
		}
		if( dp.page_index != page_index ) {
			DUMP_BP();
		}
		dp.byte_offset -= KL_DUMP_HEADER_OFFSET;
#endif
		/* write out the dump page header */
		if (write(outf, (char *)&dp, sizeof(dp)) != sizeof(dp)) {
			/* if this fails, we may have run out of space */
			fprintf(KL_ERRORFP, "Warning: write() to dump "
				"file failed!\n");
			return (1);
		}

		/* 
		 * make sure we aren't at the end of the dump 
		 */
		if (dp.flags & KL_DUMP_DH_END) {
			/* update progress count */
			if (progress)
				update_progress(dh.num_dump_pages,
						dh.num_dump_pages);
			if(dp.flags & KL_DUMP_DH_TRUNCATED) {
				fprintf(KL_ERRORFP, "NOTE: Dump Truncated (Clean)\n");
				return (1);
			}
			return (2);
		}

		/* try to read in the dump page itself */
		if (read(devf, tfile, dp.size) < dp.size) {
			/* if this fails, we may have run out of space */
			fprintf(KL_ERRORFP,
				"Warning: short dump in dumpdev!\n");
			return (1);
		}

#if DUMP_DEBUG
		/* test for test pattern */
		if(dp.flags & KL_DUMP_DH_TEST_PATTERN) {
			int i;

			/* 
			 * Debugging test pattern is for each byte
			 * to have the page_index in it,
			 */
			for ( i = 0; i < dp.size; i++ ) {
				if( tfile[i] != (page_index & 0xff) ) {
					DUMP_BP();
				}
			}
		}
#endif

		/* try to write out the dump page itself */
		if (write(outf, (char *)tfile, dp.size) != dp.size) {
			/* if this fails, we may have run out of space */
			fprintf(KL_ERRORFP, "Warning: write() to dump "
				"file failed!\n");
			DUMP_BP();
			return (1);
		}
		/* update progress count */
		if (progress)
			update_progress(page_nbr++, dh.num_dump_pages);
	}

	/* NOTREACHED */
}

/*
 * __kl_dump_erase() -- Erase a crash dump from <dumpdev>.  This just
 *                      clobbers over the dump header.
 */
int
__kl_dump_erase(char *dumpdev)
{
	int devf;
	kl_dump_header_t dh;

	/* check the types */
	if (kl_check_type_dev(dumpdev) <= 0) {
		fprintf(KL_ERRORFP, "Error: %s not a block device!\n", dumpdev);
		return (0);
	}

	/* try to open the output file */
	if ((devf = open(dumpdev, O_RDWR,
		(S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH))) < 0) {
			fprintf(KL_ERRORFP,
				"Error: open() on dumpdev failed!\n");
			return (0);
	}

	/* 
	 * move beyond the swap header 
	 */
	if (lseek(devf, KL_DUMP_HEADER_OFFSET, SEEK_SET) < 0) {
		fprintf(KL_ERRORFP, "Error: lseek() on dumpdev failed!\n");
		close(devf);
		return (0);
	}

	/* try to read the dump header */
	if (read(devf, (char *)&dh, sizeof(dh)) < sizeof(dh)) {
		fprintf(KL_ERRORFP, "Error: read() of dump header failed!\n");
		close(devf);
		return (0);
	}

	/* check the magic number */
	if (dh.magic_number != KL_DUMP_MAGIC_NUMBER) {
		close(devf);
		return (0);
	}

	/* write a goofy number for sanity checking */
	dh.magic_number = (uint64_t)0xdeadbeef;

	/* 
	 * move beyond the swap header
	 */
	if (lseek(devf, KL_DUMP_HEADER_OFFSET, SEEK_SET) < 0) {
		fprintf(KL_ERRORFP, "Error: lseek() on dumpdev failed!\n");
		close(devf);
		return (0);
	}

	/* try to re-write the dump header */
	if (write(devf, (char *)&dh, sizeof(dh)) != sizeof(dh)) {
		fprintf(KL_ERRORFP, "Error: write() of dump header failed!\n");
		close(devf);
		return (0);
	}

	close(devf);
	return (1);
}

/*
 * check_type_dir() -- run the stat() operation without having all the
 *                     inline definitions conflicted by the kernel.  Make
 *                     sure <path> is a directory.
 */
static int
kl_check_type_dir(char *path)
{
        struct stat sbuf;

        if (stat(path, &sbuf) < 0) {
                return (-1);
        }

        return (S_ISDIR(sbuf.st_mode) ? 1 : 0);
}

/*
 * check_type_dev() -- run the stat() operation without having all the
 *                     inline definitions conflicted by the kernel.  Make
 *                     sure <path> is a block device type.
 */
static int
kl_check_type_dev(char *path)
{
        struct stat sbuf;

        if (stat(path, &sbuf) < 0) {
                return (-1);
        }

	if (S_ISREG(sbuf.st_mode))
		return (2);

        return (S_ISBLK(sbuf.st_mode) ? 1 : 0);
}

/*
 * Name: __kl_dump_retrieve_s390sa()
 * Func: retrieve a s390 (standalone) dump
 *       Returns 0 on failure, 1 if the
 *       core dump is short or if there was a problem finishing the
 *       read/write of the dump, or 2 on success.
 */
static int
__kl_dump_retrieve_s390sa(int fin, int fout, char* dumpdir, int progress)
{
#if defined(DUMP_ARCH_S390) || defined(DUMP_ARCH_S390X)
        kl_dump_header_t dh;
	kl_dump_header_s390sa_t s390_dh;
	kl_compress_fn_t compress_fn = kl_compress_gzip;
	kl_dump_page_t dp;
	char dump_page_buf[KL_DUMP_BUFFER_SIZE];
	char buf[KL_DUMP_PAGE_SIZE];
	char dpcpage[KL_DUMP_PAGE_SIZE];
	uint64_t mem_loc;
	uint32_t buf_loc = 0;
	int size;
	uint32_t dp_size,dp_flags;
	int rc = 2;

        memset(&dh, 0, sizeof(dh));

	/* get the dump header
         */

        if (lseek(fin, 0, SEEK_SET) < 0) {
                fprintf(KL_ERRORFP, "Error: Cannot lseek() to get the dump header "
                        "from the dump file!\n");
                rc = 0; goto out;
        }
        if (read(fin, (char *)&s390_dh,
                sizeof(s390_dh)) != sizeof(s390_dh)) {
                        fprintf(KL_ERRORFP, "Error: Cannot read() dump header "
                                "from dump file!\n");
                rc = 0; goto out;
        }
        kl_s390sa_to_reg_header(&s390_dh,&dh);

        /* write dump header */
 
        memset(dump_page_buf, 0, KL_DUMP_BUFFER_SIZE);
        memcpy((void *)dump_page_buf, (const void *)&dh, sizeof(kl_dump_header_t));
        lseek(fout, 0L, SEEK_SET);
        if (write(fout, (char *)dump_page_buf, KL_DUMP_BUFFER_SIZE) != KL_DUMP_BUFFER_SIZE) {
		 rc = 0; goto out;
        }

	/* write dump */

	mem_loc = 0;
	lseek(fin, KL_DUMP_HEADER_SZ_S390SA, SEEK_SET);
	while (mem_loc < dh.memory_size) {
		if(read(fin, buf, KL_DUMP_PAGE_SIZE) != KL_DUMP_PAGE_SIZE){
			fprintf(KL_ERRORFP, "Error: read error\n");
			rc = 1; goto out;
		}
		memset(dpcpage, 0, KL_DUMP_PAGE_SIZE);
		/* get the new compressed page size
                 */

                size = compress_fn((char *)buf, KL_DUMP_PAGE_SIZE,
                                (char *)dpcpage, KL_DUMP_PAGE_SIZE);

                /* if compression failed or compressed was ineffective,
                 * we write an uncompressed page
                 */
                if (size < 0) {
                        dp_flags = KL_DUMP_DH_RAW;
                        dp_size  = KL_DUMP_PAGE_SIZE;
                } else {
                        dp_flags = KL_DUMP_DH_COMPRESSED;
                        dp_size  = size;
                }
	        dp.address = mem_loc;
                dp.size    = dp_size;
                dp.flags   = dp_flags;
                memcpy((void *)(dump_page_buf + buf_loc),
                        (const void *)&dp, sizeof(kl_dump_page_t));
                buf_loc += sizeof(kl_dump_page_t);
		/* copy the page of memory
                 */
                if (dp_flags & KL_DUMP_DH_COMPRESSED) {
                        /* copy the compressed page
                         */
                        memcpy((void *)(dump_page_buf + buf_loc),
                                (const void *)dpcpage, dp_size);
                } else {
                        /* copy directly from memory
                         */
                        memcpy((void *)(dump_page_buf + buf_loc),
                                (const void *)buf, dp_size);
                }
		buf_loc += dp_size;
		if(write(fout, dump_page_buf, buf_loc) != buf_loc){
                        fprintf(KL_ERRORFP, "Error: write error\n");
			rc = 1; goto out;
		}
		buf_loc = 0;
		mem_loc += KL_DUMP_PAGE_SIZE;
		if (progress)
			update_progress(mem_loc/KL_DUMP_PAGE_SIZE, dh.num_dump_pages);
	}

        /* write end marker 
         */
 
        dp.address = 0x0;
        dp.size    = KL_DUMP_DH_END;
        dp.flags   = 0x0;
	write(fout, dump_page_buf, sizeof(kl_dump_page_t)); 
out:
	return rc;
#else
        return 0;
#endif
}
