/*
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

// written by Jon Keating <jon@licq.org>

#ifndef __MSN_H
#define __MSN_H

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "licq_events.h"
#include "licq_icqd.h"
#include "licq_socket.h"

#include "msnbuffer.h"
#include "msnevent.h"

#define MSN_PPID 0x4D534E5F
#define L_MSNxSTR "[MSN] "

const char CONTACT_LIST[] = "FL";
const char ALLOW_LIST[] = "AL";
const char BLOCK_LIST[] = "BL";

const unsigned short FLAG_CONTACT_LIST = 1;
const unsigned short FLAG_ALLOW_LIST   = 2;
const unsigned short FLAG_BLOCK_LIST   = 4;
const unsigned short FLAG_REVERSE_LIST = 8;

#ifndef HAVE_STRNDUP
char *strndup(const char *s, size_t n);
#endif

#include <string>
#include <list>
#include <vector>
#include <cctype>
#include <pthread.h>

using std::string;
using std::list;
using std::vector;
using std::isalnum;

  const unsigned long MSN_DP_EVENT = 1;

class CMSNPacket;
class CMSNDataEvent;

struct SBuffer
{
  CMSNBuffer *m_pBuf;
  string m_strUser;
  bool m_bStored;
};

typedef list<SBuffer *> BufferList;

struct SStartMessage
{
  CMSNPacket *m_pPacket;
  ICQEvent *m_pEvent;
  CICQSignal *m_pSignal;
  char *m_szUser;
  unsigned long m_nSeq;
  bool m_bConnecting,
       m_bDataConnection;
};

typedef list<SStartMessage *> StartList;

class CMSN
{
public:
  CMSN(CICQDaemon *, int);
  ~CMSN();

  void Run();
  
  void MSNPing();
  bool Connected() { return m_nServerSocket != -1; }
  bool CanSendPing() { return m_bCanPing; }
  void MSNLogoff(bool = false);
  void MSNLogon(const char *, int);
 
  bool WaitingPingReply()          { return m_bWaitingPingReply; }
  void SetWaitingPingReply(bool b) { m_bWaitingPingReply = b; }

  pthread_mutex_t mutex_ServerSocket; // Ugly, but whatever.

private:
  void ProcessSignal(CSignal *);
  void ProcessPipe();
  void ProcessServerPacket(CMSNBuffer *);
  void ProcessNexusPacket(CMSNBuffer &);
  void ProcessSSLServerPacket(CMSNBuffer &);
  void ProcessSBPacket(char *, CMSNBuffer *, int);
  
  // Network functions
  void SendPacket(CMSNPacket *);
  void Send_SB_Packet(string &, CMSNPacket *, int = -1, bool = true);
  void MSNLogon(const char *, int, unsigned long);
  void MSNGetServer();
  void MSNAuthenticateRedirect(string &, string &);
  void MSNAuthenticate(char *);
  bool MSNSBConnectStart(string &, string &);
  bool MSNSBConnectAnswer(string &, string &, string &, string &);

  void MSNSendInvitation(char *, CMSNPacket *);
  void MSNSendMessage(char *, char *, pthread_t, unsigned long);
  void MSNSendTypingNotification(char *, unsigned long);
  void MSNChangeStatus(unsigned long);
  void MSNAddUser(char *);
  void MSNRemoveUser(char *);
  void MSNRenameUser(char *);
  void MSNGrantAuth(char *);
  void MSNUpdateUser(char *);
  void MSNBlockUser(char *);
  void MSNUnblockUser(char *);
  void MSNGetDisplayPicture(const string &, const string &);

  // Internal functions
  int HashValue(int n) { return n % 211; }
  void StorePacket(SBuffer *, int);
  void RemovePacket(string, int, int = 0);
  SBuffer *RetrievePacket(const string &, int);
  ICQEvent *RetrieveEvent(unsigned long);
  void HandlePacket(int, CMSNBuffer &, const char *);
  unsigned long SocketToCID(int);
  static string Decode(const string &);
  static string Encode(const string &);
  void WaitDataEvent(CMSNDataEvent *);
  bool RemoveDataEvent(CMSNDataEvent *);
  CMSNDataEvent *FetchDataEvent(const string &, int);
  CMSNDataEvent *FetchStartDataEvent(const string &);

  // Interface to CICQDaemon
  void PushPluginSignal(CICQSignal *);

  // Config
  unsigned long m_nListVersion;
    
  // Variables
  CICQDaemon *m_pDaemon;
  bool m_bExit;
  int m_nPipe;
  int m_nServerSocket;
  int m_nNexusSocket;
  int m_nSSLSocket;
  CMSNBuffer *m_pPacketBuf,
             *m_pNexusBuff,
             *m_pSSLPacket;
  vector<BufferList> m_vlPacketBucket;
  list<ICQEvent *> m_pEvents;
  list<CMSNDataEvent *> m_lMSNEvents;
  StartList m_lStart;
  bool m_bWaitingPingReply,
       m_bCanPing;
  
  // Server variables
  unsigned long m_nStatus,
                m_nOldStatus,
                m_nSessionStart;
  string m_strMSPAuth,
         m_strSID,
         m_strKV;
         
  pthread_t m_tMSNPing;
  pthread_mutex_t mutex_StartList,
                  mutex_MSNEventList,
                  mutex_Bucket;
    
  char *m_szUserName,
       *m_szPassword,
       *m_szCookie;

  friend class CMSNDataEvent;
};

extern CSocketManager gSocketMan;

#endif // __MSN_H
