/* $Id: gtkconfig.c,v 1.17 2005/05/10 17:18:40 marcusva Exp $
 *
 *  This file is part of LingoTeach, the Language Teaching program
 *  Copyright (C) 2004-2005 Marcus von Appen. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <string.h>
#include <libxml/xpath.h>

#include "../defs.h"
#include "../errors.h"
#include "gtkplugin.h"
#include "gtkconfig.h"

/* used for the val_types array.
 */
enum
{
     VAL_LEARN,
     VAL_TRAIN,
     VAL_EDIT
};

/* for both learning and teraining modes the user can currently only set
 * the default word amount, languages and a "show-hints" flag. Thus the
 * val_type_types only consists of those two enum entries.
 */
enum
{
     VAL_TYPE_WORD,
     VAL_TYPE_HINT,
     VAL_TYPE_METHOD
};

/* used by the languages. The user can set a main language (e.g natural one)
 * and languages to learn from (normal ones = NONE) 
 */
enum
{
     LANG_TYPE_MAIN,
     LANG_TYPE_NONE
};

/* The configuration specs for the gtk client only contain 3 different main
 * nodes - languages (for learning and training, main language or just used)
 * and learning as well as training entries (how many words and if hints
 * should be displayed)
 */
enum
{
     QUERY_LANG,
     QUERY_LEARN,
     QUERY_TRAIN,
};

/* used for saving the xml nodes to the config, see gtkconfig_save_... */
#define NODE_LANGUAGE "language"

static char *val_types[] = { "learn", "train", "edit", NULL };
static char *val_type_types[] = { "words", "hints", "method", NULL };
static char *lang_modes[] = { "learn", "train", "edit", NULL };
static char *lang_types[] = { "main", "none", NULL };
static char *queries[] = {
     "/lingoteach/gtk/language[@mode='%s' and @type='%s']",
     "/lingoteach/gtk/learn[@type='%s']",
     "/lingoteach/gtk/train[@type='%s']",
     NULL
};

static void set_val (lingGtkPrefs *settings, int valtype , int type,
                     char *value);
static void set_lang (lingGtkPrefs *settings, int mode, int type,
                      xmlXPathObjectPtr obj);
static void load_languages (lingGtkPrefs *settings,
                            xmlXPathContextPtr context);

/*
 * sets the value which was parsed out of a configuration file
 */ 
static void
set_val (lingGtkPrefs *settings, int valtype , int type, char *value)
{
     debug ("Setting value %s\n", value);

     switch (valtype)
     {
     case VAL_LEARN:
          switch (type)
          {
          case VAL_TYPE_WORD:
               settings->learn_words = atoi (value);
               break;
          case VAL_TYPE_HINT:
               settings->learn_hints = (lingbool) atoi (value);
               break;
          case VAL_TYPE_METHOD:
               settings->learn_method = atoi (value);
               break;
          }
          break;
     case VAL_TRAIN:
          switch (type)
          {
          case VAL_TYPE_WORD:
               settings->train_words = atoi (value);
               break;
          case VAL_TYPE_HINT:
               settings->train_hints = (lingbool) atoi (value);
               break;
          case VAL_TYPE_METHOD:
               settings->train_method = atoi (value);
               break;
          }
          break;
     }
     return;
}

/*
 * sets the languages parsed out of the configuration file
 */
static void
set_lang (lingGtkPrefs *settings, int mode, int type, xmlXPathObjectPtr obj)
{
     lingchar *value = NULL;
     int i = 0;

     switch (type)
     {
     case LANG_TYPE_MAIN: /* main language to use */
          if (obj->nodesetval->nodeNr == 0)
               break;
          value = xmlXPathCastToString (obj);
          if (!value)
          {
               error_warning (_("Main language could not be set!"),
                              _("The main language could not be set in this "
                                "session. Please set it manually."),
                              CONF_SOLUTION);
               break;
          }

          debug ("Setting main language %s\n", value);
          
          switch (mode)
          {
          case VAL_LEARN: /* learning language */
               learn_language_change_main_language (settings->learn_lang,
                                                    value);
               break;
          case VAL_TRAIN:
               learn_language_change_main_language (settings->train_lang,
                                                    value);
          case VAL_EDIT:
               settings->edit_lang = g_strdup (value);
	       break;
          }
          xmlFree (value);
          break;
     case LANG_TYPE_NONE:
          for (i = 0; i < obj->nodesetval->nodeNr; i++)
          {
               value = xmlNodeGetContent (obj->nodesetval->nodeTab[i]);
               debug ("Setting language %s\n", value);

               switch (mode)
               {
               case VAL_LEARN: /* learning language */
                    learn_language_use_language (settings->learn_lang,
                                                 value, TRUE);
                    break;
               case VAL_TRAIN:
                    learn_language_use_language (settings->train_lang,
                                                 value, TRUE);
                    break;
               }
               xmlFree (value);
          }
          break;
     }
     return;
}

/*
 * loads the languages from the configuration and sets them
 */
static void
load_languages (lingGtkPrefs *settings, xmlXPathContextPtr context)
{
     xmlXPathObjectPtr object = NULL;
     lingchar *query = NULL;
     int i = 0;
     int j = 0;

     debug ("Loading languages...\n");
     while (lang_modes[i])
     {
          j = 0;
          while (lang_types[j])
          {
               query = g_strdup_printf (queries[QUERY_LANG],
                                        lang_modes[i], lang_types[j]);
               object = xmlXPathEvalExpression (query, context);
               g_free (query);
               if (!object)
               {
		    error_warning (_("Languages could not be loaded!"),
				   _("Some languages could not be loaded "
				     "from the language file. They will not "
                                     "be available in this session."),
				   _("Please make sure, your language file "
				     "contains valid information and is in a "
				     "proper state."));
                    j++;
                    break;
               }

               /* check, if there is a node, which is known by the library,
                  so we can work with it in set_lang() */
               if (object->type != XPATH_UNDEFINED && object->nodesetval)
                    set_lang (settings, i, j, object);

               xmlXPathFreeObject (object);
               j++;
          }
               i++;
     }
     return;
}

/*
 * loads the gtk dependant stuff out of the user configuration
 */
void 
gtkconfig_load (lingGtkPrefs *settings)
{
     xmlDocPtr doc = NULL;
     xmlXPathContextPtr context = NULL;
     xmlXPathObjectPtr object = NULL;
     lingchar *value = NULL;
     lingchar *query = NULL;
     int i = 0;
     int j = 0;

     debug ("Loading gtk gui configuration...\n");

     if (settings->prefs->predef)
     {
          /* default configuration has to be loaded */
          gtkconfig_load_defaults (settings);
          return;
     }

     /* add all known languages to the modes */
     while (settings->prefs->languages[i])
     {
          debug ("Now adding %s\n", settings->prefs->languages[i]);
          settings->learn_lang =
               learn_language_append_language (settings->learn_lang,
                                               settings->prefs->languages[i],
                                               FALSE, FALSE);
          settings->train_lang =
               learn_language_append_language (settings->train_lang,
                                               settings->prefs->languages[i],
                                               FALSE, FALSE);
          i++;
     }

     /* TODO */
     if (settings->prefs->old_conf)
          return;

     doc = xmlParseFile (settings->prefs->conf_file);
     if (!doc)
     {
          error_warning (_("Configuration file could not be loaded!"),
                         _("The configuration file could not be loaded. "
                           "No configuration settings will be made."),
                         CONF_SOLUTION);
          return;
     }
     xmlXPathInit ();
     context = xmlXPathNewContext (doc);
     load_languages (settings, context);

     /* now parse all other settings */
     i = 0;
     while (i < VAL_EDIT && val_types[i])
     {
          j = 0;
          while (val_type_types[j])
          {
               query = g_strdup_printf (queries[QUERY_LEARN + i],
                                        val_type_types[j]);
               object = xmlXPathEval (query, context);
               g_free (query);
               if (!object)
	       {
                    error_warning (_("Value could not be read from "
                                     "configuration."),
                                   _("Querying the configuration file did "
                                     "not bring any result, thus a value "
                                     "will not be set."), CONF_SOLUTION);
                    j++;
                    break;
               }
               value = xmlXPathCastToString (object);
               if (!value)
               {
                    error_warning (_("Value could not be converted!"),
                                   _("A read value could not be converted "
                                     "correctly and thus will not be set."),
                                   CONF_SOLUTION);
               }
               set_val (settings, i, j, value);
               xmlFree (value);
               xmlXPathFreeObject (object);
               j++;
          }
          i++;
     }
     xmlXPathFreeContext (context);
     xmlFreeDoc (doc);
     return;
}

/*
 * initializes the gtk preferences
 */
lingGtkPrefs*
gtkconfig_init_settings (lingoteachSettings *preferences)
{
     /* create preferences */
     lingGtkPrefs *set = NULL;

     debug ("Initializing new gtk config...\n");

     if (!preferences) /* assume new preferences */
     {
          error_warning (ERR_MEM_INFO, ERR_MEM_DESC, ERR_NOT_AVAILABLE);
          return NULL;
     }

     set = g_new (lingGtkPrefs, 1);
     if (!set)
     {
          error_warning (ERR_MEM_INFO, ERR_MEM_DESC, ERR_NOT_AVAILABLE);
          return NULL;
     }

     set->prefs = preferences;

     set->learn_words = 0;
     set->learn_hints = FALSE;
     set->learn_method = RANDOM;
     set->learn_lang = NULL;

     set->train_words = 0;
     set->train_hints = FALSE;
     set->train_method = RANDOM;
     set->train_lang = NULL;

     set->edit_lang = NULL;
     set->plugins = NULL;

     set->learn_session = NULL;
     set->train_session = NULL;
     return set;
}

void
gtkconfig_free_settings (lingGtkPrefs *settings)
{
     debug ("Freeing gtk settings...\n");
     GSList *item = NULL;
     
     learn_language_free (settings->learn_lang);
     learn_language_free (settings->train_lang);

     if (settings->learn_session)
          learn_session_free (settings->learn_session);

     item = settings->plugins;
     while (item)
     {
          gtkplugin_plugin_free ((lingGtkPlugin *) item->data);
          item = item->next;
     }
     g_slist_free (settings->plugins);

     conf_free_config (settings->prefs);
     g_free (settings);
     return;
}

/*
 * saves the gtk part of the user preferences
 */
gboolean
gtkconfig_save_settings (lingGtkPrefs *settings)
{
     xmlDocPtr doc = NULL;
     xmlNodePtr root = NULL;
     xmlNodePtr parent = NULL;
     xmlNodePtr node = NULL;
     learnLanguage *tmp = NULL;
     int i = 0;
     gchar *words = NULL;
     gchar *hints = NULL;
     gchar *method = NULL;

     debug ("Saving gtk gui settings...\n");

     conf_save_settings (settings->prefs);
     doc = xmlParseFile (settings->prefs->conf_file);
     if (!doc)
     {
          error_warning (_("Configuration file error!"),
                         _("The configuration file seems to be errornous. "
                           "Your current settings will probably be lost."),
                         CONF_SOLUTION);
          return FALSE;
     }
     
     root = xmlDocGetRootElement (doc);
     parent = root->children;
     while (parent)
     {
          if (xmlStrcasecmp (parent->name, "gtk") == 0)
               break;
          parent = parent->next;
     }
     if (parent)
     {
          xmlUnlinkNode (parent);
          xmlFreeNode (parent);
     }
     parent = xmlNewChild (root, NULL, "gtk", NULL);

     /* the <gtk> section is completely new every time, so add all nodes */
     
     /* start with the languages */
     while (val_types[i])
     {
          switch (i)
          {

          case VAL_LEARN:
               tmp = settings->learn_lang;
               words = g_malloc (sizeof (settings->learn_words)); 
               g_snprintf (words, sizeof (words), "%i", settings->learn_words);

               hints = (settings->learn_hints) ? "1" : "0";
               method = g_malloc (sizeof (gint));
               g_snprintf (method, sizeof (method), "%i",
                           settings->learn_method);
               break;
          case VAL_TRAIN:
               tmp = settings->train_lang;
               words = g_malloc (sizeof (settings->train_words)); 
               g_snprintf (words, sizeof (words), "%i", settings->train_words);

               hints = (settings->train_hints) ? "1" : "0";
               method = g_malloc (sizeof (gint));
               g_snprintf (method, sizeof (method), "%i",
                           settings->train_method);
               break;
          case VAL_EDIT:
               tmp = NULL;
               node = xmlNewTextChild (parent, NULL, NODE_LANGUAGE,
                                       settings->edit_lang);
               xmlNewProp (node, "mode", lang_modes[VAL_EDIT]);
               xmlNewProp (node, "type", lang_types[LANG_TYPE_MAIN]);
               break;
          }

          while (tmp)
          {
               if (tmp->used)
               {
                    node = xmlNewTextChild (parent, NULL, NODE_LANGUAGE,
                                            tmp->language);
                    xmlNewProp (node, "mode", lang_modes[i]);
                    xmlNewProp (node, "type", lang_types[LANG_TYPE_NONE]);
               }
               else if (tmp->main)
               {
                    node = xmlNewTextChild (parent, NULL, NODE_LANGUAGE,
                                            tmp->language);
                    xmlNewProp (node, "mode", lang_modes[i]);
                    xmlNewProp (node, "type", lang_types[LANG_TYPE_MAIN]);
               }
               tmp = tmp->next;
          }
          
          if (i != VAL_EDIT)
          {
               /* now add all other settings */
               node = xmlNewTextChild (parent, NULL, val_types[i], words);
               xmlNewProp (node, "type", val_type_types[VAL_TYPE_WORD]);
               g_free (words);
               
               node = xmlNewTextChild (parent, NULL, val_types[i], hints);
               xmlNewProp (node, "type", val_type_types[VAL_TYPE_HINT]);
               
               node = xmlNewTextChild (parent, NULL, val_types[i], method);
               xmlNewProp (node, "type", val_type_types[VAL_TYPE_METHOD]);
               g_free (method);
          }

          i++;
     }

     xmlKeepBlanksDefault (0);
     if (!xmlSaveFormatFile (settings->prefs->conf_file, doc, 1))
     {
          error_warning (_("Configuration file could not be saved!"),
                         _("The configuration file could not be saved. "
                           "Your current settings will probably be lost."),
                         CONF_SOLUTION);
     }
     xmlFreeDoc (doc);
     return TRUE;
}

/*
 * fallback function, if anything goes wrong while loading the gtk part of
 * the preferences
 */
void
gtkconfig_load_defaults (lingGtkPrefs *settings)
{
     char *tmp_file;

     debug ("Loading gtk gui defaults...\n");

     settings->prefs->predef = FALSE;

     /* load config from the default config file */
     tmp_file = settings->prefs->conf_file;
     settings->prefs->conf_file = malloc (strlen (DATADIR) + strlen (CONFFILE)
                                          + 2);
     snprintf (settings->prefs->conf_file,
               strlen (DATADIR) + strlen (CONFFILE) + 2,
               "%s/%s", DATADIR, CONFFILE);
     gtkconfig_load (settings);
     free (settings->prefs->conf_file);
     settings->prefs->conf_file = tmp_file;

     /* save it */
     gtkconfig_save_settings (settings);

     settings->prefs->predef = TRUE;

     return;
}

/* checks the preferences for sanity */
gboolean
gtkconfig_check_config (lingGtkPrefs *settings, PreferenceType type)
{
     learnLanguage *lang = NULL;
     gboolean _main = FALSE;
     gboolean used = FALSE;

     switch (type)
     {
     case PREFERENCE_GENERAL:
          break;
     case PREFERENCE_LEARN:
          debug ("Checking learning settings\n");
          if (settings->learn_words == 0 || !settings->learn_lang)
               return FALSE;

          lang = settings->learn_lang;
          while (lang)
          {
               if (lang->main)
                    _main = TRUE;
               else if (lang->used)
                    used = TRUE;
               lang = lang->next;
          }
          if (!_main || !used)
               return FALSE;
          break;
     case PREFERENCE_TRAIN:
          if (settings->train_words == 0 || !settings->train_lang)
               return FALSE;
          break;
     case PREFERENCE_EDIT:
          break;
     case PREFERENCE_PLUGIN:
          break;
     default: /* impossible */
          break;
     }
     return TRUE;
}
