/*  $Id: gui.c,v 1.16 2005/03/28 16:24:40 marcusva Exp $
 *
 *  This file is part of LingoTeach, the Language Teaching program
 *  Copyright (C) 2001-2005 The LingoTeach Team. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "../li18n.h"
#include "../defs.h"
#include "../errors.h"
#include "../event.h"
#include "hig-widgets.h"
#include "gtkdefs.h"

#include "preferences.h"
#include "gtkconfig.h"
#include "menu.h"
#include "gtklearn.h"
#include "gtktrain.h"
#include "gtkedit.h"
#include "gtkerr.h"
#include "util.h"
#include "gui.h"

#define WELCOME_FILE "WELCOME"

/* global variables */
lingGtkMainWindow *main_app = NULL;
lingGtkPrefs *settings = NULL;

static lingGtkMainWindow* create_main_window (void);
static void free_main_window (lingGtkMainWindow *app);
static GtkWidget* create_init (void);
static GtkWidget* init_main_win (void);

static lingGtkMainWindow*
create_main_window (void)
{
     lingGtkMainWindow *app = g_new (lingGtkMainWindow, 1);
     if (!app)
     {
          error_critical (ERR_MEM_INFO, 
                          _("The main window structures could not be "
                            "allocated."), ERR_NOT_AVAILABLE);
          return NULL;
     }

     app->window = NULL;
     app->menu = NULL;
     app->accels = NULL;
     app->tips = NULL;
     app->status = NULL;
     app->notebook = NULL;
     app->pages = NULL;
     app->objects = NULL;
     app->editor = NULL;

     return app;
}

static void
free_main_window (lingGtkMainWindow *app)
{
     event_free_listener (app->objects);
     g_slist_free (app->pages);
     g_free (app);
}

static GtkWidget*
create_init (void)
{
     GtkWidget *box_main;
     GtkWidget *image;
     GtkWidget *view_text;
     GtkWidget *scr_win_text;
     GtkTextBuffer *buf_text;
     GtkTextIter iter;

     GtkTextChildAnchor *anchor;
     GtkWidget *btn_prefs;

     gchar *tmp = NULL;

     gchar *file = NULL;
     gchar *buf = NULL;
     GError *err = NULL;
     gsize n;

     box_main = hig_vbox_new ();
     image = gtk_image_new_from_stock (GTK_STOCK_MISSING_IMAGE,
                                       GTK_ICON_SIZE_DIALOG);
     gtk_box_pack_start (GTK_BOX (box_main), image, FALSE, TRUE, 0);
     
     scr_win_text = hig_scrolled_window_new ();
     gtk_box_pack_start (GTK_BOX (box_main), scr_win_text, TRUE, TRUE, 0);
     
     view_text = gtk_text_view_new ();
     gtk_text_view_set_editable (GTK_TEXT_VIEW (view_text), FALSE);
     gtk_text_view_set_wrap_mode (GTK_TEXT_VIEW (view_text), GTK_WRAP_WORD);
     gtk_text_view_set_cursor_visible (GTK_TEXT_VIEW (view_text), FALSE);
     gtk_text_view_set_left_margin (GTK_TEXT_VIEW (view_text), 15);
     gtk_container_add (GTK_CONTAINER (scr_win_text), view_text);

     buf_text = gtk_text_view_get_buffer (GTK_TEXT_VIEW (view_text));
     gtk_text_buffer_create_tag (buf_text, "heading",
                                 "weight", PANGO_WEIGHT_BOLD,
                                 "size", 15 * PANGO_SCALE,
                                 NULL);
     gtk_text_buffer_create_tag (buf_text, "spaced",
                                 "left_margin", 30, "right_margin", 25, NULL);
     
     gtk_text_buffer_create_tag (buf_text, "warn", "foreground", "red", NULL);
     gtk_text_buffer_create_tag (buf_text, "link", "foreground", "blue", 
                                 "underline", PANGO_UNDERLINE_SINGLE, NULL);
     /* note to translators: the complete string would look like:
        "Welcome to LingoTeach - A language teaching program" */
     tmp = g_strdup_printf (_("Welcome to %s - %s\n\n"), NAME, DESCRIPTION);
     gtk_text_buffer_get_iter_at_offset (buf_text, &iter, 0);
     gtk_text_buffer_insert_with_tags_by_name (buf_text, &iter, tmp, -1,
                                               "heading", NULL);
     g_free (tmp);

     file = util_get_file_with_locale (DOCDIR, WELCOME_FILE);
     if (!g_file_get_contents (file, &buf, &n, &err))
     {
          error_warning (_("WELCOME file could not be read!"),
                         err->message, ERR_NOT_AVAILABLE);
          g_error_free (err);
          g_free (file);
          err = NULL;
     }
     g_free (file);
     
     if (buf)
     {
          gtk_text_buffer_insert_with_tags_by_name (buf_text, &iter, buf,
                                                    -1, NULL);
          g_free (buf);
     }
     
     if (settings->prefs->old_conf || settings->prefs->predef)
     {
          /* maybe this can be replaced by a text iter movement... */
          gtk_text_buffer_insert (buf_text, &iter, "\n\n", -1);

          gtk_text_buffer_insert_with_tags_by_name
               (buf_text, &iter, _("You are running the program for the first "
                                   "time. We recommend you to visit the "),
                -1, "warn", "spaced", NULL);

          anchor = gtk_text_buffer_create_child_anchor (buf_text, &iter);
          btn_prefs = gtk_button_new_with_mnemonic (_("_Preferences"));
          g_signal_connect (G_OBJECT (btn_prefs), "clicked",
                            G_CALLBACK (preferences_activate), NULL);
          gtk_text_view_add_child_at_anchor (GTK_TEXT_VIEW (view_text),
                                             btn_prefs, anchor);

          gtk_text_buffer_insert_with_tags_by_name
               (buf_text, &iter, _(" first in order to fit LingoTeach to your "
                                   "needs."), -1, "warn", "spaced", NULL);
     }

     return box_main;
}

static GtkWidget*
init_main_win (void)
{
     GtkWidget *box_main;
     GtkWidget *box_init;

     main_app = create_main_window ();
     if (!main_app)
          return NULL;

     main_app->tips = gtk_tooltips_new ();

     /* create main window */
     main_app->window = hig_window_new (NAME " v" VERSION " - " INTVERSION);
     gtk_container_set_border_width (GTK_CONTAINER (main_app->window), 0);
     gtk_window_set_default_size (GTK_WINDOW (main_app->window), 600, 400);
     g_signal_connect (G_OBJECT (main_app->window), "destroy",
                       G_CALLBACK (gui_quit), NULL);

     /* main box */
     box_main = hig_vbox_new ();
     gtk_box_set_spacing (GTK_BOX (box_main), 0);
     gtk_container_set_border_width (GTK_CONTAINER (box_main), 0);
     gtk_container_add (GTK_CONTAINER (main_app->window), box_main);
    
     /* main menu */
     main_app->menu = menu_create_menu ();
     gtk_box_pack_start (GTK_BOX (box_main),
                         gtk_ui_manager_get_widget (main_app->menu, MAIN_MENU),
                         FALSE, FALSE, 0);

     /* notebook */
     main_app->notebook = gtk_notebook_new ();
     gtk_notebook_set_show_border (GTK_NOTEBOOK (main_app->notebook), FALSE);
     gtk_notebook_set_show_tabs (GTK_NOTEBOOK (main_app->notebook), FALSE);
     gtk_box_pack_start (GTK_BOX (box_main), main_app->notebook,
                         TRUE, TRUE, 0);

     gui_add_page_entry (create_init ());
     gui_add_page_entry (gtklearn_create_learn_page ());
     /* gui_add_page_entry (gtktrain_create_train_page ()); */

     /* activate correct menu selection */
     menu_set_active (PAGE_INIT);

     /* status bar */
     main_app->status = gtk_statusbar_new ();
     gtk_box_pack_end (GTK_BOX (box_main), main_app->status, FALSE, FALSE, 0);
     util_new_status (GTK_STATUSBAR (main_app->status), NAME " v" VERSION);

     /* create the initial window box */
     box_init = hig_vbox_new ();
     gtk_notebook_append_page (GTK_NOTEBOOK (main_app->notebook), box_init,
                               NULL);

     return main_app->window;
}

void
gui_create_gui (int argc, char *argv[], lingoteachSettings *preferences)
{
     GtkWidget *window = NULL;

     if (!g_thread_supported ())
          g_thread_init (NULL);
     g_set_prgname (NAME);
    
     gtk_init (&argc, &argv);

     settings = gtkconfig_init_settings (preferences);
     if (!settings)
          return;

     gtkconfig_load (settings);

     /* register gui errors */
     gtkerr_register_errors ();

     window = init_main_win ();
     gtk_widget_show_all (window);

     gtk_main ();
     return;
}

void
gui_add_page_entry (void *page)
{
     g_return_if_fail (page != NULL);

     if (GTK_IS_WIDGET (page))
          gtk_notebook_append_page (GTK_NOTEBOOK (main_app->notebook),
                                    GTK_WIDGET (page), NULL);
     
     return;
}

void gui_quit (void)
{
     /* TODO: we have to clean anything up here! */
     debug ("Quitting application and releasing resources...\n");

     free_main_window (main_app);
     gtkconfig_free_settings (settings);
     gtk_main_quit ();
     return;
}
