/* $Id: menu.c,v 1.23 2005/04/20 06:42:19 marcusva Exp $
 *
 *  This file is part of LingoTeach, the Language Teaching program
 *  Copyright (C) 2004-2005 Marcus von Appen. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "../li18n.h"
#include "../defs.h"
#include "../errors.h"

#include "preferences.h"
#include "gtklesson.h"
#include "gtksession.h"
#include "gtkedit.h"
#include "gtkconfig.h"
#include "gtkimport.h"
#include "util.h"
#include "about.h"
#include "gui.h"
#include "menu.h"

extern lingGtkMainWindow *main_app;
extern lingGtkPrefs *settings;

#define MENU_PATH "/MainMenu/%s"

static void init_editor (void);
static void set_sensitive (GtkUIManager *menu, const gchar *name,
                           gboolean sensitive);
static void open_help (void);

/* the various menu definitions */
static const gchar *main_menu = 
"<ui>"
"  <menubar name='MainMenu'>"
"    <menu action='LingoTeach'>"
"      <menuitem action='New'/>"
"      <menuitem action='Open'/>"
"      <menuitem action='Save'/>"
"      <menuitem action='SaveAs'/>"
"      <separator/>"
"      <menuitem action='Import'/>"
"      <menuitem action='Export'/>"
"      <menuitem action='Print'/>"
"      <separator/>"
"      <menuitem action='Properties'/>"
"      <separator/>"
"      <menuitem action='Quit'/>"
"    </menu>"
"    <menu action='EditMenu'>"
"      <menuitem action='Preferences'/>"
"    </menu>"
"    <menu action='ViewMenu'>"
"      <menuitem action='Lessons'/>"
"      <menuitem action='Editor'/>"
"      <separator/>"
"      <menuitem action='Statistics'/>"
"    </menu>"
"    <menu action='HelpMenu'>"
"      <menuitem action='Help'/>"
"      <menuitem action='About'/>"
"    </menu>"
"  </menubar>"
"</ui>";

static GtkActionEntry menu_entries[] = {
     { "LingoTeach", NULL, "_LingoTeach", NULL, NULL, NULL },
     { "EditMenu", NULL, N_("_Edit"), NULL, NULL, NULL },
     { "ViewMenu", NULL, N_("_View"), NULL, NULL, NULL },
     { "HelpMenu", NULL, N_("_Help"), NULL, NULL, NULL },

     { "New", GTK_STOCK_NEW, NULL, NULL, N_("Create a new session"),
       gtksession_new_session },
     { "Open", GTK_STOCK_OPEN, NULL, NULL, N_("Open a session"),
       gtksession_load_session },
     { "Save", GTK_STOCK_SAVE, NULL, NULL, N_("Save session"),
       gtksession_save_session },
     { "SaveAs", GTK_STOCK_SAVE_AS, NULL, NULL, N_("Save session to a file"),
       gtksession_save_session_to_file },
     { "Import", NULL, N_("I_mport"), NULL, N_("Import meanings"),
       gtkimport_activate},
     { "Export", NULL, N_("E_xport"), NULL, N_("Export meaning"),
       NULL }, /*TODO*/
     { "Print", GTK_STOCK_PRINT, NULL, NULL, N_("Print the current meaning"),
       NULL },
     { "Properties", GTK_STOCK_PROPERTIES, NULL, NULL, N_("Properties"),
       NULL /* the callback is connected in gtklearn_init_session () */ },
     { "Quit", GTK_STOCK_QUIT, NULL, NULL, N_("Quit"), gui_quit },
     
     { "Preferences", GTK_STOCK_PREFERENCES, NULL, NULL,
       N_("Preferences for the application"), preferences_activate },

     { "Lessons", NULL, N_("_Lessons"), NULL, N_("Currently loaded lessons"),
       gtklesson_activate },
     { "Editor", NULL, N_("_Editor"), NULL, N_("Edit a lesson"),
       init_editor },
     { "Statistics", NULL, N_("_Statistics"), NULL,
       N_("Learning or training statistics"), NULL },

     { "Help", GTK_STOCK_HELP, NULL, NULL, N_("Help"), NULL},
     { "About", GTK_STOCK_ABOUT, NULL, NULL, N_("About"), about_activate },
};
static guint menu_entry_amount = G_N_ELEMENTS (menu_entries); 

static void
init_editor (void)
{
     main_app->editor = gtkedit_init_editor ();
     gtk_widget_show_all (main_app->editor->window);
     return;
}
static void
set_sensitive (GtkUIManager *menu, const gchar *name, gboolean sensitive)
{
     GtkWidget *widget = menu_get_entry (menu, name);
     if (widget)
          gtk_widget_set_sensitive (widget, sensitive);
     return;
}

static void
open_help (void)
{
     gchar *file = util_get_file_with_locale (DOCDIR, "/lingoteach.html");
     util_open_link (file);
     g_free (file);
     return;
}

GtkWidget*
menu_get_entry (GtkUIManager *menu, const gchar *name)
{
     GtkWidget *widget;
     gchar *text = g_strdup_printf (MENU_PATH, name);

     widget = gtk_ui_manager_get_widget (menu, text);
     g_free (text);

     return widget;
}

GtkUIManager*
menu_create_menu (void)
{
     GtkUIManager *menu;
     GError *error = NULL;

     GtkWidget *widget;

     menu = gtk_ui_manager_new ();
     
     main_app->actions = gtk_action_group_new ("Actions");
     gtk_action_group_add_actions (main_app->actions, menu_entries,
                                   menu_entry_amount, NULL);

     gtk_ui_manager_insert_action_group (menu, main_app->actions, 0);
     g_object_unref (main_app->actions);

     main_app->accels = gtk_ui_manager_get_accel_group (menu);
     gtk_window_add_accel_group (GTK_WINDOW (main_app->window),
                                 main_app->accels);
     if (!gtk_ui_manager_add_ui_from_string (menu, main_menu, -1, &error))
     {
          error_critical (_("Main menu could not be created!"),
                          error->message, ERR_NOT_AVAILABLE);
          g_error_free (error);
     }

     /* some fixed callbacks */
     widget = menu_get_entry (menu, "HelpMenu/Help");
     if (widget)
          g_signal_connect_swapped (G_OBJECT (widget), "activate",
                                    G_CALLBACK (open_help), NULL);
     return menu;
}

void
menu_set_active (lingGtkPages no)
{
     switch (no)
     {
     case PAGE_INIT:
          set_sensitive (main_app->menu, "LingoTeach/Save", FALSE);
          set_sensitive (main_app->menu, "LingoTeach/SaveAs", FALSE);
          set_sensitive (main_app->menu, "LingoTeach/Import", TRUE);
          set_sensitive (main_app->menu, "LingoTeach/Export", FALSE);
          set_sensitive (main_app->menu, "LingoTeach/Print", FALSE);
          set_sensitive (main_app->menu, "LingoTeach/Properties", FALSE);
          set_sensitive (main_app->menu, "ViewMenu/Statistics", FALSE);
          break;
     case PAGE_LEARN:
          set_sensitive (main_app->menu, "LingoTeach/Save", TRUE);
          set_sensitive (main_app->menu, "LingoTeach/SaveAs", TRUE);
          set_sensitive (main_app->menu, "LingoTeach/Import", FALSE);
          set_sensitive (main_app->menu, "LingoTeach/Export", TRUE);
          set_sensitive (main_app->menu, "LingoTeach/Print", TRUE);
          set_sensitive (main_app->menu, "LingoTeach/Properties", TRUE);
          set_sensitive (main_app->menu, "ViewMenu/Statistics", TRUE);
          break;
     case PAGE_TRAIN:
          set_sensitive (main_app->menu, "LingoTeach/Save", TRUE);
          set_sensitive (main_app->menu, "LingoTeach/SaveAs", TRUE);
          set_sensitive (main_app->menu, "LingoTeach/Import", FALSE);
          set_sensitive (main_app->menu, "LingoTeach/Export", TRUE);
          set_sensitive (main_app->menu, "LingoTeach/Print", TRUE);
          set_sensitive (main_app->menu, "LingoTeach/Properties", TRUE);
          set_sensitive (main_app->menu, "ViewMenu/Statistics", TRUE);
          break;
     default: /* do nothing */
          break;
     }
     gtk_notebook_set_current_page (GTK_NOTEBOOK (main_app->notebook),
                                    (gint) no);
     return;
}
