/* $Id: learn.c,v 1.19 2005/03/24 10:27:23 marcusva Exp $
 *
 *  This file is part of LingoTeach, the Language Teaching program
 *  Copyright (C) 2004-2005 Marcus von Appen. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif 

#include <libxml/xpath.h>
#include <stdlib.h>
#include <string.h>

#include "learn.h"

static learnEntry* create_random_learn_entries (learnLesson *lesson);
static learnLesson *get_lesson_from_node (learnSession *session,
                                          lingConfig *config, xmlNodePtr node,
                                          lingLesson *list);
static learnLanguage *get_language_from_node (learnSession *session,
                                              lingchar **languages,
                                              xmlNodePtr node);
static lingbool fetch_entries (xmlXPathContextPtr context,
                               learnSession *session, learnLesson *current);

static learnEntry*
create_random_learn_entries (learnLesson *lesson)
{
     unsigned int i = 0;
     unsigned int id = 0;
     learnEntry *entry = NULL;
     learnEntry *tmp = NULL;

     while (i < lesson->amount)
     {
          id = learn_random_no (lesson->end);
          tmp = entry;
          tmp = learn_entry_append_entry (entry, lesson, id);
          if (!tmp)
          {
               /* clean up */
               learn_entry_free (entry);
               return NULL;
          }
          entry = tmp;
          i++;
     }
     return entry;
}

static learnLesson*
get_lesson_from_node (learnSession *session, lingConfig *config,
                      xmlNodePtr node, lingLesson *list)
{
     lingLesson *lesson = NULL;
     lingchar *val = NULL;
     unsigned int amount = 0;
     unsigned int start = 0;
     unsigned int end = 0;
     lingbool load = TRUE;
     learnMethodType method = RANDOM;
     learnLesson *tmp;

     /* load the lesson */
     val = xmlGetProp (node, "id");
     while (list)
     {
          if (xmlStrncmp (val, ling_lesson_get_path (list),
                          xmlStrlen (val)) == 0)
          {
               load = FALSE;
               break;
          }
          list = list->next;
     }
     if (load)
          lesson = ling_lesson_create (val, config);
     else
          lesson = list;

     xmlFree (val);
     if (!lesson)
          return NULL;

     val = xmlGetProp (node, "amount");
     amount = atoi (val);
     xmlFree (val);

     val = xmlGetProp (node, "start");
     start = atoi (val);
     xmlFree (val);

     val = xmlGetProp (node, "end");
     end = atoi (val);
     xmlFree (val);

     val = xmlGetProp (node, "access");
     method = atoi (val);
     xmlFree (val);
     
     tmp = learn_lesson_append_lesson (session->lessons, lesson,
                                       amount, start, end, method);
     if (!tmp)
     {
          ling_lesson_free (lesson);
          return NULL;
     }

     session->lessons = tmp;
     while (tmp->next)
          tmp = tmp->next;
     return tmp;
}

static learnLanguage*
get_language_from_node (learnSession *session, lingchar **languages,
                        xmlNodePtr node)
{
     learnLanguage *lang = NULL;
     lingchar *val = NULL;
     lingchar *val2 = NULL;
     lingbool main_lang = FALSE;
     int i = 0;

     val = xmlXPathCastNodeToString (node);
     /* only those languages are added, which exist in the loaded language.xml,
      * too */
     while (languages[i])
     {
          main_lang = FALSE;
          if (xmlStrncmp (val, languages[i], xmlStrlen (languages[i])) == 0)
          {
               /* in order to save memory, we will use the language array */
               val2 = xmlGetProp (node, "main");
               if (val2)
               {
                    main_lang = TRUE;
                    xmlFree (val2);
               }
                    
               lang = learn_language_append_language (session->language,
                                                      languages[i], !main_lang,
                                                      main_lang);
               if (!lang)
                    return NULL;
               session->language = lang;
          }
          /* if the language is not found, we simply skip it */
          i++;
     }
     xmlFree (val);
     return session->language;
}

static lingbool
fetch_entries (xmlXPathContextPtr context, learnSession *session,
               learnLesson *current)
{
     xmlXPathObjectPtr entry_obj = NULL;
     int i = 0;
     int cnt = 0;
     unsigned int id = 0;
     unsigned int cur_id = 0;
     size_t size = 0;
     lingchar *name = NULL;
     lingchar *val = NULL;
     lingchar *search = NULL;
     learnEntry *entry = NULL;
     enum { CURRENT,
            ENTRY };
     const lingchar *entries[] = { "/session/lesson[@id='%s']/current/text()",
                                   "/session/lesson[@id='%s']/entry",
                                   NULL };

     name = ling_lesson_get_path (current->lesson);

     /* fetch entry ids and current */
     while (entries[cnt])
     {
          size = xmlStrlen (entries[cnt]) + xmlStrlen (name) + 1;
          search = xmlMalloc (size);
          if (!search)
               return FALSE;

          snprintf (search, size, entries[cnt], name);

          entry_obj = xmlXPathEval (search, context);
          xmlFree (search);
          if (!entry_obj)
               return FALSE;
          
          switch (cnt)
          {
          case CURRENT:
               val = xmlXPathCastToString (entry_obj);
               cur_id = (unsigned int) atoi (val);
               xmlFree (val);
               break;
          case ENTRY:
               while (i < entry_obj->nodesetval->nodeNr)
               {
                    val = xmlNodeGetContent (entry_obj->nodesetval->nodeTab[i]);
                    id = (unsigned int) atoi (val);
                    xmlFree (val);

                    /* add entry */
                    entry = learn_entry_append_entry (session->entry, current,
                                                      id);
                    if (!entry)
                    {
                         xmlXPathFreeObject (entry_obj);
                         return FALSE;
                    }
                    session->entry = entry;
                    i++;
               }
               break;
          }
          xmlXPathFreeObject (entry_obj);
          cnt++;
     }

     /* set current entry */
     if (cur_id != 0)
     {
          entry = session->entry;
          while (entry)
          {
               if (entry->id == cur_id) /* && entry->lesson == current) */
               {
                    session->cur_entry = entry;
                    break;
               }
               entry = entry->next;
          }
     }
     return TRUE;
}

learnLanguage*
learn_language_init_new (void)
{
     learnLanguage *lang = malloc (sizeof (learnLanguage));
     if (!lang)
          return NULL;

     lang->used = FALSE;
     lang->main = FALSE;
     lang->language = NULL;
     lang->next = NULL;
     lang->prev = NULL;

     return lang;
}

learnLanguage*
learn_language_append_language (learnLanguage *lang, lingchar *language,
                                lingbool used, lingbool main)
{
     learnLanguage *list = lang;
     learnLanguage *tmp = NULL;

     while (list)
     {
          tmp = list;
          list = list->next;
     }
     
     list = learn_language_init_new ();
     if (!list)
          return NULL;
     
     list->language = language;
     list->used = used;
     list->main = main;
     list->prev = tmp;
     list->next = NULL;
     if (tmp)
          tmp->next = list;

     if (!lang)
          return list;
     return lang;
}

void
learn_language_change_main_language (learnLanguage *lang, lingchar *language)
{
     while (lang)
     {
          if (xmlStrncasecmp (lang->language, language,
                              xmlStrlen (language)) == 0)
          {
               lang->main = TRUE;
               lang->used = FALSE;
          }
          else if (lang->main)
               lang->main = FALSE;
          lang = lang->next;
     }
     return;
}

void
learn_language_use_language (learnLanguage *lang, lingchar *language,
                             lingbool used)
{
     while (lang)
     {
          if (xmlStrncasecmp (lang->language, language,
                              xmlStrlen (language)) == 0)
          {
               if (lang->main && used)
                    lang->main = FALSE;
               lang->used = used;
               break;
          }
          lang = lang->next;
     }
     return;
}

void
learn_language_free (learnLanguage *lang)
{
     learnLanguage *tmp = NULL;

     while (lang)
     {
          tmp = lang;
          lang = lang->next;
          free (tmp);
     }
     return;
}

learnLesson*
learn_lesson_new (void)
{
     learnLesson *lesson = malloc (sizeof (learnLesson));
     if (!lesson)
          return NULL;
     
     lesson->lesson = NULL;
     lesson->amount = 0;
     lesson->start = 0;
     lesson->end = 0;
     lesson->access = RANDOM;
     lesson->next = NULL;
     lesson->prev = NULL;

     return lesson;
}

learnLesson*
learn_lesson_append_lesson (learnLesson *learn_lesson, lingLesson *lesson,
                            unsigned int amount, unsigned int start,
                            unsigned int end, learnMethodType type)
{
     learnLesson *list = learn_lesson;
     learnLesson *tmp = NULL;

     while (list)
     {
          tmp = list;
          list = list->next;
     }
     
     list = learn_lesson_new ();
     if (!list)
          return NULL;
     
     list->lesson = lesson;
     list->amount = amount;
     list->start = start;
     list->end = end;
     list->access = type;
     list->prev = tmp;
     list->next = NULL;
     if (tmp)
          tmp->next = list;

     if (!learn_lesson)
          return list;
     return learn_lesson;
}

learnLesson*
learn_lesson_remove_lesson (learnLesson *learn_lesson, lingLesson *lesson)
{
     learnLesson *list = learn_lesson;

     while (list)
     {
          if (list->lesson != lesson)
               list = list->next;
          else
          {
               if (list->prev)
                    list->prev->next = list->next;
               if (list->next)
                    list->next->prev = list->prev;
               if (list == learn_lesson)
                    learn_lesson = list->next;
               free (list);
               break;
          }
     }
     return learn_lesson;
}

void
learn_lesson_free (learnLesson *lesson)
{
     learnLesson *tmp = NULL;

     while (lesson)
     {
          tmp = lesson;
          lesson = lesson->next;
          free (tmp);
     }
     return;
}


learnEntry*
learn_entry_new (void)
{
     learnEntry *entry = malloc (sizeof (learnEntry));
     if (!entry)
          return NULL;

     entry->lesson = NULL;
     entry->id = 0;
     entry->next = NULL;
     entry->prev = NULL;
     
     return entry;
}

learnEntry*
learn_entry_append_entry (learnEntry *entry, learnLesson *lesson,
                          unsigned int id)
{
     learnEntry *list = entry;
     learnEntry *tmp = NULL;

     while (list)
     {
          tmp = list;
          list = list->next;
     }

     list = learn_entry_new ();
     if (!list)
          return NULL;
     
     list->lesson = lesson;
     list->id = id;
     list->prev = tmp;
     list->next = NULL;
     if (tmp)
          tmp->next = list;

     if (!entry)
          return list;
     return entry;
}

learnEntry*
learn_entry_create_from_lesson (learnLesson *lesson)
{
     unsigned int i = 0;
     learnEntry *entry = NULL;
     learnEntry *tmp = NULL;

     switch (lesson->access)
     {
     case RANDOM:
          entry = create_random_learn_entries (lesson);
          break;
     case SEQUENCE:
          for (i = lesson->start; i < lesson->amount && i <= lesson->end; i++)
          {
               tmp = entry;
               tmp = learn_entry_append_entry (entry, lesson, i);
               entry = tmp;
          }
          break;
     case REVIEW:
     case LEARN:
     default:
          entry = create_random_learn_entries (lesson);
          break;
     }
     return entry;
}

void
learn_entry_free (learnEntry *entry)
{
     learnEntry *tmp = NULL;

     while (entry)
     {
          tmp = entry;
          entry = entry->next;
          free (tmp);
     }
     return;
}

learnSession*
learn_session_new (void)
{
     learnSession *session = malloc (sizeof (learnSession));
     if (!session)
          return NULL;

     session->file = NULL;
     session->lessons = NULL;
     session->language = NULL;
     session->entry = NULL;
     session->hints = FALSE;
     session->cur_entry = NULL;

     return session;
}

void
learn_session_free (learnSession *session)
{
     learn_lesson_free (session->lessons);
     learn_language_free (session->language);
     learn_entry_free (session->entry);
     if (session->file)
          free (session->file);
     free (session);
     return;
}

learnSession*
learn_session_load (char *file, lingConfig *config, lingchar **languages,
                    lingLesson **list)
{
     xmlDocPtr doc = NULL;
     xmlXPathContextPtr context = NULL;
     xmlXPathObjectPtr obj = NULL;
     learnSession *session = NULL;
     learnLesson *lesson = NULL;
     lingLesson *tmp = NULL;
     enum { LESSON,
            LANGUAGE };
     const lingchar *xpaths[] = { "/session/lesson",
                                  "/session/language",
                                  NULL };
     int i = 0;
     int j = 0;

     session = learn_session_new ();
     if (!session)
          return NULL;

     /* copy file name */
     session->file = malloc (strlen (file) + 1);
     if (!session->file)
     {
          free (session);
          return NULL;
     }
     strncpy (session->file, file, strlen (file) + 1);

     doc = xmlParseFile (file);
     if (!doc)
     {
          learn_session_free (session);
          return NULL;
     }

     xmlXPathInit ();
     context = xmlXPathNewContext (doc);
     while (xpaths[i])
     {
          obj = xmlXPathEval (xpaths[i], context);
          if (!obj)
          {
               xmlXPathFreeContext (context);
               xmlFreeDoc (doc);
               learn_session_free (session);
               return NULL;
          }

          switch (i)
          {
          case LESSON:
               j = 0;
               while (j < obj->nodesetval->nodeNr)
               {
                    lesson = get_lesson_from_node (session, config,
                                                   obj->nodesetval->nodeTab[j],
                                                   *list);
                    if (!lesson)
                    {
                         learn_session_free (session);
                         xmlXPathFreeObject (obj);
                         xmlXPathFreeContext (context);
                         xmlFreeDoc (doc);
                         return NULL;
                    }
                    if (!fetch_entries (context, session, lesson))
                    {
                         learn_session_free (session);
                         xmlXPathFreeObject (obj);
                         xmlXPathFreeContext (context);
                         xmlFreeDoc (doc);
                         return NULL;
                    }

                    /* add the lesson to the given list */
                    tmp = *list;
                    if (!tmp)
                         *list = lesson->lesson;
                    else
                    {
                         while (tmp->next)
                              tmp = tmp->next;
                         tmp->next = lesson->lesson;
                         tmp->next->next = NULL;
                    }
                    j++;
               }
               break;
          case LANGUAGE:
               j = 0;
               while (j < obj->nodesetval->nodeNr)
               {
                    if (!get_language_from_node (session, languages,
                                                 obj->nodesetval->nodeTab[j]))
                    {
                         learn_session_free (session);
                         xmlXPathFreeObject (obj);
                         xmlXPathFreeContext (context);
                         xmlFreeDoc (doc);
                         return NULL;
                    }
                    j++;
               }
               break;
          }
          xmlXPathFreeObject (obj);
          i++;
     }

     /* TODO: add DTD for session files to validate against! */
     if (!session->lessons || !session->language || !session->entry)
     {
          learn_session_free (session);
          xmlXPathFreeObject (obj);
          xmlXPathFreeContext (context);
          xmlFreeDoc (doc);
          return NULL;
     }

     /* set hints */
     obj = xmlXPathEval ("/session/hints", context);
     if (!obj)
     {
          learn_session_free (session);
          xmlXPathFreeObject (obj);
          xmlXPathFreeContext (context);
          xmlFreeDoc (doc);
          return NULL;
     }
     else
          session->hints = (xmlXPathCastToNumber (obj) != 0) ? TRUE : FALSE;

     xmlXPathFreeObject (obj);
     xmlXPathFreeContext (context);
     xmlFreeDoc (doc);
     return session;
}

lingbool
learn_session_save (learnSession *session, char *file)
{
     xmlDocPtr doc = NULL;
     xmlNodePtr node = NULL;
     xmlNodePtr child = NULL;
     xmlNodePtr entry_child = NULL;
     lingchar *text = NULL;
     learnEntry *entry = NULL;
     learnLesson *lesson = session->lessons;
     learnLanguage *language = session->language;

     doc = xmlNewDoc ("1.0");
     node = xmlNewNode (NULL, "session");
     
     xmlDocSetRootElement (doc, node);

     /* save all languages */
     while (language)
     {
          if (language->used || language->main)
               child = xmlNewChild (node, NULL, "language",
                                    language->language);
          if (language->main)
               xmlSetProp (child, "main", "1");

          language = language->next;
     }

     /* save all lessons */
     while (lesson)
     {
          child = xmlNewChild (node, NULL, "lesson", NULL);
          xmlSetProp (child, "id", ling_lesson_get_path (lesson->lesson));

          text = malloc (sizeof (unsigned int));
          if (!text)
          {
               xmlFreeDoc (doc);
               return FALSE;
          }
          snprintf (text, sizeof (unsigned int), "%i", lesson->amount);
          xmlSetProp (child, "amount", text);

          snprintf (text, sizeof (unsigned int), "%i", lesson->start);
          xmlSetProp (child, "start", text);

          snprintf (text, sizeof (unsigned int), "%i", lesson->end);
          xmlSetProp (child, "end", text);

          snprintf (text, sizeof (unsigned int), "%i", lesson->access);
          xmlSetProp (child, "access", text);

          /* now save all ids, which belong to the lesson */
          entry = session->entry;
          while (entry)
          {
               if (entry->lesson == lesson)
               {
                    snprintf (text, sizeof (unsigned int), "%i", entry->id);
                    entry_child = xmlNewChild (child, NULL, "entry", text);
               }
               entry = entry->next;
          }

          /* check for current id */
          if (session->cur_entry->lesson == lesson)
          {
               snprintf (text, sizeof (unsigned int), "%i",
                         session->cur_entry->id);
               entry_child = xmlNewChild (child, NULL, "current", text);
          }
          free (text);
          lesson = lesson->next;
     }
     
     /* save hints */
     text = malloc (sizeof (unsigned int));
     snprintf (text, sizeof (unsigned int), "%i", (session->hints) ? 1 : 0);
     child = xmlNewTextChild (node, NULL, "hints", text);
     free (text);

     if (!xmlSaveFormatFile (file, doc, 1))
     {
          xmlFreeDoc (doc);
          return FALSE;
     }

     /* copy file name */
     if (session->file)
          free (session->file);
     session->file = malloc (strlen (file) + 1);
     strncpy (session->file, file, strlen (file) + 1);

     xmlFreeDoc (doc);
     return TRUE;
}

lingbool
learn_session_check_lesson (learnSession *session, lingLesson *lesson)
{
     learnLesson *tmp = session->lessons;
     while (tmp)
     {
          if (tmp->lesson == lesson)
               return TRUE;
          tmp = tmp->next;
     }
     return FALSE;
}

int
learn_random_no (unsigned int max)
{
     unsigned int no = max + 1;

#ifdef HAVE_RANDOM
#ifdef HAVE_SRANDOMDEV
     srandomdev ();
#endif /* HAVE_SRANDOMDEV */
     no = (unsigned int) random () % max;
#else
     no = (unsigned int) rand () % max;
#endif /* HAVE_RANDOM */

     return no;
}
