/*b
 * Copyright (C) 2001,2002  Rick Richardson
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Rick Richardson <rickr@mn.rr.com>
b*/

/*
 * LinuxTrade main program
 *
 * N.B. error checking is very sparse here
 */
#include "config.h"
#include <locale.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <ncurses.h>
#include <panel.h>
#include <errno.h>
#include <ctype.h>
#include <time.h>
#include <sys/time.h>
#include <unistd.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <netdb.h>
#include <netinet/in.h>
#include "curse.h"
#include <pthread.h>
#include "debug.h"
#include "error.h"
#include "rc.h"
#include "streamer.h"
#include "pref.h"
#include "srpref.h"
#include "linuxtrade.h"
#include "colon.h"
#include "stocklist.h"
#include "help.h"
#include "info.h"
#include "chart.h"
#include "news.h"
#include "alert.h"
#include "arca.h"
#include "island.h"
#include "qml2.h"
#include "l2sr.h"
#include "inplay.h"
#include "updown.h"
#include "splits.h"
#include "markcal.h"
#include "symlookup.h"
#include "optchain.h"
#include "holdings.h"
#include "bloomearn.h"
#include "quiet.h"
#include "wsrnearn.h"
#include "exthours.h"
#include "alertipo.h"
#include "toolmode.h"
#include "futs.h"
#include "util.h"
#include "ma.h"
#if 0
#include <glib.h>
#endif

/*
 * Options
 */
#if __CYGWIN__
	int	ColorMode = 0;
#else
	int	ColorMode = 3;
#endif
char	WriteFileName[256];
FILE	*WriteFile = NULL;
FILE	*ReadFile = NULL;
FILE	*StreamerLog = NULL;
int	SkipCount;
int	ForceDemoMode = 0;
int	ToolMode = 0;

#undef	NEEDREG

static int	Movers = 0;
static char	*Top10Market;
static char	*Top10Mode;
static int	ShowTotals = 0;
#if 0
GTree	*Lod;
GTree	*Hod;
#endif

char		RcFileName[] = "." PROGNAMESTR "/rc";
static char	Email[256];
static char	Browser[256];

RCFILE	RcFile[] =
{
	{ "streamer",	"scottrader.com",
			RC_STR, 0, 64, "Streamer type"},
#if 0	// for testing
	{ "combo",	"choiceA|Bchoice|choice C|choice D",
		RC_COMBO, 1, 64, "Source for web browser charts"},
#endif
#ifdef NEEDREG
	{ "reg_name",	"Demo User",
			RC_STR, 0, 64, "Owners full name from registration"},
	{ "alert_mail",	Email,
			RC_STR, 0, 64,
			"Users registered email address (for alerts)"},
	{ "reg_codes",	"1c00-5445-338b-5f9b",
			RC_STR, 0, 64, "Program registration code"},
#else
	{ "alert_mail",	Email,
			RC_STR, 0, 64,
			"Users email address (for alerts)"},
#endif
	{ "colormode",
			#if __CYGWIN__
				"0",
			#else
				"3",
			#endif
			RC_NUM, 0, 3, "Background color mode. "
			"0=black, 1=white, 2=user/darkish, 3=user/whitish"
	},
	{ "news_alerts","1",
			RC_NUM, 0, 2, "Global news alerts setting. "
			"0=off, 1=on by stock, 2=on for all stocks"
	},
	{ "alert_ext",	"linuxtrade.audio",
			RC_STR, 1, 64, "External alert program"},
	{ "alert_synth", "default|viavoice|ibm-web|ibmwatson-web"
			"|speechworks|festival|btl",
			RC_LIST, 1, 64,
			"TTS synthesizer to use with linuxtrade.audio"},
	{ "top10_mkt",	"Q|E|N",
			RC_LIST, 1, 64, "Default top-10 market symbol (E,N,Q)"},
	{ "top10_mode",	"%|$",
			RC_LIST, 1, 64, "Default top-10 market mode (% or $)"},
	{ "showtotals","0",
			RC_NUM, 0, 1, "Show portfolios totals in realtime. "
			"0=off, 1=on"
	},
#if 0
	// Someday, associate a market with a symbol, so that we can handle
	// a mix of US and foreign market quotes.
	{ "symbol_mkt",	"us",
			RC_STR, 1, 6, "Default market for symbol entry"},
#endif
	{ "print_cmd","a2ps -q",
			RC_STR, 1, 64,
			"Command to print plain text (with overstrikes)"},
	{ "mouse",	"1",
			RC_NUM, 0, 1, "Enable mouse buttons"
	},
	{ "fill_charts",
			#if 0
			// equis.com's online charts have gone byebye
			"quote.com|prophetfinance.com|equis.com",
			#else
			"quote.com|prophet.net|askresearch.com|moneyam(uk)",
			#endif
			RC_LIST, 1, 64, "Source for curses chart backfills"},
	{ "web_charts",	"pcquote.com|quote.com|stockcharts.com|yahoo.com"
			"|prophet.net",
			RC_LIST, 1, 64, "Source for web browser charts"},
	{ "java_charts","livecharts.com|money.net|prophet.net(chartstream)|"
			"prophet.net(javacharts)",
			RC_LIST, 1, 64, "Source for Java live charts"},
	{ "lc_user",	"demo",
			RC_STR, 1, 64, "www.livecharts.com (Lycos) username"},
	{ "lc_pass",	"demo",
			RC_PASS, 1, 64, "www.livecharts.com (Lycos) password"},
	{ "lc_encpass",	"demo",
			RC_PASS, 1, 64,
			"www.livecharts.com (Lycos) hashed password"
			" (computed from lc_pass)"},
	{ "lc_geom",	"700 x 350",
			RC_STR, 1, 64, "WIDTH x HEIGHT for LiveChart"},
	{ "lc_opts",	"5,1,2",
			RC_STR, 1, 64,
			"Display options for LiveChart (blank for none)"},
	{ "pf_user",	"Guest",
			RC_STR, 1, 64, "www.prophet.net username"},
	{ "pf_pass",	"Guest",
			RC_PASS, 1, 64, "www.prophet.net password"},
	{ "browser",	Browser,
			// "gnome-moz-remote --newwin %s",
			RC_STR, 1, 64, "Command which launches a browser"},
	{ "broker_browser",
			"mozilla -P linuxtrade -width 400 -height 630" "|"
			"MozillaFirebird -P linuxtrade -width 400 -height 630"
			,
			RC_COMBO, 1, 64,
			"Command which launches a small browser for trading"},
	{ "broker_URL",
			"https://pb.schwab.com/dwp/loginwiz" "|"
			"https://etrade.everypath.com/pdaMyPalm.html"
			,
			RC_COMBO, 1, 64,
			"URL for broker trading (use a web clipping page)"},

	//
	// Don't really expect these to change much
	//
	{ "optchain_URL",
			"http://www.pcquote.com/options/string.php?ticker=%s",
			RC_STR, 1, 64, "URL for getting option chains"},
#if 0
	{ "URL_charts",
			"http://stockcharts.com/webcgi/wb.exe?CG.web+%s",
			RC_STR, 1, 64,
			"URL for displaying all stocklist charts on a single page"},
#endif
	{ "arca_host",	"tools.tradearca.com",
			RC_STR, 1, 64, "Archipelago hostname"},
	{ "arca_port",	"80",
			RC_NUM, 0, 65535, "Archipelago port number"},
	{ "island_host","newgritch.island.com",
			RC_STR, 1, 64, "Island hostname"},
	{ "island_port","80",
			RC_NUM, 0, 65535, "Island port number"},
	{ "island_URL",	"/SERVICE/SQUOTE?STOCK=%s",
			RC_STR, 1, 64, "Island URL"},
	{ "live_quotes","5",
			RC_NUM, 0, 60*60,
			"Live quotes within documents refresh period (secs)"},
//#define BSITE "djl"
#define BSITE "schwab2"
	{ "inplay_URL",	"http://www.briefing.com/" BSITE "/inplay.htm",
			RC_STR, 1, 64, "Briefing.com InPlay URL"},
	{ "inplay_poll","5",
			RC_NUM, 0, 24*60,
			"InPlay document refresh period (mins)"},
	{ "inplay_alert","5",
			RC_NUM, 0, 24*60,
			"InPlay new document alert period (mins)"},
	{ "updown_URL",	"http://www.briefing.com/" BSITE "/updown.htm",
			RC_STR, 1, 64, "Briefing.com Up/Down-grades URL"},
	{ "updown_poll","5",
			RC_NUM, 0, 24*60,
			"Up/downgrades document refresh period (mins)"},
	{ "updown_alert","5",
			RC_NUM, 0, 24*60,
			"Up/downgrades new document alert period (mins)"},
	{ "splits_URL",	"http://www.briefing.com/print/FreeServices"
			"/fs_splits.htm",
			RC_STR, 1, 64, "Briefing.com Splits Calendar URL"},
	{ "quiet_URL",	"http://www.ipo.com/ipoinfo/printquiet.asp",
			RC_STR, 1, 64, "Quiet Period Calendar URL"},
#if 0
	{ "earnings_URL", "http://www.wsrn.com/apps/earnings/company.xpl?"
			"client_id=PRINT&f=EE&s=%s",
#else
	{ "earnings_URL", "http://www.earnings.com/com/validation.jsp"
			"?tckr=%s&pageID=earnings",
#endif
			RC_STR, 1, 64, "Earnings info for a single stock URL"},
	{ "exthours_poll","15",
			RC_NUM, 0, 60*60,
			"Extended hours refresh period (seconds)"},
	{ "timezone",
#			ifdef __CYGWIN__
				/* Doesn't understand new style TZ's */
				"EST5EDT",
#			else
				"US/Eastern",
#			endif
			RC_STR, 1, 64,
			"Timezone (New York time is recommended)"},
	{ NULL }
};

/*
 * Usage
 */
void
usage(void)
{
fprintf(stderr,
"Usage:\n"
"	" PROGNAMESTR " [options] [stock-symbol ...]\n"
"\n"
"Options:\n"
"	-c bg		Set background color\n"
"			0=black, 1=white, 2=user/darkish, 3=user/whitish [3]\n"
"	-l listnum	Start display with stocklist # (1-9) [1]\n"
"	-t streamer	Streamer type name [scottrader.com]\n"
"			Can be none, scottrader.com, or money.net\n"
"	-u username	Set username [NONE]\n"
"	-p password	Set password [JPY9747]\n"
"	-h hostname	Scottrade host to use for data [63.240.138.21]\n"
"	-P 'pref=val'	Set the preference named 'pref' to 'value'\n"
"	-r filename	Replay session from filename\n"
"	-w filename	Save session to filename\n"
"	-s count	Skip count quotes before displaying\n"
"	-D level	Turn on debugging\n"
"	-L log		Log raw packets (for debugging)\n"
"	-T modenum	Tool mode: quotes to stdout, no user interface\n"
"			1=one connection, 2=persistent, 3=next streamer\n"
"\n"
"NOTES:\n"
"	You can get a free username/password from http://www.scottrader.com\n"
"\n"
"Values that can be set in $HOME/." PROGNAMESTR "/rc:\n"
	);

	print_rc_file(RcFile, stderr);

	exit(1);
}

/*
 * Globals
 */
int	CursesInitted = FALSE;
int	CursorX = 0;
int	StockCursor = 0;
int	Ungetch = 0;		// There is an ungotten keystroke

attr_t	RevOrBold = A_REVERSE;
int	Reverse = 0;

STOCK	Stock[NUMSTOCK];
int	NumStock;
char	StockListName[64];
char	StockList0Name[64];

char	List0[NUMSTOCK * (SYMLEN+1) + 1];		// The "temporary" list
char	List0Comments[NUMSTOCK * (COMLEN+1) + 1];	// The "temporary" list

int	ListNum = 1;		// Current stocklist

int	TsDisplayed = FALSE;

#define	STDEMOTIME	(15*60)	// Demo time for Scottrader demo login
#define	DEMOTIME	(20*60)	// Demo time for unregistered product
time_t	StDemoTime = 0;
time_t	DemoTime = 0;
time_t	LastRefresh = 0;

int	NewStreamer;

pthread_mutex_t	CurseMutex = PTHREAD_MUTEX_INITIALIZER;

/*
 * Holdings mode
 */
#		define	HOLDINGS_OFF	0
#		define	HOLDINGS_ON	1
#		define	HOLDINGS_EDIT	2	// Editing mode
#		define	HOLDINGS_OBESE	3	// Editing mode, obese width
static int	Holdings = HOLDINGS_OFF;	// Show holdings
static int	SaveHoldings = HOLDINGS_OFF;	// Show holdings
static int	PerShare = 0;

/*
 * Forward declarations (yuck)
 */
void		create_ts(STOCK	*sp);
void		destroy_ts(STOCK *sp);
int		trendx(void);
int		pressurex(void);

/*
 * Compatibility
 */
#ifndef HAVE_LRINT
long int
lrint(double x)
{
	return rint(x);
}
#endif

#if 0
gint
alphacmp(gconstpointer s1, gconstpointer s2)
{
	return (strcasecmp((char *) s1, (char *) s2));
}
#endif

/*
 * Find a stock in the housekeeping records
 */
STOCK *
find_stock(char	*sym)
{
	int	i;

	for (i = 0; i < NumStock; ++i)
	{
		if (strcmp(Stock[i].sym, sym) == 0)
			return Stock + i;
	}
	return NULL;
}

/*
 * Delete a stock, return the ordinal position of the stock that
 * was deleted, or -1 if the stock can't be found.
 */
int
del_stock(char *sym)
{
	char	*p;
	STOCK	*s;
	int	i;

	for (p = sym; *p; ++p)
		*p = toupper(*p);

	s = find_stock(sym);
	if (!s)
		return -1;

	destroy_ts(s);

	--NumStock;
	for (i = s - Stock; i < NumStock; ++i)
	{
		Stock[i] = Stock[i+1];
		Stock[i].y = 4 + i;
	}
	return s - Stock;
}

/*
 * Create STOCK data structure from a string of blank separated symbols
 */
STOCK *
add_from_stocklist(char *symbols, int scroll)
{
	int	i;
	char	*s;
	char	c;
	int	len;
	STOCK	*laststock = NULL;

	for (i = 0, s = symbols; 1; ++i)
	{
		c = symbols[i];
		if (c == ' ')
			c = ',';
		symbols[i] = toupper(c);

		if (c != ',' && c != 0)
			continue;

		//
		// Add this stock
		//
		len = i - (s-symbols);
		if (len > SYMLEN)
			len = SYMLEN;

		if (NumStock >= NUMSTOCK)
		    break;

		memset(&Stock[NumStock], 0, sizeof(Stock[NumStock]));
		strncpy(Stock[NumStock].sym, s, len);
		Stock[NumStock].sym[len] = 0;
		Stock[NumStock].y = 4 + NumStock;
		Stock[NumStock].nquotes = 0;

		if ((laststock = find_stock(Stock[NumStock].sym)) == NULL)
		{
			if (DemoTime && NumStock >= 8)
			{
				display_msg(A_BOLD,
						"A maximum of 8 stocks are "
						"allowed in demo mode.\n");
				break;
			}

			if (!ToolMode)
			{
				if (scroll)
					vscrollrect(Stock[NumStock].y, 0,
						LINES-3, COLS-1, -1);

				mvprintw(Stock[NumStock].y, 0, "%-6.6s",
					Stock[NumStock].sym);

				create_ts(&Stock[NumStock]);

				alert_bind1(&Stock[NumStock]);

				laststock = &Stock[NumStock];
			}

			++NumStock;
			if (NumStock == (LINES-4-2))
			{
				symbols[i] = 0;
				break;
			}
		}

		if (c == 0)
			break;
		s = symbols + i + 1;
	}

	return laststock;
}

void
copy_quote(QUOTE *q, QUOTE *qdst, QUICKQUOTE *qq, LIVEQUOTE *lq)
{
    static QUOTE	zero;

    if (!q)
	q = &zero;

    if (qdst)
	*qdst = *q;

    if (qq)
    {
	qq->bid = q->bid;
	qq->ask = q->ask;
	qq->last = q->last;
	qq->prev_close = q->close;
	qq->bid_size = q->bid_size;
	qq->ask_size = q->ask_size;
	qq->bid_id = q->bid_id;
	qq->ask_id = q->ask_id;
	qq->volume = q->volume;
	qq->last_size = q->last_size;
	qq->timetrade = q->time;
	qq->high = q->high;
	qq->low = q->low;
    }

    if (lq)
    {
	lq->last = q->last;
	lq->close = q->close;
    }
}

void
trigger_add(STREAMER sr, char *str, int trendfund)
{
	char	sym[256];
	float	hi, lo;
	STOCK	*sp;
	ALERT	*ap;
	int	rc;

	rc = sscanf(str, "%s %f %f", sym, &hi, &lo);
	if (rc != 3)
	{
	err:
		beep();
		return;
	}

	sp = find_stock(sym);

	if (!sp)
	{
		add_from_stocklist(sym, TRUE);
		if (!ToolMode && !sr->usenews)
			alert_news_startpolling();
		sp = find_stock(sym);
		if (!sp)
			goto err;
		(*sr->send_symbols)(sr, sym, TRUE);
		(*sr->send_symbols_end)(sr, TRUE, FALSE);
	}

	display_stock_cursor(&Stock[StockCursor], FALSE);
	StockCursor = sp - Stock;
	display_stock_cursor(sp, TRUE);

	alert_delete_all(sym);

	if (hi)
	{
		ap = alert_new(sym);
		ap->alstate = ALERT_EN;
		ap->var = 'l';
		if (trendfund)
		{
			ap->op = 'F';
			ap->val = hi;
		}
		else
		{
			ap->op = '>';
			ap->val = hi - 0.01;
		}
		ap->act = ALERT_TERM | ALERT_EXT;
	}

	if (lo)
	{
		ap = alert_new(sym);
		ap->alstate = ALERT_EN;
		ap->var = 'l';
		if (trendfund)
		{
			ap->op = 'f';
			ap->val = lo;
		}
		else
		{
			ap->op = '<';
			ap->val = lo + 0.01;
		}
		ap->act = ALERT_TERM | ALERT_EXT;
	}

	alert_bind1(sp);

	alert_save();

	time(&LastRefresh);
}

void
set_timezone(void)
{
	static char	tzbuf[128];
	sprintf(tzbuf, "TZ=%s", get_rc_value(RcFile, "timezone"));
	putenv(tzbuf);
}

/*
 * Routines to intelligently format quote values in limited space
 */
#define NQFMT	16
char	QfmtVals[NQFMT][16];
int	QfmtIndex;

char *
qfmt(double val)
{
	if (QfmtIndex == NQFMT)
		return "NOSPACE";

	if (val <= -1000)
		sprintf(QfmtVals[QfmtIndex], "%6.0f", val);
	else if (val <= -100)
		sprintf(QfmtVals[QfmtIndex], "%6.1f", val);
	else if (val < 1000)
		sprintf(QfmtVals[QfmtIndex], "%6.2f", val);
	else if (val < 10000)
		sprintf(QfmtVals[QfmtIndex], "%6.1f", val);
	else if (val < 100000)
		sprintf(QfmtVals[QfmtIndex], "%6.0f", val);
	else
		return "TOOBIG";

	return QfmtVals[QfmtIndex++];
}

void
qfmt_init(void)
{
	QfmtIndex = 0;
}

/*
 * Format a volume to fit in a 6 character field by display it
 * in 100's and switching to M's if it is really big.
 */
char *
volfmt6(long val)
{
	long vol100 = val/100;

	if (QfmtIndex == NQFMT)
		return "NSPACE";

	if (vol100 <= 999999)
		sprintf(QfmtVals[QfmtIndex], "%6ld", vol100);
	else
		sprintf(QfmtVals[QfmtIndex], "%5ldm", vol100/10000);

	return QfmtVals[QfmtIndex++];
}

char *
volfmt6b(long val)
{
	if (QfmtIndex == NQFMT)
		return "NSPACE";

	if (val <= 999999)
		sprintf(QfmtVals[QfmtIndex], "%6ld", val);
	else if (val <= 9999999)
		sprintf(QfmtVals[QfmtIndex], "%5.3fM", val/(1000.0*1000));
	else if (val <= 99999999)
		sprintf(QfmtVals[QfmtIndex], "%5.2fM", val/(1000.0*1000));
	else
		sprintf(QfmtVals[QfmtIndex], "%5.1fM", val/(1000.0*1000));

	return QfmtVals[QfmtIndex++];
}

char *
pctfmt5(double numer, double denom)
{
	double	val = denom ? (numer*100) / denom : 0;
	long	ival = lrint(val);

	if (QfmtIndex == NQFMT)
		return "NOSPC";

	if (ival >= +10000000 || ival <= -10000000)
		sprintf(QfmtVals[QfmtIndex], "%chuge", (val<0) ? '-' : '+');
	else if (ival >= +10000 || ival <= -10000)
		sprintf(QfmtVals[QfmtIndex], "%+3ldK", ival/1000);
	else if (ival >= +1000 || ival <= -1000)
		sprintf(QfmtVals[QfmtIndex], "%+4ld", ival);
	else if (val >= +100.0 || val <= -100.0)
		sprintf(QfmtVals[QfmtIndex], "%+4.0f", val);
	else
		sprintf(QfmtVals[QfmtIndex], "%+4.1f", val);

	return QfmtVals[QfmtIndex++];
}

char *   // PO added this function
pctfmt6(double numer, double denom)
{
	double	val = denom ? (numer*100) / denom : 0;
	long	ival = lrint(val);

	if (QfmtIndex == NQFMT)
		return "NOSPC";

	if (ival >= +10000000 || ival <= -10000000)
		sprintf(QfmtVals[QfmtIndex], "%chuge", (val<0) ? '-' : '+');
	else if (ival >= +10000 || ival <= -10000)
		sprintf(QfmtVals[QfmtIndex], "%+3ldK", ival/1000);
	else if (ival >= +1000 || ival <= -1000)
		sprintf(QfmtVals[QfmtIndex], "%+5ld", ival);
	else if (val >= +100.0 || val <= -100.0)
		sprintf(QfmtVals[QfmtIndex], "%+5.0f", val);
	else if (val >= +10.0 || val <= -10.0)
		sprintf(QfmtVals[QfmtIndex], "%+5.1f", val);
	else
		sprintf(QfmtVals[QfmtIndex], "%+5.2f", val);

	return QfmtVals[QfmtIndex++];
}

char *
pctfmt7(double numer, double denom)
{
	double	val = denom ? (numer*100) / denom : 0;
	long	ival = lrint(val);

	if (QfmtIndex == NQFMT)
		return "NOSPC";

	if (val >= +1000000000 || val <= -1000000000)
		sprintf(QfmtVals[QfmtIndex], "  %chuge", (val<0) ? '-' : '+');
	else if (ival >= +1000000 || ival <= -1000000)
		sprintf(QfmtVals[QfmtIndex], "%+5ldK", ival/1000);
	else if (ival >= +100000 || ival <= -100000)
		sprintf(QfmtVals[QfmtIndex], "%+6ld", ival);
	else if (val >= +10000.0 || val <= -10000.0)
		sprintf(QfmtVals[QfmtIndex], "%+6.0f", val);
	else
		sprintf(QfmtVals[QfmtIndex], "%+6.1f", val);

	return QfmtVals[QfmtIndex++];
}

char *
dolfmt8(double val)
{
	if (QfmtIndex == NQFMT)
		return "NSPACE";

	if (val >= 10000.00)
		sprintf(QfmtVals[QfmtIndex], "%7.0fK", val/1000.0);
	else if (val > -10000.00)
		sprintf(QfmtVals[QfmtIndex], "%8.2f", val);        
	else
		sprintf(QfmtVals[QfmtIndex], "%7.0fK", val/1000.0);

	return QfmtVals[QfmtIndex++];
}

char *
dolfmt9(double val)
{
	if (QfmtIndex == NQFMT)
		return "NSPACE";

	if (val >= 100000.00)
		sprintf(QfmtVals[QfmtIndex], "%8.0fK", val/1000.0);
	else if (val > -100000.00)
		sprintf(QfmtVals[QfmtIndex], "%9.2f", val);        
	else
		sprintf(QfmtVals[QfmtIndex], "%8.0fK", val/1000.0);

	return QfmtVals[QfmtIndex++];
}

char *
dolfmt10(double val)
{
	if (QfmtIndex == NQFMT)
		return "NSPACE";

	if (val >= 1000000.00)
		sprintf(QfmtVals[QfmtIndex], "%9.0fK", val/1000.0);
	else if (val > -1000000.00)
		sprintf(QfmtVals[QfmtIndex], "%10.2f", val); 
	else
		sprintf(QfmtVals[QfmtIndex], "%9.0fK", val/1000.0);

	return QfmtVals[QfmtIndex++];
}

char *
dolfmt11(double val)
{
	if (QfmtIndex == NQFMT)
		return "NSPACE";

	if (val >= 10000000.00)
		sprintf(QfmtVals[QfmtIndex], "%10.0fK", val/1000.0);
	else if (val > -10000000.00)
		sprintf(QfmtVals[QfmtIndex], "%11.2f", val); 
	else
		sprintf(QfmtVals[QfmtIndex], "%10.0fK", val/1000.0);

	return QfmtVals[QfmtIndex++];
}

char *
qfmt_etime(time_t dt)
{
	if (QfmtIndex == NQFMT)
		return "NSPACE";

	sprintf(QfmtVals[QfmtIndex], "%ldd%02ld:%02ld:%02ld",
			dt / (24*60*60),
			(dt/3600) % 24,
			(dt/60) % 60,
			dt % 60);

	return QfmtVals[QfmtIndex++];
}

/*
 * Set the color of numeric field based on its value compared to its old value
 */
void
set_color(int y, int x, int wid, double new, double old,
		attr_t ahigh, attr_t aequal, attr_t alow)
{
	attr_t	attr;

	if (new > old)
		attr = ahigh;
	else if (new < old)
		attr = alow;
	else
		attr = aequal;

	if (attr == A_INVIS)
		return;

#if 0
	// Busted in RH 7.2
	mvchgat(y, x, wid, attr, 0, NULL);
#else
	{
		int	i;

		for (i = 0; i < wid; ++i)
		{
			chtype	ch;

			ch = mvinch(y, x+i);
			ch &= 0xff;
			ch |= attr;
			mvaddch(y, x+i, ch);
		}
	}
#endif
}

WINDOW *
minmaxwin(int minh, int maxh, int anchorbot, int right)
{
	int	w, h;
	int	x, y;
	int	avail, used;

	if (COLS >= 160)
	{
	    x = right ? COLS / 2 : 0;
	    w = 80;
	}
	else
	{
	    x = 0;
	    w = COLS;
	}

	used = 4 + 2 + NumStock + 12;
	avail = LINES - used;
	if (avail >= minh)
	{
	    // Will fit below stocks *and* top10
	    h = (avail > maxh) ? maxh : avail;
	    y = anchorbot ? LINES-2-h : 4+NumStock+12;
	    return newwin(h, w, y, x);
	}

	used -= 12;
	avail = LINES - used;
	if (avail >= minh)
	{
	    // Will fit below stocks...
	    h = (avail > maxh) ? maxh : avail;
	    y = anchorbot ? LINES-2-h : 4+NumStock;
	    return newwin(h, w, y, x);
	}

	avail = LINES-2;
	if (avail >= minh)
	{
	    // Will fit, but will obscure some stocks...
	    h = minh;
	    y = anchorbot ? LINES-h : 0;
	    return newwin(h, w, y, x);
	}

	// Won't fit anywhere
	return newwin(LINES-2, w, 0, x);
}

WINDOW *
bestwin(int lines)
{
	int	avail;
	int	cols;

	if (COLS >= 160)
		cols = 80;
	else
		cols = COLS;

	avail = LINES - 4 - 2 - NumStock - 12;
	if (avail >= lines)
		return newwin(lines, cols, 4+NumStock+12, 0);
	avail += 12;
	if (avail >= lines)
		return newwin(lines, cols, 4+NumStock, 0);
	avail = LINES - 2;
	if (avail >= lines)
		return newwin(lines, cols, LINES-2-lines, 0);
	return newwin(LINES-2, cols, 0, 0);
}

WINDOW *
bestrightwin(int lines, int anchorbot)
{
	int	avail;
	int	x, cols;

	if (COLS >= 160)
	{
		x = COLS / 2;
		cols = COLS - x;
	}
	else
	{
		x = 0;
		cols = COLS;
	}

	avail = LINES - 4 - 2 - NumStock - 12;
	if (anchorbot)
	{
		if (avail >= lines)
			return newwin(lines, cols, LINES-2-lines, x);
		avail += 12;
		if (avail >= lines)
			return newwin(lines, cols, LINES-2-lines, x);
		avail = LINES - 2;
		if (avail >= lines)
			return newwin(lines, cols, LINES-2-lines, x);
	}
	else
	{
		if (avail >= lines)
			return newwin(lines, cols, 4+NumStock+12, x);
		avail += 12;
		if (avail >= lines)
			return newwin(lines, cols, 4+NumStock, x);
		avail = LINES - 2;
		if (avail >= lines)
			return newwin(lines, cols, LINES-2-lines, x);
	}

	// Last resort, use all available space except bottom 2 lines
	return newwin(LINES-2, cols, 0, x);
}

/*
 * Create Time and Sales windows 
 */
void
create_ts(STOCK	*sp)
{
	sp->tsdisplay = FALSE;
	sp->tswin = newwin(10, 25, 0, 0);
	wbkgd(sp->tswin, Reverse ? A_REVERSE : A_NORMAL);

	box(sp->tswin, 0, 0);
	mvwprintw(sp->tswin, 0, (25-strlen(sp->sym))/2, sp->sym);
	sp->tssub = derwin(sp->tswin, 8, 25-2, 1, 1);
	sp->tspanel = new_panel(sp->tswin);
	hide_panel(sp->tspanel);
}

void
destroy_ts(STOCK *sp)
{
	if (!sp || !sp->tswin)
		return;

	del_panel(sp->tspanel);
	delwin(sp->tssub);
	delwin(sp->tswin);
	sp->tswin = NULL;
	sp->tsdisplay = FALSE;
}

void
display_ts(void)
{
	int	i;
	int	x, y;
	int	cnt;
	int	xts;
	int	needrows;
	int	availrows;
	int	cols, sx;

	for (cnt = i = 0; i < NumStock; ++i)
		if (Stock[i].tsdisplay && Stock[i].tspanel && Stock[i].sym[0] != '$')
			++cnt;

	if (COLS >= 160)
	{
		sx = 80;
		cols = COLS - sx;
	}
	else
	{
		sx = 0;
		cols = COLS;
	}

	xts = cols / 25;
	needrows = (cnt + xts - 1) / xts;
	needrows *= 10;

	if (COLS >= 160)
	{
		y = 4 + NumStock + 1;
	}
	else
	{
		availrows = LINES - 2 - 4 - NumStock - 12;
		if (availrows > needrows)
			y = 4 + NumStock + 12;
		else
			y = 4 + NumStock;
	}

	x = 0;
	for (i = 0; i < NumStock; ++i)
	{
		if (!Stock[i].tspanel)
			continue;

		// Don't display index T&S :-)
		if (Stock[i].sym[0] == '$')
			continue;
		if (!Stock[i].tsdisplay)
			continue;

		mvwin(Stock[i].tswin, y, sx + x);
		show_panel(Stock[i].tspanel);
		x += 25;
		if (x >= (cols-25))
		{
			x = 0;
			y += 10;
			if (y > (LINES-2-10))
				break;
		}
	}
}

void
hide_ts(void)
{
	int	i;

	for (i = 0; i < NumStock; ++i)
	{
		if (!Stock[i].tspanel)
			continue;
		hide_panel(Stock[i].tspanel);
	}
}

/*
 * Display the title on the top row
 */
char	Title[] = "LinuxTrade";

static int	DateX;

void
display_title(void)
{
	time_t		now;
	struct tm	*tmp;
	char		buf[64];
	static int	first = 1;
	char		*datefmt = "%x %X %Z";

	if (first)
	{
		first = 0;
		mvprintw(0, 0, "%s", Title);
	}

	time(&now);
	tmp = localtime(&now);
	strftime(buf, sizeof(buf), datefmt, tmp);
	mvprintw(0, DateX = (COLS - strlen(buf) - 1), "%s", buf);

	if (DemoTime)
	{
		attrset(COLOR_PAIR(2));
		mvprintw(1, COLS-4 - 1, "DEMO");
		attrset(A_NORMAL);
	}
	else
	{
		mvprintw(1, COLS-4 - 1, "    ");
	}
}

/*
 * Display the communications status indicator on the first row
 */
void
display_status(STREAMER sr, int online)
{
	char		chars[] = "|/-\\|/-\\";
	static int	i = 0;
	int		col = sizeof(Title);
	int		x;

	if (chars[i] == 0)
		i = 0;
	if (online)
		mvaddch(0, col, COLOR_PAIR(1) | chars[i++]);
	else
		mvaddch(0, col, COLOR_PAIR(2) | 'X');

	printw(" [%s]  ", sr->id);

	if (DateX == 0)
		DateX = COLS - 1;
	attrset(A_BOLD);
	x = getcurx(stdscr);
	if (StockListName[0])
		printw("%d: %-*.*s",
			ListNum,
			DateX - x - 3,
			DateX - x - 4,
			StockListName);
	else
		printw("LIST %d%*s",
			ListNum,
			DateX - x - 5,
			"");
	attrset(A_NORMAL);
}

int
trendx(void)
{
	int	need;

	if (COLS >= (need = 80+6+39+22))	// fatbast
		return need - 5;
	else if (COLS >= (need = 80+6+39))	// obese
		return need - 5;
	else if (COLS >= (need = 80+6))		// wide
		return need - 5;
	else					// normal
		return 80 - 5;
}

int
pressurex(void)
{
	if ((COLS-trendx()) < 15)
	    return COLS;
	else
	    return COLS - 10;
}

/*
 * Display alert status
 */
void
display_alert(STOCK *sp, int on)
{
	if (on)
	{
		attrset(A_BOLD|A_BLINK);
		mvprintw(sp->y, trendx(),
			(sp->alerting & 2) ? "NEWS " : "ALERT");
		attrset(A_NORMAL);
	}
	else
	{
		attrset(A_NORMAL);
		mvprintw(sp->y, trendx(), "     ");
	}
}

void
display_delayed(STOCK *sp, int on)
{
	if (on)
	{
		attrset(A_BOLD|A_BLINK);
		mvprintw(sp->y, trendx(), "DELAY");
		attrset(A_NORMAL);
	}
	else
	{
		attrset(A_NORMAL);
		mvprintw(sp->y, trendx(), "     ");
	}
}

void
display_pressure(STOCK *sp)
{
	int	x, b, s;

	x = pressurex();
	if (x >= COLS)
	    return;

	s = (sp->press_sell+3) / 6;
	s %= 5;
	b = (sp->press_buy+3) / 6;
	b %= 5;

	attrset(A_NORMAL);
	mvprintw(sp->y, x, "%*s", 5-s, "");
	attrset(COLOR_PAIR(BLACKonRED));
	printw("%.*s", s, "-----");

	attrset(COLOR_PAIR(BLACKonGREEN));
	printw("%.*s", b, "+++++");
	attrset(A_NORMAL);
	printw("%*s", 5-b, "");
}

/*
 * Display status message
 */
void
display_msg(attr_t attr, char *fmt, ...)
{
	va_list	ap;

	if (ToolMode)
	    return;

	blankrect(LINES-2, 0, LINES-2, COLS-1, 1);

	attrset(attr);
	move(LINES-2, 0);
        va_start(ap, fmt);
        vwprintw(stdscr, fmt, ap);
        va_end(ap);
	attrset(A_NORMAL);
}

void
display_prompt(attr_t attr, char *fmt, ...)
{
	va_list	ap;

	if (ToolMode)
	    return;

	blankrect(LINES-1, 0, LINES-1, COLS-1, 1);

	attrset(attr);
	move(LINES-1, 0);
        va_start(ap, fmt);
        vwprintw(stdscr, fmt, ap);
        va_end(ap);
	attrset(A_NORMAL);
}

/*
 *	Turn on/off cursor highlighting on the specified line
 */
static void
display_cursor(int y, int on)
{
	linecursor(y, 0, trendx(), on, RevOrBold);
}

void
display_stock_cursor(STOCK *sp, int on)
{
	display_cursor(sp->y, on);
	if (on && sp->comment[0])
		display_msg(A_NORMAL, "%s: %.*s",
			sp->sym, COLS-16, sp->comment);
	else
		display_msg(A_NORMAL, "");
}

static void
display_holdings_totals(int onoff)
{
	double	totcost, totvalue, totgain, gain, yesterday;
	int	i;
	int	row = 4 + NumStock;
	STOCK	*sp;
	int	obese = (COLS >= (80+6+39));

	if (onoff == FALSE)
	{
		move(row, 0);
		clrtoeol();
		return;
	}

	if (!Holdings && !obese)
		return;

	totcost = totvalue = totgain = yesterday = 0;
	for (i = 0, sp = Stock; i < NumStock; ++i, ++sp)
	{
		totcost += sp->cost;
		yesterday += sp->nshares * sp->cur.close;
		if (sp->cost >= 0)
			gain = sp->nshares * sp->cur.last - sp->cost;
		else
			gain = -sp->cost - sp->nshares * sp->cur.last;
		totgain += gain;
	}
	totvalue = totcost + totgain;

	switch (Holdings)
	{
	default:
		break;
	case HOLDINGS_OFF:
	case HOLDINGS_OBESE:
		if (!obese)
			break;
		qfmt_init();
		if (yesterday != 0)
		{
			mvprintw(row, 44, "%9s %6s%29s%11s%11s %7s =%-12.2f",   // PO chg a lot
				 dolfmt9(totvalue-yesterday),                 // PO added
				 pctfmt6(totvalue-yesterday, yesterday), "",  // PO 6 from 7
				dolfmt11(totcost), dolfmt11(totgain),
				pctfmt7(totgain,
					totcost >= 0 ? totcost : -totcost),
				totvalue
				);
			set_color(row, 44, 16, totvalue, yesterday,     // PO change line
				COLOR_PAIR(1), A_NORMAL, COLOR_PAIR(2));
		}
		else
		{
			mvprintw(row, 89, "%11s%11s %7s =%-12.2f",     // PO made 89 from 88, -$
				dolfmt11(totcost), dolfmt11(totgain),
				pctfmt7(totgain,
					totcost >= 0 ? totcost : -totcost),
				totvalue
				);
		}		
		set_color(row, 100, 19, totgain, 0.0,                 // PO 19 from 18
			COLOR_PAIR(1), A_NORMAL, COLOR_PAIR(2));
		break;
	case HOLDINGS_ON:
	case HOLDINGS_EDIT:
		qfmt_init();
		mvprintw(row, 22, "%9s%9s =$%-12.2f",
			dolfmt9(totcost), dolfmt9(totgain),
			totvalue);
		set_color(row, 31, 9, totgain, 0.0,
			COLOR_PAIR(1), A_NORMAL, COLOR_PAIR(2));
		break;
	}
}

void
display_quote_hdr(void)
{
	int	wide = (COLS >= (80+6));
	int	obese = (COLS >= (80+6+39));
	int	fatbast = (COLS >= (80+6+39+22));
	int	x;

	char	norm_fmt[] =
			"%-6.6s%-8s"
			" %6s/%-3s %6s/%-3s"
			" %6s/%-3s";
	char	hold_fmt[] =
			"%-6.6s%-8s"
			" %7s %8s %8s"
			" %6s";
	char	narr_fmt[] =
			" %5s"
			" %6s %6s"
			" %6s %s";
	char	wide_fmt[] =
			" %5s %5s"
			" %6s %6s"
			" %6s %s";
	char	obese_fmt[] =
			" %5s %6s"
			" %6s %6s"
			" %6s"
			" %7s %10s %10s %7s %s";
	char	fatbast_fmt[] =
			" %5s %6s"   // PO made 6 from 5
			" %6s %6s"
			" %6s"
			" %7s %10s %10s %7s  %-20s %s";  // PO added 2 extra spaces
			

	attrset(A_BOLD);

	if (Holdings)
		mvprintw(2, 0, hold_fmt,
			"STOCK", "TIME",
			"SHARES", PerShare ? "SHRCOST" : "TOTCOST",
			"GAIN",
			"LAST");
	else
		mvprintw(2, 0, norm_fmt,
			"STOCK", "TIME",
			"BID", "SZ",
			"ASK", "SZ",
			"LAST", "SZ");

	if (fatbast)
		printw(fatbast_fmt,
			"CHG", "%CHG",
			"HIGH", "LOW",
			"VOL",
			"SHARES", PerShare ? "SHRCOST" : "TOTCOST",
			"GAIN", "%GAIN",
			"COMMENTS",
			"TREND");
	else if (obese)
		printw(obese_fmt,
			"CHG", "%CHG",
			"HIGH", "LOW",
			"VOL",
			"SHARES", PerShare ? "SHRCOST" : "TOTCOST",
			"GAIN", "%GAIN",
			"TREND");
	else if (wide)
		printw(wide_fmt,
			"CHG", "%CHG",
			"HIGH", "LOW",
			"VOL", "TREND");
	else
		printw(narr_fmt,
			"CHG",
			"HIGH", "LOW",
			"VOL", "TREND");

	x = pressurex();
	if (x < COLS)
	    mvprintw(2, x, " PRESSURE ");

	attrset(A_NORMAL);

	if (0)
	    for (x = 0; x < COLS; x += 10)
		mvprintw(3, x, "123456789 ");
}

/*
 * Display a quote value
 *
 * tradetype:
 * 	0 = nosale
 * 	1 = regular sale
 * 	2 = offhours sale???
 */
static int FirstDisplayQuote = 1;

void
display_quote(QUOTE *q, int tradetype)
{
	STOCK	*sp;
	int	n;
	int	x;
	double	gain, shrcost;
	int	wide = (COLS >= (80+6));
	int	obese = (COLS >= (80+6+39));
	int	fatbast = (COLS >= (80+6+39+22));

	char	quote_fmt[] =
			"%-6.6s%02d:%02d:%02d"
			" %6s %-3d %6s %-3d %6s %-3d";
	char	hold_fmt[] =
			"%-6.6s%02d:%02d:%02d"	//sym, time
			" %7d%9s%9s %6s";	//shares, cost, gain, last
	char	narr_fmt[] =
			"%6s %6s %6s %6s";	// chg, hi, lo, vol
	char	wide_fmt[] =
			"%6s %5s %6s %6s %6s";// chg, %chg, hi, lo, vol
	char	obese_fmt[] =
			"%6s %6s %6s %6s %6s"	// chg, %chg, hi, lo, vol  (PO MADE %chg 6 not 5)
			" %7d%11s"		// shares, cost
			"%11s %7s";		// gain, %gain
	char	obese1_fmt[] =
			"%6s %6s %6s %6s %6s";	// chg, %chg, hi, lo, vol
	char	obese2_fmt[] =
			"%10s %7s";		// gain, %gain

	if (ToolMode)
	{
		tool_quote(q, tradetype);
		return;
	}

	if (FirstDisplayQuote)
	{
		FirstDisplayQuote = 0;
		display_quote_hdr();
	}

	sp = find_stock(q->sym);
	if (!sp)
		return;
	if (SkipCount)
	{
		--SkipCount;
		return;
	}
	if (sp->nquotes++ == 0)
		sp->cur = *q;
	sp->old = sp->cur;
	if (Debug >= 6)
	{
		q->bid = q->bid + (random() % 10) - 5;
		q->ask = q->ask + (random() % 10) - 5;
		q->last = q->last + (random() % 10) - 5;
		q->high = q->high + (random() % 10) - 5;
		q->low = q->low + (random() % 10) - 5;

		tradetype = random() % 2;
		sp->press_sell = random() % 31;
		sp->press_buy = random() % 31;
	}

	// Quote update records don't include high/low
	if (q->low == 0)
	    q->low = q->last;
	if (q->last > q->high)
		q->high = q->last;
	if (q->last < q->low)
		q->low = q->last;

	sp->cur = *q;

	n = sp - Stock;

	qfmt_init();

	if (sp->cost >= 0)
		gain = sp->nshares * q->last - sp->cost;
	else
		gain = -sp->cost - sp->nshares * q->last;

	/*
	 * Check for new alerts
	 */
	alert_check(sp);

	switch (Holdings)
	{
	case HOLDINGS_OFF:
	case HOLDINGS_OBESE:
		// Holdings mode off or obese display
		mvprintw(sp->y, 0, quote_fmt,
			q->sym,
			q->time/3600, (q->time/60) % 60, q->time % 60,
			qfmt(q->bid),
			q->bid_size > 999 ? 999 : q->bid_size,
			qfmt(q->ask),
			q->ask_size > 999 ? 999 : q->ask_size,
			qfmt(q->last),
			q->last_size > 99999 ? 999 : q->last_size/100);
		break;
	case HOLDINGS_ON:
		// Holdings on, update all holdings info
		shrcost = sp->nshares ? (sp->cost/sp->nshares) : 0;
		mvprintw(sp->y, 0, hold_fmt,
			q->sym,
			q->time/3600, (q->time/60) % 60, q->time % 60,
			sp->nshares,
			PerShare ? dolfmt9(shrcost) : dolfmt9(sp->cost),
			dolfmt9(gain),
			qfmt(q->last));

		x = getcurx(stdscr);
		set_color(sp->y, 31, 9, gain, 0.0,
			COLOR_PAIR(1), A_NORMAL, COLOR_PAIR(2));
		move(sp->y, x);
		break;
	case HOLDINGS_EDIT:
		// Holdings on, but don't overwrite shares and totcost
		mvprintw(sp->y, 0, "%-6.6s%02d:%02d:%02d",
			q->sym,
			q->time/3600, (q->time/60) % 60, q->time % 60);

		mvprintw(sp->y, 32, "%8s %6s", dolfmt8(gain), qfmt(q->last));

		x = getcurx(stdscr);
		set_color(sp->y, 32, 8, gain, 0.0,
			COLOR_PAIR(1), A_NORMAL, COLOR_PAIR(2));
		move(sp->y, x);
		break;
	}

	if (obese)
	{
		if (Holdings == HOLDINGS_OBESE)
		{
			printw(obese1_fmt,
				qfmt(q->last - q->close),
				pctfmt5(q->last - q->close, q->close),
				qfmt(q->high),
				qfmt(q->low),
				volfmt6(q->volume));
			move(sp->y, 101);
			printw(obese2_fmt,
				dolfmt10(gain),
				pctfmt7(gain, sp->cost >= 0
					? sp->cost : - sp->cost)
				);
			set_color(sp->y, 101, 18, gain, 0.0,
				COLOR_PAIR(1), A_NORMAL, COLOR_PAIR(2));
		}
		else
		{
			shrcost = sp->nshares ? (sp->cost/sp->nshares) : 0;
			printw(obese_fmt,
				qfmt(q->last - q->close),
				pctfmt6(q->last - q->close, q->close),  // PO made 6
				qfmt(q->high),
				qfmt(q->low),
				volfmt6(q->volume),
				sp->nshares,
				PerShare?dolfmt11(shrcost) : dolfmt11(sp->cost),
				dolfmt11(gain),
				pctfmt7(gain, sp->cost >= 0
					? sp->cost : - sp->cost)
				);
			set_color(sp->y, 100, 19, gain, 0.0,
				COLOR_PAIR(1), A_NORMAL, COLOR_PAIR(2));
		}
		if (fatbast)
		{
			printw("  ");
			if (sp->comment[0])
				printw("%-20.20s", sp->comment);
			else
				alert_comment(sp, 20);
		}
	}
	else if (wide)
		printw(wide_fmt,
			qfmt(q->last - q->close),
			pctfmt5(q->last - q->close, q->close),
			qfmt(q->high),
			qfmt(q->low),
			volfmt6(q->volume)
			);
	else
		printw(narr_fmt,
			qfmt(q->last - q->close),
			qfmt(q->high),
			qfmt(q->low),
			volfmt6(q->volume)
			);

	/*
	 * Mark change
	 */
	set_color(sp->y, 47, 6, q->last - q->close, 0.0,
		COLOR_PAIR(1), A_NORMAL, COLOR_PAIR(2));
	if (wide)
		set_color(sp->y, 47+6, 7, q->last - q->close, 0.0,  // PO MADE 7 instead of 6
			COLOR_PAIR(1), A_NORMAL, COLOR_PAIR(2));

	/*
	 * Mark direction of bid/ask/last
	 */
	if (Holdings == HOLDINGS_OFF)
	{
		set_color(sp->y, 15, 6, q->bid, sp->old.bid,
			COLOR_PAIR(1), A_NORMAL, COLOR_PAIR(2));
		set_color(sp->y, 26, 6, q->ask, sp->old.ask,
			COLOR_PAIR(1), A_NORMAL, COLOR_PAIR(2));
	}
	set_color(sp->y, Holdings ? 41 : 37, 6, q->last, sp->old.last,
		COLOR_PAIR(1), A_NORMAL, COLOR_PAIR(2));

	/*
	 * Indicate new highs/lows
	 */
	if (Holdings == HOLDINGS_OFF)
	{
		if (q->bid)
			set_color(sp->y, 0, 6, q->bid, sp->old.high,
				COLOR_PAIR(1), COLOR_PAIR(1), A_NORMAL);
		if (q->ask)
			set_color(sp->y, 0, 6, q->ask, sp->old.low,
				A_INVIS, A_INVIS, COLOR_PAIR(2));
	}

	if (sp->old.high)
		set_color(sp->y, wide ? 61 : 54, 6, q->high, sp->old.high,
			COLOR_PAIR(YELLOWonBLACK), A_NORMAL, A_NORMAL);
	set_color(sp->y, wide ? 68 : 61, 6, q->low, sp->old.low,
		A_NORMAL, A_NORMAL, COLOR_PAIR(YELLOWonBLACK));

	/*
	 * Mark trends if not alerting and not delayed
	 */
	if (!sp->alerting && !q->delayed && q->sym[0] != '$')
	{
	    int		sx = trendx();
	    int		ex = pressurex() - 1;
	    QUOTE	*old = &sp->old;

	    #if 0 || defined(DATEK_STYLE)
		    #define BA_UP	(' ' | COLOR_PAIR(5))
		    #define BA_DN	(' ' | COLOR_PAIR(4))
		    #define BUP_ADN	(ACS_S3 | COLOR_PAIR(4))
		    #define BDN_AUP	(ACS_S7 | COLOR_PAIR(4))

		    if (q->bid > old->bid)
		    {
			    if (q->ask < old->ask)
				    mvinsch(sp->y, sx, BUP_ADN);
			    else if (q->ask > old->ask)
				    mvinsch(sp->y, sx, A_BOLD | BA_UP);
			    else
				    mvinsch(sp->y, sx, BA_UP);
		    }
		    else if (q->bid < old->bid)
		    {
			    if (q->ask > old->ask)
				    mvinsch(sp->y, sx, BDN_AUP);
			    else if (q->ask < old->ask)
				    mvinsch(sp->y, sx, A_BOLD | BDN_AUP);
			    else
				    mvinsch(sp->y, sx, BA_DN);
		    }
		    else
		    {
			    if (q->ask > old->ask)
				    mvinsch(sp->y, sx, BA_UP);
			    else if (q->ask < old->ask)
				    mvinsch(sp->y, sx, BA_DN);
		    }
	    #else
		    if (tradetype)
		    {
			    #define GREENPLUS	('+' | COLOR_PAIR(1))
			    #define REDMINUS	('-' | COLOR_PAIR(2))
			    #define GREENEQ		('=' | COLOR_PAIR(1))
			    #define REDEQ		('=' | COLOR_PAIR(2))

			    hscrollrect(sp->y, sx, sp->y, ex, SCROLL1RIGHT);
			    if (q->last > old->last)
				    mvaddch(sp->y, sx, GREENPLUS);
			    else if (q->last < old->last)
				    mvaddch(sp->y, sx, REDMINUS);
			    else
			    {
				    chtype	ch = mvinch(sp->y, sx+1);
				    int	pair = PAIR_NUMBER(ch);

				    mvaddch(sp->y, sx, '=' | COLOR_PAIR(pair));
			    }
		    }
	    #endif
	}

	if (n == StockCursor)
		display_cursor(sp->y, TRUE);

	/*
	 * Update Time and Sales
	 */
	if (tradetype && q->sym[0] != '$')
	{
		float	last;
		attr_t	attr = A_NORMAL;

		wvscrollrect(sp->tssub, 0, 0,
			getmaxy(sp->tssub)-1, getmaxx(sp->tssub)-1, 1);

		if (tradetype == 2)
		{
			last = q->bid;	// off-hours
			if (last > sp->old.high)
				attr = COLOR_PAIR(1) | RevOrBold;
			else if (last > q->last)
				attr = COLOR_PAIR(1) | A_NORMAL;
			else if (last < sp->old.low)
				attr = COLOR_PAIR(2) | RevOrBold;
			else if (last < q->last)
				attr = COLOR_PAIR(2) | A_NORMAL;
		}
		else
		{
			last = q->last;	// regular

			if (last > sp->old.high)
				attr = COLOR_PAIR(1) | RevOrBold;
			else if (last < sp->old.low)
				attr = COLOR_PAIR(2) | RevOrBold;
			else if (last >= q->ask)
				attr = COLOR_PAIR(1) | A_NORMAL;
			else if (last <= q->bid)
				attr = COLOR_PAIR(2) | A_NORMAL;
			else
				attr = A_NORMAL;
		}

		wmove(sp->tssub, getmaxy(sp->tssub)-1, 0);
		wattrset(sp->tssub, attr);
		wprintw(sp->tssub, "%7.2f %6d %02d:%02d:%02d",
			last, q->last_size,
			q->time/3600, (q->time/60) % 60, q->time % 60
			);
		wattrset(sp->tssub, A_NORMAL);
		touchwin(sp->tswin);

		chart_update(sp, q);
	}

	/*
	 * Update pressure
	 */
	if (tradetype && q->sym[0] != '$')
	{
		int	old = sp->press_pipe[sp->press_in];

		if (old < 0)
		    --sp->press_sell;
		else if (old > 0)
		    --sp->press_buy;

		if (q->ask == q->bid)
		    sp->press_pipe[sp->press_in++] = 0;
		else if (q->last >= q->ask)
		{
		    ++sp->press_buy;
		    sp->press_pipe[sp->press_in++] = 1;
		}
		else if (q->last <= q->bid)
		{
		    ++sp->press_sell;
		    sp->press_pipe[sp->press_in++] = -1;
		}
		else
		    sp->press_pipe[sp->press_in++] = 0;

		if (sp->press_in >= asizeof(sp->press_pipe))
		    sp->press_in = 0;

		display_pressure(sp);
	}

	/*
	 * Display alert state and/or delayed state
	 */
	if (sp->alerting)
		display_alert(sp, 1);
	else if (q->delayed)
		display_delayed(sp, 1);

	/*
	 * Display portfolio totals
	 */
	if (ShowTotals)
		display_holdings_totals(TRUE);
}

void
display_livequote(LIVEQUOTE *lqp)
{
	inplay_livequote(lqp);
	updown_livequote(lqp);
	bloomearn_livequote(lqp);
	splits_livequote(lqp);
	quiet_livequote(lqp);
}

void
redisplay_list(void)
{
	int	i;

	display_quote_hdr();

	for (i = 0; i < NumStock; ++i)
	{
		STOCK	*sp = &Stock[i];

		display_quote(&sp->cur, 0);
	}
}

void
display_top10(TOP10 *t10)
{
	int	col;
	int	row;
	char	*mkt;
	int	cols = cols80();

	row = 4 + NumStock + 1;
	if ( (row+11) >= LINES-2)
		return;
	if (t10->rank < 1 || t10->rank > 10)
		return;

	if (t10->market == 'Q') mkt = "NASD";
	else if (t10->market == 'E') mkt = "AMEX";
	else if (t10->market == 'N') mkt = "NYSE";
	else return;

	if (t10->market != Top10Market[0])
		return;

	qfmt_init();
	switch (t10->type)
	{
	case TOP_VOL:
		col = 0;
		attrset(A_BOLD);
		mvprintw(row, col, "%s Top 10 Volume    ", mkt);
		attrset(A_NORMAL);
		printw("%*s", cols/3 - 22, "");
		if (t10->change > 0) color_set(1, NULL);
		else if (t10->change < 0) color_set(2, NULL);
		mvprintw(row+t10->rank, col, "%-6.6s %+8.2f %-6.6s",
			t10->sym, t10->change, volfmt6(t10->volume));
		attrset(A_NORMAL);
		printw("%*s", cols/3 - 22, "");
		break;
	case TOP_NETGAIN:
		if (Top10Mode[0] != '$') return;
		col = cols/3;
		attrset(A_BOLD);
		mvprintw(row, col, "%s Top 10 Net Gainers", mkt);
		attrset(A_NORMAL);
		printw("%*s", cols/3 - 23, "");
		mvprintw(row+t10->rank, col, "%-6.6s %+8.2f %-6.6s",
			t10->sym, t10->change, volfmt6(t10->volume));
		printw("%*s", cols/3 - 22, "");
		break;
	case TOP_NETLOSS:
		if (Top10Mode[0] != '$') return;
		col = 2*cols/3;
		attrset(A_BOLD);
		mvprintw(row, col, "%s Top 10 Net Losers", mkt);
		attrset(A_NORMAL);
		printw("%*s", cols/3 - 22, "");
		mvprintw(row+t10->rank, col, "%-6.6s %+8.2f %-6.6s",
			t10->sym, t10->change, volfmt6(t10->volume));
		printw("%*s", cols/3 - 22, "");
		break;
	case TOP_PCTGAIN:
		if (Top10Mode[0] != '%') return;
		col = cols/3;
		attrset(A_BOLD);
		mvprintw(row, col, "%s Top 10 Pct Gainers", mkt);
		attrset(A_NORMAL);
		printw("%*s", cols/3 - 23, "");
		mvprintw(row+t10->rank, col, "%-6.6s %+8.2f %-6.6s",
			t10->sym, t10->change, volfmt6(t10->volume));
		printw("%*s", cols/3 - 22, "");
		break;
	case TOP_PCTLOSS:
		if (Top10Mode[0] != '%') return;
		col = 2*cols/3;
		attrset(A_BOLD);
		mvprintw(row, col, "%s Top 10 Pct Losers", mkt);
		attrset(A_NORMAL);
		printw("%*s", cols/3 - 22, "");
		mvprintw(row+t10->rank, col, "%-6.6s %+8.2f %-6.6s",
			t10->sym, t10->change, volfmt6(t10->volume));
		printw("%*s", cols/3 - 22, "");
		break;
	}
}

#if 0
static int
filter_lhod(MKTMOVER *mm, GTree *db)
{
	time_t	expires, now;

	if (!db)
		return 0;

	if (1)
		return 0;

	time(&now);
	expires = (time_t) g_tree_lookup(db, mm->sym);
	if (expires && now < expires)
		return 1;
	expires = now + 5*60;
	g_tree_insert(db, strdup(mm->sym), (gpointer) expires);
	return 0;
}
#endif

void
display_mktmover(MKTMOVER *mm)
{
	int	col;
	int	row;
	int	nrows;
	int	cols = cols80();
	static char	hdrlo[] =   "New Lows.......................";
	static char	hdrlo52[] = "New 52-week Lows...............";
	static char	hdrhi[] =   "New Highs......................";
	static char	hdrhi52[] = "New 52-week Highs..............";

	row = 4 + NumStock + 1 + 11 + 1;
	nrows = LINES - 2 - row - 1;
	if (nrows <= 0)
		return;

	if (mm == NULL)
	{
		blankrect(row, 0, row+nrows, cols-1, FALSE);
		if (Movers == 1)
		{
			attrset(A_BOLD);
			mvprintw(row+0, 0, hdrlo);
			mvprintw(row+0, cols/2, hdrhi);
		}
		else if (Movers == 2)
		{
			attrset(A_BOLD);
			mvprintw(row+0, 0, hdrlo52);
			mvprintw(row+0, cols/2, hdrhi52);
		}
		attrset(A_NORMAL);
		return;
	}

	if (!Movers)
		return;

	if (strcmp(mm->sym, "ATEST") == 0)
		return;
	if (mm->hilo < 1 || mm->hilo > 8)
		return;

	switch (mm->hilo)
	{
	case NEAR_LO:
	case LO_DAILY:
		if (Movers != 1)
			return;
		#if 0
			if (filter_lhod(mm, Lod))
				return;
		#endif
		col = 0;
		attrset(A_BOLD);
		if (ColorMode == 0) color_set(REDonBG, NULL);
		mvprintw(row+0, col, hdrlo);
		attrset(A_NORMAL);
		vscrollrect(row+1, col, row+nrows, col + cols/2 -1, 1);
		break;
	case NEAR_LO_52:
	case LO_52:
		#if 0
			if (filter_lhod(mm, Lod))
				return;
		#endif
		col = 0;
		attrset(A_BOLD);
		if (ColorMode == 0) color_set(REDonBG, NULL);
		mvprintw(row+0, col, Movers == 2 ? hdrlo52 : hdrlo);
		attrset(A_NORMAL);
		vscrollrect(row+1, col, row+nrows, col + cols/2 -1, 1);
		color_set(REDonBG, NULL);
		break;
	case NEAR_HI:
	case HI_DAILY:
		if (Movers != 1)
			return;
		#if 0
			if (filter_lhod(mm, Lod))
				return;
		#endif
		attrset(A_BOLD);
		col = cols/2;
		if (ColorMode == 0) color_set(GREENonBG, NULL);
		mvprintw(row+0, col, hdrhi);
		attrset(A_NORMAL);
		vscrollrect(row+1, col, row+nrows, col + cols/2 -1, 1);
		break;
	case NEAR_HI_52:
	case HI_52:
		#if 0
			if (filter_lhod(mm, Lod))
				return;
		#endif
		attrset(A_BOLD);
		col = cols/2;
		if (ColorMode == 0) color_set(GREENonBG, NULL);
		mvprintw(row+0, col, Movers == 2 ? hdrhi52 : hdrhi);
		attrset(A_NORMAL);
		vscrollrect(row+1, col, row+nrows, col + cols/2 -1, 1);
		color_set(GREENonBG, NULL);
		break;
	default:
		return;
	}

	if (mm->count == 0 && mm->change == 0 && mm->hilo > 4)
		mvprintw(row+nrows, col, "%-6.6s %7.2f %6s  %02d:%02d:%02d",
			mm->sym, mm->last, "*near*",
			mm->time/3600, (mm->time/60) % 60, mm->time % 60);
	else if (mm->count && !mm->change)
		mvprintw(row+nrows, col, "%-6.6s %7.2f %6d  %02d:%02d:%02d",
			mm->sym, mm->last, mm->count,
			mm->time/3600, (mm->time/60) % 60, mm->time % 60);
	else
		mvprintw(row+nrows, col, "%-6.6s %7.2f %+7.2f %02d:%02d:%02d",
			mm->sym, mm->last, mm->change,
			mm->time/3600, (mm->time/60) % 60, mm->time % 60);
	attrset(A_NORMAL);
}

void
display_scrollnews(char *sym, char *link, char *headline,
			time_t tim, char *src, int insertat)
{
	int	row;
	int	x;
	int	nrows;
	int	cols = cols80();
	char	srcbuf[20+3+1];
	char	lbuf[256];
	char	*datefmt = "%m/%d %H:%M";
	char	datetimebuf[64];
	struct tm *tmp;

	if (COLS >= 160)
	{
		x = 80;
		row = 4 + NumStock + 1;
	}
	else
	{
		x = 0;
		row = 4 + NumStock + 1 + 11 + 1;
	}

	nrows = LINES - 2 - row - 1;
	if (nrows <= 0)
		return;

	// Do not overwrite movers
	if (x == 0 && Movers)
		return;

	if (insertat >= nrows)
		return;

	// vscrollrect(row+1, x, row+nrows, x + cols - 1, 1);
	vscrollrect(row+insertat, x, row+nrows-1, x + cols - 1, -1);

	// format date and time
	tmp = localtime(&tim);
	strftime(datetimebuf, sizeof(datetimebuf), datefmt, tmp);

	// format news source, if any
	if (src && *src)
		sprintf(srcbuf, " (%.20s)", src);
	else
		srcbuf[0] = 0;

	// format complete line, then clip it
	sprintf(lbuf, "%-5.5s %-11s %-.128s%s",
			sym, datetimebuf, headline, srcbuf);
	lbuf[cols-1] = 0;

	if (strcmp(sym, "hot") == 0)
		attrset(COLOR_PAIR(REDonBG));
	else if (strcmp(sym, "inplay") == 0)
		attrset(COLOR_PAIR(GREENonBG));
	mvprintw(row+insertat, x, "%s", lbuf);
	attrset(A_NORMAL);
}

void
launch_url(char *url_pref, char *sym)
{
	char	*p;
	char	buf[512];
	char	urlbuf[512];
	char	*url;
	char	*launcher = get_rc_value(RcFile, "browser");

	if (strncmp(url_pref, "pref:", 5) == 0)
		url = get_rc_value(RcFile, url_pref+5);
	else
		url = url_pref;

	urlbuf[0] = '\'';
	if ( (p = strchr(url, '%')) && p[1] == 's')
		sprintf(urlbuf+1, url, sym);
	else
	{
		strcpy(urlbuf+1, url);
		if (sym)
		{
			strcat(urlbuf, " ");
			strcat(urlbuf, sym);
		}
	}
	strcat(urlbuf, "'");

	if ( (p = strchr(launcher, '%')) && p[1] == 's')
		sprintf(buf, launcher, urlbuf);
	else
	{
		strcpy(buf, launcher);
		strcat(buf, " ");
		strcat(buf, urlbuf);
	}

	strcat(buf, " > /dev/null 2>&1 &");

	system(buf);
}

void
launch_broker(STOCK *sp)
{
	char	*sym = sp ? sp->sym : NULL;
	char	*p;
	char	buf[512];
	char	urlbuf[512];
	char	*url = get_rc_value(RcFile, "broker_URL");
	char	*launcher = get_rc_value(RcFile, "broker_browser");

	urlbuf[0] = '\'';
	if ( (p = strchr(url, '%')) && p[1] == 's')
		sprintf(urlbuf+1, url, sym);
	else
		strcpy(urlbuf+1, url);
	strcat(urlbuf, "'");

	if ( (p = strchr(launcher, '%')) && p[1] == 's')
		sprintf(buf, launcher, urlbuf);
	else
	{
		strcpy(buf, launcher);
		strcat(buf, " ");
		strcat(buf, urlbuf);
	}

	strcat(buf, " > /dev/null 2>&1 &");

	system(buf);
}

#ifdef NEEDREG
void
register_user(void)
{
	char	*p;
	char	buf[512];
	char	urlbuf[512];
	char	*launcher = get_rc_value(RcFile, "browser");

	strcpy(urlbuf, "'https://www.paypal.com/cgi-bin/webscr");
	strcat(urlbuf, "?cmd=_xclick");
	strcat(urlbuf, "&business=linuxtrade@mn.rr.com");
	strcat(urlbuf, "&undefined_quantity=1");
	strcat(urlbuf, "&item_name=LinuxTrade");
	strcat(urlbuf, "&item_number=text");
	strcat(urlbuf, "&amount=19.95");
	strcat(urlbuf, "&no_shipping=1");
	strcat(urlbuf, "&cancel_return=http://linuxtrade.0catch.com");
	strcat(urlbuf, "&return=http://linuxtrade.0catch.com");
	strcat(urlbuf, "/r");
	strcat(urlbuf, "eg");
	strcat(urlbuf, "is");
	strcat(urlbuf, "te");
	strcat(urlbuf, "r.");
	strcat(urlbuf, "ht");
	strcat(urlbuf, "ml");
	strcat(urlbuf, "'");

	if ( (p = strchr(launcher, '%')) && p[1] == 's')
		sprintf(buf, launcher, urlbuf);
	else
	{
		strcpy(buf, launcher);
		strcat(buf, " ");
		strcat(buf, urlbuf);
	}

	strcat(buf, " > /dev/null 2>&1 &");
	system(buf);
}
#endif

void
update_software(void)
{
	system("linuxtrade.up > /dev/null 2>&1 &");
}

void
all_charts_from_array(char *opts)
{
	char	buf[512];
	int	i;
	int	n = 0;

	strcpy(buf, "linuxtrade.G");
	strcat(buf, " ");
	strcat(buf, opts);
	for (i = 0; i < NumStock; ++i)
	{
		if (0 && Stock[i].sym[0] == '$')
			continue;

		strcat(buf, " '");
		strcat(buf, Stock[i].sym);
		strcat(buf, "'");
		++n;
	}

	if (n)
		system(buf);
}

void
all_charts_from_list(char *opts, char *list)
{
	char	buf[512];

	strcpy(buf, "linuxtrade.G");
	strcat(buf, " ");
	strcat(buf, opts);
	strcat(buf, " '");
	strcat(buf, list);
	strcat(buf, "' ");

	system(buf);
}

/*
 * Start/stop curses
 */
static int	EnableMouse = 0;
static mmask_t	OldMouse;

void
enable_mouse(void)
{
	if (EnableMouse)
	{
		mousemask(OldMouse, NULL);
	}

	EnableMouse = atoi(get_rc_value(RcFile, "mouse"));
	if (EnableMouse)
	{
		mousemask(BUTTON1_CLICKED
			| BUTTON1_DOUBLE_CLICKED, &OldMouse);
		mouseinterval(500);
	}
}

void
disable_mouse(void)
{
	if (!EnableMouse)
		return;

	mousemask(OldMouse, NULL);
}

void
start_curses(void)
{
	// Give the X-terminal some time to process window size events
	sleep(1);

	initscr(); cbreak(); noecho();
	nonl();
	intrflush(stdscr, FALSE);
	keypad(stdscr, TRUE);

	start_color();

	Reverse = (ColorMode == 1) ? 1 : 0;

	RevOrBold = Reverse ? A_BOLD : A_REVERSE;

	wbkgd(stdscr, Reverse ? A_REVERSE : A_NORMAL);

	init_pair(1, COLOR_GREEN, COLOR_BLACK);
	init_pair(2, COLOR_RED, COLOR_BLACK);
	init_pair(3, COLOR_YELLOW, COLOR_BLACK);
	init_pair(4, COLOR_GREEN, COLOR_RED);
	init_pair(5, COLOR_RED, COLOR_GREEN);
	init_pair(6, COLOR_CYAN, COLOR_BLACK);
	init_pair(7, COLOR_RED, COLOR_WHITE);
	init_pair(8, COLOR_BLUE, COLOR_WHITE);
	init_pair(9, COLOR_RED, COLOR_BLACK);
	init_pair(10, COLOR_BLACK, COLOR_GREEN);
	init_pair(11, COLOR_BLACK, COLOR_RED);
	init_pair(12, COLOR_BLACK, COLOR_YELLOW);
	init_pair(13, COLOR_BLACK, COLOR_CYAN);

	if (ColorMode >= 2)
	{
		use_default_colors();

		if (ColorMode == 2)
			init_pair(1, COLOR_GREEN, -1);
		else
			init_pair(1, COLOR_BLUE, -1);
		init_pair(2, COLOR_RED, -1);
		init_pair(3, COLOR_YELLOW, -1);
		init_pair(6, COLOR_CYAN, -1);
		init_pair(9, COLOR_RED, COLOR_BLACK);
	}

	enable_mouse();

	CursesInitted = TRUE;

	// This hack to get the panel library initialized
	// so that update_panels() will do something even
	// if no panels have been created yet.
	del_panel(new_panel(stdscr));
}

void
end_curses(void)
{
	if (CursesInitted)
	{
		disable_mouse();
		endwin();
		CursesInitted = FALSE;
	}
}

#ifdef NEEDREG
	#include "regsha1.h"
#endif

static int
registered(void)
{
#ifndef NEEDREG
	return 1;
#else
	char	*name = get_rc_value(RcFile, "reg_name");
	char	*mail = get_rc_value(RcFile, "alert_mail");
	char	*codes = get_rc_value(RcFile, "reg_codes");
	char	prod[16];
	char	codebuf[20];
	char	*p, *b;

	if (0 && getuid() == 400) return 1;
	if (ForceDemoMode) return 0;

	codes = strdup(codes);
	if (!codes)
		error(1, "No space in registered\n");

	prod[0] = 'L';
	prod[1] = 'i';
	prod[2] = 'n';
	prod[3] = 'u';
	prod[4] = 'x';
	prod[5] = 'T';
	prod[6] = 'r';
	prod[7] = 'a';
	prod[8] = 'd';
	prod[9] = 'e';
	prod[10] = '1';
	prod[11] = 0;

	p = codes;
	for (;;)
	{
		while (*p == ' ' || *p == ',')
			++p;
		b = p;
		while (*p && *p != ' ' && *p != ',')
			++p;

		if (*p)
			*p++ = 0;

		regcode(name, mail, prod, codebuf);

		if (strcmp(b, codebuf) == 0)
			return (1);

		if (*p == 0)
			break;
	}

	free(codes);
	return (0);
#endif
}

void
set_demo_mode(int on)
{
	if (on)
	{
		if (!DemoTime)
		{
			DemoTime = time(NULL) + DEMOTIME;
			if (0 && Debug == 1)
				DemoTime = time(NULL) + 30;
		}
	}
	else
		DemoTime = 0;
}

int
savelist(int c, STREAMER sr)
{
	blankrect(LINES-1, 0, LINES-1, COLS-1, 1);
	CursorX = 0;

	// Double ess means save to current list
	if (c >= '1' && c <= '9')
		ListNum = c - '0';
	else if (c == 'q' || c == 033 || c == 'x')
		return 2;
	else if (c != 's')
	{
		beep();
		return 2;
	}

	stocklist_write(ListNum);
	display_title();
	return 2;
}

int
savestock(int c, STREAMER sr)
{
	int	listnum;

	blankrect(LINES-1, 0, LINES-1, COLS-1, 1);
	CursorX = 0;

	// Double ess means save to current list
	if (c >= '1' && c <= '9')
		listnum = c - '0';
	else
	{
		beep();
		return 2;
	}
	if (!NumStock)
	{
		beep();
		return 2;
	}

	stocklist_add(listnum, &Stock[StockCursor]);
	display_title();
	return 2;
}

int
confirm_delete(int c, STREAMER sr)
{
	int	n;

	blankrect(LINES-1, 0, LINES-1, COLS-1, 1);
	CursorX = 0;

	if (c == 'n')
		return 2;
	if (c == 'y')
		c = 'd';
	if (c != 'd')
	{
		beep();
		return 2;
	}

	(*sr->send_symbols)(sr, Stock[StockCursor].sym, FALSE);
	(*sr->send_symbols_end)(sr, FALSE, FALSE);

	n = del_stock(Stock[StockCursor].sym);
	if (n == -1)
		return 2;
	vscrollrect(4+n,0, LINES-3,COLS-1, 1);

	if (StockCursor >= NumStock)
		StockCursor = NumStock ? NumStock-1 : 0;
	display_stock_cursor(&Stock[StockCursor], TRUE);

	return 2;
}

int
confirm_quit(int c, STREAMER sr)
{
	blankrect(LINES-1, 0, LINES-1, COLS-1, 1);
	CursorX = 0;

	if (c == 'n')
		return 2;
	if (c == 'y')
		c = 'q';
	if (c != 'q')
	{
		beep();
		return 2;
	}

	if (sr->send_disconnect)
		(*sr->send_disconnect)(sr);
	if (sr->close)
		(*sr->close)(sr);

	alert_news_stoppolling();

	end_curses();
	pthread_mutex_unlock(&CurseMutex);

	/*
	 * Kill any threads that are running
	 */
	inplay_kill();
	updown_kill();

	// I do not know why this is needed, but it prevents a hang
	// in exit() from happening if there are active threads.
	pthread_exit(NULL);

	exit(0);
}

/*
 * Add a newline separated list of comments to the current stocklist
 */
void
add_comments(char *comments)
{
	int	i = 0;
	int	len;
	char	*p;

	if (!NumStock || !comments)
		return;
	while (*comments)
	{
		p = strchr(comments, '\n');
		if (!p)
			break;
		len = p - comments;
		if (len > COMLEN)
			len = COMLEN;
		memcpy(Stock[i].comment, comments, len);
		Stock[i].comment[len] = 0;
		++i;
		if (i >= NumStock)
			break;
		comments = p + 1;
	}
}

void
new_stocklist(STREAMER sr, int num, char *list, char *label)
{
	int	i;

	for (i = 0; i < NumStock; ++i)
	{
		(*sr->send_symbols)(sr, Stock[i].sym, FALSE);
		destroy_ts(&Stock[i]);
	}
	(*sr->send_symbols_end)(sr, FALSE, TRUE);

	vscrollrect(4 + 0, 0, LINES-3, COLS-1, NumStock);
	NumStock = StockCursor = 0;
	ListNum = num;
	if (list[0])
	{
		add_from_stocklist(list, TRUE);
		if (!ToolMode && !sr->usenews)
			alert_news_startpolling();
		(*sr->send_symbols)(sr, list, TRUE);
		(*sr->send_symbols_end)(sr, TRUE, TRUE);
		if (num)
			stocklist_holdings(num);
	}
	if (label)
		strcpy(StockListName, label);

	time(&LastRefresh);
	display_title();
	display_status(sr, FALSE);
}

/*
 * Run an external command for a stock and display it on the message line
 */
void
external_msg(char *prog, char *sym, char *arg)
{
	FILE	*fp;
	char	buf[BUFSIZ];
	char	*p;

	if (arg)
		sprintf(buf, PROGNAMESTR ".%s '%s' '%s'", prog, sym, arg);
	else
		sprintf(buf, PROGNAMESTR ".%s '%s'", prog, sym);
	fp = popen(buf, "r");
	if (!fp)
		return;
	fgets(buf, sizeof(buf), fp);
	pclose(fp);
	p = strchr(buf, '\n'); if (p) *p = 0;
	display_msg(A_NORMAL, buf);
}

static void (*LeaveCursor)(void);

void
leave_cursor(void)
{
	if (LeaveCursor)
		(*LeaveCursor)();
	else
		move(LINES-1, CursorX);
}

void
prompt(char *str)
{
	blankrect(LINES-1, 0, LINES-1, COLS-1, 1);
	mvprintw(LINES-1, 0, "%s", str);
	CursorX = getcurx(stdscr);
}

void
process_user_cmd(STREAMER sr)
{
	int		c;
	int		rc = 0;
	int		i, n;
	MEVENT		m;
	char		*sp;
	int		len;
	static char	collect = 0;
	static char	collectbuf[1024];
	static int	(*handler)(int c, STREAMER sr);

	c = getch();
	Ungetch = 0;
	// fprintf(stderr, "c = %d\n", c);

	if (c == '\f')
	{
		leave_cursor();
		wrefresh(curscr);
		return;
	}

	if (handler)
	{
		rc = (*handler)(c, sr);

		//
		// Cursor up/down while displaying a chart or L2 means to
		// switch to a chart/L2 of the next symbol
		//
		if (handler == chart_command &&
			(c == KEY_UP || c == KEY_DOWN))
			goto process;
		#if 0 /* needs debugging... */
			if (handler == l2sr_command &&
				(c == KEY_UP || c == KEY_DOWN))
				goto process;
		#endif

		if (rc == 1 || rc == 2)
		{
			if (handler == pref_command)
			{
				ShowTotals = atoi(get_rc_value(RcFile,
							"showtotals"));
				display_holdings_totals(ShowTotals);
				Top10Market = get_rc_value(RcFile,
						"top10_mkt");
				Top10Mode = get_rc_value(RcFile,
						"top10_mode");
				set_timezone();
				set_demo_mode( !registered() );
				enable_mouse();
				display_title();
			}
			if (handler == holdings_command)
			{
				Holdings = SaveHoldings;
				redisplay_list();
				display_stock_cursor(&Stock[StockCursor], TRUE);
			}

			handler = NULL;
			LeaveCursor = NULL;
			if (rc == 2)
				c = 'q';
			if (c == 'q')
			{
				leave_cursor();
				update_panels(); refresh(); // doupdate();
				return;
			}
			// Handle all other chars as next command
		}
		else if (rc == 0)
		{
			leave_cursor();
			update_panels(); refresh(); // doupdate();
			return;
		}
		else if (rc == 3)
		{
			// handle this char, return to popup
		}
		else
			return;
	}

	if (collect)
	{
		int	eol = 0;

		if (c == '\r' || c == '\n')
		{
			eol = 1;
			CursorX = 0;
			blankrect(LINES-1, 0, LINES-1, COLS-1, 1);
		}
		else if (c == '\b')
		{
			char	*s;

			s = strchr(collectbuf, 0);
			if (s > collectbuf)
			{
				*--s = 0;
				--CursorX;
				mvaddch(LINES-1, CursorX, ' ');
			}
			else
			{
				// Abandon command
				CursorX = 0;
				blankrect(LINES-1, 0, LINES-1, COLS-1, 1);
				enable_mouse();
				collect = 0;
			}
		}
		else if (c == '\033')
		{
			// Abandon command
			CursorX = 0;
			blankrect(LINES-1, 0, LINES-1, COLS-1, 1);
			enable_mouse();
			collect = 0;
		}
		else if (c == CTRL('u'))
		{
			char	*s = strchr(collectbuf, 0);
			// erase command
			len = strlen(collectbuf);
			for (i = 0; i < len; ++i)
			{
				*--s = 0;
				--CursorX;
				mvaddch(LINES-1, CursorX, ' ');
			}
		}
		else
		{
			char s[2];
			s[0] = c; s[1] = 0;
			strcat(collectbuf, s);
			mvaddch(LINES-1, CursorX, c);
			++CursorX;
		}

		if (eol)
		{
			int	nshares;
			double	cost;
			STOCK	*sp;

			switch (collect)
			{
			case 'a':
				if (!collectbuf[0])
					break;
                
				sp = add_from_stocklist(collectbuf, TRUE);
				if (!ToolMode && !sr->usenews)
					alert_news_startpolling();
				(*sr->send_symbols)(sr, collectbuf, TRUE);
				(*sr->send_symbols_end)(sr, TRUE, FALSE);
				display_stock_cursor(&Stock[StockCursor],FALSE);
				if (sp)
					StockCursor = sp->y - 4;
				else
					StockCursor = NumStock - 1;
				display_stock_cursor(&Stock[StockCursor], TRUE);
				time(&LastRefresh);
				break;
			case 'D':
				if (!collectbuf[0])
					break;

				n = del_stock(collectbuf);
				if (n == -1)
					break;
				vscrollrect(4+n,0, LINES-3,COLS-1, 1);
				(*sr->send_symbols)(sr, collectbuf, FALSE);
				(*sr->send_symbols_end)(sr, TRUE, FALSE);
				if (n == NumStock)
					StockCursor = NumStock ? NumStock-1 : 0;
				display_stock_cursor(&Stock[StockCursor], TRUE);
				break;
			case 'W':
				if (!collectbuf[0])
					break;
				strcpy(WriteFileName, collectbuf);
				WriteFile = fopen(collectbuf, "w");
				if (rc < 0)
				{
					beep();
					break;
				}
				(*sr->record)(sr, WriteFile);
				break;
			case '#':
				if (!collectbuf[0])
					break;
				rc = sscanf(collectbuf, "%d%*[, ]%lf\n",
						&nshares, &cost);
				if (rc != 2)
				{
					beep();
					break;
				}
				Stock[StockCursor].nshares = nshares;
				Stock[StockCursor].cost = cost;
				break;
			case ':':
				docolon(sr, collectbuf, &handler);
				break;
			case 'f':
				trigger_add(sr, collectbuf, TRUE);
				break;
			case 't':
				trigger_add(sr, collectbuf, FALSE);
				break;
			}
			enable_mouse();
			collect = 0;
		}

		leave_cursor();
		update_panels(); refresh(); // doupdate();
		return;
	}

process:
	switch (c)
	{
	case '\f':
		leave_cursor();
		wrefresh(curscr);
		return;
	case ':':
		collect = ':';
		collectbuf[0] = 0;
		prompt(":");
		break;
	case 'a':
		// Add symbols
		collect = 'a';
		collectbuf[0] = 0;
		prompt("Add Symbols: ");
		disable_mouse();
		break;
	case 'd':
		// Delete stock at cursor
		handler = confirm_delete;
		mvprintw(LINES-1, 0, "Delete %s? (press d again): ",
				Stock[StockCursor].sym);
		CursorX = getcurx(stdscr);
		printw("          ");
		break;
	case 'D':
		// Delete stock by name
		collect = 'd';
		collectbuf[0] = 0;
		prompt("Delete Symbols: ");
		disable_mouse();
		break;
	case CTRL('f'):
		// Add trendfund <symbol> <longtrig <short trig>
		collect = 'f';
		collectbuf[0] = 0;
		prompt("Trendfund Trigger <symbol> <hitrig> <lotrig>: ");
		disable_mouse();
		break;
	case CTRL('t'):
		// Add trigger <symbol> <longtrig <short trig>
		collect = 't';
		collectbuf[0] = 0;
		prompt("Trigger <symbol> <hitrig> <lotrig>: ");
		disable_mouse();
		break;
	case 'r':
		// Force a price refresh of a stock
		// (sometimes streamer doesn't work right)
		(*sr->send_symbols)(sr, Stock[StockCursor].sym, TRUE);
		(*sr->send_symbols_end)(sr, TRUE, FALSE);
		break;
	
	case '0':
	case '1': case '2': case '3': case '4': case '5':
	case '6': case '7': case '8': case '9':
		if (DemoTime && c >= '3')
		{
			display_msg(A_BOLD, "A maximum of 3 stocklists are "
					"allowed in demo mode.\n");
			beep();
			break;
		}

		ListNum = c - '0';
		if (ListNum == 0)
		{
			new_stocklist(sr, 0, List0, StockList0Name);
			add_comments(List0Comments);
		}
		else
		{
			sp = stocklist_read(ListNum);
			new_stocklist(sr, ListNum, sp, NULL);
		}
		if (TsDisplayed)
			display_ts();
		break;

	case 's':
		handler = savelist;
		blankrect(LINES-1, 0, LINES-1, COLS-1, 1);
		mvprintw(LINES-1, 0,
			"Save to Stocklist #1-9 or s for current: ");
		CursorX = getcurx(stdscr);
		break;
	case 'S':
		handler = savestock;
		prompt("Save this stock to Stocklist #1-9: ");
		break;

	case '?':
	case 'h':
		handler = help_command;
		help_popup("main");
		break;

	case 'A':
	key_alert:
		handler = alert_command;
		alert_popup(&Stock[StockCursor]);
		break;
	case CTRL('A'):
		if (Stock[StockCursor].alerting)
		{
			Stock[StockCursor].alerting = 0;
			display_alert(&Stock[StockCursor], FALSE);
		}
		else
		{
			for (i = 0; i < NumStock; ++i)
				if (Stock[i].alerting)
				{
					Stock[i].alerting = 0;
					display_alert(&Stock[i], FALSE);
				}
		}
		break;

	case 'i':
	key_info:
		handler = info_command;
		info_popup(Stock[StockCursor].sym, sr, "fund");
		break;
	case 'I':
		handler = info_command;
		info_popup(Stock[StockCursor].sym, sr, "tech");
		break;

	case 'n':
	key_news:
		handler = news_command;
		news_popup(sr, &Stock[StockCursor], NEWS_DEFAULT);
		break;

	case 'o':
		handler = optchain_command;
		optchain_popup(&Stock[StockCursor], sr,
			atoi(get_rc_value(SrCur->rcfile,"usechains")) );
		break;
	case 'O':
		launch_url("pref:optchain_URL", Stock[StockCursor].sym);
		break;

	case 'H':
	key_holdmode:
		if (trendx() >= 120)
		{
			// No need for holdings mode
			beep();
			break;
		}
		Holdings = !Holdings;
		redisplay_list();
		break;
	case '*':
		PerShare = !PerShare;
		redisplay_list();
		break;
	case '@':
		blankrect(LINES-1, 0, LINES-1, COLS-1, 1);
		mvprintw(LINES-1, 0,
			"Enter # of shares and total cost: ");
		CursorX = getcurx(stdscr);
		collect = '#';
		collectbuf[0] = 0;
		break;
	case '#':
	key_holdings:
		if (NumStock == 0)
		{
			beep();
			break;
		}
		handler = holdings_command;
		LeaveCursor = holdings_cursor;
		SaveHoldings = Holdings;
		if (trendx() >= 120)
		{
			Holdings = HOLDINGS_OBESE;
			display_stock_cursor(&Stock[StockCursor], FALSE);
			if (holdings_mode(StockCursor, TRUE, PerShare))
			{
			hold_error:
			    handler = NULL;
			    display_stock_cursor(&Stock[StockCursor], TRUE);
			    LeaveCursor = NULL;
			    Holdings = SaveHoldings;
			    redisplay_list();
			    display_msg(A_BOLD,
			    "Sorry, you need %d screen lines (%d more) to "
			    "enter holdings for this portfolio",
			    NumStock + 4 + 9, NumStock + 4 + 9 - LINES);
			}
		}
		else
		{
			Holdings = HOLDINGS_ON;
			redisplay_list();
			display_stock_cursor(&Stock[StockCursor], FALSE);
			Holdings = HOLDINGS_EDIT;
			if (holdings_mode(StockCursor, FALSE, PerShare))
			    goto hold_error;
		}
		break;

		//
		// Preferences
		//
	case 'P':
		handler = pref_command;
		disable_mouse();
		pref_popup(RcFile, RcFileName);
		break;

		//
		// Streamer switching
		//
	case CTRL('R'):
		{
			char	*ostreamer;
			char	*streamer;

			ostreamer = get_rc_value(RcFile, "streamer");
			if (strcmp(ostreamer, "none") == 0)
			{
				streamer = cycle_streamer(1);
				if (streamer)
					set_rc_value(RcFile,
							"streamer", streamer);
			}
			NewStreamer = TRUE;
		}
		break;
	case CTRL('N'):
		{
			char	*ostreamer;
			char	*streamer;

			ostreamer = get_rc_value(RcFile, "streamer");
			streamer = cycle_streamer(1);
			if (streamer&& ostreamer && strcmp(streamer, ostreamer))
			{
				set_rc_value(RcFile, "streamer", streamer);
				NewStreamer = TRUE;
			}
		}
		break;
	case CTRL('P'):
		{
			char	*ostreamer;
			char	*streamer;

			ostreamer = get_rc_value(RcFile, "streamer");
			streamer = cycle_streamer(-1);
			if (streamer&& ostreamer && strcmp(streamer, ostreamer))
			{
				set_rc_value(RcFile, "streamer", streamer);
				NewStreamer = TRUE;
			}
		}
		break;

		//
		// Briefing.com pages
		//
	case 'p':
	key_inplay:
		handler = inplay_command;
		inplay_popup(sr);
		break;
	case 'u':
	key_updown:
		handler = updown_command;
		updown_popup(sr);
		break;

	case 'e':
		handler = bloomearn_command;
		bloomearn_popup(sr);
		break;

		/*
		 * Level II books
		 */
	case 'b':
	key_arca:
		handler = arca_command;
		arca_popup(&Stock[StockCursor]);
		break;

	case 'B':
	key_island:
		handler = island_command;
		island_popup(&Stock[StockCursor]);
		break;

	case 'L':
		handler = qml2_command;
		qml2_popup(&Stock[StockCursor]);
		break;

	case 'l':
		{
			char *useL2 = get_rc_value(SrCur->rcfile, "useL2");
			if (sr && sr->send_l2 && useL2)
			{
				handler = l2sr_command;
				l2sr_popup(&Stock[StockCursor], sr);
			}
			else
			{
				handler = qml2_command;
				qml2_popup(&Stock[StockCursor]);
			}
		}
		break;

		/*
		 * Create hokey character-based chart
		 */
	case 'c':
	key_chart:
		{
			char     *usecharts;
		       	usecharts = get_rc_value(SrCur->rcfile,"usecharts");

			handler = chart_command;
			chart_popup(&Stock[StockCursor],
					atoi(usecharts) ? sr : NULL);
		}
		break;

		/*
		 * Fire off live charts
		 */
	case 'C':
		if (NumStock)
		{
			char	buf[256];
			char	*jc = get_rc_value(RcFile, "java_charts");
			char	*cmd = "linuxtrade.lc";

			if (jc[0] == 'p')
			{
				if (strstr(jc, "java"))
				    cmd = "linuxtrade.pf -j";
				else
				    cmd = "linuxtrade.pf";
			}
			else if (jc[0] == 'm')
				cmd = "linuxtrade.pf -m";

			sprintf(buf, "%s '%s' >/dev/null 2>&1 &",
						cmd, Stock[StockCursor].sym);
			system(buf);
		}
		break;

		/*
		 * Fire off heat map
		 */
	case 'M':
		system("linuxtrade.hm >/dev/null  2>&1 &");
		break;

		/*
		 * Get a web page full of charts
		 */
	case 'g':
		all_charts_from_array("-i");
		break;
	case 'G':
		all_charts_from_array("");
		break;

#ifdef NEEDREG
	case 'R':
		register_user();
		break;
#endif
	case 'U':
		update_software();
		break;

		/*
		 * Toggle display of market movers
		 */
	case 'm':
		if (*sr->send_movers)
		{
			Movers = (Movers + 1) % 3;
			display_mktmover(NULL);
			(*sr->send_movers)(sr, Movers);
			switch (Movers)
			{
			case 0:
				display_msg(A_NORMAL,
				    "Market Movers display is now off");
				break;
			case 1:
				display_msg(A_NORMAL,
				    "Market Movers display is now on");
				break;
			case 2:
				display_msg(A_NORMAL,
				    "Market Movers display is 52-week only");
				break;
			}
		}
		else
			beep();
		if (Movers)
		{
			#if 0
				if (!Lod)
					Lod = g_tree_new(alphacmp);
				if (!Hod)
					Hod = g_tree_new(alphacmp);
			#endif
		}
		break;

		/*
		 * Change market data that is displayed
		 */
	case 'E':
	case 'N':
	case 'Q':
		// Turn off old settings
		if (sr->send_top10)
			for (i = 1; i <= 5; ++i)
				(*sr->send_top10)(sr, Top10Market[0], i, FALSE);
		Top10Market[0] = c;
		if (!sr->send_top10)
			break;
		(*sr->send_top10)(sr, Top10Market[0], 1, 1);
		if (Top10Mode[0] == '$')
			for (i = 2; i <= 3; ++i)
				(*sr->send_top10)(sr, Top10Market[0], i, TRUE);
		else
			for (i = 4; i <= 5; ++i)
				(*sr->send_top10)(sr, Top10Market[0], i, TRUE);
		break;

		/*
		 * toggle %/$ in top10
		 */
	case '%':
		Top10Mode[0] = '%';
		if (!sr->send_top10)
			break;
		for (i = 2; i <= 5; ++i)
			(*sr->send_top10)(sr, Top10Market[0], i, FALSE);
		for (i = 4; i <= 5; ++i)
			(*sr->send_top10)(sr, Top10Market[0], i, TRUE);
		break;
	case '$':
		Top10Mode[0] = '$';
		if (!sr->send_top10)
			break;
		for (i = 2; i <= 5; ++i)
			(*sr->send_top10)(sr, Top10Market[0], i, FALSE);
		for (i = 2; i <= 3; ++i)
			(*sr->send_top10)(sr, Top10Market[0], i, TRUE);
		break;

		/*
		 * Time and Sales mini-windows
		 */
	case 't':
		if (Stock[StockCursor].sym[0] == '$')
			break;
		Stock[StockCursor].tsdisplay = !Stock[StockCursor].tsdisplay;
		for (i = 0; i < NumStock; ++i)
			if (Stock[i].tsdisplay)
				break;
		TsDisplayed = i != NumStock;
		hide_ts();
		if (TsDisplayed)
			display_ts();
		break;
	case 'T':
		TsDisplayed = !TsDisplayed;
		for (i = 0; i < NumStock; ++i)
			if (Stock[i].sym[0] != '$')
				Stock[i].tsdisplay = TsDisplayed;
		hide_ts();
		if (TsDisplayed)
			display_ts();
		break;

	case CTRL('b'):
		launch_broker(&Stock[StockCursor]);
		break;

		/*
		 * Cursor Motion
		 */
	case 'j':
	case KEY_DOWN:
		if (StockCursor >= (NumStock-1) )
			beep();
		else
		{
			display_stock_cursor(&Stock[StockCursor], FALSE);
			++StockCursor;
			display_stock_cursor(&Stock[StockCursor], TRUE);
		}

		if (handler == chart_command)
			goto key_chart;
		break;

	case 'k':
	case KEY_UP:
		if (StockCursor <= 0)
			beep();
		else
		{
			display_stock_cursor(&Stock[StockCursor], FALSE);
			--StockCursor;
			display_stock_cursor(&Stock[StockCursor], TRUE);
		}

		if (handler == chart_command)
			goto key_chart;
		break;

	case KEY_MOUSE:
		if (getmouse(&m) != OK)
			break;

		#if 0
			fprintf(stderr, "Mid=%d x=%d y=%d, bstate=%x\n",
				m.id, m.x, m.y, m.bstate);
		#endif

		if (m.y >= 4 && m.y < (4+NumStock))
		{
			display_stock_cursor(&Stock[StockCursor], FALSE);
			StockCursor = m.y - 4;
			display_stock_cursor(&Stock[StockCursor], TRUE);

			if (m.bstate & BUTTON1_DOUBLE_CLICKED)
			{
				if (m.x >= 0 && m.x < 5)
					goto key_info;
				if (m.x >= 6 && m.x < 13)
					goto key_news;
				if (Holdings && m.x >= 16 && m.x <= 39)
					goto key_holdings;
				if (!Holdings && m.x >= 16 && m.x <= 25)
					goto key_arca;
				if (!Holdings && m.x >= 27 && m.x <= 36)
					goto key_island;
				if (!Holdings && m.x >= 38 && m.x <= 47)
					goto key_chart;
				if (m.x >= trendx() && m.x < 9999)
					goto key_alert;
			}
		}
		else if (m.y == 1 && m.x >= 0 && m.x < 6)
			goto key_inplay;
		else if (m.y == 1 && m.x >= 7 && m.x < 13)
			goto key_updown;
		else if (m.y == 2)
			goto key_holdmode;
		break;

	case 'W':
		if (WriteFile)
		{
			fclose(WriteFile);
			WriteFile = NULL;
		}
		collect = 'W';
		strcpy(collectbuf, WriteFileName);
		mvprintw(LINES-1, 0, "Record to file: %s", WriteFileName);
		CursorX = getcurx(stdscr);
		printw("                           ");
		break;

	case KEY_F(11):
		print_rect_troff(0, 0, LINES, COLS, NULL, "screen.tr");
		break;

	case KEY_PRINT:
	case KEY_F(12):
		print_window(curscr, LINES,
				// "tee xxx.txt | a2ps -o xxx.ps");
				get_rc_value(RcFile, "print_cmd"));
		break;

	case KEY_RESIZE:
		// There are actually lots of other things that will break
		// if you dynamically change the screen size.  I don't
		// particularly care to fix these.
		//
		// Note also that we won't actually get the KEY_RESIZE
		// until the user presses a key, because we select()
		// on user input and ncurses issues only an ungetch()
		// for the KEY_RESIZE.
		display_quote_hdr();
		break;

	case 'q':
		handler = confirm_quit;
		mvprintw(LINES-1, 0, "Really quit? (press q again): ");
		CursorX = getcurx(stdscr);
		printw("          ");
		break;
	default:
		beep();
		break;
	}

	leave_cursor();
	update_panels(); refresh(); // doupdate();
}

/*
 * Main program
 */
int
main(int argc, char *argv[])
{
	extern char	*optarg;
	extern int	optind;
	int		i, c;
	int		len;
	char		*p;

	char		symbols[2048] = "";
	char		buf[256];
	char		*streamer;
	char		*hostname;
	char		*password;
	int		refreshmode;
	char		*home = getenv("HOME");
	char		*srreason;

	int		rc;
	STREAMER	sr = NULL;
	time_t		last_tick;

	setlocale(LC_ALL, "" );

	/*
	 * Setup to handle errors
	 */
	error_init(end_curses, "%s", argv[0]);

	/*
	 * Add our bin directory to user's PATH
	 */
	if (home && strlen(home) < (sizeof(buf) - 5) )
	{
		sprintf(buf, "%s/bin", home);
		add_to_PATH(buf);
	}
	add_to_PATH("/usr/share/" PROGNAMESTR "/bin");

	/*
	 * Convert old RC files and process rc file
	 */
	if (home)
	{
		sprintf(buf, "%s/." PROGNAMESTR, home);
		rc = mkdir(buf, 0700);
		if (rc == 0)
		{
			sprintf(buf, "%s", PROGNAMESTR ".conv");
			system(buf);
		}
	}

	sprintf(Email, "%s@localhost", getenv("LOGNAME"));

	if (access("/usr/bin/gnome-moz-remote", R_OK) == 0)
		strcpy(Browser, "gnome-moz-remote --newwin %s");
	else
		strcpy(Browser, "netscape -remote openURL\\(%s,new-window\\)");

	process_rc_file(RcFile, RcFileName);

	streamer = get_rc_value(RcFile, "streamer");
	SrCur = find_streamer(streamer);

	/*
	 * Change ARCA params
	 */
	{
		int	port = atoi(get_rc_value(RcFile, "arca_port"));

		if (port == 23)
		{
			set_rc_value(RcFile, "arca_host","tools.tradearca.com");
			set_rc_value(RcFile, "arca_port", "80");
		}
	}

	/*
	 * Parse args
	 */
	while ((c = getopt(argc, argv, "?dD:L:l:u:p:P:h:r:w:s:t:c:T:")) != EOF)
	{
	    switch (c)
	    {
	    case 'c':
		    c = atoi(optarg);
		    if (c >= 0 && c <= 3)
			    set_rc_value(RcFile, "colormode",
					    optarg);
		    break;
	    case 'd':
		    ForceDemoMode = 1;
		    break;
	    case 'l':
		    ListNum = atoi(optarg);
		    if (ListNum < 1 || ListNum > 99)
			    error(1, "Invalid stocklist number\n");
		    break;
	    case 'D':
		    Debug = atoi(optarg);
		    break;
	    case 'L':
		    StreamerLog = fopen(optarg, "w");
		    break;
	    case 'p':
		    set_rc_value(SrCur->rcfile, "password", optarg);
		    break;
	    case 'P':
		    p = strchr(optarg, '=');
		    if (!p)
			    error(1, "-P syntax is pref=value\n");
		    *p++ = 0;
		    if (set_rc_value(RcFile, optarg, p))
			    error(1, "No preference named '%s'\n",
					    optarg);
		    break;
	    case 'u':
		    set_rc_value(SrCur->rcfile, "username", optarg);
		    break;
	    case 'h':
		    set_rc_value(SrCur->rcfile, "hostname", optarg);
		    break;
	    case 'w':
		    WriteFile = fopen(optarg, "ab");
		    if (!WriteFile)
			    syserror(1, "Can't open '%s'\n",
				    optarg);
		    break;
	    case 'r':
		    ReadFile = fopen(optarg, "rb");
		    if (!ReadFile)
			    syserror(1, "Can't open '%s'\n",
				    optarg);
		    break;
	    case 's':
		    SkipCount = atoi(optarg);
		    break;
	    case 't':
		    streamer = match_streamer(optarg);
		    set_rc_value(RcFile, "streamer", streamer);
		    SrCur = find_streamer(streamer);
		    break;
	    case 'T':
		    ToolMode = strtol(optarg, NULL, 0);
		    switch (ToolMode)
		    {
		    case 1: case 2: case 3: case 4:
		    case 17: case 18: case 19:
			break;
		    default:
			error(1, "Mode must be 1, 2, 3, 4, 17, 18, or 19\n");
			break;
		    }
		    break;
	    default:
		    usage();
		    break;
	    }
	}

	argc -= optind;
	argv += optind;

	/*
	 * Nag if they are using the demo password
	 *
	 * Please, do not remove this code.  The Scottrader people
	 * are kind enough to make a free streamer available, and the
	 * least you can do is to register with them.  This password
	 * is strictly to be used for only a few minutes to see if
	 * LinuxTrade is something you want to use.
	 */
	password = get_rc_value(SrCur->rcfile, "password");

	if (getuid() != 400 && strcmp(streamer, "scottrader.com") == 0
			&& strcmp(password, "JPY9747") == 0)
	{
		char	buf[BUFSIZ];

		printf("Please sign up for your own FREE "
			"Scottrader login at:\n"
			"\thttp://www.scottrader.com\n");
		printf("Enter password (%s for demo): ", DEMOPW);
		fflush(stdout);
		fgets(buf, sizeof(buf), stdin);

		set_rc_value(SrCur->rcfile, "password", buf);

		if (strcmp(password, "JPY9747") == 0)
			StDemoTime = time(NULL) + STDEMOTIME;
	}

	set_demo_mode( !registered() );

#ifdef NEEDREG
	//
	// If they've been using the product for more than 2 weeks, get
	// more aggressive about booting them.
	//
	if (DemoTime)
	{
		struct	stat	stats;
		time_t		age;

		sprintf(buf, "%s/." PROGNAMESTR, home);
		stat(buf, &stats);

		age = DemoTime - stats.st_mtime;
		if (age > (28 * 24 * 60 * 60))
			DemoTime = time(NULL) +  (5*60);
		else if (age > (21 * 24 * 60 * 60))
			DemoTime = time(NULL) + (10*60);
		else if (age > (14 * 24 * 60 * 60))
			DemoTime = time(NULL) + (15*60);
	}
#endif

	if (argc == 0)
	{
		strcpy(symbols, stocklist_read(ListNum));
		if (!symbols[0])
			strcpy(symbols, "$DJI $COMP $SPX "
					"CSCO SUNW INTC MSFT ORCL "
					"JDSU DELL JNPR CIEN");
	}
	else
	{
		for (i = 0; i < argc; ++i)
		{
			if (i)
				strcat(symbols, " ");
			if (strcmp(argv[i], "...") == 0)
				strcat(symbols, stocklist_read(ListNum));
			else
				strcat(symbols, argv[i]);
		}
		ListNum = 0;
	}

	/*
	 * Set the timezone to (usually) New York Time
	 */
	set_timezone();

	/*
	 * Handle toolmode, if requested
	 */
	if (ToolMode)
	{
		tool_mode(ToolMode, symbols);
		exit(0);
	}

	/*
	 * Initialize curses
	 */
	ColorMode = atoi(get_rc_value(RcFile, "colormode"));
	start_curses();

	/*
	 * Read alerts
	 */
	alert_read("." PROGNAMESTR "/alerts");

	/*
	 * Create STOCK structure
	 */
	add_from_stocklist(symbols, FALSE);
	stocklist_holdings(ListNum);

	Top10Market = get_rc_value(RcFile, "top10_mkt");
	Top10Mode = get_rc_value(RcFile, "top10_mode");
	ShowTotals = atoi(get_rc_value(RcFile, "showtotals"));

	pthread_mutex_lock(&CurseMutex);

	srreason = "";

newstreamer:
	NewStreamer = FALSE;
	sr = (*SrCur->new)();

	if (!ToolMode)
	{
		if (sr->usenews)
			alert_news_stoppolling();
		else
			alert_news_startpolling();
	}

restart:
	attrset(A_BOLD);
	move(2, 0); clrtoeol();
	len = sprintf(buf,
			"... Waiting for %s Streamer to Connect ...",
			streamer);
	mvcenter80(2, buf);
	attrset(A_NORMAL);
	FirstDisplayQuote = 1;

	display_status(sr, FALSE);
	display_title();
	leave_cursor();
	update_panels(); refresh(); // doupdate();

	/*
	 * Open connection to streamer
	 */
	debug(5, "Opening streamer connection...\n");

	hostname = get_rc_value(SrCur->rcfile, "hostname");

	password = get_rc_value(SrCur->rcfile, "password");
	refreshmode = atoi(get_rc_value(SrCur->rcfile, "refreshmode"));

	if (strcmp(streamer, "scottrader.com") == 0 &&
			strcmp(password, "JPY9747") == 0)
	{
		move(LINES-2, 0);
		printw("*** Using DEMO password %s, "
			"Please sign up for your own FREE "
			"login at:\n"
			"\thttp://www.scottrader.com",
			password);
	}

	(*sr->record)(sr, WriteFile);

	rc = (*sr->open)(sr, SrCur->rcfile, ReadFile);

	if (rc < 0)
	{
failure:
		if (rc == SR_AUTH)
			srreason = "Authorization failed. ";
		else
			srreason = "";
		streamer_free(sr);
		set_rc_value(RcFile, "streamer", "none");
		streamer = get_rc_value(RcFile, "streamer");
		SrCur = find_streamer(streamer);
		goto newstreamer;
		// syserror(1, "Couldn't connect to '%s'\n", hostname);
	}

	debug(5, "Send symbols...\n");

	/*
	 * Send initial requests
	 */
	(*sr->send_symbols)(sr, symbols, TRUE);
	(*sr->send_symbols_end)(sr, TRUE, TRUE);

	if (Top10Market[0] && sr->send_top10)
	{
		(*sr->send_top10)(sr, Top10Market[0], TOP_VOL, 1);
		if (Top10Mode[0] == '$')
		{
			(*sr->send_top10)(sr, Top10Market[0], TOP_NETGAIN, 1);
			(*sr->send_top10)(sr, Top10Market[0], TOP_NETLOSS, 1);
		}
		else
		{
			(*sr->send_top10)(sr, Top10Market[0], TOP_PCTGAIN, 1);
			(*sr->send_top10)(sr, Top10Market[0], TOP_PCTLOSS, 1);
		}
	}

	/*
	 * If display is wide, default mover display to "on"
	 */
	if (COLS >= 160)
		Movers = 1;
	else
		Movers = 0;

	if (Movers)
	{
		if (sr->send_movers)
			(*sr->send_movers)(sr, Movers);
		else
		{
			display_mktmover(NULL);
			Movers = 0;
		}
	}

	/*
	 * Streamer is connected
	 */
	blankrect(2, 0, 2, COLS-1, 1);
	attrset(A_BOLD);
	if (strcmp(streamer, "none") == 0)
	{
		sprintf(buf, "... %sPress 'P' to select streamer type ...",
				srreason);
		mvcenter80(2, buf);
	}
	else
	{
		if (sr->fd[0] > 0)
		    mvcenter80(2, "... Waiting for Streamer to Send Data ...");
		else
		    mvcenter80(2, "... Authorization Failure ...");
		srreason = "";
	}
	attrset(A_NORMAL);
	FirstDisplayQuote = 1;

	if (NumStock)
		display_stock_cursor(&Stock[StockCursor], TRUE);
	display_status(sr, TRUE);
	display_title();
	leave_cursor();
	update_panels(); refresh(); // doupdate();

	time(&LastRefresh);
	time(&last_tick);
	for (;;)
	{
		fd_set		fds;
		int		nfds;
		int		afd, ifd, qmfd, newsfd;
		struct timeval	tv;
		time_t		now;

		time(&now);

		if (StDemoTime && now >= StDemoTime)
			error(1, "Demo password time exceeded. Get a free "
				"password at http://www.scottrader.com\n");
		if (DemoTime && now >= DemoTime)
			error(1, "Test drive time exceeded. Register at "
					"http://linuxtrade.0catch.com\n");

		if (now >= (last_tick+1))
		{
			if (sr->timetick)
				(*sr->timetick)(sr, now);

			// Update InPlay news
			inplay_poll();
			updown_poll();
			bloomearn_poll();
			exthours_poll();

			// Update Option chains
			optchain_poll();

			arca_poll();

			last_tick = now;
		}

		if (sr->refresh && refreshmode
				&& now >= (LastRefresh + sr->refresh))
		{
			// Refresh symbols.  Sometimes the server
			// doesn't automatically send us updates.
			// Scottrader does this with index symbols.
			int	n = 0;
			for (i = 0; i < NumStock; ++i)
				if (refreshmode == 2 || Stock[i].sym[0] == '$')
				{
					++n;
					(*sr->send_symbols)(sr,
						    Stock[i].sym, TRUE);
				}
			if (n)
				(*sr->send_symbols_end)(sr, TRUE, TRUE);
			LastRefresh = now;
		}

		/*
		 * Figure out what fd's we need to select on
		 */
		tv.tv_sec = 1;
		tv.tv_usec = 0;
		FD_ZERO(&fds);

		/*
		 * Set bit for stdin (keyboard)
		 */
		FD_SET(0, &fds);
		nfds = 0;

		/*
		 * Set bits for each fd opened by the selected streamer
		 */
		for (i = 0; i < sr->nfd; ++i)
		{
			// guard against fd[i] <= 0, which shouldn't happen
			if (sr->fd[i] <= 0)
				continue;
			FD_SET(sr->fd[i], &fds);
			if (sr->fd[i] > nfds)
				nfds = sr->fd[i];
		}

		/*
		 * Set bits for active book popups
		 */
		afd = arca_fd();
		if (afd >= 0)
		{
			FD_SET(afd, &fds);
			if (afd > nfds)
				nfds = afd;
		}

		ifd = island_fd();
		if (ifd >= 0)
		{
			FD_SET(ifd, &fds);
			if (ifd > nfds)
				nfds = ifd;
		}

		/*
		 * Set bit for quotemedia L2
		 */
		qmfd = qml2_fd();
		if (qmfd >= 0)
		{
			FD_SET(qmfd, &fds);
			if (qmfd > nfds)
				nfds = qmfd;
		}

		/*
		 * Set bit for new poller
		 */
		newsfd = alert_news_fd();
		if (newsfd >= 0)
		{
			FD_SET(newsfd, &fds);
			if (newsfd > nfds)
				nfds = newsfd;
		}

		/*
		 * Do the select
		 */
		pthread_mutex_unlock(&CurseMutex);
		rc = (*sr->select)(sr, nfds+1, &fds, NULL, NULL, &tv);
		if (rc == -1 && errno != EINTR)
			syserror(1, "Select error\n");
		pthread_mutex_lock(&CurseMutex);
		if (rc <= 0)
		{
			/* Update time */
			display_title();
			leave_cursor();
			update_panels(); refresh(); // doupdate();
			continue;
		}

		if (afd >= 0 && FD_ISSET(afd, &fds))
			arca_data();
		if (ifd >= 0 && FD_ISSET(ifd, &fds))
			island_data();
		if (qmfd >= 0 && FD_ISSET(qmfd, &fds))
			qml2_data();
		if (newsfd >= 0 && FD_ISSET(newsfd, &fds))
			alert_news_data();

		if (FD_ISSET(0, &fds))
		{
			do
			{
				Ungetch = 0;
				process_user_cmd(sr);
			} while (Ungetch);

			if (NewStreamer)
			{
				(*sr->close)(sr);
				streamer_free(sr);
				streamer = get_rc_value(RcFile, "streamer");
				SrCur = find_streamer(streamer);
				goto newstreamer;
			}
		}

		for (i = 0; i < sr->nfd; ++i)
		{
			if (sr->fd[i] <= 0)
				continue;
			if (!FD_ISSET(sr->fd[i], &fds))
				continue;

			rc = (*sr->process)(sr, i);
			if (rc < 0)
			{
				display_status(sr, FALSE);
				(*sr->close)(sr);
				if (rc == SR_AUTH)
				{
					goto failure;
				}
				goto restart;
			}

			display_status(sr, TRUE);
			display_title();
			leave_cursor();
			update_panels(); refresh(); // doupdate();
		}
	}

	exit(0);
}
