/*b
 * Copyright (C) 2001,2002  Rick Richardson
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Rick Richardson <rickr@mn.rr.com>
b*/

/*
 * money.net
 *
 * Line oriented ASCII
 *
 * N.B. As of 11/23/2002 this is a work in progress due to new
 * streamer protocol.
 */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ncurses.h>
#include <panel.h>
#include <errno.h>
#include <time.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include "error.h"
#include "rc.h"
#include "streamer.h"
#include "linuxtrade.h"
#include "optchain.h"
#include "info.h"
#include "util.h"
#include "news.h"
#include "article.h"

extern int Debug;

#define		BUFLEN		65536
#define		SYMCNT		200
#define		SYMBUFLEN	(SYMCNT * (SYMLEN+1) + 1)

typedef struct streamerpriv
{
	char	buf[BUFLEN+1];
	char	*bufp;
	char	*bufe;

	time_t	last_keepalive;
	time_t	last_flush;
	time_t	last_sendnews;

	char	symbuf[SYMBUFLEN];
	int	symcnt;
} STREAMERDATA;

typedef struct
{
	char	*canon, *sym;
} SYMMAP;

static SYMMAP SymMap[] =
{
	{	"$DJI",		".DJIA",	},
	{	"$DJT",		".DJT",		},
	{	"$DJU",		".DJU",		},

	{	"$NYA",		"NYA",		},
	{	"$TRIN",	".TRIN",	},
	{	"$TICK",	".TICK",	},

	{	"$COMP",	".COMP",	},
	{	"$NDX",		"NDX",		},
	{	"$SPX",		".SPX",		},
	{	"$OEX",		"OEX",		},
	{	"$MID",		"MID",		},
	{	"$SML",		"SML",		},
	{	"$RLX",		"RLX",		},

	{	"$XAL",		"XAL",		},
	{	"$BTK",		"BTK",		},
	{	"$XBD",		"XBD",		},
	{	"$XAX",		"XAX",		},
	{	"$XCI",		"XCI",		},
	{	"$IIX",		"IIX",		},
	{	"$NWX",		"NWX",		},
	{	"$XOI",		"XOI",		},
	{	"$DRG",		"DRG",		},
	{	"$XTC",		"XTC",		},

	{	"$GSO",		"GSO",		},
	{	"$HWI",		"HWI",		},
	{	"$RUI",		"RUI",		},
	{	"$RUT",		"RUT",		},
	{	"$RUA",		"RUA",		},
	{	"$SOX",		"SOX",		},
	{	"$OSX",		"OSX",		},

	{	"$BKX",		"BKX",		},
	{	"$XAU",		"XAU",		},
	{	"$GOX",		"GOX",		},
	{	"$YLS",		"YLS",		},
	{	"$DDX",		"DDX",		},
	{	"$DOT",		"DOT",		},

	{	"$RXP",		"RXP",		},
	{	"$RXH",		"RXH",		},
	{	"$XNG",		"XNG",		},
	{	"$FPP",		"FPP",		},
	{	"$DJR",		"DJR",		},
	{	"$UTY",		"UTY",		},

	{	"$TYX",		"TYX",		},
	{	"$TRINQ",	".TRIQE",	},
	{	"$TICKQ",	".TICQE",	},
	{	"$TRINA",	".TRIA",	},
	{	"$TICKA",	".TICA",	},
	{	"$VIX",		"VIX",		},
	{	"$VXN",		"VXN",		},

	{	NULL,		NULL		}
};

//
// Convert canonical index names to/from streamer index names
//
static void
streamer_canon2sym(char *out, char *in)
{
	char	*ip, *op;
	char	*p;
	int	len;
	SYMMAP	*map;

	ip = in;
	op = out;
	for (;;)
	{
		p = strchr(ip, '|');
		if (!p) p = strchr(ip, ',');
		if (!p) p = strchr(ip, ' ');
		if (!p) p = strchr(ip, 0);

		len = p - ip;
		if (ip[len-1] != '.')
		{
			memcpy(op, ip, len);
			op[len] = 0;
		}
		else
		{
			// option symbol, insert blank after root
			sprintf(op, "%.*s %.2s", len-2-1, ip, ip+len-2-1);
		}

		for (map = SymMap; map->canon; ++map)
			if (strcmp(op, map->canon) == 0)
			{
				strcpy(op, map->sym);
				break;
			}

		if (*p == 0)
			break;

		ip += len + 1;
		op = strchr(op, 0);
		*op++ = *p;
		*op = 0;
	}
}

static void
streamer_sym2canon(char *out, char *in)
{
	SYMMAP	*map;
	char	*p;

	if (strchr(in, ' '))
	{
		// Implies option symbol
		char buf[SYMLEN+1];
		strcpy(buf, in);
		in = buf;

		for (p = in; *p; ++p)
			if (*p != ' ')
				*out++ = *p;
		*out++ = '.';
		*out = 0;
		return;
	}
	for (map = SymMap; map->sym; ++map)
		if (strcmp(in, map->sym) == 0)
		{
			strcpy(out, map->canon);
			return;
		}

	if (in != out)
		strcpy(out, in);
}

/*
 * Read a line from the streamer.  Discard the \n.  Buffer
 * unused data to be returned later.  Return length of line read.
 */
static int
streamer_gets(STREAMER sr, char *buf, int buflen)
{
	int	rc;
	char	*sp, *ep;

	if (sr->priv->bufp == NULL)
	{
		rc = read(sr->fd[0], sr->priv->buf, BUFLEN);
		if (rc <= 0)
			return (rc);
		sr->cnt_rx += rc;
		sr->priv->bufp = sr->priv->buf;
		sr->priv->bufe = &sr->priv->buf[rc];
		*sr->priv->bufe = '\0';
		if (StreamerLog)
			fprintf(StreamerLog, "< %s\n", sr->priv->buf);
	}

	sp = sr->priv->bufp;
	ep = strchr(sp, '\n');
	while (!ep)
	{
		rc = read(sr->fd[0], sr->priv->bufe,
				&sr->priv->buf[BUFLEN] - sr->priv->bufe);
		if (rc <= 0)
			return (rc);
		if (StreamerLog)
		{
			sr->priv->bufe[rc] = '\0';
			fprintf(StreamerLog, "< %s\n", sr->priv->bufe);
		}
		sr->cnt_rx += rc;
		ep = strchr(sr->priv->bufe, '\n');
		sr->priv->bufe += rc;
		*sr->priv->bufe = '\0';
	}
	*ep++ = 0;
	if (ep >= sr->priv->bufe)
		sr->priv->bufp = NULL;
	else
		sr->priv->bufp = ep;

	if ( (ep-sp) > buflen)
		error(1, "Buffer too small for streamer line\n");

	memcpy(buf, sp, ep-sp);

	if (Debug >= 5)
	{
		timestamp(stderr);
		if (ep-sp)
			fprintf(stderr, "< %s\n", buf);
		else
			fprintf(stderr, "< 0 length\n");
	}
	return (ep-sp);
}

static int
streamer_select(STREAMER sr,
		int numfd, fd_set *readfds, fd_set *writefds,
		fd_set *exceptfds, struct timeval *timeout)
{
	if (readfds && FD_ISSET(sr->fd[0], readfds) && sr->priv->bufp)
	{
			FD_ZERO(readfds);
			FD_SET(sr->fd[0], readfds);
			if (writefds)
				FD_ZERO(writefds);
			if (exceptfds)
				FD_ZERO(exceptfds);
			return 1;
	}

	if (0)
		fprintf(stderr, "do select\n");
	return select(numfd, readfds, writefds, exceptfds, timeout);
}

static void
streamer_close(STREAMER sr)
{
	close(sr->fd[0]);
	sr->fd[0] = -1;
}

static void
streamer_record(STREAMER sr, FILE *fp)
{
	sr->writefile = fp;
}

static void
streamer_init(STREAMER sr)
{
	sr->refresh = 0;
	sr->fd[0] = -1;
	sr->nfd = 1;
	sr->id[0] = 0;

	sr->priv->symcnt = 0;
	sr->priv->bufp = NULL;
	sr->priv->bufe = NULL;
	time(&sr->priv->last_keepalive);
	time(&sr->priv->last_sendnews);
	time(&sr->priv->last_flush);
}

static void
streamer_send_news(STREAMER sr, char *sym, int narticles)
{
	streamer_printf(sr->fd[0], "NS %d:%s\r\n", narticles, sym);
}

/*
 * Open connection to the streamer
 */
static int
streamer_open(STREAMER sr, RCFILE *rcp, FILE *readfile)
{
	struct hostent		*hep;
	struct sockaddr_in	sockaddr;
	int			rc;
	char			buf[512];
	int			conn_tries;
	int			read_tries;
	char			*username = get_rc_value(rcp, "username");
	char			*password = get_rc_value(rcp, "encpasswd");
	char			*hostname = get_rc_value(rcp, "hostname");
	int			port = atoi(get_rc_value(rcp, "port"));

	streamer_init(sr);
	++sr->cnt_opens;
	++sr->cnt_realopens;
	time(&sr->time_open);
	time(&sr->time_realopen);

	if (readfile)
	{
		sr->fd[0] = fileno(readfile);
		sr->readfile = readfile;
		return sr->fd[0];
	}

	sr->readfile = NULL;

	hep = mygethostbyname(hostname);
	if (!hep)
		return (-1);

	memcpy(&sockaddr.sin_addr, hep->h_addr, hep->h_length);
	sockaddr.sin_family = AF_INET;
	sockaddr.sin_port = htons(port);

	conn_tries = 0;
reconnect:
	if (++conn_tries >= 5)
		return -4;

	if (Debug >= 5)
		fprintf(stderr, "Open socket...\n");

	sr->fd[0] = socket(AF_INET, SOCK_STREAM, 0);
	if (sr->fd[0] < 0)
		return -2;

	if (Debug >= 5)
		fprintf(stderr, "Socket fd=%d...\n", sr->fd[0]);

	if (Debug >= 5)
		fprintf(stderr, "Connect to '%s'...\n", hostname);

	rc = connect_timeout(sr->fd[0], (SA *) &sockaddr, sizeof(sockaddr), 15);
	if (rc < 0)
	{
		if (Debug >= 5)
			syserror(0, "Couldn't connect\n");
		streamer_close(sr);
		sleep(1);
		goto reconnect;
	}

	sr->priv->bufp = NULL;

	if (Debug >= 5)
		fprintf(stderr, "Authorize...\n");
	streamer_printf(sr->fd[0], "AM %s,%s\r\n", username, password);

	sleep(1);

	/* get AUTH responses */
	if (Debug >= 5)
		fprintf(stderr, "Get handshake...\n");

	read_tries = 0;
	for (;;)
	{
		fd_set		fds;
		struct timeval	tv;

		tv.tv_sec = 5;
		tv.tv_usec = 0;
		FD_ZERO(&fds);
		FD_SET(0, &fds);
		FD_SET(sr->fd[0], &fds);
		rc = streamer_select(sr, sr->fd[0]+1, &fds, NULL, NULL, &tv);
		if (rc == -1 && errno != EINTR)
			syserror(1, "Select error\n");
		if (rc == 0)
		{
			if (Debug >= 5)
				fprintf(stderr, "Timeout...\n");
			if (++read_tries == 3)
			{
				streamer_close(sr);
				sleep(2);
				goto reconnect;
			}

			streamer_printf(sr->fd[0], "AM %s,%s\r\n",
					username, password);
			continue;
		}
		if (rc <= 0)
		{
			streamer_close(sr);
			sleep(2);
			goto reconnect;
		}

		rc = streamer_gets(sr, buf, sizeof(buf));
		if (rc <= 0)
		{
			streamer_close(sr);
			sleep(2);
			goto reconnect;
		}

		if (sr->writefile)
		{
			fputs(buf, sr->writefile);
			putc('\n', sr->writefile);
		}

		if (strcmp(buf, "A  OK\r") == 0)
			break;
		else
			return SR_AUTH;
	}

	sprintf(sr->id, "money.net");

	streamer_send_news(sr, "*FH*", 15);

	// Clear all symbols
	streamer_printf(sr->fd[0], "QC\r\n");

	return 0;
}

static void
streamer_send_quickquote(STREAMER sr, char *sym)
{
	if (sr->fd[0] < 0 || sr->readfile)
		return;

	streamer_printf(sr->fd[0], "QA %s\r\n", sym);
}

static void
streamer_send_livequote(STREAMER sr, char *sym)
{
	STOCK	*sp;

	if (sr->fd[0] < 0 || sr->readfile)
		return;

	sp = find_stock(sym);
	if (sp)
		streamer_printf(sr->fd[0], "QA %s\r\n", sym);
	else
		streamer_printf(sr->fd[0], "QA %s\r\nQR %s\r\n", sym, sym);
}

static void
streamer_send_symbols(STREAMER sr, char *symbols, int add)
{
	char	*p;
	int	cnt;

	if (sr->fd[0] < 0 || sr->readfile)
		return;

	if (sr->priv->symcnt >= SYMCNT)
		return;

	if (sr->priv->symcnt == 0)
		strcpy(sr->priv->symbuf, "");

	for (cnt = 1, p = symbols; *p; ++p)
		if (*p == ' ' || *p == ',')
			++cnt;

	if (sr->priv->symcnt)
		strcat(sr->priv->symbuf, ",");
	sr->priv->symcnt += cnt;
	strcat(sr->priv->symbuf, symbols);
}

static void
streamer_send_symbols_end(STREAMER sr, int add, int all)
{
	char	*p;
	char	sbuf[SYMBUFLEN];

	if (add)
	{
		streamer_canon2sym(sbuf, sr->priv->symbuf);

		streamer_printf(sr->fd[0], "QA %s\r\n", sbuf);
	}
	else
	{
		if (all)
			streamer_printf(sr->fd[0], "QC\r\n");
		else
		{
			for (p = sr->priv->symbuf; *p; ++p)
				if (*p == ' ')
					*p = ',';

			streamer_canon2sym(sbuf, sr->priv->symbuf);

			streamer_printf(sr->fd[0], "QR %s\r\n", sbuf);
		}
	}

	sr->priv->symcnt = 0;
}

static void
streamer_send_keepalive(STREAMER sr)
{
	if (sr->fd[0] < 0 || sr->readfile)
		return;

	streamer_printf(sr->fd[0], "P  PING\r\n");
}

static void
do_quote(char *buf, int update)
{
	//int		avevol;
	char		*p;
	int		rc;
	double		mktcap;
	int		hh, mm, ss;
	double		pe;
	double		beta;
	double		open;
	char		*fld[64];
	int		i;
	int		nfld;
	int		gottrade = 0;
	QUOTE		q;
	QUICKQUOTE	qq;
	LIVEQUOTE	lq;

	//QS CSCO|DCISCO SYS INC|Y15.24|B14.86|A14.87|H15.2|L14.68|V86880418|121.92|28.12|N0.0|E0.37|P40.243244|34|49|M11/22/2002|F23:59:12|O14.93|C108847.63|XQ|61.996|d+|v1000|xQ|T14.89

	nfld = strsplit(fld, asizeof(fld), buf+3, '|');
	if (nfld < 2)
		return;

	memset(&q, 0, sizeof(q));
	memset(&qq, 0, sizeof(qq));
	memset(&lq, 0, sizeof(lq));

	streamer_sym2canon(q.sym, fld[0]);

	if (update)
	{
		STOCK	*sp;

		sp = find_stock(q.sym);
		if (!sp)
		{
			// char	buf[64];

			// Must have been a live quote request
			//strcpy(lq.sym, q.sym);
			//len = adv(p, '|'); p += len;
			//len = adv(p, '|'); lq.last = atof(p); p += len;
			//lq.close = lq.last;
			//display_livequote(&lq);

			return;
		}

		copy_quote(&sp->cur, &q, &qq, &lq);
	}

	strcpy(qq.sym, q.sym);
	strcpy(lq.sym, q.sym);

	for (i = 1; i < nfld; ++i)
	{
		p = &fld[i][1];
		switch (fld[i][0])
		{
		case 'D': strcpy(qq.fullname, p); break;
		case 'T': lq.last = qq.last = q.last = atof(p); break;
		case 't': lq.last = qq.last = q.last = atof(p);
			  if (update)
				  gottrade = 1;
			  break;
		case 'O': open = atof(p); break;
		case 'Y': lq.close = qq.prev_close = q.close = atof(p); break;
		case 'B': qq.bid = q.bid = atof(p); break;
		case 'A': qq.ask = q.ask = atof(p); break;
		case 'H': qq.high = q.high = atof(p); break;
		case 'L': qq.low = q.low = atof(p); break;
		case 'V':
			  // For options we store volume in last_size,
			  // and open interest in volume.
			  if (strchr(fld[0], ' '))
				  qq.last_size = q.last_size = atoi(p);
			  else
				  qq.volume = q.volume = atoi(p);
			  break;
		case 'v':
			  if (!strchr(fld[0], ' '))
			  {
				  if (update)
					  gottrade = 1;
				  qq.last_size = q.last_size = atoi(p);
			  }
			  break;
		case '1': qq.high52 = atof(p); break;
		case '2': qq.low52 = atof(p); break;
		case 'X': qq.exchange = *p; break;
		case 'x': /* last_exchange = *p; */ break;
		case '3': q.bid_size = atoi(p); break;
		case '4': q.ask_size = atoi(p); break;
		case 'F':
			rc = sscanf(p, "%d:%d:%d", &hh, &mm, &ss);
			if (rc != 3) break;;
			qq.timetrade = q.time = hh * 3600 + mm * 60 + ss;
			break;
		case 'M': /* MM/DD/YYYY of last trade */ break;
		case 'd': q.bid_tick = *p; break;
		case 'C': mktcap = atof(p); /* in M$ */ break;
		case 'P': pe = atof(p); break;
		case 'N': qq.annualdiv = atof(p) * 4; break;
		case 'E': qq.year_eps = atof(p); break;
		case '6': beta = atof(p); break;
		case 'S': /* option strike price */ break;
		case 'R': /* option expiration MM/DD/YYYY */ break;
		case 'I': qq.volume = q.volume = atoi(p); /* open interest */
			  break;
		default:
			if (Debug >= 5)
				fprintf(stderr, "<%s> <%c> <%s>\n",
					fld[0], fld[i][0], p);
			break;
		}
	}

	qq.bid_id = qq.ask_id = '?';
	qq.last_eps = 12345678;
	qq.cur_eps = 12345678;
	qq.sharesout = 12345678;

	display_quote(&q, gottrade);
	optchain_quote(&q);

	info_quickquote(&qq);

	display_livequote(&lq);
}

static void
streamer_send_article(STREAMER sr, char *artkey)
{
	char	buf[512];
	char	*url = "http://www.money.net/marketview/readStory.php";

	if (sr->fd[0] < 0 || sr->readfile)
		return;

	if (Debug > 1)
		fprintf(stderr, "Send article '%s'\n", artkey);

	sprintf(buf, "%s?src=BETA&storyID=%s", url, artkey);
	article_display_mini(buf, "\t\t<p class=\"head\">");
}

static int
duplicate_headline(char	*hline)
{
	static char	oldhline[256];

	// Simple attempt to get rid of some, but not all, duplicates
	if (strcmp(hline, oldhline) == 0)
		return TRUE;

	strncpy(oldhline, hline, sizeof(oldhline));
	oldhline[sizeof(oldhline)-1] = 0;
	return FALSE;
}

/*
 * Keep track of newest article we have seen
 */
static long long	LastArticleNum;

static void
do_newnews(STREAMER sr, char *buf)
{
	char		*afld[256];
	int		nafld;
	char		*p;
	int		a;
	int		rc;
	long long	newartnum = 0;

	//*FH*:Headline|10380097801404563544-PNW#Headline|10380034203643474543-ELN#
	//First 10 digits of article number are Unix time

	p = strchr(buf, ':');
	if (!p) return;
	*p++ = 0;

	nafld = strsplit(afld, asizeof(afld), p, '#');
	if (nafld < 1)
		return;

	// Find first article we haven't seen yet
	for (a = 0; a < nafld; ++a)
	{
		long long	artnum;

		p = strchr(afld[a], '|');
		if (!p) continue;

		artnum = strtoll(++p, (char **)NULL, 10);
		if (a == 0)
			newartnum = artnum;
		if (artnum == LastArticleNum)
			break;
	}
	// a is now the index of the last article we've seen,
	// or nafld if we haven't seen any of these.
	LastArticleNum = newartnum;

	// Re-process oldest to newest, but only the new articles.
	for (a = a - 1; a >= 0; --a)
	{
		char		*ifld[3];
		int		nifld;
		int		itim;
		time_t		tim;
		char		*sym;

		nifld = strsplit(ifld, asizeof(ifld), afld[a], '|');
		if (nifld != 2)
			continue;

		rc = sscanf(ifld[1], "%10d", &itim);
		if (rc != 1)
			continue;

		if (duplicate_headline(ifld[0]))
			continue;

		sym = strchr(ifld[1], '-');
		if (sym) ++sym;
		else sym = "hot";

		tim = itim;
		if (!ToolMode)
		    display_scrollnews(sym, ifld[1], ifld[0], tim, "", 0);
		save_headline(sym, ifld[1], ifld[0], tim, "", sr, 0);
	}
}

static int
streamer_process(STREAMER sr, int fdindex)
{
	int	len;
	char	buf[4096];

	len = streamer_gets(sr, buf, sizeof(buf));
	if (len <= 0)
	{
		if (sr->readfile)
		{
			if (sr->fd[0])
				fclose(sr->readfile);
			sr->fd[0] = -1;
			return (0);
		}
		return (-1);
	}

	if (sr->writefile)
	{
		fputs(buf, sr->writefile);
		putc('\n', sr->writefile);
	}

	if (strncmp(buf, "QS ", 3) == 0)
		do_quote(buf, FALSE);
	else if (strncmp(buf, "QU ", 3) == 0)
		do_quote(buf, TRUE);
	else if (strncmp(buf, "NS ", 3) == 0)
		do_newnews(sr, buf+3);

	return (0);
}

static void
streamer_timetick(STREAMER sr, time_t now)
{
	if (now > sr->priv->last_keepalive + 20)
	{
		streamer_send_keepalive(sr);
		sr->priv->last_keepalive = now;
	}

	if (now > sr->priv->last_sendnews + 60)
	{
		streamer_send_news(sr, "*FH*", 15);
		sr->priv->last_sendnews = now;
	}

	if (now > sr->priv->last_flush + 5)
	{
		if (sr->writefile)
			fflush(sr->writefile);
	}
}

static void
nullsr(STREAMER sr)
{
}

STREAMER
moneynet_new(void)
{
	STREAMER	sr;

	sr = (STREAMER) malloc(sizeof(*sr));
	if (!sr)
		return NULL;
	memset(sr, 0, sizeof(*sr));

	sr->open = streamer_open;
	sr->select = streamer_select;
	sr->close = streamer_close;
	sr->record = streamer_record;
	sr->timetick = streamer_timetick;

	sr->send_quickquote = streamer_send_quickquote;
	sr->send_livequote = streamer_send_livequote;
	sr->send_livequote_end = nullsr;
	sr->send_symbols = streamer_send_symbols;
	sr->send_symbols_end = streamer_send_symbols_end;

	// sr->send_disconnect = streamer_send_disconnect;
	// sr->send_top10 = streamer_send_top10;
	// sr->send_movers = streamer_send_movers;
	// sr->send_info = streamer_send_info;
	// sr->send_optchain = streamer_send_optchain;
	// sr->send_chart = streamer_send_chart;

	sr->send_article = streamer_send_article;

	sr->process = streamer_process;

	sr->priv = (STREAMERDATA *) malloc(sizeof(*sr->priv));
	if (!sr->priv)
	{
		free(sr);
		return NULL;
	}

	time(&sr->time_start);

	streamer_init(sr);

	return (sr);
}
