/*b
 * Copyright (C) 2001,2002  Rick Richardson
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Rick Richardson <rickr@mn.rr.com>
b*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <ncurses.h>
#include <panel.h>
#include <errno.h>
#include <time.h>
#include <sys/time.h>
#include <ctype.h>
#include <pthread.h>
#include <signal.h>
#include "error.h"
#include "rc.h"
#include "streamer.h"
#include "curse.h"
#include "linuxtrade.h"
#include "minihtml.h"
#include "updown.h"

/*
 * TODO. Oh yeah, there's bugs in here.
 */
static WINDOW	*Win;
static WINDOW	*Subwin;
static PANEL	*Panel;

static int	Cursor;		// Article line number at top of display
static int	MaxCursor;	// Maximum value for above
static int	NumLines;	// Number of lines in the article
static time_t	LastDocPoll;	// Last time we grabbed the InPlay data
static time_t	LastLivePoll;	// Last time we grabbed the live quotes
static time_t	LastAlertPoll;	// Last time we checked for new document
static STREAMER	Sr = NULL;
static int	ValDocPoll;
static int	ValLivePoll;
static int	ValAlertPoll = -1;
static int	Alerting = 0;

#define	PADLINES	700

typedef struct
{
	char	sym[SYMLEN+1];
	int	x;
} QUOTELOC;

static QUOTELOC	Live[PADLINES];

static void
get_live_quotes(void)
{
	int	y;
	int	cnt = 0;

	for (y = Cursor; y < Cursor+getmaxy(Win)-2; ++y)
		if (Live[y].sym[0])
		{
			(Sr->send_livequote)(Sr, Live[y].sym);
			++cnt;
		}
	if (cnt)
		(Sr->send_livequote_end)(Sr);

	time(&LastLivePoll);
}

void
updown_livequote(LIVEQUOTE *lq)
{
	int	y;

	if (!Win)
		return;

	for (y = Cursor; y < Cursor+getmaxy(Win)-2; ++y)
	{
		attr_t	attr;

		if (strcmp(lq->sym, Live[y].sym))
			continue;

		if (lq->last > lq->close)
			attr = COLOR_PAIR(1) | A_NORMAL;
		else if (lq->last < lq->close)
			attr = COLOR_PAIR(2) | A_NORMAL;
		else
			attr = A_NORMAL;

		wattrset(Subwin, attr);
		mvwprintw(Subwin, y, Live[y].x,
			"%7.2f %+7.2f", lq->last, lq->last - lq->close);
		wattrset(Subwin, A_NORMAL);
	}

	// TODO: optimize this to do 1 line and move into loop
	copywin(Subwin, Win,
			Cursor, 0,
			1, 1,
			getmaxy(Win)-2, getmaxx(Win)-2,
			FALSE);
	move(LINES-1, CursorX);
	update_panels();
	doupdate();
}

//
// TODO: make this asynchronous
//

static void
display_more(void)
{
	mvwaddch(Win, 1, getmaxx(Win)-1,
		Cursor ? ACS_UARROW : ACS_VLINE);

	mvwaddch(Win, getmaxy(Win)-2, getmaxx(Win)-1,
		(Cursor < MaxCursor) ? ACS_DARROW : ACS_VLINE);
}

static TBLCOL	Col[] =
{
	{  0, 99999},
	{  0, 11 },	//Company
	{ 13, 18 },	//Symbol
	{ 20, 28 },	//Brokerage
	{ 30, 41 },	//From
	{ 43, 53 },	//To
	{ 55, 60 },	//Target
	{ 61, 68 },
	{ 70, 78 },
	{ 0, 99999 }
};
#define NUMCOL	asizeof(Col)

static int Nstock0;

static void
postwordhook(int *tblcolp, char *wbuf, int wlen)
{
	int	x, y;
	int	dolive = ValLivePoll > 1;

	// Add live quotes and add it to stocklist 0
	if (dolive && (wlen-0) < SYMLEN && *tblcolp == 2)
	{
		x = getcurx(Subwin);
		y = getcury(Subwin);

		if (strcmp(wbuf, "Symbol"))
		{
			memcpy(Live[y].sym, wbuf, wlen);
			Live[y].sym[wlen] = 0;
			Live[y].x = getmaxx(Subwin) - 15;

			if (Nstock0)
				strcat(List0, " ");
			strcat(List0, Live[y].sym);
			strcpy(StockList0Name, "Up/Downgrades");
			++Nstock0;

			if (Nstock0 == 1)
			{
				wattrset(Subwin, A_BOLD);
				mvwprintw(Subwin,
					y-2, getmaxx(Subwin)-15,
					"   Live Quotes ");
				mvwprintw(Subwin,
					y-1, getmaxx(Subwin)-15,
					"  Price  Change");
				wattrset(Subwin, A_NORMAL);
				wmove(Subwin, y, x);
			}
		}
	}
}

static void
get_updown(void)
{
	FILE	*fp;
	int	y;
	char	buf[BUFSIZ];
	char	*url = get_rc_value(RcFile, "updown_URL");

	sprintf(buf, "%s \"%s\"", SUCKURL, url);

	fp = popen(buf, "r");
	if (!fp)
	{
		mvwprintw(Subwin, getmaxy(Subwin)/2, 20,
				"Can't access updown");
		touchwin(Win);
		LastAlertPoll = LastDocPoll = time(NULL);
		return;
	}

	/*
	 * Eat lines until we see the table
	 */
	minihtml_skip_past_line(fp, "<HR><!--HTML-ONLY-END-->");

	/*
	 * Parse HTML
	 */
	Nstock0 = 0;
	List0[0] = 0;
	List0Comments[0] = 0;

	for (y = 0; y < PADLINES; ++y)
		Live[y].sym[0] = 0;

	minihtml_parse(Subwin, fp, Col, NUMCOL, MHP_DEFAULT,
			NULL, postwordhook, NULL, NULL);

	pclose(fp);

	LastAlertPoll = LastDocPoll = time(NULL);

	Cursor = 0;
	NumLines = getcury(Subwin) + 1;
	MaxCursor = NumLines - (getmaxy(Win) - 2);
	if (MaxCursor < 0)
		MaxCursor = 0;

	display_more();

	copywin(Subwin, Win,
			Cursor, 0,
			1, 1,
			getmaxy(Win)-2, getmaxx(Win)-2,
			FALSE);
	move(LINES-1, CursorX);
	update_panels();
	doupdate();
	get_live_quotes();
}

/*
 * Window is down, but check Up/Downgrades for a new document
 */
static pthread_t       Tid;

static void *
check_updown(void *arg)
{
	FILE		*fp;
	char		buf[BUFSIZ];
	static char	lastbuf[256];
	char		*url = get_rc_value(RcFile, "updown_URL");
	extern pthread_mutex_t CurseMutex;

	pthread_detach(pthread_self());

	sprintf(buf, "%s \"%s\"", SUCKURL, url);

	LastAlertPoll = time(NULL);

	fp = popen(buf, "r");
	if (!fp)
		return NULL;

	/*
	 * Eat lines until we see the table
	 */
	while (fgets(buf, sizeof(buf), fp))
	{
		static char checkfor[] = "<H4>Updated:";

		if (strncmp(buf, checkfor, strlen(checkfor)-1))
			continue;

		if (strcmp(buf, lastbuf) == 0)
			break;

		memcpy(lastbuf, buf, sizeof(lastbuf));

		pthread_mutex_lock(&CurseMutex);
		if (CursesInitted)
		{
			attrset(COLOR_PAIR(2));
			mvprintw(1, 7, "UPDOWN");
			attrset(A_NORMAL);
		}
		pthread_mutex_unlock(&CurseMutex);
		Alerting = 1;
		break;
	}

	pclose(fp);
	return (NULL);
}

void
updown_poll(void)
{
	time_t	now;
	int		rc;

	if (!Win)
	{
		if (ValAlertPoll < 0)
			ValAlertPoll = 60 * atoi(get_rc_value(RcFile,
						"updown_alert"));
		if (!ValAlertPoll)
			return;

		// Don't bother checking again if we've already notified him
		if (Alerting)
			return;

		now = time(NULL);

		// Defer first poll for 35 seconds
		if (LastAlertPoll == 0)
			LastAlertPoll = now + 35 - ValAlertPoll;

		if (ValAlertPoll && now >= (LastAlertPoll + ValAlertPoll))
		{
			rc = pthread_create(&Tid, NULL, check_updown, NULL);
			if (rc)
				error(1, "Couldn't create inplay thread.\n");
		}

		return;
	}

	if (!ValLivePoll && !ValDocPoll)
		return;

	now = time(NULL);

	if (ValLivePoll && now >= (LastLivePoll + ValLivePoll))
		get_live_quotes();

	if (ValDocPoll && !Cursor && now >= (LastDocPoll + ValDocPoll))
		get_updown();
}

void
updown_kill(void)
{
	if (Tid)
		pthread_cancel(Tid);
}

void
updown_popup(STREAMER sr)
{
	int	n;
	int	cols;

	// Clear alert, if any
	mvprintw(1, 7, "      ");
	Alerting = 0;

	n = LINES - 4 - 2 - NumStock - 12;
	if (n < 24)
		n = 24;
	Win = bestwin(n);
	if (!Win)
		error(1, "Can't create updown window\n");

	cols = getmaxx(Win);

	wbkgd(Win, Reverse ? A_REVERSE : A_NORMAL);

	box(Win, 0, 0);
	mvwcenter(Win, 0, "Briefing.com Up/Downgrades");

	//
	// Turns out, this works best for most news articles and an 80
	// column xterm. We will clip columns 78 and 79, but there are
	// only blanks there in the news articles I've seen.  For smaller
	// displays, we are screwed anyway.
	//
	Subwin = newpad(PADLINES, cols - 2);
	if (!Subwin)
		error(1, "Can't create updown pad\n");
	wbkgd(Subwin, Reverse ? A_REVERSE : A_NORMAL);

	Panel = new_panel(Win);

	ValDocPoll = 60 * atoi(get_rc_value(RcFile, "updown_poll"));

	//
	// Lots of stocks on this list, limit poll period
	//
	ValLivePoll = atoi(get_rc_value(RcFile, "live_quotes"));
	if (ValLivePoll && ValLivePoll < 10)
		ValLivePoll = 10;

	// Paint empty frame to give user some feedback
	clearok(Win, TRUE);
	update_panels(); doupdate();

	Sr = sr;
	get_updown();
}

static void
popdown(void)
{
	hide_panel(Panel);
	update_panels();
	del_panel(Panel);
	delwin(Subwin);
	delwin(Win);
	Win = 0;
	Subwin = 0;
	Sr = NULL;
}

int
updown_command(int c, STREAMER sr)
{
	MEVENT	m;

	switch (c)
	{
	case '\f':
		move(LINES-1, CursorX);
		wrefresh(curscr);
		return -1;
	case 'j':
	case KEY_DOWN:
		if (++Cursor > MaxCursor)
		{
			--Cursor;
			beep();
			break;
		}
		break;
	case 'k':
	case KEY_UP:
		if (--Cursor < 0)
		{
			++Cursor;
			beep();
			break;
		}
		break;
	case '-':
	case KEY_PPAGE:
		if (Cursor == 0)
		{
			beep();
			break;
		}
		Cursor -= getmaxy(Win) - 2 - 1;
		if (Cursor < 0)
			Cursor = 0;
		break;
	case '+':
	case ' ':
	case KEY_NPAGE:
		if (Cursor == MaxCursor)
		{
			beep();
			break;
		}
		Cursor += getmaxy(Win) - 2 - 1;
		if (Cursor > MaxCursor)
			Cursor = MaxCursor;
		break;
	case '0':
	case KEY_HOME:
		Cursor = 0;
		break;
	case '$':
	case KEY_END:
		Cursor = MaxCursor;
		break;

	case KEY_F(11):
		print_rect_troff(getbegy(Win), getbegx(Win),
				getmaxy(Win), getmaxx(Win),
				NULL, "screen.tr");
		break;
	case KEY_F(12):
	case CTRL('P'):
	case KEY_PRINT:
		print_window(Subwin, NumLines,
				get_rc_value(RcFile, "print_cmd"));
		break;

		// Change stocklist on main screen
	case '1': case '2': case '3': case '4': case '5':
	case '6': case '7': case '8': case '9':
		return 3;

	case KEY_MOUSE:
		if (getmouse(&m) != OK)
			break;

		// Ignore clicks in our window
		if (m.y >= getbegy(Win)
			&& m.y < getbegy(Win) + getmaxy(Win))
			break;

		// popdown and reprocess clicks in main window
		if (ungetmouse(&m) == OK)
			Ungetch = 1;
		popdown();
		return 2;

		// Quick switches to another popup
	case 'p':
	case 'e':
		popdown();
		return 1;

		// Regular exit
	case 033:
	case 'q':
	case 'x':
		popdown();
		return 2;
	default:
		beep();
		break;
	}

	display_more();
	copywin(Subwin, Win,
			Cursor, 0,
			1, 1,
			getmaxy(Win)-2, getmaxx(Win)-2,
			FALSE);
	move(LINES-1, CursorX);
	update_panels();
	doupdate();
	get_live_quotes();

	return (-1);
}
