/*
 *  Linux snipes, a text-based maze-oriented game for linux.
 *  Copyright (C) 1997 Jeremy Boulton.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Jeremy Boulton is reachable via electronic mail at
 *  boultonj@ugcs.caltech.edu.
 */

#include <stdlib.h>
#include <curses.h>
#include <assert.h>
#include "snipes.h"
#include "enemies.h"
#include "coords.h"
#include "collision.h"
#include "chars.h"
#include "sound.h"
#include "screen.h"

#define MAX_HIVES			40
#define NEW_ENEMY_DELAY_TIME		240


int num_hives = 0;
int num_dead_hives = 0;
int hive_color = 0;
int hive_creation_delay_counter;

struct hive_info {
  char in_use;
  char has_collided;
  coordinate position;
} hive[MAX_HIVES];


void init_hives( int color, screen_coords *sc, int init_num_hives )
{
  int i;

  hive_color = color;
  hive_creation_delay_counter = 0;
  num_hives = init_num_hives;
  num_dead_hives = 0;
  
  assert( num_hives < MAX_HIVES );

  for( i=0; i<MAX_HIVES; i++ )
    hive[i].in_use = 0;

  for( i=0; i<num_hives; i++ ) {
    hive[i].in_use = 1;
    hive[i].has_collided = 0;
    do {
      hive[i].position.x = sc->square_width/2-1 +
	(random()%sc->square_count_x)*(sc->square_width-1);
      hive[i].position.y = sc->square_height/2-1 +
       	(random()%sc->square_count_y)*(sc->square_height-1);
    } while( check_collisions( hive[i].position.x, hive[i].position.y, 2, 2,
			       MAZE_OBJECT_EVERYTHING ) );

    collision_map_place_obj( hive[i].position.x,
			     hive[i].position.y, 2, 2,
			     MAZE_OBJECT_SNIPE_MAKER );
  }
}


void hive_remove_from_collision_map( int index )
{
  collision_map_remove_obj( hive[index].position.x,
			    hive[index].position.y, 2, 2,
			    MAZE_OBJECT_SNIPE_MAKER );
}


void delete_hive( int index )
{
  hive_remove_from_collision_map( index );
  hive[index].in_use = 0;
  num_hives--;
  num_dead_hives++;
}


int get_num_live_hives( void )
{
  return num_hives;
}


int get_num_dead_hives( void )
{
  return num_dead_hives;
}


void show_hives( screen_coords *sc )
{
  int i;
  coordinate screen_pos;
  static int show_or_hide=0;
  
  for( i=0; i<MAX_HIVES; i++ ) {
    if( hive[i].in_use && !hive[i].has_collided )
      if( coords_check_on_screen( sc, hive[i].position, 0 ) ) {
	if( show_or_hide ) {
	  coords_to_screen_pos( sc, hive[i].position, &screen_pos );

	  screen_setcolorpair(hive_color);

	  screen_move( screen_pos.y, screen_pos.x );
	  screen_addch( HIVE_TOPLEFT );
	  screen_addch( HIVE_TOPRIGHT );

	  screen_move( screen_pos.y+1, screen_pos.x );
	  screen_addch( HIVE_BOTTOMLEFT );
	  screen_addch( HIVE_BOTTOMRIGHT );
	}
      }
  }
  
  show_or_hide = show_or_hide?0:1;
}


/* kill_hive( coordinate pos )
 * 
 * Hives get killed by weapons.  Rather than having both weapons
 * and hives check for collisions, the weapons take care of
 * everything.  When a weapon collides with a hive, this function
 * is called with the coordinate of the weapon that collided with
 * it.
 */

void kill_hive( coordinate pos )
{
  int i;

  for( i=0; i<MAX_HIVES; i++ )
    if( hive[i].in_use )
      if( hive[i].position.x == pos.x || hive[i].position.x == pos.x-1 )
	if( hive[i].position.y == pos.y || hive[i].position.y == pos.y-1 ) {
	  sound_post_enemy_death();
	  hive[i].has_collided = 1;
	}
}


/* do_hive_stuff( screen_coords *sc )
 * 
 * If a hive got hit by a bullet, get rid of it.  If it's time
 * to create a new enemy, pick a random hive and have it create
 * one.
 */

void do_hive_stuff( screen_coords *sc )
{
  static coordinate deltas[6] = {
    {  2,  0 },
    {  2,  1 },
    { -2,  0 },
    { -2,  1 },
    {  0, -1 },
    {  0,  1 }
  };
  int count, i, j;
  coordinate tp;

  for( i=0; i<MAX_HIVES; i++ ) {
    if( hive[i].in_use && hive[i].has_collided )
      delete_hive( i );
  }

  hive_creation_delay_counter += num_hives;

  if( hive_creation_delay_counter >= NEW_ENEMY_DELAY_TIME ) {
    hive_creation_delay_counter = 0;
    
    /* Find a hive at random */
    count = (random()%num_hives) + 1;
    for( i=0; i<MAX_HIVES; i++ )
      if( hive[i].in_use )
	if( --count == 0 )
	  break;
    
    assert( i < MAX_HIVES );

    if( !hive[i].has_collided )
      for( j=0; j<6; j++ ) {
	tp = hive[i].position;
	coords_add_delta( sc, &tp, deltas[j] );
	if( !check_collisions( tp.x, tp.y, 2, 1,
			       MAZE_OBJECT_EVERYTHING^MAZE_OBJECT_PLAYER ) ) {
	  add_enemy( tp, 0 );
	  break;
	}
      }
  }
}
