/*
 * $Id: services.c,v 1.9 2000/10/23 16:54:25 linus Exp $
 * Copyright (C) 1991  Lysator Academic Computer Association.
 *
 * This file is part of the LysKOM server.
 * 
 * LysKOM is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by 
 * the Free Software Foundation; either version 1, or (at your option) 
 * any later version.
 * 
 * LysKOM is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with LysKOM; see the file COPYING.  If not, write to
 * Lysator, c/o ISY, Linkoping University, S-581 83 Linkoping, SWEDEN,
 * or the Free Software Foundation, Inc., 675 Mass Ave, Cambridge, 
 * MA 02139, USA.
 *
 * Please mail bug reports to bug-lyskom@lysator.liu.se. 
 */
/*
 *  services.c  --  All the services the LysKOM server makes available
 *		    for clients.
 *
 *  Created by ceder 1990-07-17
 */

#include <config.h>
#ifdef HAVE_STRING_H
#include <string.h>
#else
#include <strings.h>
#endif
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <stdio.h>
#include <ctype.h>

#include <kom-types.h>
#include <services.h>
#include <kom-errno.h>

#include "client.h"
#include "parse.h"
#include "output.h"
#include "komserv.h"
#include "input.h"

ISC *kom_server = NULL;
#define KS kom_server


free_function	 isc_free	= NULL;
malloc_function	 isc_malloc	= NULL;
realloc_function isc_realloc	= NULL;

/*
 * Session control
 */
extern  Success
kom_open_connection (const char	* server, int portno)
{
    char *user;
    char reply[7];
    int i, c;
    
    user = getlogin();
    if ( user == NULL )
    {
	fprintf(stderr, "Can't get user id with getlogin().\n");
	fflush(stderr);
	user = "(unknown)";
    }
    
    if ( (kom_server = isc_connect(server, portno)) == NULL )
    {
	kom_errno = KOM_NO_CONNECT;
	return FAILURE;
    }
    
    fprintf(KS->out, "A%dH%s\n", strlen(user), user);
    fflush(KS->out);

    reply[ 6 ] = '\0';
    for ( i = 0; i < 6; i++)
    {
	if ( !isascii(c = my_getc(KS->in)))
	{
	    isc_disconnect(KS);
	    kom_errno = KOM_SERVER_IS_CRAZY;
	    return FAILURE;
	}
	reply[ i ] = c;
    }
    
    if ( strcmp(reply, "LysKOM") != 0 )
    {
	isc_disconnect(KS);
	kom_errno = KOM_SERVER_IS_CRAZY;
	return FAILURE;
    }
    return OK;    
}


/*
 *  Set the functions to use for storage management.  These must
 *  be call compatible with the normal functions malloc(),
 *  realloc() and free().
 */
extern  void
kom_set_storage_management (malloc_function	new_malloc,
			    realloc_function	new_realloc,
			    free_function	new_free)
{
    isc_malloc = new_malloc;
    isc_realloc = new_realloc;
    isc_free = new_free;
}


extern  Success
kom_login (Pers_no	 person,
	   const String	 passwd,
	   Bool		 visible)
{
    output_login(person, passwd, visible);
    return input_header();
}


extern  Success
kom_logout (void)	/* can never fail */
{
    output_logout( );
    return input_header( );
}


extern  Success
kom_change_conference (Conf_no	conference )
{
    output_change_conference( conference );
    return input_header( );
}



extern  Success
kom_change_name (Conf_no       conf_no,
		 const String  new_name)
{
    output_change_name( conf_no, new_name );
    return input_header( );
}

extern  Success
kom_change_what_i_am_doing (const String  what_am_i_doing )
{
    output_change_what_i_am_doing( what_am_i_doing );
    return input_header();
}



/******************************
*     Person-related calls    *
******************************/

/*
 *  Create a new person.  Returns 0 if any error occured.
 */
extern  Pers_no
kom_create_person (const String  name,
		   const String  passwd )
{
    output_create_person( name, passwd );
    if ( input_header( ) != OK )
	return 0;

    return parse_long( KS->in );
}

extern  Success
kom_get_person_stat (Pers_no	      person,
		     Person	    * result )
{
    output_get_person_stat( person );
    return input_get_person_stat( result );
}

extern  Success
kom_get_created_texts(Pers_no       pers_no,
		      Local_text_no first,
		      unsigned long len,
		      Text_list   * result)
{
    output_get_created_texts( pers_no, first, len );
    return input_get_created_texts( result );
}


extern  Success
kom_get_membership (Pers_no		  person,
		    unsigned short	  first,
		    unsigned short	  no_of_confs,
		    Bool		  want_read_texts,
		    Membership_list	* memberships )
{
    output_get_membership( person, first, no_of_confs, want_read_texts);
    return input_get_membership( memberships );
}

extern  Success
kom_set_priv_bits (Pers_no    person,
		   Priv_bits  privileges )
{
    output_set_priv_bits( person, privileges );
    return input_header( );
}


extern  Success
kom_set_passwd (Pers_no	      person,
		const String  old_pwd,
		const String  new_pwd )
{
    output_set_passwd( person, old_pwd, new_pwd );
    return input_header( );
}


extern  Success
kom_query_read_texts (Pers_no      pers_no,
		  Conf_no      conf_no,
		  Membership * result   )
{
    output_query_read_texts( pers_no, conf_no );
    return input_query_read_texts( result );
}



/****************************************
*	Conference-related calls	*
****************************************/


extern  Conf_no
kom_create_conf (const String  name,
		 Conf_type     type )
{
    output_create_conf( name, type );
    if ( input_header( ) != OK )
	return 0;

    return parse_long( KS->in );
}


extern  Success
kom_delete_conf (Conf_no	conf )
{
    output_delete_conf( conf );
    return input_header( );
}


extern  Success
kom_get_conf_stat (Conf_no	   conf_no,
		   Conference    * result )
{
    output_get_conf_stat( conf_no );
    return input_get_conf_stat( result );
}

extern  Success
kom_get_members (Conf_no	  conf,
		 unsigned short	  first,
		 unsigned short	  no_of_members,
		 Member_list	* members	)
{
    output_get_members(conf, first, no_of_members);
    return input_get_members( members );
}


extern  Success
kom_add_member (Conf_no		conf_no,
		Pers_no		pers_no,
		unsigned char	priority,
		unsigned short	where	)
{
    output_add_member( conf_no, pers_no, priority, where );
    return input_header( );
}


extern  Success
kom_sub_member (Conf_no	conf_no,
		Pers_no	pers_no )
{
    output_sub_member( conf_no, pers_no );
    return input_header( );
}

extern  Success
kom_set_unread (Conf_no   conf_no,
		Text_no   no_of_unread)
{
    output_set_unread( conf_no, no_of_unread );
    return input_header( );
}

extern  Success
kom_set_presentation (Conf_no   conf_no,
		      Text_no   text_no ) /* 0 to delete pres. */
{
    output_set_presentation( conf_no, text_no );
    return input_header( );
}

extern  Success
kom_set_etc_motd (Conf_no	  conf_no,
		  Text_no	  text_no )
{
    output_set_etc_motd( conf_no, text_no );
    return input_header( );
}


extern  Success
kom_set_supervisor (Conf_no	  conf_no,
		    Conf_no	  admin	)
{
    output_set_supervisor( conf_no, admin );
    return input_header( );
}

extern  Success
kom_set_permitted_submitters (Conf_no  conf_no,
			      Conf_no  perm_sub )
{
    output_set_permitted_submitters( conf_no, perm_sub );
    return input_header( );
}

extern  Success
kom_set_super_conf (Conf_no	  conf_no,
		    Conf_no	  super_conf )
{
    output_set_super_conf( conf_no, super_conf );
    return input_header( );
}

extern  Success
kom_set_conf_type (Conf_no    conf_no,
		   Conf_type  type	)
{
    output_set_conf_type( conf_no, type );
    return input_header( );
}

extern  Success
kom_set_garb_nice (Conf_no    conf_no,
		   Garb_nice  ndays	)
{
    output_set_garb_nice( conf_no, ndays );
    return input_header( );
}



/********************************
*     Calls to handle marks     *
********************************/


extern  Success
kom_get_marks (Mark_list *result )
{
    output_get_marks( );
    return input_get_marks( result );
}



extern  Success
kom_mark_text (Text_no  	text,
	       unsigned char    mark_type )
{
    output_mark_text( text, mark_type );
    return input_header( );
}




/*******************************
*     Calls to handle texts    *
*******************************/


extern  Success
kom_get_text (Text_no      text,
	      String_size  start_char,
	      String_size  end_char,
	      String      *result)
{
    output_get_text( text, start_char, end_char );
    return input_get_text( result );
}


extern  Success
kom_get_text_stat (Text_no    text,
		   Text_stat *result)
{
    output_get_text_stat( text );
    return input_get_text_stat( result );
}

extern  Success
kom_mark_as_read (Conf_no                conference,
		  int		         no_of_texts,
		  const Local_text_no  * text_arr   )
{
    output_mark_as_read( conference, no_of_texts, text_arr );
    return input_header( );
}


extern  Text_no
kom_create_text (const String       	message,
		     unsigned short	no_of_misc,
		     const Misc_info  * misc	     )
{
    output_create_text( message, no_of_misc, misc );
    return input_create_text( );
}

extern  Text_no
kom_create_anonymous_text (const String       message,
			   unsigned short     no_of_misc,
			   const Misc_info  * misc	     )
{
    output_create_anonymous_text( message, no_of_misc, misc );
    return input_create_anonymous_text( );
}

extern Success
kom_delete_text( Text_no text_no )
{
    output_delete_text( text_no );
    return input_header( );
}



extern  Success
kom_add_recipient (Text_no    text_no,
		   Conf_no    conf_no,
		   Info_type  type	)
{
    output_add_recipient( text_no, conf_no, type );
    return input_header( );
}

extern  Success
kom_sub_recipient (Text_no	  text_no,
		   Conf_no	  conf_no )
{
    output_sub_recipient( text_no, conf_no);
    return input_header( );
}


extern  Success
kom_add_comment (Text_no	  comment,
		 Text_no 	  comment_to )
{
    output_add_comment( comment, comment_to );
    return input_header( );
}


extern  Success
kom_sub_comment (Text_no	  comment,
		 Text_no	  comment_to )
{
    output_sub_comment( comment, comment_to );
    return input_header( );
}


extern Success
kom_add_footnote (Text_no	footnote,
		  Text_no 	footnote_to )
{
    output_add_footnote( footnote, footnote_to );
    return input_header();
}


extern Success
kom_sub_footnote (Text_no footnote,
		  Text_no parent )
{
    output_sub_footnote( footnote, parent );
    return input_header();
}


extern  Success
kom_get_map (Conf_no	     conf_no,
	     Local_text_no   first_local_no,
	     Local_text_no   no_of_texts,
	     Text_list	   * result)
{
    output_get_map( conf_no, first_local_no, no_of_texts );
    return input_get_map( result );
}


extern  Success
kom_get_time ( struct tm  * clock_res )
{
    output_get_time( );
    return input_get_time( clock_res );
}

extern  Success
kom_get_info (Info *result )
{
    output_get_info( );
    return input_get_info( result );
}


extern Success
kom_set_motd_of_lyskom (Text_no motd)
{
    output_set_motd_of_lyskom( motd );
    return input_header( );
}


extern Success
kom_enable (unsigned char ena_level)
{
    output_enable( ena_level );
    return input_header( );
}


extern Success
kom_sync (void) 
{
    output_sync( );
    return input_header( );
}


extern Success
kom_shutdown (int exit_val) 
{
    output_shutdown( exit_val );
    return input_header( );
}


extern Success
kom_send_message (Pers_no recipient,
		  const String message)
{
    output_send_message( recipient, message );
    return input_header( );
}

extern Success
kom_get_unread_confs(Pers_no       pers_no,
		     Conf_no_list *result)
{
    output_get_unread_confs( pers_no );
    return input_get_unread_confs( result );
}


extern Success
kom_get_last_text(struct tm 	* clock_res,
		  Text_no	* result)
{
    output_get_last_text( clock_res );
    return input_get_last_text( result );
}


/*
 * Return next/previous existing text-no.
 */

extern  Success
kom_find_next_text_no(Text_no	  start, 
		      Text_no	* result)
{
    output_get_next_text_no(start);
    return input_get_next_text_no(result);
}


extern  Success
kom_find_previous_text_no (Text_no	  start, 
			   Text_no	* result)
{
    output_get_previous_text_no(start);
    return input_get_previous_text_no(result);
}

/*
 * Lookup names.
 */
extern  Success
kom_lookup_z_name(const String 		name,
		  int	 		want_persons,
		  int			want_confs,
		  Conf_z_info_list    * result)
{
    output_lookup_z_name(name, want_persons, want_confs);
    return input_lookup_z_name(result);
}


/*
 * Lookup names.
 */
extern  Success
kom_re_z_lookup(const String 		name,
		int	 		want_persons,
		int			want_confs,
		Conf_z_info_list    * result)
{
    output_re_z_lookup(name, want_persons, want_confs);
    return input_re_z_lookup(result);
}


/*
 * Who_is_on_dynamic
 */
extern  Success
kom_who_is_on_dynamic(int			want_visible,
		      int			want_invisible,
		      long			active_last,
		      Dynamic_session_info_list*result)
{
    output_who_is_on_dynamic(want_visible, want_invisible, active_last);
    return input_who_is_on_dynamic(result);
}
