/* $Id: reading.c,v 1.11 2001/12/29 08:23:36 linus Exp $
 * reading.c
 * Implements the reading order, handling of comments.
 *
 *
 * Copyright (C) 2000	Lysator Computer Club,
 *			Linkoping University,  Sweden
 *
 * Everyone is granted permission to copy, modify and redistribute
 * this code, provided the people they give it to can.
 *
 *
 * Author:	Linus Tolke
 *		Lysator Computer Club
 *		Linkoping University
 *		Sweden
 *
 * email:	linus@lysator.liu.se
 *
 *
 * Any opinions expressed in this code are the author's personal opinions,
 * and does not represent any official standpoint of Lysator, even if so
 * stated.
 */

#include <config.h>

#ifdef HAVE_ASSERT_H
#include <assert.h>
#endif

#include <kom-types.h>
#include <misc-types.h>
#include <kom-errno.h>
#include <services.h>
#include <zmalloc.h>

#include "internal.h"
#include "xoutput.h"
#include "copy.h"
#include "cache.h"
#include "misc-parser.h"
#include "error.h"
#include "offline.h"
#include "main.h"

#include "reading.h"
#include "gentypes.h"

/*
 *****************************************************************
 */

/*
 * New reading functions.
 */
LINKED_LIST_TYPE_DEFINITION(Text_no, 0);
LINKED_LIST_FIRST_ELEMENT_FUNCTION(Text_no);
LINKED_LIST_INSERT_FIRST_FUNCTION(Text_no);
LINKED_LIST_REMOVE_FIRST_FUNCTION(Text_no);
LINKED_LIST_REMOVE_ALL_FUNCTION(Text_no);
LINKED_LIST_CLEAN_FROM_FUNCTION(Text_no);

A_STACK(Text_no, footnote, 0);
A_STACK(Text_no, comment, 0);
A_STACK(Text_no, read_texts, 0);

A_STACK_CLEAN_FROM_FUNCTION(Text_no, footnote);
A_STACK_CLEAN_FROM_FUNCTION(Text_no, comment);
A_STACK_CLEAN_FROM_FUNCTION(Text_no, read_texts);

A_STACK_CONTAINS_FUNCTION(Text_no, read_texts);

A_STACK_REMOVE_ALL_FUNCTION(Text_no, footnote);
A_STACK_REMOVE_ALL_FUNCTION(Text_no, comment);

A_STACK(Text_no, temp_footnote, 0);
A_STACK(Text_no, temp_comment, 0);

A_STACK_REMOVE_ALL_FUNCTION(Text_no, temp_footnote);
A_STACK_REMOVE_ALL_FUNCTION(Text_no, temp_comment);



static void
clean_temp_stacks(void)
{
#ifdef TRACE
    if (TRACE(6))
	xprintf("DEBUG: clean_temp_stacks()\n");
#endif

    temp_footnote_remove_all();
    temp_comment_remove_all();
}

/*
 * Cache for keeping track of what local numbers I have left unread in the
 * current conference.
 * This is updated when we first access a conference trying to determine what
 * text to read, kept updated by the reading_* functions and async messages
 * from the server.
 * This is instead of the unread_texts_left/decrease_texts_left.
 *
 * The implementation is an ordered list of local numbers (linked list).
 * The interface is these functions:
 */
static void rs_unread_texts_set(Membership * ms, Conference * cs);
static Local_text_no rs_unread_texts_get_first(void);
static unsigned int rs_unread_texts_get_count(void);
static void rs_unread_texts_read(Local_text_no);
static void rs_unread_texts_new_text(Local_text_no);
static void rs_unread_texts_empty(void);
/*
 */
LINKED_LIST_TYPE_DEFINITION(Local_text_no, 0);
LINKED_LIST_FIRST_ELEMENT_FUNCTION(Local_text_no);
LINKED_LIST_INSERT_FIRST_FUNCTION(Local_text_no);
LINKED_LIST_REMOVE_FIRST_FUNCTION(Local_text_no);
LINKED_LIST_REMOVE_ALL_FUNCTION(Local_text_no);
LINKED_LIST_COUNT_ELEMENTS_FUNCTION(Local_text_no);
LINKED_LIST_CLEAN_FROM_FUNCTION(Local_text_no);
LINKED_LIST_CLEAN_FROM_SORTED_FUNCTION(Local_text_no);
LINKED_LIST_SORTED_INSERT_LT_FUNCTION(Local_text_no);

static LINKED_LIST_TYPE(Local_text_no) * unread_texts = NULL;
static Conf_no unread_texts_is_set = 0;

static void
rs_unread_texts_set(Membership * ms, Conference * cs)
{
    Local_text_no ltno;
    int i;

#ifdef TRACE
    if (TRACE(6))
    {
	xprintf("DEBUG: rs_unread_texts_set(ms{ltr=%lu,nor=%lu}, cs{fln=%lu,not=%lu)\n",
		ms->last_text_read, ms->no_of_read,
		cs->first_local_no, cs->no_of_texts);
    }
#endif

    /* Build a complete list of all from last in conf to last complete read */
    /* We work from the end to get a sorted list to begin with. */
    for (ltno = cs->first_local_no + cs->no_of_texts - 1;
	 ltno > ms->last_text_read;
	 ltno--)
    {
	linked_list_Local_text_no_insert_first(&unread_texts, ltno);
    }

    /* Remove the texts in the list. */
    if (ms->read_texts)
    {
	for (i = 0; i < ms->no_of_read; i++)
	{
	    linked_list_Local_text_no_clean_from_sorted(&unread_texts, 
							ms->read_texts[i]);
	}
    }

    unread_texts_is_set = ccn;
}

/*
 * rs_unread_texts_get_first
 * Returns 0 if there is no text in the list.
 */
static Local_text_no
rs_unread_texts_get_first(void)
{
#ifdef TRACE
    if (TRACE(6))
	xprintf("DEBUG: rs_unread_texts_get_first()\n");
#endif
    return linked_list_Local_text_no_first_element(unread_texts);
}

static unsigned int
rs_unread_texts_get_count(void)
{
#ifdef TRACE
    if (TRACE(6))
	xprintf("DEBUG: rs_unread_texts_get_count()\n");
#endif
    return linked_list_Local_text_no_count_elements(unread_texts);
}

static void
rs_unread_texts_read(Local_text_no ltno)
{
#ifdef TRACE
    if (TRACE(6))
	xprintf("DEBUG: rs_unread_texts_read(%lu)\n", ltno);
#endif
    linked_list_Local_text_no_clean_from(&unread_texts, ltno);
}

static void
rs_unread_texts_new_text(Local_text_no ltno)
{
    /* Enter the new Local_text_no sorted in the list. */
#ifdef TRACE
    if (TRACE(6))
	xprintf("DEBUG: rs_unread_texts_new_text()\n");
#endif
    linked_list_Local_text_no_sorted_insert_lt(&unread_texts, ltno);
}

static void
rs_unread_texts_empty(void)
{
#ifdef TRACE
    if (TRACE(6))
	xprintf("DEBUG: rs_unread_texts_empty()\n");
#endif

    linked_list_Local_text_no_remove_all(&unread_texts);
    unread_texts_is_set = 0;
}


/*
 * For getting the text number
 */
Text_no
reading_next_footnote_to_be_read(void)
{
    Text_no tno = peek_footnote();

#ifdef TRACE
    if (TRACE(6))
	xprintf("DEBUG: reading_next_footnote_to_be_read() => %lu\n", tno);
#endif

    return tno;
}

Text_no
reading_next_comment_to_be_read(void)
{
    Text_no tno = peek_comment();

#ifdef TRACE
    if (TRACE(6))
	xprintf("DEBUG: reading_next_comment_to_be_read() => %lu\n", tno);
#endif

    return tno;
}


static void
initiate_unread_texts(void)
{
    if (unread_texts_is_set != ccn
	&& ccn != 0)
    {
	/* Fetch the information from this conference. */
	Membership ship = EMPTY_MEMBERSHIP;
	Conference * stat;
	Kom_err error;

#ifdef TRACE
	if (TRACE(6))
	    xprintf("DEBUG: initiate_unread_texts()\n");
#endif

	if (locate_membership(ccn, cpn, &ship) < 0)
	{
	    /* We could not find the the membership. Somehow we went to/are in
	     * a conference where we are not member. Confusion. This is however
	     * not the responsibility of this function so lets just ignore it.
	     */
	    return;
	}

	stat = conf_stat(ccn, &error);
	if (stat == NULL)
	{
	    /* For some reason we could not fetch the conference. */
	    return;
	}

	if (unread_texts_is_set)
	{
	    /* Empty the read texts cache */
	    rs_unread_texts_empty();
	}

	rs_unread_texts_set(&ship, stat);
	release_conf(stat);
	zfree(stat);

	/* Don't release the membership. It is an "unmarked" copy. */
    }
}


/*
 * reading_next_text_to_be_read_with_local_no
 *
 * Returns the next text to be read in this conference.
 * If there is no more text to be read, then returns 0.
 * If the list of unread texts in this conference is not set, then fetch it.
 * Let the argument *ltno be the local text no found.
 */
Text_no
reading_next_text_to_be_read_with_local_no(Local_text_no * ltnop)
{
    Text_no global_no_to_read = 0;

#ifdef TRACE
    if (TRACE(6))
	xprintf("DEBUG: reading_next_text_to_be_read_with_local_no()\n");
#endif

    if (ccn == 0)
	return 0;

    initiate_unread_texts();

    /*
     * Now we know that the unread list contains all unread texts in the
     * current conference.
     */
    do {
	Kom_err error;
	*ltnop = rs_unread_texts_get_first();

	if (*ltnop == 0)
	{
	    global_no_to_read = 0;
	    break;
	}

	global_no_to_read = map_local_to_global (ccn, *ltnop, &error);

	if (global_no_to_read == 0)
	{
	    rs_unread_texts_read(*ltnop);
	}
    } while (global_no_to_read == 0);

    /* BUG: Doesn't check for error */

    /* BUG: We don't discount for the newly read texts. */

#ifdef TRACE
    if (TRACE(6))
	xprintf("DEBUG: reading_next_text_to_be_read_with_local_no() => %lu\n",
		global_no_to_read);
#endif

    return global_no_to_read;
}

/*
 * reading_next_text_to_be_read_with_local_no
 *
 * Returns the next text to be read in this conference.
 * If there is no more text to be read, then returns 0.
 * If the list of unread texts in this conference is not set, then fetch it.
 */
Text_no
reading_next_text_to_be_read(void)
{
    Text_no global_no_to_read;
    Local_text_no ltno;

#ifdef TRACE
    if (TRACE(6))
	xprintf("DEBUG: reading_next_text_to_be_read()\n");
#endif

    global_no_to_read = reading_next_text_to_be_read_with_local_no(&ltno);

#ifdef TRACE
    if (TRACE(6))
	xprintf("DEBUG: reading_next_text_to_be_read() => %lu\n",
		global_no_to_read);
#endif

    return global_no_to_read;
}



unsigned int
reading_unread_texts_count(void)
{
    unsigned int i;

    initiate_unread_texts();

    i = rs_unread_texts_get_count();

#ifdef TRACE
    if (TRACE(6))
	xprintf("DEBUG: reading_unread_texts_count() => %u\n", i);
#endif

    return i;
}

/*
 * reading_next_conf_with_unread
 * Returns the conf_no of the next conference with unread.
 * By next we mean the first conf after ccn in the reading order, or if there
 * are no confs with unread after ccn, the first conf in the reading order
 * with unread.
 * ccn can be returned if it is the only conf with unread.
 * Returns 0 if there are no unread.
 */
Conf_no
reading_next_conf_with_unread(void)
{
#define FUNCTION "reading_next_conf_with_unread"

    Kom_err error;

    Conf_no_list list = EMPTY_CONF_NO_LIST;
    Membership ship = EMPTY_MEMBERSHIP;
    int pos;

#ifdef TRACE
    if (TRACE(6))
	xprintf("DEBUG: %s()\n", FUNCTION);
#endif

    if (offline)
    {
        fetch_person_number(&cpn);

        offline_get_unread_confs(&list);
    }
    else
    {
        if (kom_get_unread_confs (cpn, &list) == FAILURE)
	{
	    switch (kom_errno)
	    {
	    case  KOM_OUT_OF_MEMORY:
		fatal1 (CLIENT_OUT_OF_MEMORY, "get_conf_no_list() failed");
		break;

	    case  KOM_NO_CONNECT:
		fatal1 (CLIENT_SERVER_ERROR, "get_conf_no_list() failed");
		break;

	    default:
		fatal1 (CLIENT_UNEXPECTED_SERVER_ERROR,
			"get_conf_no_list() failed");
		break;
	    }
	}    
    }

    /*
     * set pos to point to confernce after the current conf (ccn), i.e. the
     * first candidate.
     */
    pos = -1;
    if (ccn != 0)
    {
	for (pos = first_membership(cpn, &ship);
	     pos != -1;
	     pos = next_membership(&ship))
	{
	    if (ship.conf_no == ccn)
	    {
		pos = next_membership(&ship);
		break;
	    }
	}
    }
    /*
     * Now pos points to the first candidate or is -1 if the current conf was
     * not found in the membership (or is 0).
     */
    if (pos == -1)
    {
	/*
	 * We start from the beginning again.
	 */
	pos = first_membership(cpn, &ship);
    }

    while (1)
    {
	if (pos == -1)
	{
	    /*
	     * We have found the end of the membership.
	     */
	    if (ccn == 0)
		break;
	    else
	    {
		pos = first_membership(cpn, &ship);
	    }
	}

	if (unread_texts_is_set != 0
	    && unread_texts_is_set == ship.conf_no)
	{
	    if (rs_unread_texts_get_count() > 0)
		return ship.conf_no;
	}
	else if (membership_in_conf_no_list(ship, list))
	{
	    int unread;

	    unread = conf_last(ship.conf_no, &error)
		- (ship.last_text_read + ship.no_of_read);
	    if (unread > 0)
	    {
		/* Free the Conf_no_list => noop */
		/* Free the Membership => noop */

#ifdef TRACE
		if (TRACE(6))
		    xprintf("DEBUG: %s() => %lu\n", FUNCTION, ship.conf_no);
#endif

		return ship.conf_no;
	    }	    
	}

	if (ship.conf_no == ccn)
	{
	    /*
	     * We have gone through the loop once and still have not found any.
	     */
	    break;
	}
	pos = next_membership(&ship);
    }

#ifdef TRACE
    if (TRACE(6))
	xprintf("DEBUG: %s() => 0 (none found)\n", FUNCTION);
#endif

    return 0;
#undef FUNCTION
}

/*
 * reading_read_text_local_no
 *
 * For removing a text from the local conference.
 */
void
reading_read_text_local_no(Conf_no cno, Local_text_no ltno)
{
    if (unread_texts_is_set == cno)
	rs_unread_texts_read(ltno);
}


/*
 * reading_read_text_simple
 *
 * For marking the text as read.
 */
void
reading_read_text_simple(Text_no tno)
{
#ifdef TRACE
    if (TRACE(6))
	xprintf("DEBUG: reading_read_text_simple(%lu)\n", tno);
#endif

    /* Put in list */
    push_read_texts(tno);

    /* Remove from comment and footnote stack */
    clean_footnote_stack_from(tno);
    clean_comment_stack_from(tno);
}

/*
 * reading_read_text
 *
 * For marking the text as read
 * Also pushes comments and such to be read next.
 */
void
reading_read_text(Text_no tno)
{
    Text_no	commentno;

#ifdef TRACE
    if (TRACE(6))
	xprintf("DEBUG: reading_read_text(%lu)\n", tno);
#endif

    reading_read_text_simple(tno);

    /* When we do the pushing here, we do this in the opposite direction 
     * meaning that the first found text is pushed last in order to appear
     * first.
     * This is done by using two temporary stacks, one for the footnotes
     * and one for the comments while scanning the text only once.
     */
    clean_temp_stacks();

    {
	Text_stat * stat = NULL;
	const Misc_info * misc;
	Misc_info_group misc_data;

	stat = text_stat(tno);

	if (stat == NULL)
	{
	    return;
	}

	for (misc = stat->misc_items;
	     (misc_data = parse_next_misc(&misc,
					  stat->misc_items + 
					  stat->no_of_misc)).type
	     != m_end_of_list;
	     )
	{
#ifdef HAVE_ASSERT_H
	    assert(misc_data.type != m_error);
#endif

#ifdef TRACE
	    if (TRACE(6))
		xprintf("DEBUG: reading_confirm_read_texts() "
			"misc_data: %d\n",
			misc_data.type);
#endif

	    if (misc_data.type == m_footn_in
		|| misc_data.type == m_comm_in)
	    {
		commentno = misc_data.text_ref;
	    } 
	    else if (misc_data.type == m_recpt
		     || misc_data.type == m_cc_recpt)
	    {
		/* This is a text in a conference */
		/*
		 * If it is the current conf, update the reading_unread
		 * structure.
		 */
		if (misc_data.recipient == ccn)
		{
#ifdef TRACE
		    if (TRACE(6))
			xprintf("DEBUG: reading_read_text(%lu) "
				"(local removing %lu)\n", 
				tno, misc_data.local_no);
#endif

		    rs_unread_texts_read(misc_data.local_no);
		}
		continue;
	    }
	    else
	    {
#ifdef TRACE
		if (TRACE(6))
		    xprintf("DEBUG: reading_confirm_read_texts() "
			    "misc_data: %d was not handled\n",
			    misc_data.type);
#endif


		continue;
	    }

	    /* Time to check if this article is unread by the user */
	    if (has_been_read(commentno) == FALSE)
	    {
		if (misc_data.type == m_footn_in)
		{
#ifdef TRACE
		    if (TRACE(6))
			xprintf("DEBUG: reading_read_text(%lu) "
				"adding %lu in footnote stack\n",
				tno, commentno);
#endif

		    push_temp_footnote(commentno);
		}
		else if (misc_data.type == m_comm_in)
		{
#ifdef TRACE
		    if (TRACE(6))
			xprintf("DEBUG: reading_read_text(%lu) "
				"adding %lu in comment stack\n",
				tno, commentno);
#endif

		    push_temp_comment(commentno);
		}
		else
		{
		    continue;
		}
	    }
	    else
	    {
#ifdef TRACE
		if (TRACE(6))
		    xprintf("DEBUG: reading_read_text(%lu) was read\n", tno);
#endif
	    }
	}
    }

    /* Push unread footnotes to footnote stack. */
    while ((commentno = pop_temp_footnote()) != 0)
    {
#ifdef TRACE
	if (TRACE(6))
	    xprintf("DEBUG: reading_read_text(%lu) "
		    "adding2 %lu in footnote stack\n",
		    tno, commentno);
#endif

	push_footnote(commentno);
    }

    /* Push unread comments to comment stack. */
    while ((commentno = pop_temp_comment()) != 0)
    {
#ifdef TRACE
	if (TRACE(6))
	    xprintf("DEBUG: reading_read_text(%lu) "
		    "adding2 %lu in comment stack\n",
		    tno, commentno);
#endif

	push_comment(commentno);
    }
}

/*
 * confirm_read_texts
 * Does the actual marking of read texts.
 */
void
reading_confirm_read_texts(void)
{
    Text_no tno;

    while ((tno = pop_read_texts()) != 0)
    {
#ifdef TRACE
	if (TRACE(6))
	    xprintf("DEBUG: reading_confirm_read_texts() marking %lu\n", tno);
#endif

	mark_global_as_read(tno, 0);
    }
}

/*
 * text_is_among_read_texts
 */
static int
text_is_among_read_texts(Text_no tno)
{
    int res = read_texts_contains(tno);

#ifdef TRACE
    if (TRACE(6))
	xprintf("DEBUG: text_is_among_read_texts(%lu) => %d\n", tno, res);
#endif

    return res;
}


/*
 * Go to conference
 */
void
reading_goto_conf(Conf_no cno)
{
#define FUNCTION "reading_goto_conf()"

    Success retval;
    Kom_err error;

#ifdef TRACE
    if (TRACE(6))
	xprintf("DEBUG: %s()\n", FUNCTION);
#endif

    /* Reset all lists. */
    footnote_remove_all();
    comment_remove_all();

    /* Set current conference */
    retval = cache_change_conf(cno, &error);

    if (retval == 0)
    {
	fatal2(CLIENT_UNEXPECTED_SERVER_ERROR, "cache_change_conf(%d)", cno);
    }
#undef FUNCTION
}


/*
 * reading_forget_about_conf
 *
 * Forget all we know about the current conf.
 */
void
reading_forget_about_conf(void)
{
    rs_unread_texts_empty();
}
