/*
 *  xoutput.c  --  Print text to a file, and if it is a tty, then stop
 *		   and wait when printed a screenful.
 *	
 *
 *
 *  Copyright (C) 1990	Lysator Computer Club,
 *			Linkoping University,  Sweden
 *
 *  Everyone is granted permission to copy, modify and redistribute
 *  this code, provided the people they give it to can.
 *
 *
 *  Author:	Thomas Bellman
 *		Lysator Computer Club
 *		Linkoping University
 *		Sweden
 *
 *  email:	Bellman@Lysator.LiU.SE
 *
 *
 *  Any opinions expressed in this code are the author's PERSONAL opinions,
 *  and does NOT, repeat NOT, represent any official standpoint of Lysator,
 *  even if so stated.
 */


#include <config.h>
#include <stdio.h>
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <stdarg.h>
#if STDC_HEADERS || HAVE_STRING_H
#include <string.h>
#else
#include <strings.h>
#endif
#include <sys/types.h>

#include <libintl.h>

#include "xoutput.h"
#include <errno.h>
#include <signal.h>

#include <s-string.h>

#include "internal.h"
#include "error.h"
#include "edit.h"
#include "quit.h"

#include "do-printf.h"


#define	Export


static  XO_stream	  kom_dfl_stream = { XO_terminal, NULL, -1, 0, "" };


/*  ==================================================================  */
/*		    	    Global variables				*/

Export  XO_stream	* kom_outstream		= &kom_dfl_stream;
Export	int		  printed_lines;
Export	int		  current_column;

/* Default characterset is ISO 8859-1 */
Export  int               charset 		= 0;

#if defined (HAVE_TERMIO) || defined (HAVE_TERMIOS)
/* The eof character is forgotten when setting term type. */
Export	unsigned char	  original_eof_char;
#endif

/*  ==================================================================  */
/*			    Local variables				*/

static	int		  window_width	= 80;	/* Default value */
static	int		  window_height	= 24;	/* Default value */




/*  ==================================================================  */
/*		    Forward declarations of functions			*/

static  int	do_xputc (int ch, void *stream);

static  int	printf_String  (va_list    * arg_p,
				putc_fun_T   putc_func,
				char	     sign,
				int	     flags,
				int	     width,
				int	     precision,
				void	   * stream);

static  int	printf_goto_column (va_list	 * arg_p,
				    putc_fun_T	   putc_func,
				    char	   sign,
				    int		   flags,
				    int		   width,
				    int		   precision,
				    void	 * stream);

void suspend_lyskom(int sig);


/*  ==================================================================  */
/*			Constants and other things			*/

#define	BUFFER_SIZE	  8192		/* Max length of one printout */


/* Get terminal size from system.
   Store number of lines into *heightp and width into *widthp.
   If zero or a negative number is stored, the value is not valid.  */

static
void
get_frame_size (int *widthp, int *heightp)
{

#if defined(HPUX_5) || defined(HPUX_6) || defined(HPUX_7) || defined(HPUX_8)
    /* Yes this is my first try with HPUX */
    /* TIOCGWINSZ and TIOCGSIZE does not work */
    return;

#else
#ifdef TIOCGWINSZ

  /* BSD-style.  */
  struct winsize size;

  if (ioctl (0, TIOCGWINSZ, &size) == -1)
    *widthp = *heightp = 0;
  else
    {
      *widthp = size.ws_col;
      *heightp = size.ws_row;
    }

#else
#ifdef TIOCGSIZE

  /* SunOS - style.  */
  struct ttysize size;  

  if (ioctl (0, TIOCGSIZE, &size) == -1)
    *widthp = *heightp = 0;
  else
    {
      *widthp = size.ts_cols;
      *heightp = size.ts_lines;
    }

#else
#ifdef VMS

  struct sensemode tty;
  ?????? No clue on what input_fd should be.
  SYS$QIOW (0, input_fd, IO$_SENSEMODE, &tty, 0, 0,
	    &tty.class, 12, 0, 0, 0, 0);
  *widthp = tty.scr_wid;
  *heightp = tty.scr_len;

#else /* system doesn't know size */

  *widthp = 0;
  *heightp = 0;

#endif /* not VMS */
#endif /* not SunOS-style */
#endif /* not BSD-style */
#endif /* early HPUX */
}


/* This is a badly named function. Better call it something like
   initially setup of sizes...
 */
void catch_sigwinch(int signal_number);
void
catch_sigwinch(int signal_number)
{
#define FUNCTION "catch_sigwinch()"

    if (signal_number != SIGWINCH)
	fatal1(CLIENT_ERROR_TOO_boring,
	       "Your signal mechanism is malfunctioning.");
    get_frame_size(&window_width,&window_height);

    /* Howcome I always manage to get the size 0, 0 from the default 
       emacs routine and then it hangs. */
    /* Stupid default: */
    if (!window_height)
	window_height = 24;
    if (!window_width)
	window_width = 80;

#ifdef SIGWINCH
    (void) signal(SIGWINCH, &catch_sigwinch);
#else
    /* Just forget the possibility to resize. */
#endif
#undef FUNCTION
}


/* Getting and setting emacs_tty structures.  */

/* Set *TC to the parameters associated with the terminal FD.
   Return zero if all's well, or -1 if we ran into an error we
   couldn't deal with.  */
int
emacs_get_tty (fd, settings)
     int fd;
     struct emacs_tty *settings;
{
  /* Retrieve the primary parameters - baud rate, character size, etcetera.  */
#ifdef HAVE_TCATTR
  /* We have those nifty POSIX tcmumbleattr functions.  */
  if (tcgetattr (fd, &settings->main) < 0)
    return -1;

#else
#ifdef HAVE_TERMIO
  /* The SYSV-style interface?  */
  if (ioctl (fd, TCGETA, &settings->main) < 0)
    return -1;

#else
#ifdef VMS
  /* Vehemently Monstrous System?  :-)  */
  if (! (SYS$QIOW (0, fd, IO$_SENSEMODE, settings, 0, 0,
		   &settings->main.class, 12, 0, 0, 0, 0)
	 & 1))
    return -1;

#else
  /* I give up - I hope you have the BSD ioctls.  */
  if (ioctl (fd, TIOCGETP, &settings->main) < 0)
    return -1;

#endif
#endif
#endif

  /* Suivant - Do we have to get struct ltchars data?  */
#ifdef HAVE_LTCHARS
  if (ioctl (fd, TIOCGLTC, &settings->ltchars) < 0)
    return -1;
#endif

  /* How about a struct tchars and a wordful of lmode bits?  */
#ifdef HAVE_TCHARS
  if (ioctl (fd, TIOCGETC, &settings->tchars) < 0
      || ioctl (fd, TIOCLGET, &settings->lmode) < 0)
    return -1;
#endif

  /* We have survived the tempest.  */
  return 0;
}


/* Set the parameters of the tty on FD according to the contents of
   *SETTINGS.  If WAITP is non-zero, we wait for all queued output to
   be written before making the change; otherwise, we forget any
   queued input and make the change immediately.
   Return 0 if all went well, and -1 if anything failed.  */
int
emacs_set_tty (fd, settings, waitp)
     int fd;
     struct emacs_tty *settings;
     int waitp;
{
  /* Set the primary parameters - baud rate, character size, etcetera.  */
#ifdef HAVE_TCATTR
  int i;
  /* We have those nifty POSIX tcmumbleattr functions.
     William J. Smith <wjs@wiis.wang.com> writes:
     "POSIX 1003.1 defines tcsetattr() to return success if it was
     able to perform any of the requested actions, even if some
     of the requested actions could not be performed.
     We must read settings back to ensure tty setup properly.
     AIX requires this to keep tty from hanging occasionally."  */
  /* This make sure that we don't loop indefinitely in here.  */
  for (i = 0 ; i < 10 ; i++)
    if (tcsetattr (fd, waitp ? TCSAFLUSH : TCSADRAIN, &settings->main) < 0)
      {
	if (errno == EINTR)
	  continue;
	else
	  return -1;
      }
    else
      {
	struct termios new;

	/* Get the current settings, and see if they're what we asked for.  */
	tcgetattr (fd, &new);
	/* We cannot use memcmp on the whole structure here because under
	 * aix386 the termios structure has some reserved field that may
	 * not be filled in.
	 */
	if (   new.c_iflag == settings->main.c_iflag
	    && new.c_oflag == settings->main.c_oflag
	    && new.c_cflag == settings->main.c_cflag
	    && new.c_lflag == settings->main.c_lflag
	    && memcmp(new.c_cc, settings->main.c_cc, NCCS) == 0)
	  break;
	else
	  continue;
      }

#else
#ifdef HAVE_TERMIO
  /* The SYSV-style interface?  */
  if (ioctl (fd, waitp ? TCSETAW : TCSETAF, &settings->main) < 0)
    return -1;

#else
#ifdef VMS
  /* Vehemently Monstrous System?  :-)  */
  if (! (SYS$QIOW (0, fd, IO$_SETMODE, &input_iosb, 0, 0,
		   &settings->main.class, 12, 0, 0, 0, 0)
	 & 1))
    return -1;

#else
  /* I give up - I hope you have the BSD ioctls.  */
  if (ioctl (fd, (waitp) ? TIOCSETP : TIOCSETN, &settings->main) < 0)
    return -1;

#endif
#endif
#endif

  /* Suivant - Do we have to get struct ltchars data?  */
#ifdef HAVE_LTCHARS
  if (ioctl (fd, TIOCSLTC, &settings->ltchars) < 0)
    return -1;
#endif

  /* How about a struct tchars and a wordful of lmode bits?  */
#ifdef HAVE_TCHARS
  if (ioctl (fd, TIOCSETC, &settings->tchars) < 0
      || ioctl (fd, TIOCLSET, &settings->lmode) < 0)
    return -1;
#endif
  
  /* We have survived the tempest.  */
  return 0;
}




static struct emacs_tty saved_tty_param;

/*  ==================================================================  */
/*			    Exported functions				*/


/*
 *  Do some initializations for the terminal and XO-output functions.
 *  Save the old terminal settings in *OLD_TTY_PARAM.
 */
static int raw_terminal = 0;

Export  Success
init_terminal (FILE	* infile,
	       FILE	* outfile)
{
#define FUNCTION "init_terminal()"
    struct emacs_tty	  tty_param;
    int			  infileno;
    static int		  firsttimearound = 1;
    
    if (raw_terminal)
	return OK;

    if (firsttimearound) {
	extend_printf ('S', &printf_String);
	extend_printf ('T', &printf_goto_column);

	firsttimearound = 0;
    }

    infileno = fileno(infile);
    /* Fetch tty settings */ /* Lets do like emacs does. */

    if (emacs_get_tty (infileno, &saved_tty_param) < 0)
    {
	/* This did not work. Lets ignore any such attempts in the future
	   but let the client run along as nothing has happened.
	 */
	raw_terminal = 1;
	return OK;
	/*NOTREACHED*/
    	perror("Oops!  You shouldn't see this.  Anyway, here's the error:\n"
	       "Fetching tty parameters: ");
	fatal2 (CLIENT_ERROR_TOO_boring, 
	       "Fetching tty parameters. Error: %d", errno);
    }

    /*  Turn off echo and on cbreak  */
    tty_param = saved_tty_param;
#if defined (HAVE_TERMIO) || defined (HAVE_TERMIOS)
    tty_param.main.c_oflag |= OPOST;	/* Enable output postprocessing */
#if 0 /* This is not emacs */
    tty_param.main.c_oflag &= ~ONLCR;	/* Disable map of NL to CR-NL on output */
#endif
#ifdef HAVE_TERMIO
    tty_param.main.c_oflag &= ~(NLDLY|CRDLY|TABDLY|BSDLY|VTDLY|FFDLY);
  				/* No output delays */
#endif
    tty_param.main.c_lflag &= ~ECHO;	/* Disable echo */
    tty_param.main.c_lflag |= ISIG;	/* Enable signals */
#ifdef HAVE_TERMIO
    tty_param.main.c_iflag &= ~IUCLC;	/* Disable map of upper case to lower on
				   input */
    tty_param.main.c_oflag &= ~OLCUC;	/* Disable map of lower case to upper on
				   output */
#endif

    tty_param.main.c_lflag &= ~ICANON;	/* Enable erase/kill and eof processing */
    original_eof_char = tty_param.main.c_cc[VEOF];
    tty_param.main.c_cc[VMIN] = 1;	/* One char is enough. */
#if 0
    tty_param.main.c_cc[VERASE] = 0377;	/* disable erase processing */
    tty_param.main.c_cc[VKILL] = 0377;	/* disable kill processing */
#endif
#ifdef HPUX
    tty_param.main.c_cflag = (tty_param.main.c_cflag & ~CBAUD) | B9600; /* baud rate sanity */
#endif /* HPUX */

#ifdef AIX
/* AIX enhanced edit loses NULs, so disable it */
#ifndef IBMR2AIX
    tty_param.main.c_line = 0;
    tty_param.main.c_iflag &= ~ASCEDIT;
#endif
    /* Also, PTY overloads NUL and BREAK.
       don't ignore break, but don't signal either, so it looks like NUL.  */
    tty_param.main.c_iflag &= ~IGNBRK;
    tty_param.main.c_iflag &= ~BRKINT;
    /* QUIT and INTR work better as signals, so disable character forms */
    tty_param.main.c_cc[VINTR] = 0377;
#ifdef SIGNALS_VIA_CHARACTERS
    /* the QUIT and INTR character are used in process_send_signal
       so set them here to something useful.  */
    if (tty_param.main.c_cc[VQUIT] == 0377)
	tty_param.main.c_cc[VQUIT] = '\\'&037;	/* Control-\ */
    if (tty_param.main.c_cc[VINTR] == 0377)
	tty_param.main.c_cc[VINTR] = 'C'&037;	/* Control-C */
#else /* no TIOCGPGRP or no TIOCGLTC or no TIOCGETC */
    /* QUIT and INTR work better as signals, so disable character forms */
    tty_param.main.c_cc[VQUIT] = 0377;
    tty_param.main.c_cc[VINTR] = 0377;
    tty_param.main.c_lflag &= ~ISIG;
#endif /* no TIOCGPGRP or no TIOCGLTC or no TIOCGETC */
    tty_param.main.c_cc[VEOL] = 0377;
    tty_param.main.c_cflag = (tty_param.main.c_cflag & ~CBAUD) | B9600; /* baud rate sanity */
#endif /* AIX */

#else /* not HAVE_TERMIO */

    tty_param.main.sg_flags &= ~(ECHO | ANYP | ALLDELAY | RAW | LCASE
				 | TANDEM);
    tty_param.main.sg_flags |= CBREAK;

#if 0 /* Dont change these */
    tty_param.main.sg_erase = 0377;
    tty_param.main.sg_kill = 0377; 
#endif
#endif /* not HAVE_TERMIO */

/*    tty_param.sg_flags = (tty_param.sg_flags | O_CBREAK) & (~ ECHO); */

    emacs_set_tty(infileno, &tty_param, 1);

    signal(SIGTSTP, &suspend_lyskom);
    catch_sigwinch(SIGWINCH /* This is a simulated call */);
 
    return  OK;		/* Well, in the first version... */
}


/*
 *  Restore the terminal settings as saved in saved_tty_param.
 */
Export  Success
restore_terminal (FILE		* infile,
		  FILE		* outfile)
{
    int infileno = fileno(infile);

    if (raw_terminal)
	return OK;

    if (emacs_set_tty(infileno,&saved_tty_param,1) < 0)
    {
	perror("Oops!  tcsetattr got an error. Here it is the error:\n"
	       "Setting tty parameters: ");
	fatal2 (CLIENT_ERROR_TOO_boring,
		"Setting tty parameters. Error: %d", errno);
    }	
    return  OK;
}


/*
 * Setting the page length explicitly.
 */
Success
set_window_height(int	newlen)
{
    if (newlen < 0)
    {
	return FAILURE;
    }
    window_height = newlen;
    return OK;
}


/* What to do when suspending */
String *theprompt;
void
suspend_lyskom(int sig)
{
    restore_terminal(stdin, stdout);
    kill (getpid(), SIGSTOP);
    init_terminal(stdin, stdout);
    if (theprompt) {
	xprintf("\n%S - ", *theprompt);
	fflush(stdout);
    }
}


/* Print a single character to a XO_stream. */
Export  int
xputc (char	      char_to_print,
       XO_stream    * outstream)
{
    return  do_xputc (char_to_print, (void *)outstream);
}   /* END: xputchar() */



Export  int
s_xfputs (String	  str,
	  XO_stream	* outstream)

{
    String_size		  pos;


    for ( pos = 0 ;  pos < s_strlen (str) ;  pos++ )
	do_xputc (str.string[pos], (void *)outstream);

    return  s_strlen (str);
}   /* END: s_xsputs() */



/*
 *  Print the argument list ARGS to OUTSTREAM according to the format string
 *  FORMAT.
 */
Export  int
vxfprintf (XO_stream		* outstream,
	   const unsigned char	* format,
	   va_list	  	args	    )

{
    return  do_printf (&do_xputc, format, args, (void *)outstream);
}   /* END: vxfprintf */


/*
 *  The fprintf style function.
 */
Export  int
xfprintf (XO_stream		* outstream,
	  const unsigned char	* format,
	  ...					)
{
    va_list	args;
    int		retval;

    va_start (args, format);
    retval = do_printf (&do_xputc, (const char *) format, args, outstream);
    va_end (args);
    return  retval;
}   /* END: xfprintf() */




Export  int
xprintf (const unsigned char	* format,
	 ...				 )

{
    va_list		  args;
    int			  retval;

    va_start (args, format);
    retval = do_printf (&do_xputc, (const char *) format, args, kom_outstream);
    va_end (args);
    return  retval;
}   /* END: xprintf() */



/*
 *  Flush all pending output to the XO_stream OUTSTREAM.
 */
Export  int
xflush (XO_stream	* outstream)
{
    String		  temp_str;

    switch (outstream->stream_type)
    {
    case  XO_terminal:
	fflush (stdout);
	break;

    case  XO_file:
	fflush (outstream->f_stream);
	break;

    case  XO_edited_text:
	temp_str.len = outstream->used_buf;
	temp_str.string = (unsigned char *) outstream->buffer;
	edit_add_string (outstream->text_no, temp_str);
	outstream->used_buf = 0;
	break;
    }

    return  0;
}



/*
 *  Move cursor to a specific column.  Moves to that column on the next
 *  line if cursor has passed that column already.  Wraps if specified
 *  column is outside screen.
 */
Export  int
xo_goto_column (int		  column,
		XO_stream	* outstream)

{
    return  printf_goto_column (NULL, &do_xputc, '\0', 0,
				column, 0, (void *)outstream);
}



/*  ==================================================================  */
/*			    Local functions				*/

/* This is copied from elisp-client and svenskmud */
static int
  xo_shrink (int ch)			/* shrink a char to 7 bits */
{
    ch = ((int) ch) & 0377;

    if (ch < 0200)
	return ch;

    return "\0\t\n\r  !c#$Y|$\"c+?!-R~C+23'u$-,10?????AAAA[][CE@EEIIIIDNOOOO\\*\\UUU^YTBaaaa{}{ce`eeiiiidnoooo|/|uuu~yty"[ch-0200];

}

/*
 *  Print one character to a XO_stream, as called from do_printf().
 */
static  int
do_xputc (int	  ch,
	  void	* stream)
{
    int		  to_next_tab;
    XO_stream	* outstream	= (XO_stream *)stream;

    if (ch < 0)				/* Garlic against signed chars */
      ch += 0400;
    
    switch (ch)
    {
    case  '\n':
	printed_lines++;
	current_column = 0;
	break;

    case  '\r':
	current_column = 0;
	break;

    case  '\b':
	current_column--;
	break;

    case  '\t':
	/* Assumes tab stops every 8th position */
	to_next_tab = 8 - current_column % 8;
	current_column += to_next_tab;
	break;

    case  '\a':
	break;

    default:
	current_column++;
	break;
    }  /* switch (ch) */


    switch (outstream->stream_type)
    {
    case  XO_file:
	putc (ch, outstream->f_stream);
	break;

    case  XO_edited_text:
	outstream->buffer[outstream->used_buf++] = ch;

	if (outstream->used_buf >= XO_EDIT_BUFSIZE)
	{
	    xflush (outstream);
	    outstream->used_buf -= XO_EDIT_BUFSIZE;
	}
	break;
	

    case  XO_terminal:
	switch(charset)
	{
	    case 0: putc(ch, stdout); break;
	    case 1: ch=iso_to_pc8(ch); putc(ch, stdout); break;
	    case 2:
	        if (ch > 127)
		    putc (xo_shrink (ch), stdout); /* Aronsson was here */
		else
		    putc (ch, stdout);
	        break;
	    case 3: ch=iso_to_mac(ch); putc(ch, stdout); break;
	    default: putc(ch, stdout); break;
	}
	while (current_column >= window_width)/* 'if' should suffice... */
	{
	    /* Hmm... I'll assume the window is at least 8 characters
	     * wide.  This is for tab characters....		*/
	    current_column -= window_width;
	    printed_lines++;
	}

	/* Check if we have printed a screenful */
	if (window_height && (printed_lines >= window_height - 1))
	  /* MAGIC NUMBER! */
	{
	    int		  c, i;
	    FILE	* instream;

	    /* This is the only sensible file I know of. */
	    instream = stdin;
	    /* Cannot call xputs here... */
	    /* xputs (gettext("(Tryck retur)")); */
	    c = strlen ((char *)gettext("(Tryck retur)"));
	    if (charset == 2)
	      for (i = 0; i < c; i++)
		putchar (xo_shrink ((unsigned char)
				    gettext("(Tryck retur)")[i]));
	    else
	        fputs ((char *)gettext("(Tryck retur)"), stdout);
	    fflush (stdout);
	    /* Assuming cbreak and no echo */
	    do
	    {
		c = getc (instream);
	    } while (    c != '\r'  &&  c != '\n'
		     &&  c != ' '   &&  c != 'q'  &&  c != 'Q');
	    putchar ('\r');
	    for (i = strlen ((char *)gettext("(Tryck retur)")) ;  i > 0 ;  i--)
		putchar (' ');
	    putchar ('\r');
	    printed_lines = 0;
	    current_column = 0;
	    if (c == 'q'  ||  c == 'Q')
		longjmp (toploop_jmpbuf, 17);
	}


	break;

    }  /* switch (kom_outstream->stream_type) */

    return  1;
}   /* END: do_xputc() */



/*
 *  Handle the 'S' format for do_printf(), i e an argument of type 'String'.
 */
static  int
printf_String  (va_list	   * arg_p,
		putc_fun_T   putc_func,
		char	     sign,
		int	     flags,
		int	     width,
		int	     precision,
		void	   * stream)

{
    String		   str;
    String_size		   i;


    str = va_arg (*arg_p, String);

    /* BUG!  Doesn't handle sign, flags, width och precision. */
    for ( i = 0 ;  i < s_strlen (str) ;  i++ )
	(*putc_func)(str.string[i], stream);

    return  s_strlen (str);
}   /* END: printf_String() */


#define BUF_SZ		128

static  int
printf_goto_column (va_list	 * arg_p,
		    putc_fun_T	   putc_func,
		    char	   sign,
		    int		   flags,
		    int		   column,
		    int		   precision,
		    void	 * stream)

{
    int		  no_of_tabs;
    int		  no_of_spaces;
    int		  no_of_printed_chars;

    if (current_column >= column)
	(*putc_func) ('\n', stream);

    no_of_tabs = column/8 - current_column/8;
    no_of_spaces = column - (current_column/8 + no_of_tabs)*8;
    no_of_printed_chars = no_of_tabs + no_of_spaces;

    while (no_of_tabs-- > 0)
	(*putc_func) ('\t', stream);
    while (no_of_spaces-- > 0)
	(*putc_func) (' ', stream);

    return  no_of_printed_chars;
}   /* END: printf_goto_column() */
