/*
  audiofile.c - part of ruby-audiofile

  Copyright (C) 2001 Rik Hemsley (rikkus) <rik@kde.org>

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to
  deal in the Software without restriction, including without limitation the
  rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
  sell copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
  AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
  ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
  WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#include <stdio.h>
#include <ctype.h>

#include <ruby.h>
#include <rubysig.h>

#include <ao/ao.h>

typedef struct rao_info
{
  ao_device * device;
} rao_info;

static VALUE cAO;

static int rao_ao_initialized = 0;

static void rao_ao_initialize(void)
{
  if (0 != rao_ao_initialized)
    return;

  ao_initialize();
  rao_ao_initialized = 1;
}

/*
  ao_option *
rao_hash_to_options(v)
  VALUE v;
{
  VALUE keys;
  VALUE key;
  int   i = 0;

  ao_option * options = 0;

  keys = rb_hash_keys(v);

  while (Qnil != (key = rb_ary_entry(keys, i++)))
  {
    (void) ao_append_option
      (
        &options,
        RSTRING(key)->ptr,
        RSTRING(rb_hash_ref(v, key))->ptr
      );
  }

  return options;
}
*/

  VALUE
rao_ao_driver_info_to_hash(ao_info * info)
{
  VALUE driver_info   = rb_hash_new();
  int   option        = 0;
  VALUE option_list;

  /**
   * Fill driver_info with the stuff from the C struct.
   */

  rb_hash_aset
    (
      driver_info,
      rb_str_new2("id"),
      INT2NUM(ao_driver_id(info->short_name))
    );

  rb_hash_aset
    (
      driver_info,
      rb_str_new2("name"),
      rb_str_new2(info->name)
    );

  rb_hash_aset
    (
      driver_info,
      rb_str_new2("type"),
      INT2NUM(info->type)
    );

  rb_hash_aset
    (
      driver_info,
      rb_str_new2("author"),
      rb_str_new2(info->author)
    );

  rb_hash_aset
    (
      driver_info,
      rb_str_new2("comment"),
      rb_str_new2(info->comment)
    );

  rb_hash_aset
    (
      driver_info,
      rb_str_new2("preferred_byte_format"),
      INT2NUM(info->preferred_byte_format)
    );

  rb_hash_aset
    (
      driver_info,
      rb_str_new2("priority"),
      INT2NUM(info->priority)
    );

  option_list = rb_ary_new();

  for (option = 0; option < info->option_count; option++)
  {
    rb_ary_push(option_list, rb_str_new2(info->options[option]));
  }

  rb_hash_aset
    (
      driver_info,
      rb_str_new2("options"),
      option_list
    );

  return driver_info;
}

  static VALUE
rao_s_new(argc, argv, klass)
  int     argc;
  VALUE * argv;
  VALUE   klass;
{
  VALUE obj = Data_Wrap_Struct(klass, 0, free, 0);
  rb_obj_call_init(obj, argc, argv);
  return obj;
}

  static VALUE
rao_initialize(argc, argv, obj)
  int     argc;
  VALUE * argv;
  VALUE   obj;
{
  rao_info * rao = 0;

  rao_ao_initialize();

  rao = ALLOC(rao_info);

  rao->device = 0;

  DATA_PTR(obj) = rao;

  return obj;
}

  static VALUE
rao_open_live
(
  klass,
  driver_id,
  bits,
  bitrate,
  channels,
  byte_format
)
  VALUE klass;
  VALUE driver_id;
  VALUE bits;
  VALUE bitrate;
  VALUE channels;
  VALUE byte_format;
{
  rao_info        * rao       = 0;
  ao_option       * opt       = 0;

  ao_sample_format  format;

  Data_Get_Struct(klass, rao_info, rao);

  format.bits         = NUM2INT(bits);
  format.rate         = NUM2INT(bitrate);
  format.channels     = NUM2INT(channels);
  format.byte_format  = NUM2INT(byte_format);

  if (0 != rao->device)
    ao_close(rao->device);

  rao->device = ao_open_live(NUM2INT(driver_id), &format, 0);

  return (0 == rao->device) ? Qfalse : Qtrue;
}

  static VALUE
rao_open_file
(
  klass,
  driver_id,
  filename,
  overwrite,
  bits,
  bitrate,
  channels,
  byte_format
)
  VALUE klass;
  VALUE driver_id;
  VALUE filename;
  VALUE overwrite;
  VALUE bits;
  VALUE bitrate;
  VALUE channels;
  VALUE byte_format;
{
  rao_info        * rao       = 0;
  ao_option       * opt       = 0;

  ao_sample_format  format;

  Data_Get_Struct(klass, rao_info, rao);

  format.bits         = NUM2INT(bits);
  format.rate         = NUM2INT(bitrate);
  format.channels     = NUM2INT(channels);
  format.byte_format  = NUM2INT(byte_format);

  if (0 != rao->device)
    ao_close(rao->device);

  rao->device =
    ao_open_file
    (
      NUM2INT(driver_id),
      RSTRING(filename)->ptr,
      (Qtrue == overwrite) ? 1 : 0,
      &format,
      opt
    );

  return (0 == rao->device) ? Qfalse : Qtrue;
}

  static VALUE
rao_play(klass, data)
  VALUE klass;
  VALUE data;
{
  rao_info        * rao       = 0;
  Data_Get_Struct(klass, rao_info, rao);

  if (0 == rao->device)
    return Qnil;

  return INT2NUM(ao_play(rao->device, RSTRING(data)->ptr, RSTRING(data)->len));
}

  static VALUE
rao_close(klass)
  VALUE klass;
{
  rao_info        * rao       = 0;
  Data_Get_Struct(klass, rao_info, rao);

  if (0 != rao->device)
    ao_close(rao->device);

  return Qnil;
}

  static VALUE
rao_s_default_driver_name(klass)
  VALUE klass;
{
  ao_info * info = 0;

  rao_ao_initialize();

  info = ao_driver_info(ao_default_driver_id());

  if (0 == info)
    return Qnil;

  return rb_str_new2(info->short_name);
}

  static VALUE
rao_s_default_driver_id(klass)
  VALUE klass;
{
  rao_info * rao = 0;
  rao_ao_initialize();
  return INT2NUM(ao_default_driver_id());
}

  static VALUE
rao_s_driver_info_list(klass)
  VALUE klass;
{
  int               i             = 0;
  ao_info         * info          = 0;
  VALUE             driver_hash;

  rao_ao_initialize();

  driver_hash = rb_hash_new();

  while (0 != (info = ao_driver_info(i++)))
  {
    rb_hash_aset
      (
        driver_hash,
        rb_str_new2(info->short_name),
        rao_ao_driver_info_to_hash(info)
      );
  }

  return driver_hash;
}

  static VALUE
rao_s_driver_info(klass, id)
  VALUE klass;
  VALUE id;
{
  ao_info         * info          = 0;

  rao_ao_initialize();

  info = ao_driver_info(NUM2INT(id));

  if (0 == info)
    return Qnil;

  else
    return rao_ao_driver_info_to_hash(info);
}

  static VALUE
rao_s_driver_id(klass, name)
  VALUE klass;
  VALUE name;
{
  rao_info        * rao       = 0;
  rao_ao_initialize();
  return INT2NUM(ao_driver_id(RSTRING(name)->ptr));
}

  void
Init_audiooutput()
{
  cAO = rb_define_class("AudioOutput", rb_cObject);

  rb_define_singleton_method(cAO, "new", rao_s_new, -1);

  rb_define_singleton_method
    (cAO, "default_driver_name",  rao_s_default_driver_name,  0);

  rb_define_singleton_method
    (cAO, "default_driver_id",    rao_s_default_driver_id,    0);

  rb_define_singleton_method
    (cAO, "driver_info",          rao_s_driver_info,          1);

  rb_define_singleton_method
    (cAO, "driver_info_list",     rao_s_driver_info_list,     0);

  rb_define_singleton_method
    (cAO, "driver_id",            rao_s_driver_id,            1);

  rb_define_method(cAO, "initialize",           rao_initialize,          -1);

  rb_define_method(cAO, "open_live",            rao_open_live,            5);
  rb_define_method(cAO, "open_file",            rao_open_file,            7);

  rb_define_method(cAO, "play",                 rao_play,                 1);
  rb_define_method(cAO, "close",                rao_close,                0);

#define rao_define_const(c) rb_define_const(cAO,#c,UINT2NUM(c))

  rao_define_const(AO_TYPE_LIVE);
  rao_define_const(AO_TYPE_FILE);

  rao_define_const(AO_FMT_LITTLE);
  rao_define_const(AO_FMT_BIG);
  rao_define_const(AO_FMT_NATIVE);
}

#if 0
  ao_sample_format
rao_hash_to_format(v)
  VALUE v;
{
  ao_sample_format format;

  format.bits =
    NUM2INT(rb_hash_aref(v, rb_str_new("bits", 4)));

  format.rate =
    NUM2INT(rb_hash_aref(v, rb_str_new("rate", 4)));

  format.channels =
    NUM2INT(rb_hash_aref(v, rb_str_new("channels", 8)));

  format.byte_format =
    NUM2INT(rb_hash_aref(v, rb_str_new("byte_format", 11)));

  return format;
}

static void rao_ao_shutdown(void)
{
  if (0 == rao_ao_initialized)
    return;

  ao_shutdown();
  rao_ao_initialized = 0;
}
#endif
