/*
 *  output-tarball.c
 *  mod_musicindex
 *
 *  $Id: output-tarball.c 662 2006-07-09 21:05:07Z varenet $
 *
 *  Created by Thibaut VARENE on Mon Jun 26 2006.
 *  Copyright (c) 2006 Thibaut VARENE
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License version 2.1,
 *  as published by the Free Software Foundation.
 *
 */

/**
 * @file
 * Tarballs generation
 *
 * @author Thibaut Varene
 * @version $Revision: 662 $
 * @date 2006
 *
 * This file generates and send on the fly a tarball of the list given as
 * parameter, using libarchive.
 *
 * @todo use apr_file I/O
 */

#include "mod_musicindex.h"

#include <http_core.h>
#include <archive.h>	/* libarchive */
#include <archive_entry.h>
#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>	/* stat */
#endif
#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>	/* stat */
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>	/* stat */
#endif
#include <stdio.h>	/* fopen... */

#define RWBUFSIZE	8192	/**< Size of our r/w buffer */

/** Holds our client data for libarchive */
struct cdata {
	request_rec *r;	/**< Apache request_rec */
};

static int
wrapopen(struct archive *a, void *client_data)
{
	return (ARCHIVE_OK);
}

static ssize_t
wrapwrite(struct archive *a, void *client_data, void *buff, size_t n)
{
	struct cdata *mydata = client_data;

	return (ap_rwrite(buff, n, mydata->r));
}

static int
wrapclose(struct archive *a, void *client_data)
{
	return (0);
}
    
/**
 * Sends tarball.
 *
 * @param p A list of music entries.
 * @param conf MusicIndex configuration paramaters struct.
 * @param r Apache request_rec struct to handle connection details.
 *
 * It is a known "feature" that when tar'ing a custom playlist, the generated
 * tarball will contain the full path as published on the webserver, including
 * '/'. We could get rid of the leading '/' if needed, but there's one good reason
 * not to put all the files at the root of one big tarball: in case there are name
 * doublons.
 */
void send_tarball(request_rec *r, const mu_ent *const p, const mu_config *const conf)
{
	const mu_ent *q;
	struct cdata *mydata;
	struct archive *a;
	struct archive_entry *entry;
	struct stat st;
	static char buff[RWBUFSIZE];
	int len;
	FILE *file;

	if (!p)
		return;
		
	if (!(mydata = malloc(sizeof(struct cdata))))
		return;

	if (!(a = archive_write_new())) {
		free(mydata);
		return;
	}
	
	mydata->r = r;
	archive_write_set_compression_none(a);
	archive_write_set_format_ustar(a);
	archive_write_open(a, mydata, wrapopen, wrapwrite, wrapclose);

	for (q = p; q != NULL; q = q->next) {
		if (q->filetype < 0)
			continue;

		if (!(entry = archive_entry_new()))
			goto out;
			
		stat(q->filename, &st);
		archive_entry_copy_stat(entry, &st);
		archive_entry_set_pathname(entry, q->file);
		archive_write_header(a, entry);

		file = fopen(q->filename, "r");
		len = fread(buff, 1, sizeof(buff), file);

		while (len > 0) {
			archive_write_data(a, buff, len);
			len = fread(buff, 1, sizeof(buff), file);
		}
		archive_entry_free(entry);
		fclose(file);
	}
out:
	archive_write_finish(a);
}
