/*
 *  playlist-ogg.c
 *  mod_musicindex
 *
 *  $Id: playlist-vorbis.c 634 2006-06-26 20:23:42Z varenet $
 *
 *  Created by Regis BOUDIN on Thu Jan 22 2004.
 *  Copyright (c) 2003-2004 Regis BOUDIN
 *  Copyright (c) 2003-2005 Thibaut VARENE
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License version 2.1,
 *  as published by the Free Software Foundation.
 *
 */

/**
 * @file
 * Ogg files management system.
 *
 * @author Regis Boudin
 * @author Thibaut Varene
 * @version $Revision: 634 $
 * @date 2003-2005
 *
 * This file contains everything needed to produce music entries from
 * Ogg Vorbis files.
 */

#include "playlist.h"
#include "playlist-vorbis.h"

#include <vorbis/codec.h>
#include <vorbis/vorbisfile.h>

#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

/**
 * Fills in the information fields about OGG data.
 *
 * This function reads the ogg tags (using libvorbisfile) from the ogg file
 * and fills in the struct mu_ent fields accordingly. If the file is not an
 * ogg one, the function returns p = head.
 *
 * @todo find the real name of "tpos" field for Vorbis files.
 *
 * @param pool Pool
 * @param head Head
 * @param in ogg file to parse (closed on normal exit)
 * @param conf MusicIndex configuration paramaters struct
 * @param names Names
 * @param r Apache request_rec struct to handle log writings (debugging)
 *
 * @return When possible, struct mu_ent correctly set up, file stream closed.
 */
mu_ent *make_ogg_entry(request_rec *r, apr_pool_t *pool, FILE *const in,
	mu_ent *const head, const mu_config *const conf, mu_ent_names *const names)
{
	mu_ent			*p = NULL;
	OggVorbis_File		vf;
	const char 		*t;
	struct stat		filestat;

	if (ov_test(in, &vf, NULL, 0) != 0)
		return head;
	
	p = NEW_ENT(pool);
	if (p == NULL) {
		ov_clear(&vf);
		return head;
	}
	
	p->filetype = FT_OGG;
	p->flags &= ~EF_VBR;

	fstat(fileno(in), &filestat);
	p->size = filestat.st_size;
	p->mtime = filestat.st_mtime;

	if (ov_test_open(&vf) == 0) {

		vorbis_comment *comment = ov_comment(&vf, -1);
		if (comment != NULL) {
			if ((t = vorbis_comment_query(comment, "album", 0)))
				p->album = ap_pstrdup(pool, t);
			if ((t = vorbis_comment_query(comment, "artist", 0)))
				p->artist = ap_pstrdup(pool, t);
			if ((t = vorbis_comment_query(comment, "title", 0)))
				p->title = ap_pstrdup(pool, t);
			if ((t = vorbis_comment_query(comment, "tracknumber", 0)))
				p->track = atoi(t);
			if ((t = vorbis_comment_query(comment, "date", 0)))
				p->date = atoi(t);
			if ((t = vorbis_comment_query(comment, "tpos", 0)))	/* XXX does it exist? */
				p->posn = atoi(t);
			if ((t = vorbis_comment_query(comment, "genre", 0)))
				p->genre = ap_pstrdup(pool, t);
		}

		if (conf->options & MI_QUICKPL) {
			p->bitrate = p->length = p->freq = 0;
		}
		else {
			vorbis_info *info = ov_info(&vf, -1);
			p->freq = info->rate;

#ifndef NO_BR_NOMINAL
			if (info->bitrate_nominal)
				p->bitrate = info->bitrate_nominal;	/* when available this is what players show */
			else
#endif
				p->bitrate = (long)ov_bitrate(&vf, -1);
			if ((info->bitrate_upper > 0) && (info->bitrate_upper == info->bitrate_lower));
			else	/* we have a VBR file */
				p->flags |= EF_VBR;
			p->length = (long)ov_time_total(&vf, -1);
		}

	}
	ov_clear(&vf);
	return p;
}
