/**
 *
 * mod_ifier
 *  Simple filtering module for Apache 2.x which allows you to deny
 * incoming connections based upon semi-arbitary criterea.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 * Furthermore, Steve Kemp gives permission to link mod_ifier
 * with OpenSSL and Apache Licensed works as well as distributing the 
 * linked executables. 
 *
 *  Steve Kemp
 *  ---
 *  http://www.steve.org.uk/
 *
 */


/**
 ** This file contains the routines which are related to CGI parameter
 ** parsing and fetching.
 **
 **/


/* Size of buffer for reading POST requests. */
#ifndef MAX_STRING
# define MAX_STRING 4096
#endif 




/**
 * Parse the CGI parameters from the incoming request into
 * a table of key + value pairs.
 *
 * This table of Key + Value pairs is used for all the CGI parameter
 * matching.
 *
 */
static apr_array_header_t * mod_ifier_parse_cgi_parameters(request_rec *req)
{
    /**
     * Make a new array to host our CGI parameter values.
     */
    apr_array_header_t *results = apr_pcalloc(req->pool,
                                              sizeof(mod_ifier_cgi_arguments));

    /**
     * The request details
     */
    const char *args;   /* Actual query string, either GET or POST */
    int arg_len;        /* Length */
    const char *start;  /* Pointer to start of string */

    /**
     * If we're a POST then read the data.
     */
    if ( req->method_number == M_POST )
    {
        char buffer[MAX_STRING];
        const char *s = NULL;
        short i = 0;

        ap_setup_client_block(req, REQUEST_CHUNKED_DECHUNK);
        
        do {
            i = ap_get_client_block(req, buffer, MAX_STRING-1);
            buffer[i] = '\0';
            if (s == NULL)
                s = ap_pstrdup(req->pool, buffer);
            else
                s = ap_pstrcat(req->pool, s, buffer, NULL);
        } while (i == (MAX_STRING-1));
        
        args = s;

        debug_printf("POST payload: '%s'\n", args );
    }
    else
    {
        /**
         * GET request?
         */
        args = apr_pstrdup(req->pool, req->args);
    }

    /* Setup our loop */
    arg_len = strlen( args );
    start   = args;

    /* Setup the array we're going to return our results in. */
    results = apr_array_make(req->pool, 1, sizeof(mod_ifier_cgi_arguments));

    /**
     * Find the end if the first parameter.
     */
    while( start <= (args + arg_len ) )
    {
        const char *end = start;

        char *marker;

        /* Move to the end of the first parameter. */
        while( ( *end != '\0' ) && 
               ( *end != ';' ) && 
               ( *end != '&' ) )
            end += 1;
        
        /*
         * We now have a string of the form 'key=value'.  Split it into
         * the two halves.
         */

        marker = strchr( start, '=' );
        if ( marker )
        {
            /* New object to host the name + value we found. */
            mod_ifier_cgi_arguments *arg;

            /* Get size of each one */
            char *key = NULL;
            char *val = NULL;

            int key_len = 0, val_len = 0;
            key_len = marker-start;
            val_len = end - marker - 1;

            /* Create a new CGI parameter holder */
            arg = (mod_ifier_cgi_arguments *)apr_array_push(results);

            /* Allocate memory. */
            key = apr_pcalloc(req->pool, key_len + 1 );
            val = apr_pcalloc(req->pool, val_len + 1);

            /* Copy over the values. */
            strncpy( key, start,    key_len );
            strncpy( val, marker+1, val_len );

            /* Now copy them into the structure */
            arg->name   = apr_pstrdup(req->pool, key);
            arg->value  = apr_pstrdup(req->pool, val);

            debug_printf("CGI added : %s=>%s\n", arg->name, arg->value );

            /* Unescape arguments. */
            ap_unescape_url(arg->value);
        }

        /**
         * Move onto the next pair.
         */
        start = end + 1;
    }


    return( results );

}



/**
 * Test to see if *any* of the CGI parameters which were submitted match
 * against the regular expression.
 * 
 * @return 1 If a submitted CGI parameter value matches the given pattern.
 * @return 0 if no matches were made.
 */
static int does_any_cgi_param_value_match( apr_array_header_t *results, request_rec *req, ap_regex_t *pattern )
{
    /**
     * Iterate over our records and find the value of any matching
     * parameters.
     */
    if ( ( results != NULL ) && ( results->nelts ) )
    {
        int i;
        mod_ifier_cgi_arguments *params;
        params = (mod_ifier_cgi_arguments *) results->elts;

        for( i = 0; i < results->nelts; i++ )
        {
            /* Something we're matching against. */
            mod_ifier_cgi_arguments *r  = &params[i];

            /* Did the value of this parameter, regardless of what it was,
             * match the specified regexp?  If so return. */
            if (!ap_regexec(pattern, r->value, 0, NULL, 0) )
                return 1;
        }
    }
    else
    {
        debug_printf("There were no submitted CGI parameters\n" );
    }

    return 0;
}


/**
 * Look for the value which was submitted for the specified CGI parameter.
 *
 * @return The value found, or NULL if the parameter wasn't present.
 */
const char * get_cgi_param_value( apr_array_header_t *results, request_rec *req, const char *wanted )
{
    debug_printf( "Looking for the value of the parameter : '%s'\n", wanted );

    /**
     * Iterate over our records and find the value of any matching
     * parameters.
     */
    if ( ( results != NULL ) && ( results->nelts ) )
    {
        int i;
        mod_ifier_cgi_arguments *params;
        params = (mod_ifier_cgi_arguments *) results->elts;

        for( i = 0; i < results->nelts; i++ )
        {
            /* Something we're matching against. */
            mod_ifier_cgi_arguments *r  = &params[i];

            /* If we found the CGI parameter with the correct
             * name return its value. */               
            if ( strcmp( r->name, wanted ) == 0 )
            {
                debug_printf("Found value '%s' for '%s'\n", r->value, wanted );
                return  r->value;
            }

        }
    }
    else
    {
        debug_printf("There were no submitted CGI parameters\n" );
    }

    debug_printf("Value not found for parameter.\n" );

    return( NULL );
}



/*
 * Local variables:
 * tab-width: 4
 * c-basic-offset: 4
 * End:
 * vim600: sw=4 ts=4 fdm=marker
 * vim<600: sw=4 ts=4
 */
