/**
 *
 * mod_ifier
 *  Simple filtering module for Apache 2.x which allows you to deny
 * incoming connections based upon semi-arbitary criterea.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 * Furthermore, Steve Kemp gives permission to link mod_ifier
 * with OpenSSL and Apache Licensed works as well as distributing the 
 * linked executables. 
 *
 *  Steve Kemp
 *  ---
 *  http://www.steve.org.uk/
 *
 */


/**
 ** This file contains the routines which match incoming requests
 ** in various ways.
 **
 ** These are called as part of the iteration over the mod_ifier_rule
 ** structures we have.
 **
 **/

/* Our CGI parsing code. */
#include "cgi.c"


/**
 * Test the request we've received to see if the header in the rule
 * we've been given is present.
 *
 * @return 1 if the request should be blocked.
 * @return 0 if the request should proceed.
 */
static int match_header_rule( request_rec *req, mod_ifier_rule *rule )
{
    /* Name and value we should match against */
    const char *name        = rule->name;
    const char *val_text    = rule->val_text;
    const ap_regex_t *value = rule->value;

    /* Get the value of the specified header from the request */
    const char *sent = apr_table_get(req->headers_in, name );

    /* The header we're testing wasn't sent */
    if ( sent == NULL )
        return( 0 );

    /* Does the value sent match our regexp? */
    if (!ap_regexec(value, sent, 0, NULL, 0) )
    {
        modifier_write_log( req, 
                            "filtered - Header '%s' value '%s' matches regexp '%s'\n",
                            name, sent, val_text
                            );

        return( 1 );
    }

    /* Header didn't match, so we should allow the request. */
    return( 0 );
}



/**
 * Test the request we've received to see if the path requested in the rule
 * we've been given is present.
 *
 * @return 1 if the request should be blocked.
 * @return 0 if the request should proceed.
 */
static int match_path( request_rec *req, mod_ifier_rule *rule )
{
    /* Path we should match against */
    const char *val_text = rule->val_text;
    ap_regex_t *pattern  = rule->value;

    /* If the request matches then it should be denied. */
    if (!ap_regexec(pattern, req->uri, 0, NULL, 0) )
    {
        modifier_write_log( req, 
                            "filtered - requested path '%s' matches regexp '%s'\n",
                            req->uri, val_text
                            );


        return 1;
    }

    /* Path didn't match, so we should allow the request. */
    return( 0 );
}



/**
 * Test the request we've received to see if the method requested in the rule
 * was used in the request
 *
 * @return 1 if the request should be blocked.
 * @return 0 if the request should proceed.
 */
static int match_method( request_rec *req, mod_ifier_rule *rule )
{
    /* method we should match against */
    const char *val_text = rule->val_text;
    ap_regex_t *pattern  = rule->value;

    /* If the request matches then it should be denied. */
    if (!ap_regexec(pattern, req->method, 0, NULL, 0) )
    {
        modifier_write_log( req, 
                            "filtered - requested method '%s' matches regexp '%s'\n",
                            req->method, val_text
                            );


        return 1;
    }

    /* Path didn't match, so we should allow the request. */
    return( 0 );
}



/**
 * Test the incoming request against the CGI parameters we received.
 *
 * @return 1 if the request should be denied.
 * @return 0 if the request should be allowed to proceed.
 */
static int match_cgi( request_rec *req, mod_ifier_rule *rule, apr_array_header_t *cgi )
{
    /* Get the CGI name / parameter we're to match */
    const char *param     = rule->name;
    const char *value_txt = rule->val_text;
    ap_regex_t *value     = rule->value;

    if ( param == NULL )
    {
        debug_printf("Just looking for value match\n" );

        /**
         * Just testing on the contents of each value.
         */
        if ( does_any_cgi_param_value_match( cgi, req, value ) )
        {
            modifier_write_log( req, 
                                "filtered - cgi parameter value matches regexp '%s'\n",
                                value_txt
                            );


            debug_printf("Value matched - name ignored\n" );
            return( 1 );
        }
    }
    else
    {
        /**
         * See if a parameter is present, and if so get its value
         */
        const char *arg = get_cgi_param_value( cgi, req, param );
        if ( arg )
        {
            /*
             * Parameter is present.
             */
            
            /*
             * See if we're matching on the presence of the value.
             */
            if ( value == NULL )
            {
                modifier_write_log( req, 
                                    "filtered - cgi parameter '%s' present - value ignored.\n",
                                    
                                    param );
                return( 1 );
            }
            else
            {
                /* The value must match the specified rexexp. */
                if (!ap_regexec(value, arg, 0, NULL, 0) )
                {
                    modifier_write_log( req, 
                                        "filtered - cgi parameter '%s' value of '%s' matches regexp '%s'\n", 
                                        param, value, value_txt );
                    return( 1 );
                }
            }
        }
    }

    /* Request should be allowed to proceed */
    return( 0 );
}


/*
 * Local variables:
 * tab-width: 4
 * c-basic-offset: 4
 * End:
 * vim600: sw=4 ts=4 fdm=marker
 * vim<600: sw=4 ts=4
 */
