#ifndef BUFFY_CONFIG_CONFIG_H
#define BUFFY_CONFIG_CONFIG_H

#include <buffy/config/Storage.h>
#include <buffy/MailFolder.h>

#include <map>

namespace xmlpp {
class Element;
class Document;
class DomParser;
}

namespace buffy {
namespace config {

class Config;

/**
 * Access a subhierarchy of values.
 */
class Node
{
protected:
	Config* m_config;
	std::string m_path;

public:
	Node(Config& config, const std::string& path)
		: m_config(&config), m_path(path) {}

	// Return true if this item is present in the configuration file
	bool isSet(const std::string& key) const;

	// Delete the given key (and all subkeys) from the configuration.  After
	// this call, querying the given key will return the default value.
	void unset(const std::string& key);

	// Get the string value of this item
	std::string get(const std::string& key) const;

	// Get the value of this item, converted to bool
	bool getBool(const std::string& key) const;

	// Get the value of this item, converted to an int
	int getInt(const std::string& key) const;

#if 0
	// Return the item name (or name[attribute]) of the child item which
	// contains a child named 'selected' which has value 'true'
	// If none are found, returns the empty string.
	std::string selected() const;
#endif

	void addDefault(const std::string& key, const std::string& val);

	void set(const std::string& key, const std::string& val);
	void setBool(const std::string& key, bool val);
	void setInt(const std::string& key, int val);

	friend class Config;
};

class ViewNode : protected Node
{
public:
	ViewNode(Config& config, const std::string& path)
		: Node(config, path) {}

	bool empty() const { return getBool("empty"); }
	bool read() const { return getBool("read"); }
	bool important() const { return getBool("important"); }

	void setEmpty(bool val) { setBool("empty", val); }
	void setRead(bool val) { setBool("read", val); }
	void setImportant(bool val) { setBool("important", val); }
};

class GeneralNode : protected Node
{
public:
	GeneralNode(Config& config, const std::string& path)
		: Node(config, path) {}

	int interval() const { return getInt("interval"); }

	void setInterval(int val) { setInt("interval", val); }
};

class FolderNode : protected Node
{
public:
	FolderNode(Config& config, const std::string& path)
		: Node(config, path) {}

	bool forceview() const { return getBool("forceview"); }
	bool forcehide() const { return getBool("forcehide"); }

	void setForceView(bool val)
	{
		if (val)
		{
			setBool("forceview", true);
			unset("forcehide");
		} else
			unset("forceview");
	}
	void setForceHide(bool val)
	{
		if (val)
		{
			setBool("forcehide", true);
			unset("forceview");
		} else
			unset("forcehide");
	}
};


class LocationsNode : protected Node
{
public:
	LocationsNode(Config& config, const std::string& path)
		: Node(config, path) {}

	std::vector<std::string> get() const;
	void set(const std::vector<std::string>& vals);
};

class MailProgramNode : protected Node
{
public:
	MailProgramNode(Config& config, const std::string& path)
		: Node(config, path) {}

	std::string name() const;
	std::string command() const { return get("command"); }
	bool selected() const;

	void setName(const std::string& val);
	void setCommand(const std::string& val) { set("command", val); }
	void setSelected();

	void run(const MailFolder& folder);
};

struct MailProgramInfo
{
	std::string name;
	std::string command;
	bool selected;
	MailProgramInfo() : selected(false) {}
	MailProgramInfo(const std::string& name, const std::string& command, bool selected = false)
		: name(name), command(command), selected(selected) {}
	bool operator==(const MailProgramInfo& info) const
	{
		return name == info.name && command == info.command && selected == info.selected;
	}
};

class MailProgramsNode : protected Node
{
protected:
	void convertOld(xmlpp::Element* el) const;

public:
	MailProgramsNode(Config& config, const std::string& path)
		: Node(config, path) {}

	/// Get a vector with accessors to all defined mail programs
	std::vector<MailProgramNode> get() const;
	std::vector<MailProgramInfo> getInfo() const;
	void set(const std::vector<MailProgramInfo>& vals);

	/// Get the name of the mail program that is currently selected
	MailProgramNode selected() const;

#if 0
	/// Select the given mail program
	void select(const std::string& name);
#endif

#if 0
	/// Remove all mail programs from the configuration
	void clear();

	/// Replace the current list of mail programs with the given one
	void add(const std::string& name, const std::string& command, bool selected = false);
#endif
};

class Config : protected Storage
{
protected:
	// State directory
	std::string rcfile;

	void init();

public:
	Config() : Storage() { init(); load(rcfile); }
	Config(const std::string& fname) : Storage() { init(); load(fname); }
	
	void load(const std::string& file);
	void save();
	void save(const std::string& file) { Storage::save(file); }
	void dump(std::ostream& out) { Storage::dump(out); }

	void addDefault(const std::string& key, const std::string& val) { Storage::addDefault(key, val); }
	void addDefault(const std::string& key, const std::vector<std::string>& val) { Storage::addDefault(key, val); }
	bool isSet(const std::string& key) const { return Storage::isSet(key); }

	//
	// Accessors for specific nodes
	//

	ViewNode view() { return ViewNode(*this, "general/view"); }

	GeneralNode general() { return GeneralNode(*this, "general"); }

	Node application(const std::string& name);

	MailProgramsNode mailPrograms();
	MailProgramNode mailProgram(const std::string& name);

	FolderNode folder(const MailFolder& folder) { return this->folder(folder.path()); }
	FolderNode folder(const std::string& folder);

	LocationsNode locations() { return LocationsNode(*this, "general/locations"); }

	friend class Node;
	friend class LocationsNode;
	friend class MailProgramNode;
	friend class MailProgramsNode;
};

}
}

// vim:set ts=4 sw=4:
#endif
