/*
* libcafix - communicates with CASIO graphing calculators.
* Copyright (C) 2001  Gran Weinholt <weinholt@linux.nu>
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*
* $Id: cafix.h,v 1.3 2001/09/07 17:57:27 weinholt Exp $
*/
#ifndef _CAFIX_H
#define _CAFIX_H

enum formats {
    FMT_UNKNOWN,
    FMT_BACKUP,		/* Memory backup */
    FMT_BWDUMP,		/* B/W screendump */
    FMT_CDUMP,		/* Color screendump */
    FMT_FUNCTION,	/* Function */
    FMT_PICTURE,	/* Picture */
    FMT_PROGRAM,	/* Program file */
    FMT_REQUEST,	/* Request for data */
    FMT_VARIABLE,	/* Variable */
    FMT_END,		/* End header */
    FMT_EDITOR		/* Some magic 9800-thingie */
};

/* datatype in the cafixdata structure will always be one of these. */
enum datatypes {
    DATA_UNKNOWN,
    DATA_BU,		/* Backup from (9xxx-calculators, I think) */
    DATA_FT,
    DATA_GF,		/* Graph function (Yx=) */
    DATA_GM,		/* G-Mem? */
    DATA_LT,		/* List */
    DATA_MT,		/* Matrix */
    DATA_PC,		/* Picture (9xxx-calculators, I think) */
    DATA_PG,		/* Program (9xxx-calculators, I think) */
    DATA_RF,
    DATA_RR,
    DATA_SE,		/* Equation? */
    DATA_TR,
    DATA_VM,		/* Variable (A-Z, etc.) */
    DATA_WD,		/* Window data */
    
    DATA_9800_TYPEA02,	/* FIXME: from the 9800, a backup */
    DATA_9800_PZ	/* FIXME: from the 9800, a PZ-program */
};

/* status in the cafixdata structure will always be one of these. */
enum status_types {
    STAT_NULL,			/* Empty status. */
    STAT_SENDING,
    STAT_SENDING_HDR,
    STAT_SENDING_DATA,
    STAT_RECEIVING,
    STAT_RECEIVING_HDR,
    STAT_RECEIVING_DATA,
    STAT_ERROR_COMM,		/* Error in communications. */
    STAT_ERROR_NOT_READY,	/* Calc. not in right mode? */
    STAT_ERROR_HEADER,		/* There's an error in the header we sent. */
    STAT_ERROR_MEMORY,		/* Calc. is out of memory. */
    STAT_ERROR_CRC,		/* Bad CRC. */
    STAT_ERROR_UNKNOWN,		/* Unknown error. */
    STAT_NOT_IMPLEMENTED,	/* Function not yet implemented. */
    STAT_OVERWRITE,		/* User should be asked if to overwrite. */
    STAT_USER_ABORT,		/* User aborted. */
    STAT_OK			/* Everything ended OK. */
};

/* This structure contains everything about a transfer that has or
   will be made. */
typedef struct {
    int format;		/* One of formats. */
    char *name;		/* Name of the program, variable etc. */
    unsigned int length;/* Size of data. */
    char *data;		/* Pointer to data. */

    char *password;	/* Password, not implemented */
    int interactive;	/* See README.Interactive. */
    int datatype;	/* One of datatypes. */
    char crc;		/* This is set by cafix_send. */
    int transmitted;	/* Number of bytes of data transmitted. */
    int status;		/* One of status_types. */
    int unstackable;	/* Can more data come after this? */
    int hlength;	/* Length of header, filled in by library. */
    char rawheader[50];	/* This is the raw header (to be) transmitted. */
    void *next;		/* For nested lists maybe.. */
    void *future;	/* For future private variables... */
} cafixdata;

/* Open a serial device. Ex: cafix_open("/dev/ttyS1");
   Returns file descriptor or -1 if open failed. timeout is
   0 (no time-out) or time-out specified in 100ms.
   Use these instead of open() and close(), please. */
int cafix_open(char *devfile, int timeout);
void cafix_close(int fd);

/* Try to handshake sending. This uses data->interactive to determine
   how the handshake will be made. Returns 0 on success and -1 on failure. */
int cafix_handshake_send(int fd, cafixdata *data);

/* Transmit the data contained in *data. This function will call
   status_callback so, for example, a progress bar can be showed.
   Returns 0 on success and -1 on failure. */
int cafix_send(int fd, cafixdata *data, int (*status_callback) (cafixdata *data));

/* Wait for handshake. data->interactive will be set by this function.
   Returns 0 if handshake was made and -1 if there was no handshake. */
int cafix_handshake_receive(int fd, cafixdata *data);

/* Receives data and put it in *data. This function allocates memory
   for data->data and data->name. This function will call status_callback
   so, for example, a progress bar can be showed. Returns 0 on success
   and -1 on failure. */
int cafix_receive(int fd, cafixdata *data, int (*status_callback) (cafixdata *data));

/* This will allocate memory for data->data and put the value(s) pointed
   to by *insert_data in the data->data structure. It will depend on data->type
   and data->subtype to know how to format the data. */
void cafix_data_generate(cafixdata *data, void *insert_data);

/* If this function knows how it will return you the double value kept
   in data->data. Otherwise it return -1, whether you like it or not. */
double cafix_data_get_double_value(cafixdata *data);

/* This appends *data to list->next or does a memcpy(list,data) if list == NULL */
void cafix_list_append(cafixdata *list, cafixdata *data);

/* Traverse a list and free everything(!) */
void cafix_list_free_all(cafixdata *list);

/* This is a status callback that does nothing. */
int cafix_empty_status_callback(cafixdata *data);

#endif
