/**
 * @file libgalago-gtk/galago-gtk-contact-chooser-widget.c
 *       Contact chooser widget
 *
 * @Copyright (C) 2005-2006 Christian Hammond.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libgalago-gtk/galago-gtk-contact-chooser-widget.h>
#include <libgalago-gtk/galago-gtk-contact-list.h>
#include <libgalago-gtk/galago-gtk-contact-path-bar.h>
#include <libgalago-gtk/galago-gtk-person-list.h>
#include <libgalago-gtk/galago-gtk-private.h>
#include <libgalago-gtk/galago-gtk-service-list.h>
#include <libgalago-gtk/galago-gtk-stock.h>
#include <libgalago-gtk/galago-gdk-pixbuf.h>
#include <gtk/gtk.h>

struct _GalagoGtkContactChooserWidgetPriv
{
	gboolean show_offline;

	GtkWidget *path_bar;
	GtkWidget *notebook;
	GtkWidget *info_pane;
	GtkWidget *service_list;
	GtkWidget *contact_list;
	GtkWidget *person_list;

	struct
	{
		GtkWidget *image;
		GtkWidget *account_box;

	} people_info_pane;

	struct
	{
	} contact_info_pane;

	GtkListStore *contacts_model;
};

enum
{
	PAGE_CONTACTS,
	PAGE_PEOPLE
};

static void galago_gtk_contact_chooser_widget_class_init(GalagoGtkContactChooserWidgetClass *klass);
static void galago_gtk_contact_chooser_widget_iface_init(GalagoGtkContactChooserIface *iface);
static void galago_gtk_contact_chooser_widget_init(GalagoGtkContactChooserWidget *chooser_widget);
static void galago_gtk_contact_chooser_widget_finalize(GObject *obj);
static void galago_gtk_contact_chooser_widget_destroy(GtkObject *obj);

static void build_people_page(GalagoGtkContactChooserWidget *chooser_widget);
static void build_contacts_page(GalagoGtkContactChooserWidget *chooser_widget);

static GtkVBoxClass *parent_class = NULL;

GType
galago_gtk_contact_chooser_widget_get_type(void)
{
	static GType type = 0;

	if (!type)
	{
		static const GTypeInfo info =
		{
			sizeof(GalagoGtkContactChooserWidgetClass),
			NULL,
			NULL,
			(GClassInitFunc)galago_gtk_contact_chooser_widget_class_init,
			NULL,
			NULL,
			sizeof(GalagoGtkContactChooserWidget),
			0,
			(GInstanceInitFunc)galago_gtk_contact_chooser_widget_init
		};

		static const GInterfaceInfo contact_chooser_info =
		{
			(GInterfaceInitFunc)galago_gtk_contact_chooser_widget_iface_init,
			NULL,
			NULL
		};

		type = g_type_register_static(GTK_TYPE_VBOX,
									  "GalagoGtkContactChooserWidget",
									  &info, 0);
		g_type_add_interface_static(type, GALAGO_GTK_TYPE_CONTACT_CHOOSER,
									&contact_chooser_info);
	}

	return type;
}

static void
_galago_gtk_contact_chooser_widget_set_service_type(
	GalagoGtkContactChooser *chooser, GalagoGtkServiceListItemType type)
{
	GalagoGtkContactChooserWidget *chooser_widget =
		GALAGO_GTK_CONTACT_CHOOSER_WIDGET(chooser);

	if (type != GALAGO_GTK_SERVICE_LIST_ITEM_SERVICE)
	{
		galago_gtk_service_list_set_item_type(
			GALAGO_GTK_SERVICE_LIST(chooser_widget->priv->service_list), type);
	}

	if (type == GALAGO_GTK_SERVICE_LIST_ITEM_PEOPLE)
	{
		if (chooser_widget->priv->person_list == NULL)
			build_people_page(chooser_widget);

		gtk_notebook_set_current_page(
			GTK_NOTEBOOK(chooser_widget->priv->notebook), PAGE_PEOPLE);
		gtk_notebook_set_current_page(
			GTK_NOTEBOOK(chooser_widget->priv->info_pane), PAGE_PEOPLE);

		galago_gtk_contact_path_bar_set_root_type(
			GALAGO_GTK_CONTACT_PATH_BAR(chooser_widget->priv->path_bar),
			GALAGO_GTK_CONTACT_PATH_BAR_ROOT_ALL_PEOPLE);

	}
	else
	{
		if (chooser_widget->priv->contact_list == NULL)
			build_contacts_page(chooser_widget);

		gtk_notebook_set_current_page(
			GTK_NOTEBOOK(chooser_widget->priv->notebook), PAGE_CONTACTS);
		gtk_notebook_set_current_page(
			GTK_NOTEBOOK(chooser_widget->priv->info_pane), PAGE_CONTACTS);

		galago_gtk_contact_path_bar_set_root_type(
			GALAGO_GTK_CONTACT_PATH_BAR(chooser_widget->priv->path_bar),
			GALAGO_GTK_CONTACT_PATH_BAR_ROOT_ALL_CONTACTS);
	}
}

static void
_galago_gtk_contact_chooser_widget_set_service(
	GalagoGtkContactChooser *chooser, GalagoService *service)
{
	GalagoGtkContactChooserWidget *chooser_widget =
		GALAGO_GTK_CONTACT_CHOOSER_WIDGET(chooser);

	galago_gtk_service_list_set_service(
		GALAGO_GTK_SERVICE_LIST(chooser_widget->priv->service_list), service);
	galago_gtk_contact_list_set_service(
		GALAGO_GTK_CONTACT_LIST(chooser_widget->priv->contact_list), service);
	galago_gtk_contact_path_bar_set_service(
		GALAGO_GTK_CONTACT_PATH_BAR(chooser_widget->priv->path_bar), service);
}

static void
_galago_gtk_contact_chooser_widget_set_contact(
	GalagoGtkContactChooser *chooser, GalagoAccount *contact)
{
	GalagoGtkContactChooserWidget *chooser_widget =
		GALAGO_GTK_CONTACT_CHOOSER_WIDGET(chooser);

	galago_gtk_contact_list_set_contact(
		GALAGO_GTK_CONTACT_LIST(chooser_widget->priv->contact_list), contact);
}

static void
_galago_gtk_contact_chooser_widget_set_person(
	GalagoGtkContactChooser *chooser, GalagoPerson *person)
{
	GalagoGtkContactChooserWidget *chooser_widget =
		GALAGO_GTK_CONTACT_CHOOSER_WIDGET(chooser);

	galago_gtk_person_list_set_person(
		GALAGO_GTK_PERSON_LIST(chooser_widget->priv->person_list), person);
}

static GalagoAccount *
_galago_gtk_contact_chooser_widget_get_contact(
	const GalagoGtkContactChooser *chooser)
{
	GalagoGtkContactChooserWidget *chooser_widget =
		GALAGO_GTK_CONTACT_CHOOSER_WIDGET(chooser);

	return galago_gtk_contact_list_get_contact(
		GALAGO_GTK_CONTACT_LIST(chooser_widget->priv->contact_list));
}

static GalagoPerson *
_galago_gtk_contact_chooser_widget_get_person(
	const GalagoGtkContactChooser *chooser)
{
	GalagoGtkContactChooserWidget *chooser_widget =
		GALAGO_GTK_CONTACT_CHOOSER_WIDGET(chooser);

	return galago_gtk_person_list_get_person(
		GALAGO_GTK_PERSON_LIST(chooser_widget->priv->person_list));
}

static void
galago_gtk_contact_chooser_widget_iface_init(
	GalagoGtkContactChooserIface *iface)
{
	iface->set_service_type = _galago_gtk_contact_chooser_widget_set_service_type;
	iface->set_service      = _galago_gtk_contact_chooser_widget_set_service;
	iface->set_contact      = _galago_gtk_contact_chooser_widget_set_contact;
	iface->set_person       = _galago_gtk_contact_chooser_widget_set_person;
	iface->get_contact      = _galago_gtk_contact_chooser_widget_get_contact;
	iface->get_person       = _galago_gtk_contact_chooser_widget_get_person;
}

static void
galago_gtk_contact_chooser_widget_class_init(
	GalagoGtkContactChooserWidgetClass *klass)
{
	GObjectClass *gobject_class;
	GtkObjectClass *object_class;

	galago_gtk_stock_init();

	parent_class = g_type_class_peek_parent(klass);

	gobject_class = G_OBJECT_CLASS(klass);
	object_class  = GTK_OBJECT_CLASS(klass);

	gobject_class->finalize = galago_gtk_contact_chooser_widget_finalize;

	object_class->destroy = galago_gtk_contact_chooser_widget_destroy;

}

static void
service_list_changed_cb(GalagoGtkServiceList *service_list,
						GalagoGtkContactChooserWidget *chooser_widget)
{
	GalagoGtkServiceListItemType item_type;
	GalagoService *service;

	item_type = galago_gtk_service_list_get_item_type(service_list);
	service   = galago_gtk_service_list_get_service(service_list);

	galago_gtk_contact_chooser_set_service_type(
		GALAGO_GTK_CONTACT_CHOOSER(chooser_widget), item_type);

	if (item_type != GALAGO_GTK_SERVICE_LIST_ITEM_PEOPLE)
	{
		galago_gtk_contact_chooser_set_service(
			GALAGO_GTK_CONTACT_CHOOSER(chooser_widget), service);
	}
}

static GtkWidget *
build_services_pane(GalagoGtkContactChooserWidget *chooser_widget)
{
	GtkWidget *sw;
	GtkWidget *service_list;

	sw = gtk_scrolled_window_new(NULL, NULL);
	gtk_widget_show(sw);
	gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(sw),
										GTK_SHADOW_IN);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw),
								   GTK_POLICY_AUTOMATIC,
								   GTK_POLICY_AUTOMATIC);

	service_list = chooser_widget->priv->service_list =
		galago_gtk_service_list_new();
	gtk_widget_show(service_list);
	gtk_container_add(GTK_CONTAINER(sw), service_list);

	galago_gtk_service_list_set_show_people(
		GALAGO_GTK_SERVICE_LIST(service_list), TRUE);
	galago_gtk_service_list_set_show_all_contacts(
		GALAGO_GTK_SERVICE_LIST(service_list), TRUE);

	g_signal_connect(G_OBJECT(service_list), "selection-changed",
					 G_CALLBACK(service_list_changed_cb), chooser_widget);

	return sw;
}

static void
show_offline_toggled_cb(GtkToggleButton *check,
						GalagoGtkContactChooserWidget *chooser_widget)
{
	gboolean show_offline = gtk_toggle_button_get_active(check);

	chooser_widget->priv->show_offline = show_offline;

	if (chooser_widget->priv->contact_list != NULL)
	{
		galago_gtk_contact_list_set_show_offline(
			GALAGO_GTK_CONTACT_LIST(chooser_widget->priv->contact_list),
			show_offline);
	}

	if (chooser_widget->priv->person_list != NULL)
	{
		galago_gtk_person_list_set_show_offline(
			GALAGO_GTK_PERSON_LIST(chooser_widget->priv->person_list),
			show_offline);
	}
}

static void
contact_list_changed_cb(GalagoGtkContactList *contact_list,
						GalagoGtkContactChooserWidget *chooser_widget)
{
	g_signal_emit_by_name(chooser_widget, "selection-changed", 0);
}

static void
contact_list_activated_cb(GalagoGtkContactList *contact_list,
						  GalagoAccount *contact,
						  GalagoGtkContactChooserWidget *chooser_widget)
{
	g_signal_emit_by_name(chooser_widget, "activated", 0);
}

static void
build_contacts_page(GalagoGtkContactChooserWidget *chooser_widget)
{
	GtkWidget *sw;
	GtkWidget *contact_list;

	g_return_if_fail(chooser_widget->priv->contact_list == NULL);

	/*
	 * Contacts list
	 */
	/* Treeview and scrolled window */
	sw = gtk_scrolled_window_new(NULL, NULL);
	gtk_widget_show(sw);
	gtk_notebook_insert_page(GTK_NOTEBOOK(chooser_widget->priv->notebook), sw,
							 NULL, PAGE_CONTACTS);
	gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(sw),
										GTK_SHADOW_IN);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw),
								   GTK_POLICY_AUTOMATIC,
								   GTK_POLICY_AUTOMATIC);

	contact_list = chooser_widget->priv->contact_list =
		galago_gtk_contact_list_new(TRUE);
	gtk_widget_show(contact_list);
	gtk_container_add(GTK_CONTAINER(sw), contact_list);

	galago_gtk_contact_list_set_show_offline(
		GALAGO_GTK_CONTACT_LIST(contact_list),
		chooser_widget->priv->show_offline);

	g_signal_connect(G_OBJECT(contact_list), "selection-changed",
					 G_CALLBACK(contact_list_changed_cb), chooser_widget);
	g_signal_connect(G_OBJECT(contact_list), "contact-activated",
					 G_CALLBACK(contact_list_activated_cb), chooser_widget);
}

static void
person_list_changed_cb(GalagoGtkPersonList *person_list,
					   GalagoGtkContactChooserWidget *chooser_widget)
{
	GalagoPerson *person;
	GalagoImage *photo;

	person = galago_gtk_person_list_get_person(person_list);

	if (person == NULL)
		return;

	photo = galago_person_get_photo(person, TRUE);

	if (photo == NULL)
		gtk_widget_hide(chooser_widget->priv->people_info_pane.image);
	else
	{
		GdkPixbuf *pixbuf = galago_gdk_pixbuf_new_from_image(photo, -1, 100);

		if (pixbuf != NULL)
		{
#if 0
			gtk_image_set_from_pixbuf(
				GTK_IMAGE(chooser_widget->priv->people_info_pane.image),
				pixbuf);
#endif

			g_object_unref(G_OBJECT(pixbuf));

			gtk_widget_show(chooser_widget->priv->people_info_pane.image);
		}
	}
}

static void
person_list_activated_cb(GalagoGtkPersonList *person_list,
						 GalagoPerson *person,
						 GalagoGtkContactChooserWidget *chooser_widget)
{
	galago_gtk_contact_path_bar_set_person(
		GALAGO_GTK_CONTACT_PATH_BAR(chooser_widget->priv->path_bar), person);
	galago_gtk_contact_list_set_person(
		GALAGO_GTK_CONTACT_LIST(chooser_widget->priv->contact_list), person);
	gtk_notebook_set_current_page(GTK_NOTEBOOK(chooser_widget->priv->notebook),
								  PAGE_CONTACTS);
	gtk_notebook_set_current_page(GTK_NOTEBOOK(chooser_widget->priv->info_pane),
								  PAGE_CONTACTS);
}

static void
build_people_page(GalagoGtkContactChooserWidget *chooser_widget)
{
	GtkWidget *sw;
	GtkWidget *person_list;
	GtkWidget *preview_image;

	g_return_if_fail(chooser_widget->priv->person_list == NULL);

	/*
	 * People list
	 */
	/* Treeview and scrolled window */
	sw = gtk_scrolled_window_new(NULL, NULL);
	gtk_widget_show(sw);
	gtk_notebook_insert_page(GTK_NOTEBOOK(chooser_widget->priv->notebook), sw,
							 NULL, PAGE_PEOPLE);
	gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(sw),
										GTK_SHADOW_IN);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw),
								   GTK_POLICY_AUTOMATIC,
								   GTK_POLICY_AUTOMATIC);

	person_list = chooser_widget->priv->person_list =
		galago_gtk_person_list_new();
	gtk_widget_show(person_list);
	gtk_container_add(GTK_CONTAINER(sw), person_list);

	galago_gtk_person_list_set_show_offline(
		GALAGO_GTK_PERSON_LIST(person_list),
		chooser_widget->priv->show_offline);

	g_signal_connect(G_OBJECT(person_list), "selection-changed",
					 G_CALLBACK(person_list_changed_cb), chooser_widget);
	g_signal_connect(G_OBJECT(person_list), "person-activated",
					 G_CALLBACK(person_list_activated_cb), chooser_widget);

	/*
	 * Preview pane
	 */
	preview_image = chooser_widget->priv->people_info_pane.image =
		gtk_image_new();
	gtk_widget_show(preview_image);
	gtk_misc_set_alignment(GTK_MISC(preview_image), 0.5, 0.0);
	gtk_notebook_insert_page(GTK_NOTEBOOK(chooser_widget->priv->info_pane),
							 preview_image, NULL, PAGE_PEOPLE);
}

static void
path_bar_root_activated_cb(GalagoGtkContactPathBar *path_bar,
						   GalagoGtkContactChooserWidget *chooser_widget)
{
	if (galago_gtk_contact_path_bar_get_root_type(path_bar) !=
		GALAGO_GTK_CONTACT_PATH_BAR_ROOT_ALL_PEOPLE)
	{
		return;
	}

	gtk_notebook_set_current_page(
		GTK_NOTEBOOK(chooser_widget->priv->notebook), PAGE_PEOPLE);
	gtk_notebook_set_current_page(
		GTK_NOTEBOOK(chooser_widget->priv->info_pane), PAGE_PEOPLE);
}

static GtkWidget *
build_contacts_pane(GalagoGtkContactChooserWidget *chooser_widget)
{
	GtkWidget *vbox;
	GtkWidget *hbox;
	GtkWidget *check;
	GtkWidget *path_bar;
	GtkWidget *notebook;
	GtkWidget *info_pane;

	vbox = gtk_vbox_new(FALSE, 6);
	gtk_widget_show(vbox);

	path_bar = chooser_widget->priv->path_bar =
		galago_gtk_contact_path_bar_new();
	gtk_widget_show(path_bar);
	gtk_box_pack_start(GTK_BOX(vbox), path_bar, FALSE, TRUE, 0);

	g_signal_connect(G_OBJECT(path_bar), "root_activated",
					 G_CALLBACK(path_bar_root_activated_cb), chooser_widget);

	hbox = gtk_hbox_new(FALSE, 12);
	gtk_widget_show(hbox);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);

	/* People and contacts notebook */
	notebook = chooser_widget->priv->notebook = gtk_notebook_new();
	gtk_widget_show(notebook);
	gtk_box_pack_start(GTK_BOX(hbox), notebook, TRUE, TRUE, 0);
	gtk_notebook_set_show_tabs(GTK_NOTEBOOK(notebook), FALSE);
	gtk_notebook_set_show_border(GTK_NOTEBOOK(notebook), FALSE);

	build_contacts_page(chooser_widget);

	/* Info pane */
	info_pane = chooser_widget->priv->info_pane = gtk_notebook_new();
	gtk_widget_show(info_pane);
	gtk_box_pack_start(GTK_BOX(hbox), info_pane, FALSE, TRUE, 0);
	gtk_notebook_set_show_tabs(GTK_NOTEBOOK(info_pane), FALSE);
	gtk_notebook_set_show_border(GTK_NOTEBOOK(info_pane), FALSE);

	/*
	 * Checkboxes
	 */
	check = gtk_check_button_new_with_mnemonic(_("Show _offline contacts"));
	gtk_widget_show(check);
	gtk_box_pack_start(GTK_BOX(vbox), check, FALSE, TRUE, 0);

	g_signal_connect(G_OBJECT(check), "toggled",
					 G_CALLBACK(show_offline_toggled_cb), chooser_widget);

	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check), TRUE);

	/* XXX */
	gtk_notebook_set_current_page(GTK_NOTEBOOK(notebook), PAGE_CONTACTS);
	gtk_notebook_set_current_page(GTK_NOTEBOOK(info_pane), PAGE_CONTACTS);
	galago_gtk_contact_list_set_service(
		GALAGO_GTK_CONTACT_LIST(chooser_widget->priv->contact_list),
		galago_get_service(GALAGO_SERVICE_ID_AIM, GALAGO_REMOTE, TRUE));

	return vbox;
}

static void
galago_gtk_contact_chooser_widget_init(
	GalagoGtkContactChooserWidget *chooser_widget)
{
	GtkWidget *hpaned;
	GtkWidget *widget;

	chooser_widget->priv = g_new0(GalagoGtkContactChooserWidgetPriv, 1);
	chooser_widget->priv->show_offline = TRUE;

	/* Horizontal pane */
	hpaned = gtk_hpaned_new();
	gtk_widget_show(hpaned);
	gtk_box_pack_start(GTK_BOX(chooser_widget), hpaned, TRUE, TRUE, 0);
	gtk_paned_set_position(GTK_PANED(hpaned), 200);

	/* Services/people list */
	widget = build_services_pane(chooser_widget);
	gtk_paned_add1(GTK_PANED(hpaned), widget);

	/* Contacts list */
	widget = build_contacts_pane(chooser_widget);
	gtk_paned_add2(GTK_PANED(hpaned), widget);
}

static void
galago_gtk_contact_chooser_widget_finalize(GObject *obj)
{
	GalagoGtkContactChooserWidget *chooser_widget;

	g_return_if_fail(obj != NULL);
	g_return_if_fail(GALAGO_GTK_IS_CONTACT_CHOOSER_WIDGET(obj));

	chooser_widget = GALAGO_GTK_CONTACT_CHOOSER_WIDGET(obj);

	g_free(chooser_widget->priv);

	if (G_OBJECT_CLASS(parent_class)->finalize)
		G_OBJECT_CLASS(parent_class)->finalize(obj);
}

static void
galago_gtk_contact_chooser_widget_destroy(GtkObject *obj)
{
	GalagoGtkContactChooserWidget *chooser_widget;

	g_return_if_fail(obj != NULL);
	g_return_if_fail(GALAGO_GTK_IS_CONTACT_CHOOSER_WIDGET(obj));

	chooser_widget = GALAGO_GTK_CONTACT_CHOOSER_WIDGET(obj);

	if (GTK_OBJECT_CLASS(parent_class)->destroy)
		GTK_OBJECT_CLASS(parent_class)->destroy(obj);
}

GtkWidget *
galago_gtk_contact_chooser_widget_new(void)
{
	return GTK_WIDGET(g_object_new(GALAGO_GTK_TYPE_CONTACT_CHOOSER_WIDGET,
								   NULL));
}

void
galago_gtk_contact_chooser_widget_set_service_type(
	GalagoGtkContactChooserWidget *chooser_widget,
	GalagoGtkServiceListItemType type)
{
	g_return_if_fail(chooser_widget != NULL);
	g_return_if_fail(GALAGO_GTK_IS_CONTACT_CHOOSER_WIDGET(chooser_widget));

	if (type != GALAGO_GTK_SERVICE_LIST_ITEM_SERVICE)
	{
		galago_gtk_service_list_set_item_type(
			GALAGO_GTK_SERVICE_LIST(chooser_widget->priv->service_list), type);
	}
}

void
galago_gtk_contact_chooser_widget_set_service(
	GalagoGtkContactChooserWidget *chooser_widget, GalagoService *service)
{
	g_return_if_fail(chooser_widget != NULL);
	g_return_if_fail(GALAGO_GTK_IS_CONTACT_CHOOSER_WIDGET(chooser_widget));

	galago_gtk_service_list_set_service(
		GALAGO_GTK_SERVICE_LIST(chooser_widget->priv->service_list), service);
}

void
galago_gtk_contact_chooser_widget_set_contact(
	GalagoGtkContactChooserWidget *chooser_widget, GalagoAccount *contact)
{
	g_return_if_fail(chooser_widget != NULL);
	g_return_if_fail(GALAGO_GTK_IS_CONTACT_CHOOSER_WIDGET(chooser_widget));
	g_return_if_fail(contact != NULL);
	g_return_if_fail(GALAGO_IS_ACCOUNT(contact));

	galago_gtk_contact_chooser_widget_set_service(chooser_widget,
		galago_account_get_service(contact));
	galago_gtk_contact_list_set_contact(
		GALAGO_GTK_CONTACT_LIST(chooser_widget->priv->contact_list), contact);
}

void
galago_gtk_contact_chooser_widget_set_person(
	GalagoGtkContactChooserWidget *chooser_widget, GalagoPerson *person)
{
	g_return_if_fail(chooser_widget != NULL);
	g_return_if_fail(GALAGO_GTK_IS_CONTACT_CHOOSER_WIDGET(chooser_widget));
	g_return_if_fail(person != NULL);
	g_return_if_fail(GALAGO_IS_PERSON(person));

	galago_gtk_contact_chooser_widget_set_service_type(chooser_widget,
		GALAGO_GTK_SERVICE_LIST_ITEM_PEOPLE);
	galago_gtk_person_list_set_person(
		GALAGO_GTK_PERSON_LIST(chooser_widget->priv->person_list), person);
}

GalagoAccount *
galago_gtk_contact_chooser_widget_get_contact(
	const GalagoGtkContactChooserWidget *chooser_widget)
{
	g_return_val_if_fail(chooser_widget != NULL, NULL);
	g_return_val_if_fail(GALAGO_GTK_CONTACT_CHOOSER_WIDGET(chooser_widget),
						 NULL);

	return galago_gtk_contact_list_get_contact(
		GALAGO_GTK_CONTACT_LIST(chooser_widget->priv->contact_list));
}

GalagoPerson *
galago_gtk_contact_chooser_widget_get_person(
	const GalagoGtkContactChooserWidget *chooser_widget)
{
	g_return_val_if_fail(chooser_widget != NULL, NULL);
	g_return_val_if_fail(GALAGO_GTK_CONTACT_CHOOSER_WIDGET(chooser_widget),
						 NULL);

	return galago_gtk_person_list_get_person(
		GALAGO_GTK_PERSON_LIST(chooser_widget->priv->person_list));
}
