/**
 * @file libgalago-gtk/galago-gtk-presence-icon.c Presence Icon widget
 *
 * @Copyright (C) 2004-2006 Christian Hammond.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libgalago-gtk/galago-gtk-presence-icon.h>
#include <libgalago-gtk/galago-gdk-pixbuf.h>
#include <libgalago-gtk/galago-gtk-private.h>
#include <gtk/gtk.h>
#include <gdk/gdk.h>

struct _GalagoGtkPresenceIconPriv
{
	GalagoGtkIconPrecedence precedence;

	GtkIconSize size;

	GalagoPresence *presence;

	GalagoPerson *person;
	GalagoAccount *account;
	char *username;

	gulong changed_id;
	gulong account_destroy_id;
};

static void galago_gtk_presence_icon_class_init(GalagoGtkPresenceIconClass *klass);
static void galago_gtk_presence_icon_init(GalagoGtkPresenceIcon *icon);
static void galago_gtk_presence_icon_finalize(GObject *obj);
static void galago_gtk_presence_icon_destroy(GtkObject *obj);
static void update_icon(GalagoPresence *presence, GalagoGtkPresenceIcon *icon);

static GtkImageClass *parent_class = NULL;

GType
galago_gtk_presence_icon_get_type(void)
{
	static GType type = 0;

	if (!type)
	{
		static const GTypeInfo info =
		{
			sizeof(GalagoGtkPresenceIconClass),
			NULL,
			NULL,
			(GClassInitFunc)galago_gtk_presence_icon_class_init,
			NULL,
			NULL,
			sizeof(GalagoGtkPresenceIcon),
			0,
			(GInstanceInitFunc)galago_gtk_presence_icon_init
		};

		type = g_type_register_static(GTK_TYPE_IMAGE, "GalagoGtkPresenceIcon",
									  &info, 0);
	}

	return type;
}

static void
galago_gtk_presence_icon_class_init(GalagoGtkPresenceIconClass *klass)
{
	GObjectClass *gobject_class;
	GtkObjectClass *object_class;

	parent_class = g_type_class_peek_parent(klass);

	gobject_class = G_OBJECT_CLASS(klass);
	object_class  = GTK_OBJECT_CLASS(klass);

	gobject_class->finalize = galago_gtk_presence_icon_finalize;

	object_class->destroy = galago_gtk_presence_icon_destroy;
}

static void
galago_gtk_presence_icon_init(GalagoGtkPresenceIcon *icon)
{
	icon->priv = g_new0(GalagoGtkPresenceIconPriv, 1);
	icon->priv->precedence = GALAGO_GTK_ICON_PHOTO;
	icon->priv->size       = GTK_ICON_SIZE_DIALOG;
}

static void
galago_gtk_presence_icon_finalize(GObject *obj)
{
	GalagoGtkPresenceIcon *icon;

	g_return_if_fail(obj != NULL);
	g_return_if_fail(GALAGO_GTK_IS_PRESENCE_ICON(obj));

	icon = GALAGO_GTK_PRESENCE_ICON(obj);

	g_free(icon->priv);

	if (G_OBJECT_CLASS(parent_class)->finalize)
		G_OBJECT_CLASS(parent_class)->finalize(obj);
}

static void
galago_gtk_presence_icon_destroy(GtkObject *obj)
{
	GalagoGtkPresenceIcon *icon;

	g_return_if_fail(obj != NULL);
	g_return_if_fail(GALAGO_GTK_IS_PRESENCE_ICON(obj));

	icon = GALAGO_GTK_PRESENCE_ICON(obj);

	GALAGO_GTK_DISCONNECT_HANDLER(icon->priv->presence,
								  icon->priv->changed_id);
	GALAGO_GTK_DISCONNECT_HANDLER(icon->priv->presence,
								  icon->priv->account_destroy_id);

	icon->priv->presence = NULL;

	if (GTK_OBJECT_CLASS(parent_class)->destroy)
		GTK_OBJECT_CLASS(parent_class)->destroy(obj);
}

static char *
secs_to_string(dbus_uint32_t secs)
{
	GString *gstr;
	dbus_uint32_t days, hrs, mins;

	days = secs / (60 * 60 * 24);
	secs = secs % (60 * 60 * 24);
	hrs  = secs / (60 * 60);
	secs = secs % (60 * 60);
	mins = secs / 60;
	secs = secs % 60;

	gstr = g_string_new("");

	if (days > 0)
	{
		g_string_append_printf(gstr, "%d %s, ", days,
							   ngettext("day", "days", days));
	}

	if (hrs > 0)
	{
		g_string_append_printf(gstr, "%d %s, ", hrs,
							   ngettext("hour", "hours", hrs));
	}

	if (mins > 0)
	{
		g_string_append_printf(gstr, "%d %s, ", mins,
							   ngettext("minute", "minutes", mins));
	}

	g_string_append_printf(gstr, "%d %s", secs,
						   ngettext("second", "seconds", secs));

	return g_string_free(gstr, FALSE);
}

static void
update_icon(GalagoPresence *presence, GalagoGtkPresenceIcon *icon)
{
	GdkPixbuf *pixbuf = NULL;

	if (presence == NULL)
	{
		GalagoAccount *account;

		account = (icon->priv->person == NULL
				   ? icon->priv->account
				   : galago_person_get_priority_account(icon->priv->person));

		if (account != NULL)
		{
			pixbuf = galago_gdk_pixbuf_new_from_account_with_size(account,
				galago_gtk_presence_icon_get_size(icon));
		}
	}
	else
	{
		pixbuf = galago_gdk_pixbuf_new_from_presence_with_size(presence,
			galago_gtk_presence_icon_get_size(icon),
			galago_gtk_presence_icon_get_precedence(icon));
	}

	gtk_image_set_from_pixbuf(GTK_IMAGE(icon), pixbuf);

	if (pixbuf != NULL)
		g_object_unref(pixbuf);
}

static void
account_destroy_cb(GalagoAccount *account, GalagoGtkPresenceIcon *icon)
{
	galago_gtk_presence_icon_set_account(icon, NULL);
}

static void
queue_update_icon(GalagoGtkPresenceIcon *icon)
{
	GALAGO_GTK_DISCONNECT_HANDLER(icon->priv->presence,
								  icon->priv->changed_id);

	if (icon->priv->person == NULL && icon->priv->account == NULL)
	{
		gtk_image_set_from_file(GTK_IMAGE(icon), NULL);
	}
	else
	{
		GalagoPresence *presence = NULL;
		GalagoAccount *account;

		account = (icon->priv->person == NULL
				   ? icon->priv->account
				   : galago_person_get_priority_account(icon->priv->person));

		if (account != NULL)
			presence = galago_account_get_presence(account, TRUE);

		icon->priv->presence = presence;

		if (presence != NULL)
		{
			icon->priv->changed_id =
				g_signal_connect(G_OBJECT(presence), "changed",
								 G_CALLBACK(update_icon), icon);
		}

		update_icon(presence, icon);
	}
}

GtkWidget *
galago_gtk_presence_icon_new(void)
{
	return GTK_WIDGET(g_object_new(GALAGO_GTK_TYPE_PRESENCE_ICON, NULL));
}

void
galago_gtk_presence_icon_set_account(GalagoGtkPresenceIcon *icon,
									 GalagoAccount *account)
{
	g_return_if_fail(icon != NULL);
	g_return_if_fail(GALAGO_GTK_IS_PRESENCE_ICON(icon));

	if (icon->priv->account == account)
		return;

	GALAGO_GTK_DISCONNECT_HANDLER(icon->priv->account,
								  icon->priv->account_destroy_id);

	icon->priv->account = account;

	if (account != NULL)
	{
		icon->priv->account_destroy_id =
			g_signal_connect(G_OBJECT(account), "destroy",
							 G_CALLBACK(account_destroy_cb), icon);
	}

	queue_update_icon(icon);
}

void
galago_gtk_presence_icon_set_person(GalagoGtkPresenceIcon *icon,
									GalagoPerson *person)
{
	g_return_if_fail(icon != NULL);
	g_return_if_fail(GALAGO_GTK_IS_PRESENCE_ICON(icon));

	if (icon->priv->person == person)
		return;

	icon->priv->person = person;

	queue_update_icon(icon);
}

void
galago_gtk_presence_icon_set_precedence(GalagoGtkPresenceIcon *icon,
										GalagoGtkIconPrecedence precedence)
{
	g_return_if_fail(icon != NULL);
	g_return_if_fail(GALAGO_GTK_IS_PRESENCE_ICON(icon));
	g_return_if_fail(precedence != GALAGO_GTK_ICON_NONE);

	if (icon->priv->precedence == precedence)
		return;

	icon->priv->precedence = precedence;

	queue_update_icon(icon);
}

void
galago_gtk_presence_icon_set_size(GalagoGtkPresenceIcon *icon,
								  GtkIconSize size)
{
	g_return_if_fail(icon != NULL);
	g_return_if_fail(GALAGO_GTK_IS_PRESENCE_ICON(icon));
	g_return_if_fail(size != GTK_ICON_SIZE_INVALID);

	if (icon->priv->size == size)
		return;

	icon->priv->size = size;

	queue_update_icon(icon);
}

void
galago_gtk_presence_icon_update(GalagoGtkPresenceIcon *icon)
{
	g_return_if_fail(icon != NULL);
	g_return_if_fail(GALAGO_GTK_IS_PRESENCE_ICON(icon));

	queue_update_icon(icon);
}

char *
galago_gtk_presence_icon_get_tooltip_text(const GalagoGtkPresenceIcon *icon)
{

	GalagoAccount *account;
	GalagoPresence *presence = NULL;
	char *str;

	g_return_val_if_fail(icon != NULL,                      NULL);
	g_return_val_if_fail(GALAGO_GTK_IS_PRESENCE_ICON(icon), NULL);

	if (icon->priv->person == NULL && icon->priv->account == NULL)
		return NULL;

	account = (icon->priv->person == NULL
			   ? icon->priv->account
			   : galago_person_get_priority_account(icon->priv->person));

	if (account != NULL)
		presence = galago_account_get_presence(account, TRUE);

	if (presence != NULL)
	{
		GalagoStatus *status;
		gboolean idle;
		dbus_uint32_t idle_time = 0;
		char *idle_text = NULL;
		char *status_text = NULL;

		idle = galago_presence_is_idle(presence);

		if (idle)
		{
			idle_time = galago_presence_get_idle_time(presence);

			if (idle_time > 0)
			{
				char *str = NULL;

				str = secs_to_string(idle_time);

				idle_text = g_strdup_printf(_("\n<b>Idle:</b> %s"), str);

				g_free(str);
			}
			else
			{
				idle_text = g_strdup_printf(_("\n<b>Idle</b>"));
			}
		}

		status = galago_presence_get_active_status(presence);

		if (status != NULL)
		{
			const char *name = galago_status_get_name(status);

			/* TODO: Status attributes. */

			if (name != NULL)
				status_text = g_strdup_printf(_("\n<b>Status:</b> %s"), name);
		}

		str = g_strdup_printf(
			"<span size='larger' weight='bold'>%s</span>"
			"%s"  /* Idle */
			"%s", /* Status */
			galago_account_get_username(account),
			(idle_text   != NULL ? idle_text   : ""),
			(status_text != NULL ? status_text : ""));

		if (idle_text != NULL)
			g_free(idle_text);

		if (status_text != NULL)
			g_free(status_text);
	}
	else
	{
		str = g_strdup_printf(
			"<span size='larger' weight='bold'>%s</span>"
			"<i>%s</i>",
			galago_account_get_username(account),
			_("No presence"));
	}

	return str;
}

GalagoAccount *
galago_gtk_presence_icon_get_account(const GalagoGtkPresenceIcon *icon)
{
	g_return_val_if_fail(icon != NULL, NULL);
	g_return_val_if_fail(GALAGO_GTK_IS_PRESENCE_ICON(icon), NULL);

	return icon->priv->account;
}

GalagoPerson *
galago_gtk_presence_icon_get_person(const GalagoGtkPresenceIcon *icon)
{
	g_return_val_if_fail(icon != NULL, NULL);
	g_return_val_if_fail(GALAGO_GTK_IS_PRESENCE_ICON(icon), NULL);

	return icon->priv->person;
}

GalagoGtkIconPrecedence
galago_gtk_presence_icon_get_precedence(const GalagoGtkPresenceIcon *icon)
{
	g_return_val_if_fail(icon != NULL, FALSE);
	g_return_val_if_fail(GALAGO_GTK_IS_PRESENCE_ICON(icon), FALSE);

	return icon->priv->precedence;
}

GtkIconSize
galago_gtk_presence_icon_get_size(const GalagoGtkPresenceIcon *icon)
{
	g_return_val_if_fail(icon != NULL, GTK_ICON_SIZE_INVALID);
	g_return_val_if_fail(GALAGO_GTK_IS_PRESENCE_ICON(icon),
						 GTK_ICON_SIZE_INVALID);

	return icon->priv->size;
}
