/* $Id: color.c,v 1.6 2005/07/30 16:31:59 cegger Exp $
   ******************************************************************************

   LIBGCP: color conversion

   Copyright (C) 2000 Christoph Egger   [Christoph_Egger@t-online.de]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

   ******************************************************************************
 */

#include <math.h>

#include "config.h"
#include <ggi/internal/gcp.h>
#include <ggi/internal/gcp_debug.h>
#include <ggi/internal/color.h>



/*  Converts from RGBA colorspace to YUV values.
 */
int gcpRGBA2YUV(ggi_visual_t vis,
		const gcp_RGBAcolor *rgba, gcp_YUVcolor *yuv)
{
	ggi_float r,g,b;


/* Formula:
 *
 * Y =  0.299 * Red + 0.587 * Green + 0.114 * Blue
 * U = -0.147 * Red - 0.289 * Green + 0.436 * Blue
 * V =  0.615 * Red - 0.515 * Green - 0.100 * Blue
 */

	r = (ggi_float)rgba->r / 65535.0f;
	g = (ggi_float)rgba->g / 65535.0f;
	b = (ggi_float)rgba->b / 65535.0f;

	yuv->y =  0.299f * r + 0.587f * g + 0.114f * b;	
	yuv->u = -0.147f * r - 0.289f * g + 0.436f * b;
	yuv->v =  0.615f * r - 0.515f * g - 0.100f * b;


	return 0;
}	/* gcpRGBA2YUV */


/*  Converts from YUV colorspace to RGBA values.
 */
int gcpYUV2RGBA(ggi_visual_t vis,
		const gcp_YUVcolor *yuv, gcp_RGBAcolor *rgba)
{

	ggi_float r,g,b;

/* Formula:
 *
 * Red   = Y + 0.000f * U + 1.140f * V
 * Green = Y - 0.396f * U - 0.581f * V
 * Blue  = Y + 2.029f * U + 0.000f * V
 */

	r = yuv->y + 0.000f * yuv->u + 1.140f * yuv->v;
	g = yuv->y - 0.396f * yuv->u - 0.581f * yuv->v;
	b = yuv->y + 2.029f * yuv->u + 0.000f * yuv->v;


	r *= 65535.0f;
	g *= 65535.0f;
	b *= 65535.0f;

	rgba->r = fabs(r);
	rgba->g = fabs(g);
	rgba->b = fabs(b);
	rgba->a = 0xFFFF;


	return 0;
}	/* gcpYUV2RGBA */




/*  Converts from RGBA colorspace to CMYK values.
 */
int gcpRGBA2CMYK(ggi_visual_t vis,
		const gcp_RGBAcolor *rgba, gcp_CMYKcolor *cmyk)
{


/* Formula:
 *
 * Black = MIN(1 - Red, 1 - Green, 1 - Blue);
 *
 * Cyan = (1 - Red - Black) / (1 - Black)
 * Magenta = (1 - Green - Black) / (1 - Black)
 * Yellow = (1 - Blue - Black) / (1 - Black)
 */


	cmyk->k = MIN(MIN(0xFFFF - rgba->r, 0xFFFF - rgba->g), 0xFFFF - rgba->b);

	if (cmyk->k != 0xFFFF) {
		cmyk->c = (0xFFFF - rgba->r - cmyk->k) / (0xFFFF - cmyk->k);
		cmyk->m = (0xFFFF - rgba->g - cmyk->k) / (0xFFFF - cmyk->k);
		cmyk->y = (0xFFFF - rgba->b - cmyk->k) / (0xFFFF - cmyk->k);
	} else {
		cmyk->c = 0x0000;
		cmyk->m = 0x0000;
		cmyk->y = 0x0000;
	}	/* if */

	return 0;
}	/* gcpRGBA2CMYK */


/*  Converts from CMYK colorspace to RGBA values.
 */
int gcpCMYK2RGBA(ggi_visual_t vis,
		const gcp_CMYKcolor *cmyk, gcp_RGBAcolor *rgba)
{

/* Formula:
 *
 * Red   = 1 - MIN(1, Cyan * (1 - Black) + Black)
 * Green = 1 - MIN(1, Magenta * (1 - Black) + Black)
 * Blue  = 1 - MIN(1, Yellow * (1 - Black) + Black)
 */

	rgba->r = 0xFFFF - MIN(0xFFFF, cmyk->c * (0xFFFF - cmyk->k) + cmyk->k);
	rgba->g = 0xFFFF - MIN(0xFFFF, cmyk->m * (0xFFFF - cmyk->k) + cmyk->k);
	rgba->b = 0xFFFF - MIN(0xFFFF, cmyk->y * (0xFFFF - cmyk->k) + cmyk->k);
	rgba->a = 0xFFFF;


	return 0;
}	/* gcpCMYK2RGBA */



/*  Converts from HSV colorspace to RGBA values.
 */
int gcpHSV2RGBA(ggi_visual_t vis,
		const gcp_HSVcolor *hsv, gcp_RGBAcolor *rgba)
{
	ggi_float f, x, y, z;
	ggi_float h, v;

	int i;


	h = (hsv->h * 180.0f / M_PI);
	v = hsv->v;

	v *= 65535.0f;

	if (hsv->s == 0.0f) {
		rgba->r = rgba->g = rgba->b = (int)v;
	} else {
		while (h < 0) {
			h += 360.0f;
		}	/* while */

		h = fmod(h, 360.0f) / 60.0f;
		i = (int)h;
		f = h - i;
		x = v * (1.0f - hsv->s);
		y = v * (1.0f - (hsv->s * f));
		z = v * (1.0f - (hsv->s * (1.0f - f)));

		switch (i) {
		case 0:
			rgba->r = v;
			rgba->g = z;
			rgba->b = x;
			break;

		case 1:
			rgba->r = y;
			rgba->g = v;
			rgba->b = x;
			break;

		case 2:
			rgba->r = x;
			rgba->g = v;
			rgba->b = z;
			break;

		case 3:
			rgba->r = x;
			rgba->g = y;
			rgba->b = v;
			break;

		case 4:
			rgba->r = z;
			rgba->g = x;
			rgba->b = v;
			break;

		case 5:
			rgba->r = v;
			rgba->g = x;
			rgba->b = y;
			break;
		}	/* switch */
	}	/* if */

	rgba->a = 0xFFFF;

	return 0;

}	/* gcpHSV2RGBA */


/*  Converts an RGBA value into the HSV colorspace.
 */
int gcpRGBA2HSV(ggi_visual_t vis,
		const gcp_RGBAcolor *rgba, gcp_HSVcolor *hsv)
{
	ggi_float min, max, delta, rc, gc, bc;

	rc = (ggi_float)rgba->r / 65535.0f;
	gc = (ggi_float)rgba->g / 65535.0f;
	bc = (ggi_float)rgba->b / 65535.0f;

	max = MAX(rc, MAX(gc, bc));
	min = MIN(rc, MIN(gc, bc));
	delta = max - min;
	hsv->v = max;

	if (max != 0.0f) {
		hsv->s = delta / max;
	} else {
		hsv->s = 0.0f;
	}	/* if */

	if (hsv->s == 0.0f) {
		hsv->h = 0.0f;
	} else {
		if (rc == max) {
			hsv->h = (gc - bc) / delta;
		} else if (gc == max) {
			hsv->h = 2 + (bc - rc) / delta;
		} else if (bc == max) {
			hsv->h = 4 + (rc - gc) / delta;
		}	/* if */

		hsv->h *= 60.0f;
		if (hsv->h < 0) {
			hsv->h += 360.0f;
		}	/* if */
	}	/* if */

	hsv->h = (hsv->h / 180.0f * M_PI);

	return 0;

}	/* gcpRGBA2HSV */



/*  Converts from YCC (Kodak PhotoCD) colorspace to RGBA values.
 */
int gcpYCC2RGBA(ggi_visual_t vis,
		const gcp_YCCcolor *ycc, gcp_RGBAcolor *rgba)
{
	ggi_float r,g,b;
	ggi_float y,c1,c2;

/* Formula:
 *
 * Y'    = 1.3584 * Y;
 * C1'   = 2.2179 * (C1 - 156);
 * C2'   = 1.8215 * (C2 - 137);
 *
 * Red   = Y' + C2'
 * Green = Y' - 0.194 * C1' - 0.509 * C2'
 * Blue  = Y' + C1'
 */


	/* Note this formula assumes, that rgb and ycc have a
	 * range from 0 to 255. So we have to modify it a little.
	 */
	y  = (ggi_float)ycc->y  / 255.0f;
	c1 = (ggi_float)ycc->c1 / 255.0f;
	c2 = (ggi_float)ycc->c2 / 255.0f;


	y  = 1.3584 * y;
	c1 = 2.2179 * (c1 - 156);
	c2 = 1.8215 * (c2 - 137);

	r = y + c2;
	g = y - 0.194 * c1 - 0.509 * c2;
	b = y + c1;

	rgba->r = (uint16_t)r;
	rgba->g = (uint16_t)g;
	rgba->b = (uint16_t)b;
	rgba->a = 0xFFFF;

	printf("rgba (%X,%X,%X,%X)\n",
		rgba->r, rgba->g, rgba->b, rgba->a);

	rgba->r = (rgba->r << 8) | rgba->r;
	rgba->g = (rgba->g << 8) | rgba->g;
	rgba->b = (rgba->b << 8) | rgba->b;
	rgba->a = 0xFFFF;

	printf("rgba (%X,%X,%X,%X)\n",
		rgba->r, rgba->g, rgba->b, rgba->a);
	
	return 0;

}	/* gcpYCC2RGBA */


/*  Converts an RGBA value into the YCC (Kodak PhotoCD) colorspace.
 */
int gcpRGBA2YCC(ggi_visual_t vis,
		const gcp_RGBAcolor *rgba, gcp_YCCcolor *ycc)
{
	ggi_float r,g,b;
	ggi_float y,c1,c2;

/* Formula:
 *
 *  Red  =f(red')
 *  Green=f(Green')
 *  Blue =f(Blue')
 *  where { f(t)=-1.099*abs(t)^0.45+0.999 if t<=-0.018
 *	  { f(t)=4.5*t if -0.018<t<0.018
 * 	  { f(t)=1.099*t^0.45-0.999 if t>=0.018
 *
 * Y'  =  0.299 * Red + 0.587 * Green + 0.114 * Blue
 * C1' = -0.299 * Red - 0.587 * Green + 0.886 * Blue
 * C2' =  0.701 * Red - 0.587 * Green - 0.114 * Blue
 *
 * Y   = (255 / 1.402) * Y'
 * C1  = 111.40 * C1'  + 156
 * C2  = 135.64 * C2'  + 137
 */

#if 0

#define f(x)						\
	if (x <= -0.018) {				\
		x = -1.099 * (fabs(x)^0.45) + 0.999;	\
	} else if (x >= 0.018) {			\
		x = 1.099 * (x^0.45) - 0.999;		\
	} else {					\
		x = 4.5 * x;				\
	}	/* if */

	r = (ggi_float)rgba->r;
	g = (ggi_float)rgba->g;
	b = (ggi_float)rgba->b;

	gcpGetRGBAGamma(vis, &r, &g, &b);

	r /= 255.0f;
	g /= 255.0f;
	b /= 255.0f;

	f(r); f(g); f(b);
#else
	r = (ggi_float)(rgba->r / 255.0f);
	g = (ggi_float)(rgba->g / 255.0f);
	b = (ggi_float)(rgba->b / 255.0f);
#endif

	y  =  0.299 * r + 0.587 * g + 0.114 * b;
	c1 = -0.299 * r - 0.587 * g + 0.886 * b;
	c2 =  0.701 * r - 0.587 * g - 0.114 * b;

	y  = (255 / 1.402) * y;
	c1 = 111.40 * c1 + 156;
	c2 = 135.64 * c2 + 137;

	ycc->y  = (uint16_t)y;
	ycc->c1 = (uint16_t)c1;
	ycc->c2 = (uint16_t)c2;

	ycc->y  = (ycc->y  << 8) | ycc->y;
	ycc->c1 = (ycc->c1 << 8) | ycc->c1;
	ycc->c2 = (ycc->c2 << 8) | ycc->c2;

	return 0;

}	/* gcpRGBA2YCC */
