/* $Id: util.c,v 1.5 2005/07/30 16:57:37 cegger Exp $
******************************************************************************

   Utility functions for the Snazzy Config Manager.
  
   Copyright (c) 1999 Andrew Apted  [andrew@ggi-project.org]
  
   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
  
******************************************************************************
*/

#include <stdio.h>
#include <stdlib.h>

#include "snazzy.h"


unsigned char *decode_image(unsigned char *img, int width, int height)
{
	unsigned char *buf, *cur;

	int x, y;
	int pix=0, count=0;
	
	buf = malloc((size_t)(width * height));

	if (buf == NULL) {
		return NULL;
	}

	cur = buf;

	for (y=0; y < height; y++)
	for (x=0; x < width;  x++) {

		if (count > 0) {
			*cur++ = pix;
			count--;
			continue;
		}
		
		if (*img > 96) {
			count = (*img++) - 97;
		}
		
		if (! *img) {
			/* Odd, an error in the image */
			free(buf);
			return NULL;
		}

		*cur++ = pix = (*img++) - 64;
	}

	return buf;
}


/* ---------------------------------------------------------------------- */


static uint8_t line_buffer[8192];

void draw_image(ggi_visual_t vis, int x, int y, int w, int h,
		unsigned char *img, int ix, int iy, int iw, int ih,
		ggi_pixel *colors)
{
	int ww;

	ggi_mode mode;

	ix %= iw;
	iy %= ih;

	ggiGetMode(vis, &mode);

	for (; h > 0; h--, y++, iy = (iy+1) % ih) {

		uint8_t  *buf1 = (uint8_t  *) line_buffer;
		uint16_t *buf2 = (uint16_t *) line_buffer;
		uint32_t *buf4 = (uint32_t *) line_buffer;

		switch ((GT_SIZE(mode.graphtype) + 7) / 8) {

		case 1: for (ww=0; ww < w; ww++) {
				*buf1++ = colors[img[iy*iw + (ix+ww)%iw]];
			}
			break;

		case 2: for (ww=0; ww < w; ww++) {
				*buf2++ = colors[img[iy*iw + (ix+ww)%iw]];
			}
			break;

		case 3: for (ww=0; ww < w; ww++) {
				ggi_pixel q = colors[img[iy*iw + (ix+ww)%iw]];
				*buf1++ = q; q >>= 8;
				*buf1++ = q; q >>= 8;
				*buf1++ = q;
			}
			break;

		case 4: for (ww=0; ww < w; ww++) {
				*buf4++ = colors[img[iy*iw + (ix+ww)%iw]];
			}
			break;
		}

		ggiPutHLine(vis, x, y, w, line_buffer);
	}
#if 0
	for (hh=0; hh < h; hh++) {
	for (ww=0; ww < w; ww++) {

		/* No points here for speed :-> */
		
		ggiPutPixel(vis, x+ww, y+hh, 
			colors[img[((iy+hh) % ih) * iw + (ix+ww) % iw]]);
	}
#endif
}

void draw_char (ggi_visual_t vis, int x, int y, int c, 
		unsigned char *font, int fw, int fh, ggi_pixel *colors)
{
	int ww, hh;
	int fx, fy;

	if (c < 0x20) {
		return;
	} else if (c < 0x80) {
		c -= 0x20;
	} else if (c < 0xA0) {	
		return;
	} else if (c < 0x100) {
		c -= 0x40;
	} else {
		return;
	}
	
	fx = (c % 16) * fw;
	fy = (c / 16) * fh;
	
	for (hh=0; hh < fh; hh++)
	for (ww=0; ww < fw; ww++) {

		/* No points here for speed :-> */
		
		int pix = font[(fy+hh) * 16 * fw + fx+ww];

		if (pix) {
			ggiPutPixel(vis, x+ww, y+hh, colors[pix-1]);
		}
	}
}

void draw_string(ggi_visual_t vis, int x, int y, const unsigned char *s, 
		 unsigned char *font, int fw, int fh, int step,
		 ggi_pixel *colors)
{
	for (; *s; s++, x += (fw + step)) {
	
		draw_char(vis, x, y, *s, font, fw, fh, colors);
	}
}


/* ---------------------------------------------------------------------- */


static
void draw_snazzy_char (ggi_visual_t vis, int x, int y, 
			int x_scale, int y_scale, int c, 
			unsigned char *font, int fw, int fh,
			unsigned char *img, int iw, int ih, 
			ggi_pixel *colors)
{
	int ww, hh;
	int fx, fy;
	int dx, dy;

	if (c < 0x20) {
		return;
	} else if (c < 0x80) {
		c -= 0x20;
	} else if (c < 0xA0) {	
		return;
	} else if (c < 0x100) {
		c -= 0x40;
	} else {
		return;
	}
	
	fx = (c % 16) * fw;
	fy = (c / 16) * fh;
	
	for (hh=0; hh < fh; hh++)
	for (ww=0; ww < fw; ww++) {

		/* This ain't no speed demon either :-> */
		
		int pix = font[(fy+hh) * 16 * fw + fx+ww];

		if (! pix) {
			continue;
		}

		for (dx=0; dx < x_scale; dx++)
		for (dy=0; dy < y_scale; dy++) {

			int sx = x + (fh-1-hh)*x_scale + dx;
			int sy = y + (     ww)*y_scale + dy;

			ggi_pixel q = colors[(pix == 1) ? 0 : 1 +
					img[(sy%ih)*iw + (sx%iw)]];

			ggiPutPixel(vis, sx, sy, q);
		}
	}
}

void draw_snazzy_string(ggi_visual_t vis, int x, int y,
			int x_scale, int y_scale,
			const unsigned char *s, unsigned char *font,
			int fw, int fh, unsigned char *img,
			int iw, int ih, ggi_pixel *colors)
{
	for (; *s; s++, y += fw * y_scale) {
	
		draw_snazzy_char(vis, x, y, x_scale, y_scale, *s, 
				 font, fw, fh, img, iw, ih, colors);
	}
}
