/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4 c-style: "K&R" -*- */

/*
   libgpiv - library for Particle Image Velocimetry

   Copyright (C) 2002, 2003, 2004 Gerber van der Graaf

   This file is part of libgpiv.
   Libgpiv is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  



-------------------------------------------------------------------------------
FILENAME:                imgproc.c
LIBRARY:                 libgpiv:
EXTERNAL FUNCTIONS:
                        gpiv_imgproc_mktstimg
                        gpiv_imgproc_smooth
                        gpiv_imgproc_highlow
                        gpiv_imgproc_clip
                        gpiv_imgproc_getbit
                        gpiv_imgproc_fft
                        gpiv_imgproc_lowpass
                        gpiv_imgproc_highpass
                        gpiv_imgproc_correlate
                        gpiv_imgproc_convolve

LAST MODIFICATION DATE:  $Id: imgproc.c,v 1.1 2006/01/31 13:30:13 gerber Exp $
 --------------------------------------------------------------------------- */

#include <gpiv.h>


void
gpiv_imgproc_mktstimg(GpivImagePar image_par,
                      GpivImageProcPar image_proc_par,
                      guint16 **img_out
                      )
/* ----------------------------------------------------------------------------
 * Generates a test image for filtering and manipulating
 * with band widths of window in hor. dir. and 
 * window + threshold in vert. dir.
 */
{
    int i, j, k, l;
    int M = image_par.nrows;
    int N = image_par.ncolumns;
    int window = image_proc_par.window;
    int threshold = image_proc_par.threshold;
    int bit = image_proc_par.bit;


    for (i = 0; i < M; i++) {
        for (j = 0; j < N; j++) {
            img_out[i][j] = 100;
        }
    }

    for (k = 0; k < M; k = k + window + k * 0.25) {
        for (l = 0; l < N; l = l + threshold + l * 0.25) {
            for (i = 0; i < (window + k * 0.25)/2; i++) {
                for (j = 0; j < (threshold + l * 0.25) / 2; j++) {
/*                     gpiv_warning("i=%d j=%d j_end=%d k=%d l=%d i+k=%d j+l=%d",  */
/*                                  i, j, (threshold + l *bit)/2, k, l, i+k, j+l); */
                    if (M/2 + (i+k) < M && N/2 + (j+l) < N 
                        && M/2 - (i+k) >= 0 && N/2 - (j+l) >=0
                        ) {
/*                         gpiv_warning("passed"); */
                            img_out[M/2 + (i+k)][N/2 + (j+l)] = 200 /* ~0 */;
                            img_out[M/2 - (i+k)][N/2 + (j+l)] = 200 /* ~0 */;
                            img_out[M/2 + (i+k)][N/2 - (j+l)] = 200 /* ~0 */;
                            img_out[M/2 - (i+k)][N/2 - (j+l)] = 200 /* ~0 */;
                        /*             img_out[i][j] << GPIV_MAX_IMG_DEPTH - 1; */
                    }
                }
            }
        }
    }
    
}



void
gpiv_imgproc_smooth(GpivImagePar image_par, 
                    GpivImageProcPar image_proc_par, 
                    guint16 **img_in,
                    guint16 **img_out
                    )
/*-----------------------------------------------------------------------------
 * Smoothing filter by taking mean value of surrounding 
 * window x window pixels
 */
{
    long int img_sum = 0;
    float img_mean = 0.0;
    int i, j, k, l, count = 0;
    int nrows = image_par.nrows;
    int ncolumns = image_par.ncolumns;
    int window = image_proc_par.window;

     
     for (i = 0; i < nrows; i++) {
         for (j = 0; j < ncolumns; j++) {
             img_mean = 0.0;
             img_sum = 0;
             count = 0;
             for (k = -(window-1)/2; k <= (window-1)/2; k++) {
                 if ((i+k >= 0) && (i+k < nrows)) {
                     for (l = -(window-1)/2; l <= (window-1)/2; l++) {
                         if ((j+l >= 0) && (j+l < ncolumns)) {
                             img_sum += (long) img_in[i+k][j+l];
                             count++;
                         }
                     }
                 }
             }
             img_mean = (float) img_sum /(float) count;

             if (image_proc_par.operator == GPIV_EQUAL) {
                 img_out[i][j] = img_mean;
             } else if (image_proc_par.operator == GPIV_SUBTRACT) {
                 img_out[i][j] = img_in[i][j] - img_mean;
             } else if (image_proc_par.operator == GPIV_ADD) {
                 img_out[i][j] = img_in[i][j] + img_mean;
             } else if (image_proc_par.operator == GPIV_MULTIPLY) {
                 img_out[i][j] = img_in[i][j] * img_mean;
             } else if (image_proc_par.operator == GPIV_DIVIDE) {
                 img_out[i][j] = img_in[i][j] / img_mean;
             } else {
                 gpiv_error("gpiv_imgproc_smooth: unvalid operator %d", 
                            image_proc_par.operator);
             }
         }
     }
     
     
}



void
gpiv_imgproc_highlow(GpivImagePar image_par,
                     GpivImageProcPar image_proc_par,
                     guint16 **img_in, 
                     guint16 **img_out
                     )
/* ----------------------------------------------------------------------------
 * High-low filter to maximize contrast by stretching pixel values
 * to local max and min within window x window area
 */
{
    int i, j, k, l, img_top =  (img_top << image_par.depth) - 1;
    guint16 img_max = 0, img_min = (1 << image_par.depth);
    guint16 **max, **min, **max_smooth, **min_smooth;
    float denomerator, factor;

    int nrows = image_par.nrows;
    int ncolumns = image_par.ncolumns;
    int window = image_proc_par.window;

    max = gpiv_matrix_guint16(nrows, ncolumns);
    min = gpiv_matrix_guint16(nrows, ncolumns);
    max_smooth = gpiv_matrix_guint16(nrows, ncolumns);
    min_smooth = gpiv_matrix_guint16(nrows, ncolumns);



/*
 * Calculate local min and max for each pixel within window
 */

    for (i = 0; i < nrows; i++) {
        for (j = 0; j < ncolumns; j++) {
            img_max = 0; 
            img_min = img_top;
            for (k = -(window-1)/2; k <= (window-1)/2; k++) {
                if ((i+k >= 0) && (i+k < nrows)) {
                    for (l = -(window-1)/2; l <= (window-1)/2; l++) {
                        if ((j+l >= 0) && (j+l < ncolumns)) {
                            
                            if (img_in[i+k][j+l] > img_max) {
                                img_max = img_in[i+k][j+l];
                            }
                            
                            if (img_in[i+k][j+l] < img_min) {
                                img_min = img_in[i+k][j+l];
                            }
                        }
                    }
                }
            }
            max[i][j] = img_max;
            min[i][j] = img_min;
            img_max = 0; 
            img_min = img_top;
        }
    }



/*
 * Spatial interpolation/smoothing of min and max
 */

    image_proc_par.operator = GPIV_EQUAL;
    gpiv_imgproc_smooth(image_par, image_proc_par, max, max_smooth);
    gpiv_imgproc_smooth(image_par, image_proc_par, min, min_smooth);
    

/*
 * Subtracting min_smooth from each pixel value (background), stretching to 
 * max_smooth to top image value
 */

    for (i = 0; i < nrows; i++) {
        for (j = 0; j < ncolumns; j++) {
            denomerator = img_top - ((int) max_smooth[i][j] 
                                     - (int) min_smooth[i][j]);
            if (denomerator == 0) {
                /* 	  factor = 0; */
                /* 	img_out[i][j] = max_smooth[i][j]; */
                img_out[i][j] = img_in[i][j];
            } else {
                factor = (float) img_top/(float) denomerator;
                img_out[i][j] = (img_in[i][j] - min_smooth[i][j]) * factor;
            }
        }
    }


    gpiv_free_matrix_guint16(max);
    gpiv_free_matrix_guint16(min); 
    gpiv_free_matrix_guint16(max_smooth);
    gpiv_free_matrix_guint16(min_smooth); 

}



void
gpiv_imgproc_clip(GpivImagePar image_par,
                  GpivImageProcPar image_proc_par,
                  guint16 **img_in, 
                  guint16 **img_out
                  )
/* ----------------------------------------------------------------------------
 * Sets all pixel values lower than threshold to zero
 */
{
    int i, j;
    int nrows = image_par.nrows;
    int ncolumns = image_par.ncolumns;
    int threshold = image_proc_par.threshold;
    
    for (i = 0; i < nrows; i++) {
        for (j = 0; j < ncolumns; j++) {
	fflush(stdout);
            if (img_in[i][j] <= threshold) {
                img_out[i][j] = 0;
            } else {
                img_out[i][j] = img_in[i][j];
            }
/*             printf(" ==> img_out[%d][%d] = %d\n",i, j, img_out[i][j]); */
        }
    }

}


void
gpiv_imgproc_getbit(GpivImagePar image_par,
                    GpivImageProcPar image_proc_par,
                    guint16 **img_in, 
                    guint16 **img_out
                    )
/* ----------------------------------------------------------------------------
 * Pointer operation to get the N least significant bits and moves them to 
 * most the significant bits
 */
{
    int i, j;
    int nrows = image_par.nrows;
    int ncolumns = image_par.ncolumns;
    int depth = image_par.depth = GPIV_MAX_IMG_DEPTH;
    int bit = image_proc_par.bit;
    unsigned char mask = ~0;

    mask = (mask & ~0) >> (depth - bit);
/*     gpiv_warning("getbit:: bit = %d depth = %d ==> mask = %X",  */
/*                  bit, depth, mask); */
    for (i = 0; i < nrows; i++) {
        for (j = 0; j < ncolumns; j++) {
            img_out[i][j] = (img_in[i][j] & mask);
            img_out[i][j] = img_in[i][j] << (depth - bit);
        }
    }

}


/* char * */
/* gpiv_imgproc_invfft(GpivImagePar image_par, */
/*                GpivImageProcPar image_proc_par, */
/*                guint16 **img_in,  */
/*                guint16 **img_out) */
/* 
 * Inverse Fast Fourier Transformation
 */
/* { */
/*     char *err_msg = NULL; */
/*     gpiv_fread_fftw_wisdom(-1); */


/*     pinv = fftw_plan_dft_r2c_2d(M, N, FFTW_COMPLEX_TO_REAL, */
/* 			       FFTW_MEASURE | FFTW_USE_WISDOM); */



/*     gpiv_fwrite_fftw_wisdom(-1); */
/* } */


char *
gpiv_imgproc_fft(GpivImagePar img_par, 
                 GpivImageProcPar img_proc_par,
                 guint16 **img_in, 
                 guint16 **img_out
                 /*             guint16 **re_img_out */
                 /*             guint16 **im_img2_out */
                 )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Fast Fourier Transformation  on an image
 *
 * INPUT:
 *     nrows: number of rows (hight) of images
 *     ncolumns: number of columns (width) of images
 *     img-in: first input image
 *
 * OUTPUT:
 *     img_out: resulting output image 
 -----------------------------------------------------------------------------*/
{
    char *err_msg = NULL;
/*     int img_size_0 = ZEROPAD_FACT  */
/*         * gpiv_max(image_par.nrows, image_par.ncolumns); */
    int i, j, ij;
    int M = img_par.nrows, N = img_par.ncolumns;

    fftw_real **a, **b;
    fftw_complex *A, **B;
    rfftwnd_plan p, pinv;
     fftw_real scale = 1.0 / (M * N);


    a = gpiv_fftw_real_matrix(M, 2 * (N / 2 + 1));
    A = (fftw_complex *) &a[0][0];

    b = gpiv_fftw_real_matrix(M, N);
    B = gpiv_fftw_complex_matrix(M, N /2 + 1);

    for (i = 0; i < M; i++) {
	for (j = 0; j < N; j++) {
	    a[i][j] = (double) img_in[i][j];
	}
    }
    

/*
 * import_wisdom from string and make plans
 * FFT transform to get A and B from a and b;
 */
    gpiv_fread_fftw_wisdom(1);
    gpiv_fread_fftw_wisdom(-1);
/* BUGFIX rfftw2d_create_plan or fftw_plan_dft_r2c_2d*/
    p = rfftw2d_create_plan(M, N, FFTW_REAL_TO_COMPLEX,
                            FFTW_MEASURE | FFTW_USE_WISDOM | FFTW_IN_PLACE
                            );

    rfftwnd_one_real_to_complex(p, &a[0][0], NULL);

    gpiv_fwrite_fftw_wisdom(1);
    rfftwnd_destroy_plan(p);


    for (i = 0; i < M; i++) {
        for (j = 0; j < N/2+1; j++) {
            ij = i*(N/2+1) + j;
            B[i][j].re = (A[ij].re) * scale;
            B[i][j].im = (A[ij].im) * scale;
        }
    }
/*
 * import_wisdom from string and make plans
 * inverse transform to get c, the convolution of a and b;
 * this has the side effect of overwriting C
 */
    pinv = /* fftw_plan_dft_r2c_2d */ rfftw2d_create_plan
        (M, N, FFTW_COMPLEX_TO_REAL,
                               FFTW_MEASURE| FFTW_USE_WISDOM );

    rfftwnd_one_complex_to_real(pinv, &B[0][0], &b[0][0]);

    gpiv_fwrite_fftw_wisdom(-1);
    rfftwnd_destroy_plan(pinv);

    for (i = 0; i < M; i++) {
        for (j = 0; j < N; j++) {
            img_out[i][j] = /* (char) */ (guint16) b[i][j];
        }
    }


    gpiv_free_fftw_real_matrix(a);
    gpiv_free_fftw_real_matrix(b);
    gpiv_free_fftw_complex_matrix(B);
}



char *
gpiv_imgproc_lowpass(GpivImagePar img_par, 
                     GpivImageProcPar img_proc_par,
                     guint16 **img_in, 
                     guint16 **img_out
                     )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Lowpass filter on an image
 *
 * INPUT:
 *     nrows: number of rows (hight) of images
 *     ncolumns: number of columns (width) of images
 *     img-in: first input image
 *
 * OUTPUT:
 *     img_out: resulting output image 
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL;
    int i, j, ij;
    int M = 2 * img_par.nrows, N = 2 * img_par.ncolumns;
    int window = img_proc_par.window;
    fftw_complex **a, **b;
    fftw_complex **A, **B;
    fftwnd_plan p = NULL, pinv = NULL;
    fftw_real scale = 1.0 / (M * N);
    double mean_img;


    gpiv_fread_fftw_wisdom(1);
    gpiv_fread_fftw_wisdom(-1);

    a = gpiv_fftw_complex_matrix(M, N);
/* gpiv_gdouble_matrix(M, 2 * (N / 2 + 1)); */
    A = gpiv_fftw_complex_matrix(M, N);
/* (fftw_complex *) &a[0][0]; */

/*     b = gpiv_gdouble_matrix(M, N); */
    b = gpiv_fftw_complex_matrix(M, N);
    B = gpiv_fftw_complex_matrix(M, N);
/* gpiv_fftw_complex_matrix(M, N /2 + 1); */

    for (i = 0; i < img_par.nrows; i++) {
	for (j = 0; j < img_par.ncolumns; j++) {
	    a[i][j].re = (double) img_in[i][j];
	    a[i][j].im = 0.0;
	}
    }
    

/*
 * import_wisdom from string and make plans
 * FFT transform to get A and B from a and b;
 */
    
 /* FFTW_REAL_TO_COMPLEX */
 /*| FFTW_USE_WISDOM  | FFTW_IN_PLACE */
    p = fftw2d_create_plan(M, N, FFTW_FORWARD,
                           FFTW_MEASURE | FFTW_USE_WISDOM
                           );

/*     rfftwnd_one_real_to_complex(p, &a[0][0], &A[0][0]); */
    fftwnd_one(p, &a[0][0], &A[0][0]);
    gpiv_fwrite_fftw_wisdom(1);
    fftwnd_destroy_plan(p);

/*     mean_img = A[0][0].re; */
    



/* 
 *lowpass filter: passing data data from 0,..,window/2
 */

    for (i = 0; i < M; i++) {
        for (j = 0; j < N; j++) {
            B[i][j].re = 0.0;
            B[i][j].im = 0.0;
        }
    }
    
    B[0][0].re = A[0][0].re * scale;
    B[0][0].im = A[0][0].im * scale;
    
    
    
    for (j = 0, i = 1; i <= window; i++) {

/*         gpiv_warning("lowpass:: A.re[%d][%d] = %f A.im[%d][%d] = %f " */
/*                      "=> A.re[%d][%d] = %f A.im[%d][%d] = %f",  */
/*                      i, j, A[i][j].re, i, j, A[i][j].im, */
/*                      M - i, j, A[M - i][j].re,  */
/*                      M - i, j, A[M - i][j].im); */
        
        B[i][j].re = A[i][j].re * scale;
        B[i][j].im = A[i][j].im * scale;
        B[M - i][j].re = (A[M - i][j].re) * scale;
        B[M - i][j].im = (A[M - i][j].im) * scale;
    }



    for (i = 0, j = 1; j <= window; j++) {

/*         gpiv_warning("lowpass:: A.re[%d][%d] = %f A.im[%d][%d] = %f " */
/*                      "=> A.re[%d][%d] = %f A.im[%d][%d] = %f",  */
/*                      i, j, A[i][j].re, i, j, A[i][j].im, */
/*                      i, N - j, A[i][N - j].re,  */
/*                      i, N - j, A[i][N - j].im); */
        
        B[i][j].re = A[i][j].re * scale;
        B[i][j].im = A[i][j].im * scale;
        B[i][N - j].re = (A[i][N - j].re) * scale;
        B[i][N - j].im = (A[i][N - j].im) * scale;
    }
/*         } */


    if (window >= 1) {
        for (i = 1; i <= window; i++) {
            for (j = 1; j <= window; j++) {
                
/*             gpiv_warning("lowpass:: A.re[%d][%d] = %f A.im[%d][%d] = %f " */
/*                          "=> A.re[%d][%d] = %f A.im[%d][%d] = %f",  */
/*                          i, j, A[i][j].re, i, j, A[i][j].im, */
/*                          M - i, N - j, A[M - i][N - j].re,  */
/*                          M - i, N - j, A[M - i][N - j].im); */
            
                B[i][j].re = A[i][j].re * scale;
                B[i][j].im = A[i][j].im * scale;
                
                B[M - i][j].re = (A[M - i][j].re) * scale;
                B[M - i][j].im = (A[M - i][j].im) * scale;
                
                B[i][N - j].re = (A[i][N - j].re) * scale;
                B[i][N - j].im = (A[i][N - j].im) * scale;
                
                B[M - i][N - j].re = (A[M - i][N - j].re) * scale;
                B[M - i][N - j].im = (A[M - i][N - j].im) * scale;
            }
        }
    }



/*     for (i = 0; i < M; i++) { */
/*         for (j = 0; j < N; j++) { */
/*             if (B[i][j].re == 0.0 && B[i][j].im == 0.0) */
/*                 gpiv_warning("B[%d][%d].re = 0.0 B[%d][%d].im = 0.0",  */
/*                              i, j, i, j);  */
/*         } */
/*     } */




/*
 * import_wisdom from string and make plans
 * inverse transform to get c, the convolution of a and b;
 * this has the side effect of overwriting C
 */
    pinv = fftw2d_create_plan(M, N, FFTW_BACKWARD/* FFTW_COMPLEX_TO_REAL */,
                              FFTW_MEASURE | FFTW_USE_WISDOM
                              );

/*     fftwnd_one_complex_to_real(pinv, &B[0][0], &b[0][0]); */
    fftwnd_one(pinv, &B[0][0], &b[0][0]);
    
    
    gpiv_fwrite_fftw_wisdom(-1);
    rfftwnd_destroy_plan(pinv);
    
    
    
    for (i = 0; i < img_par.nrows; i++) {
        for (j = 0; j < img_par.ncolumns; j++) {
/*             if (b[i][j].re == 0.0 || b[i][j].im == 0.0) { */
/*                 gpiv_warning("inv FT image:: b[%d][%d].re=%f b[%d][%d].im=%f", */
/*                              i, j, b[i][j].re, i, j, b[i][j].im); */
/*             } */
            img_out[i][j] = /* (char)  */(guint16) b[i][j].re;
        }
    }






/*     gpiv_free_fftw_real_matrix(a); */
/*     gpiv_free_fftw_real_matrix(b); */
    gpiv_free_fftw_complex_matrix(a);
    gpiv_free_fftw_complex_matrix(b);
    gpiv_free_fftw_complex_matrix(B);
}



char *
gpiv_imgproc_highpass(GpivImagePar img_par, 
                      GpivImageProcPar img_proc_par,
                      guint16 **img_in, 
                      guint16 **img_out
                      )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Highpass filter on an image; 
 *     passing data from M - window,..,M/2, N - window,..,N/2
 *
 * INPUT:
 *     nrows: number of rows (hight) of images
 *     ncolumns: number of columns (width) of images
 *     img-in: first input image
 *
 * OUTPUT:
 *     img_out: resulting output image 
 ----------------------------------------------------------------------------*/
{
    char *err_msg = NULL;
/*     int img_size_0 = ZEROPAD_FACT  */
/*         * gpiv_max(image_par.nrows, image_par.ncolumns); */
    int i, j, ij;
    int M = 2 * img_par.nrows, N = 2 * img_par.ncolumns;
    int window = img_proc_par.window;
/*      fftw_real **a, **b; */
    fftw_complex **a, **b;
    fftw_complex **A, **B;
    rfftwnd_plan p = NULL, pinv = NULL;
    fftw_real scale = 1.0 / (M * N);
    double mean_img;


    gpiv_warning("highpass: 0, window = %d", window);
    gpiv_fread_fftw_wisdom(1);
    gpiv_fread_fftw_wisdom(-1);

    a = gpiv_fftw_complex_matrix(M, N);
/* gpiv_fftw_real_matrix(M, 2 * (N / 2 + 1)); */
    A = gpiv_fftw_complex_matrix(M, N);
/* (fftw_complex *) &a[0][0]; */

/*     b = gpiv_fftw_real_matrix(M, N); */
    b = gpiv_fftw_complex_matrix(M, N);
    B = gpiv_fftw_complex_matrix(M, N);
/* gpiv_fftw_complex_matrix(M, N /2 + 1); */

    for (i = 0; i < img_par.nrows; i++) {
	for (j = 0; j < img_par.ncolumns; j++) {
	    a[i][j].re = (double) img_in[i][j];
	    a[i][j].im = 0.0;
	}
    }
    

/*
 * import_wisdom from string and make plans
 * FFT transform to get A and B from a and b;
 */

 /* FFTW_REAL_TO_COMPLEX */
 /*| FFTW_USE_WISDOM  | FFTW_IN_PLACE */
    p = fftw2d_create_plan(M, N, FFTW_FORWARD,
                            FFTW_MEASURE | FFTW_USE_WISDOM
                            );

/*     rfftwnd_one_real_to_complex(p, &a[0][0], &A[0][0]); */
    fftwnd_one(p, &a[0][0], &A[0][0]);
    gpiv_fwrite_fftw_wisdom(1);
    rfftwnd_destroy_plan(p);

/*     mean_img = A[0][0].re; */
    


    for (i = 0; i < M; i++) {
        for (j = 0; j < N; j++) {
            B[i][j].re = 0.0;
            B[i][j].im = 0.0;
        }
    }
    
    
    
    if (window == M/2 || window == N/2) {
        gpiv_warning("highpass: window >= M/2 && window >= N/2: A.re[0][0] = %f A.im[0][0] = %f ",
                     A[0][0].re, A[0][0].im);
        B[0][0].re = A[0][0].re * scale;
        B[0][0].im = A[0][0].im * scale;
    }

    if (window == N/2) {
    for (j = 0, i = M/2 - window + 1; i <= M/2; i++) {
            
/*         gpiv_warning("highpass: window >= N/2: A.re[%d][%d] = %f A.im[%d][%d] = %f " */
/*                      "=> A.re[%d][%d] = %f A.im[%d][%d] = %f",  */
/*                      i, j, A[i][j].re, i, j, A[i][j].im, */
/*                      M - i, j, A[M - i][j].re,  */
/*                      M - i, j, A[M - i][j].im); */
        
        B[i][j].re = A[i][j].re * scale;
        B[i][j].im = A[i][j].im * scale;
        B[M - i][j].re = A[M - i][j].re * scale;
        B[M - i][j].im = A[M - i][j].im * scale;
    }
    }


    if (window == M/2) {
    for (i = 0, j = N/2 - window + 1; j <= N/2; j++) {

/*         gpiv_warning("highpass: window >= M/2: A.re[%d][%d] = %f A.im[%d][%d] = %f " */
/*                      "=> A.re[%d][%d] = %f A.im[%d][%d] = %f",  */
/*                      i, j, A[i][j].re, i, j, A[i][j].im, */
/*                      i, N - j, A[i][N - j].re,  */
/*                      i, N - j, A[i][N - j].im); */
        
        B[i][j].re = A[i][j].re * scale;
        B[i][j].im = A[i][j].im * scale;
        B[i][N - j].re = A[i][N - j].re * scale;
        B[i][N - j].im = A[i][N - j].im * scale;
    }
    }


        for (i = M/2 - window; i <= M/2; i++) {
            for (j = N/2 - window; j <= N/2; j++) {
                
                if (i >= 1 && j >= 1) {
/*                     gpiv_warning("highpass: passing data: A.re[%d][%d] = %f A.im[%d][%d] = %f " */
/*                                  "=> A.re[%d][%d] = %f A.im[%d][%d] = %f",  */
/*                                  i, j, A[i][j].re, i, j, A[i][j].im, */
/*                                  M - i, N - j, A[M - i][N - j].re,  */
/*                                  M - i, N - j, A[M - i][N - j].im); */
                    
                    B[i][j].re = A[i][j].re * scale;
                    B[i][j].im = A[i][j].im * scale;
                    
                    B[M - i][j].re = A[M - i][j].re * scale;
                    B[M - i][j].im = A[M - i][j].im * scale;
                    
                    B[i][N - j].re = A[i][N - j].re * scale;
                    B[i][N - j].im = A[i][N - j].im * scale;
                    
                    B[M - i][N - j].re = A[M - i][N - j].re * scale;
                    B[M - i][N - j].im = A[M - i][N - j].im * scale;
                }
            }
        }




/*     for (i = 0; i < M; i++) { */
/*         for (j = 0; j < N; j++) { */
/*             if (B[i][j].re == 99.9 && B[i][j].im == 99.9) */
/*                 gpiv_warning("B[%d][%d].re = 99.9 ",  */
/*                              i, j);  */
/*         } */
/*     } */




/*
 * import_wisdom from string and make plans
 * inverse transform to get c, the convolution of a and b;
 * this has the side effect of overwriting C
 */
    pinv = fftw2d_create_plan(M, N, FFTW_BACKWARD/* FFTW_COMPLEX_TO_REAL */,
                              FFTW_MEASURE | FFTW_USE_WISDOM
                              );

/*     fftwnd_one_complex_to_real(pinv, &B[0][0], &b[0][0]); */
    fftwnd_one(pinv, &B[0][0], &b[0][0]);


    gpiv_fwrite_fftw_wisdom(-1);
    rfftwnd_destroy_plan(pinv);



    for (i = 0; i < img_par.nrows; i++) {
        for (j = 0; j < img_par.ncolumns; j++) {
/*             if (b[i][j].re == 0.0 || b[i][j].im == 0.0) { */
/*                 gpiv_warning("inv FT image:: b[%d][%d].re=%f b[%d][%d].im=%f", */
/*                              i, j, b[i][j].re, i, j, b[i][j].im); */
/*             } */
            img_out[i][j] = /* (char) */ (guint16) b[i][j].re;
        }
    }






/*     gpiv_free_fftw_real_matrix(a); */
/*     gpiv_free_fftw_real_matrix(b); */
    gpiv_free_fftw_complex_matrix(a);
    gpiv_free_fftw_complex_matrix(b);
    gpiv_free_fftw_complex_matrix(B);
}



void
gpiv_imgproc_correlate(GpivImagePar img_par, 
                       GpivImageProcPar img_proc_par,
                       guint16 **img1_in, 
                       guint16 **img2_in, 
                       guint16 **img_out
                       )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Correlates two images
 *
 * INPUT:
 *     nrows: number of rows (hight) of images
 *     ncolumns: number of columns (width) of images
 *     img1-in: first input image
 *     img2_in: second input image
 *
 * OUTPUT:
 *     img_out: resulting output image 
 */
{
    int i, j, ij;
    int M = img_par.nrows, N = img_par.ncolumns;

    fftw_real **a, **b, **c;
    fftw_complex *A, *B, **C;
    rfftwnd_plan p, pinv;
    fftw_real scale = 1.0 / (M * N);

    gpiv_fread_fftw_wisdom(1);
    gpiv_fread_fftw_wisdom(-1);

    a = gpiv_fftw_real_matrix(M, 2 * (N / 2 + 1));
    A = (fftw_complex *) &a[0][0];
    b = gpiv_fftw_real_matrix(M, 2 * (N / 2 + 1));
    B = (fftw_complex *) &b[0][0];
    c = gpiv_fftw_real_matrix(M, N);
    C = gpiv_fftw_complex_matrix(M, N /2 + 1);

    for (i = 0; i < M; i++) {
	for (j = 0; j < N; j++) {
	    a[i][j] = (double) img1_in[i][j];
	    b[i][j] = (double) img2_in[i][j];
	}
    }
    

/*
 * import_wisdom from string and make plans
 * FFT transform to get A and B from a and b;
 */
    p = rfftw2d_create_plan(M, N, FFTW_REAL_TO_COMPLEX,
                            FFTW_MEASURE | FFTW_USE_WISDOM | FFTW_IN_PLACE
                            );

    rfftwnd_one_real_to_complex(p, &a[0][0], NULL);
    rfftwnd_one_real_to_complex(p, &b[0][0], NULL);


    gpiv_fwrite_fftw_wisdom(1);
    rfftwnd_destroy_plan(p);


/*
 * Correlates FFT(b) = B with FFT(a) = A to C following
 * B * conjugate(A) result in correct sign of displacements!
 */
    for (i = 0; i < M; i++) {
	for (j = 0; j < N / 2 + 1; j++) {
	    ij = i * (N / 2 + 1) + j;
	    C[i][j].re = (B[ij].re * A[ij].re + B[ij].im * A[ij].im) * scale;
	    C[i][j].im = (B[ij].im * A[ij].re - B[ij].re * A[ij].im) * scale;
	}
    }


/*
 * import_wisdom from string and make plans
 * inverse transform to get c, the convolution of a and b;
 * this has the side effect of overwriting C
 */
    pinv = rfftw2d_create_plan(M, N, FFTW_COMPLEX_TO_REAL,
                               FFTW_MEASURE| FFTW_USE_WISDOM );

    rfftwnd_one_complex_to_real(pinv, &C[0][0], &c[0][0]);

    gpiv_fwrite_fftw_wisdom(-1);
    rfftwnd_destroy_plan(pinv);

    for (i = 0; i < M; i++) {
        for (j = 0; j < N; j++) {
            img_out[i][j] = /* (char) */ (guint16) c[i][j];
        }
    }


    gpiv_free_fftw_real_matrix(a);
    gpiv_free_fftw_real_matrix(b);
    gpiv_free_fftw_real_matrix(c);
    gpiv_free_fftw_complex_matrix(C);
}


void
gpiv_imgproc_convolve(GpivImagePar img_par, 
                      GpivImageProcPar img_proc_par,
                      guint16 **img1_in, 
                      guint16 **img2_in, 
                      guint16 **img_out
                      )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Convolves two images
 *
 * INPUT:
 *     image_par: structure to image parameters
 *     image_proc_par: image processing parameters
 *     img1-in: first input image
 *     img2_in: second input image
 *
 * OUTPUT:
 *     img_out: resulting output image 
 */
{
    int i, j, ij;
    int M = img_par.nrows, N = img_par.ncolumns;

    fftw_real **a, **b, **c;
    fftw_complex *A, *B, **C;
    rfftwnd_plan p, pinv;
    fftw_real scale = 1.0 / (M * N);

    gpiv_fread_fftw_wisdom(1);
    gpiv_fread_fftw_wisdom(-1);

    a = gpiv_fftw_real_matrix(M, 2 * (N / 2 + 1));
    A = (fftw_complex *) &a[0][0];
    b = gpiv_fftw_real_matrix(M, 2 * (N / 2 + 1));
    B = (fftw_complex *) &b[0][0];
    c = gpiv_fftw_real_matrix(M, N);
    C = gpiv_fftw_complex_matrix(M, N /2 + 1);

    for (i = 0; i < M; i++) {
	for (j = 0; j < N; j++) {
	    a[i][j] = (double) img1_in[i][j];
	    b[i][j] = (double) img2_in[i][j];
	}
    }
    

/*
 * import_wisdom from string and make plans
 * FFT transform to get A and B from a and b;
 */
    p = rfftw2d_create_plan(M, N, FFTW_REAL_TO_COMPLEX,
                            FFTW_MEASURE | FFTW_USE_WISDOM | FFTW_IN_PLACE
                            );

    rfftwnd_one_real_to_complex(p, &a[0][0], NULL);
    rfftwnd_one_real_to_complex(p, &b[0][0], NULL);


    gpiv_fwrite_fftw_wisdom(1);
    rfftwnd_destroy_plan(p);


/*
 * Convolves  FFT(a) = A with FFT(b) = B to C following
 * FFT(a) * FFT(b) = C
 */
    for (i = 0; i < M; i++) {
        for (j = 0; j < N/2+1; j++) {
            ij = i*(N/2+1) + j;
            C[i][j].re = (A[ij].re * B[ij].re
                          - A[ij].im * B[ij].im) * scale;
            C[i][j].im = (A[ij].re * B[ij].im
                          + A[ij].im * B[ij].re) * scale;
        }
    }


/*
 * import_wisdom from string and make plans
 * inverse transform to get c, the convolution of a and b;
 * this has the side effect of overwriting C
 */
    pinv = rfftw2d_create_plan(M, N, FFTW_COMPLEX_TO_REAL,
                               FFTW_MEASURE| FFTW_USE_WISDOM );

    rfftwnd_one_complex_to_real(pinv, &C[0][0], &c[0][0]);

    gpiv_fwrite_fftw_wisdom(-1);
    rfftwnd_destroy_plan(pinv);

    for (i = 0; i < M; i++) {
        for (j = 0; j < N; j++) {
            img_out[i][j] = /* (char) */ (guint16) c[i][j];
        }
    }


    gpiv_free_fftw_real_matrix(a);
    gpiv_free_fftw_real_matrix(b);
    gpiv_free_fftw_real_matrix(c);
    gpiv_free_fftw_complex_matrix(C);
}

