(**
   A imgae class that implements a fuel gauge, needed for displaying the
   amount a process has yet proceded.
**)

MODULE VO:FuelGauge;

(*
    A fuelgauge.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT  D := VO:Base:Display,
        F := VO:Base:Frame,
        O := VO:Base:Object,
        Z := VO:Base:Size,

        V := VO:Model:Value,

        G := VO:Object;

TYPE
  Prefs*     = POINTER TO PrefsDesc;

  (**
    In this class all preferences stuff of the button is stored.
  **)

  PrefsDesc* = RECORD (G.PrefsDesc)
                 width*,
                 height* : Z.SizeDesc;
               END;

  FuelGauge*     = POINTER TO FuelGaugeDesc;
  FuelGaugeDesc* = RECORD (G.ImageDesc)
                     current  : V.ValueModel;
                     min,max  : LONGINT;
                     vertical : BOOLEAN;
                   END;

VAR
  prefs* : Prefs;

  PROCEDURE (p : Prefs) Init*;

  BEGIN
    p.Init^;

    p.frame:=F.double3DIn;

    p.width.Init;
    p.height.Init;

    p.width.SetSize(Z.unit,3);
    p.height.SetSize(Z.unit,3);
  END Init;

  PROCEDURE (f : FuelGauge) Init*;

  BEGIN
    f.Init^;

    f.SetPrefs(prefs);

    f.current:=NIL;

    f.max:=100;
    f.vertical:=FALSE;
  END Init;

  PROCEDURE (f : FuelGauge) SetDirection*(vertical : BOOLEAN);

  BEGIN
    f.vertical:=vertical;
  END SetDirection;

  (**
    Set the minimal and maximal value the fuel gauge should display.

    NOTE
    The image currently does not cut down the value to the given range,
    so be carefull.
  **)

  PROCEDURE (f : FuelGauge) SetRange*(min,max : LONGINT);

  BEGIN
    IF (f.min#min) OR (f.max#max) THEN
      f.min:=min;
      f.max:=max;
      IF f.visible THEN
        f.Redraw;
      END;
    END;
  END SetRange;

  (**
    Set the integer model which represents the current value.
    Therefore the actual argument is expected to be of type
    V.ValueModel. Anything else is ignored.
  **)

  PROCEDURE (f : FuelGauge) SetModel*(model : O.Model);

  BEGIN
    IF f.current#NIL THEN
      f.UnattachModel(f.current);
    END;
    IF (model#NIL) & (model IS V.ValueModel) THEN
      f.current:=model(V.ValueModel);
      f.AttachModel(f.current);
      IF f.visible THEN
        f.Redraw;
      END;
    ELSE
      f.current:=NIL;
    END;
  END SetModel;


  (**
    This function is used to check if an argument to SetModel
    was successfully accepted.
   **)

  PROCEDURE (f : FuelGauge) ModelAccepted * (m : O.Model):BOOLEAN;

  BEGIN
    RETURN m=f.current
  END ModelAccepted;


  PROCEDURE (f : FuelGauge) CalcSize*;

  BEGIN
    IF f.vertical THEN
      f.minWidth:=1;
      f.width:=f.prefs(Prefs).width.GetSize();
      f.minHeight:=4;
      f.height:=f.minHeight;
    ELSE
      f.minWidth:=4;
      f.width:=f.minWidth;
      f.minHeight:=1;
      f.height:=f.prefs(Prefs).height.GetSize();
    END;


    f.CalcSize^;
  END CalcSize;

  PROCEDURE (f : FuelGauge) Draw*(x,y,w,h : LONGINT);

  VAR
    size : LONGINT;
    draw : D.DrawInfo;

  BEGIN
    f.Draw^(x,y,w,h);

    IF ~f.Intersect(x,y,w,h) THEN
      RETURN;
    END;

    draw:=f.GetDrawInfo();

    IF f.vertical THEN
      size:=((f.height)*f.current.GetLongint()-f.min) DIV (f.max-f.min);
      f.DrawBackground(f.x,f.y,f.width,f.height-size);
      draw.PushForeground(D.fillColor);
      draw.FillRectangle(f.x,f.y+f.height-size,f.width,size);
      draw.PopForeground;
    ELSE
      size:=((f.width)*f.current.GetLongint()-f.min) DIV (f.max-f.min);
      draw.PushForeground(D.fillColor);
      draw.FillRectangle(f.x,f.y,size,f.height);
      draw.PopForeground;
      f.DrawBackground(f.x+size,f.y,f.width-size,f.height);
    END;
  END Draw;

  PROCEDURE (f : FuelGauge) Hide*;

  BEGIN
    IF f.visible THEN
      f.DrawHide;
      f.Hide^;
    END;
  END Hide;

  PROCEDURE (f : FuelGauge) Resync*(model : O.Model; msg : O.ResyncMsg);

  BEGIN
    IF f.visible THEN
      f.Redraw;
    END;
  END Resync;

  PROCEDURE CreateFuelGauge*():FuelGauge;

  VAR
    fuelGauge : FuelGauge;

  BEGIN
    NEW(fuelGauge);
    fuelGauge.Init;

    RETURN fuelGauge;
  END CreateFuelGauge;

BEGIN
  NEW(prefs);
  prefs.Init;
END VO:FuelGauge.