/*! This is not really -*-c++-*-, but it tricks Emacs into believing so anyway. */
//!
//!  Roxen Photo Album Module by Johan Sundstrm <johan@id3.org>
//!
//!  Distributed under the GNU General Public License
//!
//!
//! TODO:
//!       Thumbnail images on its own
//!       Build directory index on its own
//!       "mountpoint/." listing ought to list only one subdirectory level
//!       Rewrite stat_file() in a readable, optimized way

#include <module.h>
#include <stat.h>
inherit "module";
inherit "roxenlib";

constant cvs_version = "$Id: album.pike,v 1.11 1998/12/23 14:11:21 johan Exp $";

//#undef DEBUG_ALL
//#define DEBUG_ALL
//#define DEBUG_FIND_DIR
//#define DEBUG_FIND_FILE
//#define DEBUG_STAT_FILE

#ifdef DEBUG_ALL
#define DEBUG_FIND_FILE
#define DEBUG_FIND_DIR
#define DEBUG_STAT_FILE
#endif

//! location:album
mapping albums = ([ ]);
string mountpoint;

#define BELOW_MOUNTPOINT(TAG, ID) do\
{                                    \
  if(zero_type(ID->misc->album))      \
    return "<!-- &lt;"+TAG+"&gt; "     \
           "can only be used in "       \
           "photo album templates -->";  \
} while(0)

mixed *register_module()
{
  return({ MODULE_PARSER | MODULE_LOCATION,
	   "Photo Album",
	   "<p>"
	   "Aids you in creating and maintaining photo albums. Just set up an index "
	   "page for the album, defining what pictures to show in the album, perhaps "
	   "create a template defining the layout of an album page, and the module "
	   "will do the rest of the job for you, providing navigational features and "
	   "some other goodies as well."
	   "</p>\n<p>"
	   "To get you going, here is how to define a very basic album with the three "
	   "images \"foo.jpg\", \"bar.gif\" and \"baz.png\". Thumbnails of these files, "
	   "with the same names, are located in a subdirectory \"thumbnails\". Think of "
	   "the album tag as of the container for the index page (it is), and of the "
	   "photo tag as a container tag inserting the thumbnail as well as defining an "
	   "image description. The index.html file:"
	   "</p>\n<pre>"
	   "&lt;html&gt;&lt;body&gt;&lt;center&gt;\n"
	   " &lt;album title=\"A flat example\" nails=thumbnails&gt;\n"
	   "  &lt;photo file=\"foo.jpg\"&gt;King of metasyntactics.&lt;/photo&gt;\n"
	   "  &lt;photo file=\"bar.gif\"&gt;...and his queen.&lt;/photo&gt;\n"
	   "  &lt;photo file=\"baz.png\"&gt;Third in fame.&lt;/photo&gt;\n"
	   " &lt;/album&gt;\n"
	   "&lt;/center&gt;&lt;/body&gt;&lt;/html&gt;"
	   "</pre>\n<p>"
	   "This would render you a photoalbum using the default template for each "
	   "album subpage, and an indexpage with the three image thumbnails, centered "
	   "on a line. Any HTML/RXML is allowed within both the album and photo tags, "
	   "but don't try too much magic with the &lt;if&gt; tag; once a photo tag has "
	   "been parsed once, its image will show up in the album."
	   "</p>\n<p>"
	   "More help on the various tags can be found by giving the help attribute to "
	   "the tag in question, for example &lt;album help&gt;."
	   "</p>\n<p>"
	   "In-depth documentation as well as information about the latest version "
	   "of the module can be found at the <a href=\"http://a205.ryd.student.liu.se/"
	   "(Photo.Album)/docs/\">module documentation</a> page at the developer's "
	   "site. The module author would be only too happy about hearing from you if "
	   "you use the module. Have fun!</p>\n"
	   "<p align=right>/<a href=\"mailto:johan@id3.org\">Johan Sundstrm</a></p>",
	   ({ }),
	   1 });
}

void create()
{
  string sep = "&nbsp;-&nbsp;";
  string bar = "<center>"
	     + "<first>First</first>" + sep
	     + "<previous>Previous</previous>" + sep
	     + "<index>Index</index>" + sep
	     + "<next>Next</next>" + sep
	     + "<last>Last</last>"
	     + "</center>";
  string template = ({ "<html><head>",
		       "<title><albumtitle></title>",
		       "</head><body>",
		       "",
		       bar + "<br>",
		       "<image><br>",
		       "",
		       "<center><caption></center>",
		       "",
		       bar,
		       "</body></html>"
		    }) * "\n";

  defvar( "template", template, "Default page template", TYPE_TEXT_FIELD,
	  "This RXML template will be used for the pages in the album. "
	  "The navigating (container) tags &lt;first&gt;, &lt;previous&gt;. "
	  "&lt;index&gt;, &lt;next&gt; and &lt;last&gt; do what you expect from "
	  "them; they are all actually disguised &lt;a&gt; tags with the href "
	  "attribute already set."
	  "<p>"
	  "The tag &lt;image&gt; will insert the current image, &lt;caption&gt; will "
	  "insert the corresponding image description / comment. &lt;pagetitle&gt; "
	  "inserts the album title and &lt;phototitle&gt; inserts the current photo "
	  "title, if defined." );

  defvar( "mountpoint", "/fotoalbum", "Mount point", TYPE_LOCATION,
	  "This is where the module will be inserted in the "
	  "namespace of your server." );

  defvar( "autosize", 1, "Calculate imagesizes?", TYPE_FLAG,
	  "If this is set to yes, the photo album module will try to figure "
	  "out the size of images when it can, and set the img tag attributes "
	  "height and width. This only works with png, gif and jpeg images; "
	  "dimensions will not be specified for other formats." );

  defvar( "index", 1, "Allow index", TYPE_FLAG | VAR_MORE,
	  "If this variable is set, you will get a normal directory listing of "
	  "all initialized albums by appending '.' or '/' to the mountpoint, or "
	  "any other directory name below, just like you're used to using your "
	  "directory module (well, at least almost)."
	  "<p>"
	  "The photo album module will respect .nodiraccess, .www_not_browsable, "
	  ".nodir_access and .www_browsable files found in the directory where "
	  "the corresponding album was defined, and behave accordingly just like "
	  "the ordinary directory module would. See your Roxen manual for deatils." );
}

class album
{
  string album_title = "";          //\ Subpage title, common for the entire album
  string thumbnail_path = "";        //\ relative path to thumbnail directory
  string template;                    //\ when defined, the template for the album
  int access_count = 0;                //\ How many times the album has been accessed
  string *image = ({ });                //\ array of the image names
  mapping(string:string) text = ([ ]);   //\ map each image to its description
  mapping(string:int)     index = ([ ]);  //\ ...and what index it has in *image
  mapping(string:string)    title = ([ ]); //\ ...and what title has been assigned to it
  mapping(string:array) stat_cache = ([ ]); //\ ...and when it was last changed etc
  mapping(string:array(int)) thumbdims=([]); //\ ...and its thumbnail's dimenstions
  mapping(string:array(int)) dimensions=([]); //\ ...and its own dimensions

  int _sizeof() { return sizeof(image); }
  string prev(string name) { int i=index[name]; return i ? image[i-1] : 0; }
  string next(string name) { int i=index[name]; return i!=sizeof(image)-1 ? image[i+1] : 0; }
  string first(string name) { return index[name] ? image[0] : 0; }
  string last(string name) { int last=sizeof(image)-1; return index[name]!=last ? image[last] : 0; }
}

string dir_name(string path)
{ return sizeof( path ) ? dirname( path ) : ""; }

string remove_trailing(string s, string unwanted)
{
  if(sscanf(reverse(s), reverse(unwanted) + "%s", s))
    return reverse(s);
  else
    return s;
}

string file_name(string path)
{ return reverse((reverse( path ) / "/")[0]); }

string|void check_variable(string name, string value)
{
  switch(name)
  {
    case "mountpoint":
      if(sizeof( value ) && value[-1] == '/')
	call_out(set, 0, "mountpoint", remove_trailing( value, "/" ));
      break;
  }
}

string s(int n)   { return n==1?"":"s"; }
string are(int n) { return n==1?"is":"are"; }
string have(int n){ return n==1?"has":"have"; }

string status()
{
  string ret;
  int number = sizeof( albums );
  if(!number)
    return "No albums have been viewed since this module was loaded.";
  int images=`+(@ Array.map(values(albums), lambda(object album) { return sizeof(album); } ) );

  ret = "There "+are(number)+" currently "+number+" photo album"+s(number)
      + " loaded, featuring a total of "+images+" image"+s(images)+".<br>"
      + "The loaded album"+s(number)+" "+are(number)+":<br>"
      + "<table><tr><td>Title</td><td>Images</td><td>Access count</td></tr>";
  string *index = indices( albums );
  object *data = values( albums );
  string host = my_configuration()->query( "MyWorldLocation" );
  host = host[0..sizeof(host)-2];

  for(int i = 0 ; i<number ; i++)
  {
    object album = data[i];
    string path = index[i];
    ret+="<tr><td><a href=\""+host+path+"/\">"+album->album_title+"</a></td><td>"+sizeof(album)
        +"</td><td>"+album->access_count+"</td></tr>";
  }
  ret += "</table>";
  return ret;
}

//! Add a <base href> to the given directory.
//! FIXME: Doesn't work for protocols other than http://
string add_base_element(string html, string dir, object id)
{
  string host;
  if(id->misc->host)
    host = "http://"+id->misc->host;
  else
    host = id->conf->query( "MyWorldLocation" );

  string lower = lower_case( html );
  int header_end;
  if((header_end = search( lower, "</head>" ))>0)
  {
    //! Find a possible base element
    if(search( lower, "<base" )!=-1)
    { //! there was a base tag for us to replace!
      object b = Regexp("(.*)<[bB][aA][sS][eE]([^>]*)(.*)");
      string *parts = b->split(html);
      return sprintf("%s<base href=\"%s/\"%s",
		     parts[0], //! Everything before "<base"
		     host + dir,
		     parts[2]); //! Everything from ">" onwards
    } else //! Add a <base> just before </head>
      return sprintf("%s<base href=\"%s/\">\n%s",
		     html[0..header_end-1],
		     host + dir,
		     html[header_end..] );
  } else
    return sprintf("<head><base href=\"%s/\"></head>%s",
		   host + dir, html );
}

int hidden(string dir, object id)
{ //! Should this directory be hidden?
  //! FIXME: Ought to cache directory listings for a short while;
  //!        stat_file() calls us to check, which results in lots of get_dir():s
  string idx = remove_trailing( dir, "/" );
  if(idx == "")
    return !QUERY(index);
  else if(zero_type(albums[ idx ]))
    return 1; //! What album?
  else
  {
    array files = get_dir(id->conf->real_file( dir, id ));
    if(!QUERY(index) && search(files, ".www_browsable") == -1)
      //! Is access to this dir is allowed nevertheless?
      return 1; //! Naah.

    //! Is access to this dir disallowed?
    if(sizeof(files & ({".nodiraccess",".www_not_browsable",".nodir_access"})))
      return 1; //! Yep.

    return 0; //! Not hidden!
  }
}

#ifdef DEBUG_FIND_FILE
#define RETURN(X) do                   \
{                                       \
  report_notice(sprintf("Photo Album: "  \
			"find_file(\"%s"  \
			"\", id) returned" \
			" %O", path, (X))); \
  return X;                                  \
} while(0)
#else
#define RETURN(X) return X
#endif

/* Build a page showing an image from some album */
mixed find_file(string path, object id)
{
#ifdef DEBUG_FIND_FILE
  report_notice(sprintf("Photo Album: find_file(\"%s\", id) called", path));
#endif
  string  dir = dir_name( path ),
         file = file_name( path ),
         bar, title, temp;
  int i = sizeof( file )-1;
  if(file[i-4..i]==".html") file = file[0..i-5];

//#ifdef DEBUG_FIND_FILE
//  report_notice(sprintf("Photo Album: find_file(\"%s\", id): dir=%s, file=%s", path, dir, file));
//#endif

  if((path=="/." || path=="/") && QUERY(index))
    RETURN(-1); //! Is directory; show all (non-hidden) albums

  if(!albums[ dir ])
  { //! The requested album was not cached!
    //! Cache it if it exists
    id->conf->try_get_file( dir + "/", id );
    if(!albums[ dir ])
      RETURN(0); //! Didn't.
  }

  if(!sizeof( file ))
    RETURN(http_redirect( dir, id )); //! No file requested; show album index.
  if(search(albums[dir]->image, file)==-1)
    if(file == "." && QUERY(index))
      RETURN(-1); //! Is directory
    else if(albums[ path ]) //! No slash after existing directory?
      RETURN(http_redirect( dir, id )); //! show album index
    else
      RETURN(0); //! Nonexistent file

  albums[dir]->access_count++;

  id->misc->album = ([ "file" : file,
		        "dir" : dir ]);
  return http_string_answer(
	    add_base_element(
		   parse_rxml(
			      albums[dir]->template || QUERY(template)
			      , id)
			     , dir, id)
	                    , "text/html");
}

array stat_file(string path, object id)
{
#ifdef DEBUG_STAT_FILE
  report_notice(sprintf("Photo Album: stat_file(\"%s\", id) called", path));
#endif
  string dir = dir_name( path ),
        file = file_name( path );
  array stat = 0;

  //! This code really *stinks*, and yes, I am rather ashamed of it. :-]
  if(!hidden( dir, id ))
  {
    if(sizeof(({ remove_trailing( path, "/") }) & indices( albums )))
      stat = file_stat(id->conf->real_file( path, id ));
    else if(albums[ dir ])
      stat = albums[ dir ]->stat_cache[ file ];
    else if(path=="/")
    { //! Fake a stat saying "I'm a fresh, readable directory and root is my owner!"
      int t = time();
      stat = ({ 040555, 0, t, t, t, 0, 0 });
    }
  } else if(!hidden( path, id ) && sizeof(({ path }) & indices( albums )))
    stat = file_stat(id->conf->real_file( path, id ));

#ifdef DEBUG_STAT_FILE
  report_notice(sprintf("Photo Album: stat_file(\"%s\", id) returned %O", path, stat));
#endif
  return stat;
}

#ifdef DEBUG_FIND_DIR
#define RETURN(X) do                   \
{                                       \
  report_notice(sprintf("Photo Album: "  \
			"find_dir(\"%s"   \
			"\", id) returned" \
			" %O", path, (X))); \
  return X;                                  \
} while(0)
#else
#define RETURN(X) return X
#endif

//! For showing directory listings
array find_dir(string path, object id)
{
#ifdef DEBUG_FIND_DIR
  report_notice(sprintf("Photo Album: find_dir(\"%s\", id) called", path));
#endif
  array files;

  if(path == "/")
  {
    files = Array.map(Array.filter(indices( albums ),
				   lambda(string dir, object id)
				   { return !hidden( dir, id ); },
				   id ),
		      lambda(string s) { return s[1..]; } );
    RETURN(files);
  }
  else
  {
    if(hidden( path, id ))
      RETURN(0);
    else
      RETURN(albums[ remove_trailing( path, "/" ) ]->image);
  }
}

//! From a mapping "arg", find the value for some given index
//! If not found, return the value of "def"
string|int find_element(mapping(string:string) arg,
			array(string) alternatives,
			string def )
{
  string *result = indices( arg ) & alternatives;
  return sizeof( result ) ? arg[result[0]] : def;
}

string container_album(string tag_name, mapping arg,
		       string contents, object id, mapping defines)
{
  string path = id->not_query;
  string dir = dirname( path );
  string template;

  //! If no album exists at this URL, initiate one
  if(!albums[dir])
    albums[dir] = album();

  //! If no array of album images exists, initiate it
  if(!arrayp(albums[dir]->image))
    albums[dir]->image = ({ });

  albums[dir]->album_title = find_element(arg,({"title", "titel"}),
	                                  albums[dir]->album_title?
					  albums[dir]->album_title:
					  "Foton");
  albums[dir]->thumbnail_path = find_element(arg,
					     ({"nails", "thumbnails", "naglar", "tumnaglar"}),
					     "naglar");

  if(template = find_element(arg, ({"template", "sidmall"}), 0))
  {
    object file = Stdio.File();
    if(file->open(id->conf->real_file(fix_relative(template, id), id), "r"))
    {
      template = file->read();
      file->close();
    } else
      template = 0;
  }
  albums[dir]->template = template;

  return parse_rxml(contents, id);
}

string container_photo(string tag_name, mapping arg,
		       string contents, object id, mapping defines)
{
  string dir = dirname(id->not_query),
         file = find_element(arg, ({ "src", "bild", "image", "file" }), 0),
         realfile, realthumb, thumbnail, temp;
  int changetime,
      autosize = QUERY(autosize);

  if(!albums[ dir ])
    return "<tt>&lt;"+tag_name+"&gt;</tt> can only be used within an <tt>&lt;album&gt;</tt> tag!";

  if(!file)
    return "[no image given]"; //! No image was stated

  realfile = id->conf->real_file(fix_relative( file, id ), id);
  //report_notice(sprintf("file:%O, realfile:%O", file, realfile));
  array stat = file_stat( realfile );
  changetime = stat && stat[ ST_MTIME ];
  if(!changetime)
    return "[The image " + realfile + " is missing]";

  thumbnail = find_element(arg, ({ "thumbnail", "nail", "nagel", "tumnagel",
			   "nails", "thumbnails", "naglar", "tumnaglar" }),
			   albums[ dir ]->thumbnail_path + "/" + file);
  realthumb = id->conf->real_file(fix_relative( thumbnail, id ), id);

  if(search(albums[ dir ]->image, file )==-1)
  { //! The file wasn't already known
    albums[ dir ]->index[ file ] = sizeof(albums[ dir ]);
    albums[ dir ]->image = albums[ dir ]->image+({ file });
    if( autosize && sizeof(stat) )
      stat[ ST_MTIME ] = 0;
    albums[ dir ]->stat_cache[ file ] = stat;
  }
  catch { if( autosize && changetime>albums[ dir ]->stat_cache[ file ][ ST_MTIME ])
  { //! The image has been changed since its dimensions were fetched last time
    albums[ dir ]->dimensions[ file ] = Dims.dims()->get( realfile );
    albums[ dir ]->thumbdims[ file ] = Dims.dims()->get( realthumb );
    albums[ dir ]->stat_cache[ file ] = stat;
  } };
  albums[ dir ]->text[ file ] = contents;
  albums[ dir ]->title[ file ] = find_element(arg, ({ "title", "titel" }), 0);

  mapping imgargs = ([ "border" : "0",
		          "src" : thumbnail ]);
  if(array dims = albums[dir]->thumbdims[file])
  {
    imgargs->width = (string)dims[0];
    imgargs->height = (string)dims[1];
  }
  if(array dims = albums[dir]->dimensions[file])
    imgargs->alt = sprintf( "[%d%d]", dims[0], dims[1] );
  else
    imgargs->alt = "";

  return make_container("a", ([ "href" : mountpoint+dir+"/"+file+".html" ]),
			  make_tag( "img", imgargs ));
}

string container_reference(string tag, mapping arg, string contents, object id, mapping def)
{ //! Link to another image in the album
  BELOW_MOUNTPOINT(tag, id);

  string dir = id->misc->album->dir,
        file = id->misc->album->file,
         url = mountpoint + dir + "/",
        temp;

  switch( tag )
  {
    case "first":
      temp = albums[ dir ]->first( file );
      break;
    case "previous":
      temp = albums[ dir ]->prev( file );
      break;
    case "next":
      temp = albums[ dir ]->next( file );
      break;
    case "last":
      temp = albums[ dir ]->last( file );
      break;
  }

  return temp ? make_container( "a", arg | ([ "href" : url + temp + ".html" ]), contents ) : contents;
}

string container_index(string tag, mapping arg, string content, object id, mapping def)
{ //! Link to the album's index page
  BELOW_MOUNTPOINT(tag, id);
  arg->href = id->misc->album->dir + "/";
  return make_container( "a", arg, content );
}

string tag_insert(string tag, mapping arg, object id, object file)
{
  BELOW_MOUNTPOINT(tag, id);
  string file = id->misc->album->file,
	  dir = id->misc->album->dir;
  switch( tag )
  {
    case "albumtitel":
    case "albumtitle":
      return parse_rxml(albums[ dir ]->album_title, id);
    case "bildtitel":
    case "phototitle":
      return parse_rxml(albums[ dir ]->title[ file ], id);
    case "bildtext":
    case "caption":
      return parse_rxml(albums[ dir ]->text[ file ], id);
    case "bild":
    case "image":
      string title = albums[ dir ]->title[ file ];
      if(title)
	title = parse_rxml( title, id );
      array dims;
      arg->src = dir + "/" + file;
      if(dims = albums[ dir ]->dimensions[ file ])
      {
	if(zero_type( arg->alt ))
	  arg->alt = sprintf( "%s[%d%d]",
			      (title ? title + " " : ""),
			      dims[0], dims[1] );
	else
	  arg->alt = parse_rxml( arg->alt, id );
	arg->width = (string)dims[0];
	arg->height = (string)dims[1];
      } else
	if(zero_type( arg->alt ))
	  arg->alt = title ? title : "";
	else
	  arg->alt = parse_rxml( arg->alt, id );
      return make_tag( "img", arg );
  }
}

string query_location() { return mountpoint; }

void start()
{
  mountpoint = query( "mountpoint" );
}

mapping query_tag_callers()
{
  return([ "caption"   :tag_insert,//!
	   "bildtext"  :tag_insert,//! synonyms for <caption>
	   "albumtitle":tag_insert, //!
	   "albumtitel":tag_insert, //! synonyms for <albumtitle>
           "phototitle":tag_insert,  //!
           "bildtitel" :tag_insert,  //! synonyms for <phototitle>
	   "image"     :tag_insert,   //!
	   "bild"      :tag_insert,   //! synonyms for <image>
	 ]);
}

mapping query_container_callers()
{
  return([ "album"     :container_album, //!
	   "fotoalbum" :container_album, //!
	   "photoalbum":container_album, //! synonyms for <album>
	   "photo"     :container_photo,  //!
	   "fotobild"  :container_photo,  //! synonyms for <photo>
	   "index"     :container_index,   //! Navigation
	   "first"     :container_reference,//!
	   "previous"  :container_reference,//!
	   "next"      :container_reference,//!
	   "last"      :container_reference,//! Navigation
	 ]);
}
