/*!
	========== licence begin GPL
    Copyright (C) 2002-2003 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end
*/
package com.sap.dbtech.jdbc.translators;

import java.io.PrintStream;

import com.sap.dbtech.util.StructuredBytes;
import com.sap.dbtech.util.StructuredMem;
import com.sap.dbtech.vsp00.StreamHandle;

/**
 * Basic class for handling features that are common to both output
 * and input ABAP/OMS Streams.
 */
public class ABAPStreamDescriptor {
	
	protected ABAPStreamTranslator translator;
	protected StructuredBytes      descriptor;
	protected StructuredMem        descriptorMark;
	protected int                  rowSize;
	/**
	 * Constructor.
	 * @param translator
	 */	
	public ABAPStreamDescriptor(ABAPStreamTranslator translator) {
		this.translator     = translator;
		this.descriptor     = new StructuredBytes(translator.physicalLength - 1);
		this.descriptorMark = null;
		this.rowSize        = 0;
	}
	
	public void clearColumns() {
		this.descriptor.putInt2(0,StreamHandle.TabHandle_ColCount_O);
		this.descriptor.putInt2(0, StreamHandle.TabHandle_Rowsize_O);
	}
	
	public void addColumn(int inout,
						  int abaptype,
						  int abaptypelength,
						  int dec,
						  int length,
						  int offset) {
						  	
		// System.err.println("ADDING COLUMN OF LENGTH " + abaptypelength + " AT OFFSET " + offset);						  	
		int columnCount = this.descriptor.getInt2(StreamHandle.TabHandle_ColCount_O);
		if(this.descriptor.size() < (StreamHandle.Header_Length_C 
			+ (columnCount + 1) * StreamHandle.ColDesc_Length_C) ) {
			expandDescriptor(StreamHandle.Header_Length_C + (columnCount+1) 
				* StreamHandle.ColDesc_Length_C);		
		}
		this.descriptor.putInt2(columnCount + 1, StreamHandle.TabHandle_ColCount_O);
		this.rowSize = this.descriptor.getInt2(StreamHandle.TabHandle_Rowsize_O);
		rowSize = Math.max(rowSize, abaptypelength + offset);
		this.descriptor.putInt2(rowSize, StreamHandle.TabHandle_Rowsize_O);
	
		
		// Input or output ?
		this.descriptor.putInt1(inout, StreamHandle.TabHandle_ColDesc_0_O  
			+ (StreamHandle.ColDesc_Length_C * columnCount) 
			+ StreamHandle.ColDesc_InOut_O);
		// Type
		this.descriptor.putInt1(abaptype, StreamHandle.TabHandle_ColDesc_0_O  
			+ (StreamHandle.ColDesc_Length_C * columnCount) 
			+ StreamHandle.ColDesc_ABAPType_O);
		//	Decimal places
		this.descriptor.putInt2(dec, StreamHandle.TabHandle_ColDesc_0_O
			+ (StreamHandle.ColDesc_Length_C * columnCount) 
			+ StreamHandle.ColDesc_Dec_O);
		// Length	
		this.descriptor.putInt2(abaptypelength, StreamHandle.TabHandle_ColDesc_0_O
			+ (StreamHandle.ColDesc_Length_C * columnCount) 
			+ StreamHandle.ColDesc_Length_O);
		// Offset
		this.descriptor.putInt2(offset, StreamHandle.TabHandle_ColDesc_0_O
			+ (StreamHandle.ColDesc_Length_C * columnCount) 
			+ StreamHandle.ColDesc_Offset_O);
	}
	
	/**
	 * Expands the descriptor.
	 * @param descriptorlength The new length in bytes.
	 */
	private void expandDescriptor(int descriptorlength) {
		byte[] oldba = descriptor.bytes();
		byte[] newba = new byte[descriptorlength];
		System.arraycopy(oldba, 0, newba, 0, oldba.length);		
		this.descriptor = new StructuredBytes(newba);
	}
	
	public void updateIndex(int index) {
		this.descriptorMark.putInt4(index, StreamHandle.TabHandle_ABAPTabID_O);		
	}
	/**
	 * Writes the ABAP Stream descriptor into the data part, including the
	 * 'defined byte' at <code>bufpos -1</code>.
	 * @param mem    The data part.
	 * @param bufpos The position.
	 */
	public void putDescriptor(com.sap.dbtech.jdbc.packet.DataPart mem, int bufpos) {
		// clearColumns();
		mem.putDefineByte (0, bufpos - 1);
		mem.putBytes (this.descriptor.bytes(), bufpos);
		// traceOn(System.err);
		this.descriptorMark = mem.getPointer(bufpos);
	}
	
	public void traceOn(PrintStream ps) {
		int colcount;
		ps.println("DESCRIPTOR");
		ps.println("  ROW SIZE       : " + descriptor.getInt2(StreamHandle.TabHandle_Rowsize_O));
		ps.println("  COLUMN COUNT   : " + (colcount = descriptor.getInt2(StreamHandle.TabHandle_ColCount_O)));
		ps.println("  ROW COUNT      : " + descriptor.getInt4(StreamHandle.TabHandle_RowCount_O));
		ps.println("  TAB ID         : " + descriptor.getInt4(StreamHandle.TabHandle_ABAPTabID_O));
		for(int i=0; i<colcount; ++i)  {
			int io = descriptor.getInt1((i * StreamHandle.ColDesc_Length_C) + StreamHandle.TabHandle_ColDesc_0_O + StreamHandle.ColDesc_InOut_O);
			int abaptype = descriptor.getInt1((i * StreamHandle.ColDesc_Length_C) + StreamHandle.TabHandle_ColDesc_0_O + StreamHandle.ColDesc_ABAPType_O);
			int length   = descriptor.getInt2((i * StreamHandle.ColDesc_Length_C) + StreamHandle.TabHandle_ColDesc_0_O + StreamHandle.ColDesc_Length_O);
			int dec      = descriptor.getInt2((i * StreamHandle.ColDesc_Length_C) + StreamHandle.TabHandle_ColDesc_0_O + StreamHandle.ColDesc_Dec_O);
			int offset   = descriptor.getInt2((i * StreamHandle.ColDesc_Length_C) + StreamHandle.TabHandle_ColDesc_0_O + StreamHandle.ColDesc_Offset_O);
			
			ps.println("    COLDESC[" + (i+1) + "]");
			ps.println("      INOUT   : " + inout(io));
			ps.println("      TYPE    : " + type(abaptype)+ " ("+abaptype+")");
			ps.println("      LENGTH  : " + length);
			ps.println("      PREC    : " + dec);
			ps.println("      OFFSET  : " + offset);
			
		}
	}
	
	private static String type(int t) 
	{
		switch(t) {
		case StreamHandle.StreamType_Bool_C:
			return "BOOL";
		case StreamHandle.StreamType_Char_C:
			return "CHAR";
		case StreamHandle.StreamType_Double_C:
			return "DOUBLE";
		case StreamHandle.StreamType_Float_C:
			return "FLOAT";
		case StreamHandle.StreamType_Int1_C:
			return "INT1";
		case StreamHandle.StreamType_Int2_C:
			return "INT2";
		case StreamHandle.StreamType_Int4_C:
			return "INT4";
		case StreamHandle.StreamType_Int8_C:
			return "INT8";
		case StreamHandle.StreamType_WYDE_C:
			return "WYDE";
		}
		return "Unknown type (" + t + ")";
	}
	
	private static String inout(int inout) {
		switch(inout) {
		case StreamHandle.Stream_IN_C:
			return "IN";
		case StreamHandle.Stream_OUT_C:
			return "OUT";
		case StreamHandle.Stream_INOUT_C:
			return "IN/OUT";
		}
		return "???";
	}

}
