/*


    ========== licence begin GPL
    Copyright (C) 2002-2003 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


*/

package com.sap.dbtech.jdbc.translators;

import java.sql.*;
import com.sap.dbtech.util.*;
import java.util.Calendar;
import com.sap.dbtech.jdbc.packet.*;

/**
 *
 */
public class DateTranslator extends CharDataTranslator {
    public static final String zeroPadC = "0000";
    /**
     * @param mode
     * @param ioType
     * @param dataType
     * @param len
     * @param ioLen
     * @param bufpos
     * @param readOnly
     * @param autoIncr
     */
    DateTranslator(int mode, int ioType, int dataType, int len, int ioLen,
            int bufpos, boolean readOnly, boolean autoIncr) {
        super(mode, ioType, dataType, len, ioLen, bufpos, readOnly, autoIncr);
    }
    /**
     *
     * @return java.lang.String
     */
    public String getString (
            SQLParamController controller,
            StructuredMem mem) throws SQLException
    {
        String result = null;
        if (!this.isNull (controller, mem))
            result = mem.getString(this.bufpos, this.physicalLength - 1);
        return result;
    }
    /**
     *
     * @return int
     */
    protected int parseYear (
        byte [] raw)
    {
        int year=((int)raw[0]-'0')*1000;
        year +=((int)raw[1]-'0')*100;
        year +=((int)raw[2]-'0')*10;
        year +=((int)raw[3]-'0');
        return year;
    }
    /**
     *
     * @return int
     */
    protected int parseMonth (
        byte [] raw)
    {
        int month=((int)raw[5]-'0')*10;
        month +=((int)raw[6]-'0');
        return month;
    }
    /**
     *
     * @return int
     */
    protected int parseDay (
        byte [] raw)
    {
        int day=((int)raw[8]-'0')*10;
        day +=((int)raw[9]-'0');
        return day;
    }
    /**
     *
     * @return java.sql.Date
     */
    public Date getDate (
        SQLParamController controller,
        StructuredMem mem,
        Calendar cal) throws SQLException
    {
        byte[] raw;
        Date result = null;

        if (!this.isNull (controller, mem)) {
            raw = mem.getBytes(this.bufpos, this.physicalLength - 1);

            int year = this.parseYear (raw);
            int month = this.parseMonth (raw);
            int day = this.parseDay (raw);

            if (cal == null)
              result = new Date(year-1900,month-1,day);
            else {
              cal.clear();
              cal.set(year,month-1,day,0,0,0);
              result = DBTechTranslator.CalendarUtil.getDateFromCalendar(cal);
            }
        }
        return result;
    }
    /**
     *
     * @return java.sql.Date
     */
    public Object getObject (
        SQLParamController controller,
        StructuredMem mem)
    throws SQLException
    {
        return this.getDate (controller, mem, null);
    }
    /**
     *
     */
    public Timestamp getTimestamp (
        SQLParamController controller,
        StructuredMem mem,
        Calendar cal)
    throws SQLException
    {
        Date dateValue = this.getDate (controller, mem, cal);
        if (dateValue == null) {
            return null;
        }
        return new Timestamp (dateValue.getTime ());
    }
    /**
     *
     * @return boolean
     */
    public boolean isCaseSensitive () {
        return false;
    }
    /**
     *
     * @param newValue java.sql.Date
     */
    public Object transDateForInput (Date arg, Calendar cal)
    throws java.sql.SQLException
    {
        if (arg == null)
            return null;

        cal.setTime(arg);

        int year = cal.get(Calendar.YEAR);
        if (year < 0 || year > 9999){
          throw this.newParseException (arg.toString(), "Date");
        }
        int month = cal.get(Calendar.MONTH)+1;
        int day = cal.get(Calendar.DAY_OF_MONTH);

        return this.formatDate (year, month, day);
    }
    final private static int DateSize_C = 10;

    /**
     *
     */
    protected byte [] formatDate (
        int year,
        int month,
        int day)
    {
        byte[] formattedDate = new byte[DateSize_C];
        formattedDate[0] = (byte)('0'+(year / 1000));
        year %= 1000;
        formattedDate[1] = (byte)('0'+(year / 100));
        year %= 100;
        formattedDate[2] = (byte)('0'+(year / 10));
        year %= 10;
        formattedDate[3] = (byte)('0'+(year));
        formattedDate[4] = (byte)'-';

        formattedDate[5] = HighTime_C[month];
        formattedDate[6] = LowTime_C [month];
        formattedDate[7] = (byte)'-';

        formattedDate[8] = HighTime_C[day];
        formattedDate[9] = LowTime_C [day];
//        System.out.println(new String(formattedDate));
        return formattedDate;
    }
    /**
     *
     * @param dataPart com.sap.dbtech.jdbc.packet.DataPart
     * @param data java.lang.Object
     */
    protected void putSpecific (
        DataPart dataPart,
        Object data)
    throws SQLException
    {
        byte [] bytes = (byte[])data;
        if (bytes.length > this.physicalLength - 1) {
            throw new com.sap.dbtech.jdbc.exceptions.ValueOverflow (
                com.sap.dbtech.vsp00.DataType.stringValuesC [this.dataType],
                -1);
        }
        dataPart.putDefineByte ((byte) ' ', this.bufpos - 1);
        dataPart.putStringBytes (bytes, this.bufpos, this.physicalLength - 1);
    }    /**
     *
     * @param obj java.lang.Object
     */
    public Object transSpecificForInput (Object obj)
    throws SQLException
    {
        Object result = null;

        if (obj instanceof Date) {
            result = this.transDateForInput ((Date) obj, Calendar.getInstance());
        }
        return result;
    }
    /**
     *
     * @param val java.lang.String
     */
    public Object transStringForInput (String val)
    throws SQLException
    {
        if (val == null) {
            return null;
        }
        try {
            return this.transDateForInput (Date.valueOf (val), Calendar.getInstance());
        }
        catch (IllegalArgumentException exc) {
            // ignore, try timestamp
        }
        try {
            Timestamp ts = Timestamp.valueOf (val);
            return this.transDateForInput (new Date (ts.getTime ()), Calendar.getInstance());
        }
        catch (IllegalArgumentException exc) {
            // possibly: try other parsing strategies
        }
        throw this.newParseException (val, "Date");
    }
}