/*


    ========== licence begin GPL
    Copyright (C) 2002-2003 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


*/
package com.sap.dbtech.jdbc.translators;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.sql.SQLData;
import java.sql.SQLException;
import java.sql.Struct;

import com.sap.dbtech.jdbc.DBProcParameterInfo;
import com.sap.dbtech.jdbc.exceptions.SQLExceptionSapDB;
import com.sap.dbtech.jdbc.packet.DataPart;
import com.sap.dbtech.util.MessageKey;
import com.sap.dbtech.util.MessageTranslator;
import com.sap.dbtech.util.StructuredBytes;
import com.sap.dbtech.util.StructuredMem;

/**
 * Translator for a structure.
 */
public class StructureTranslator extends DBTechTranslator {

	DBProcParameterInfo      parameterStructure;
	StructMemberTranslator[] structureConverter;
	boolean unicode;
	
	public StructureTranslator(
		int mode,
		int ioType,
		int dataType,
		int len,
		int ioLen,
		int bufpos,
		boolean unicode, boolean readOnly, boolean autoIncre) {
		super(mode, ioType, dataType, len, ioLen, bufpos, readOnly, autoIncre);
		this.unicode = unicode;
        this.structureConverter = new StructMemberTranslator[0];
	}

	protected void putSpecific(DataPart dataPart, Object data)
		throws SQLException {
		byte[] bytes = (byte[]) data;
		dataPart.putDefineByte(0, this.bufpos - 1);
		dataPart.putBytes(bytes, this.bufpos, this.physicalLength - 1);
	}

	public byte getByte(SQLParamController controller, StructuredMem mem)
		throws SQLException {
		byte[] result = null;
		if (this.isNull(controller, mem)) {
			return 0;
		} else {
			result = mem.getBytes(this.bufpos, 1);
		}
		return result[0];
	}

	public byte[] getBytes(SQLParamController controller, StructuredMem mem)
		throws SQLException {
		byte[] result = null;
		if (!this.isNull(controller, mem)) {
			result = mem.getBytes(this.bufpos, this.logicalLength);
		}
		return result;
	}

	public Object getObject(SQLParamController controller, 
	                        StructuredMem mem)
		throws SQLException {
		byte[] ba = this.getBytes(controller, mem);
		if(ba != null) {	
			Object [] objArr = new Object[structureConverter.length];
			StructuredBytes sb = new StructuredBytes(ba);
			for(int i=0; i<objArr.length; ++i) {
				objArr[i] = structureConverter[i].getObject(sb, 0);
			}
			return new Structure(objArr, this.parameterStructure.getSQLTypeName());
		} else {
			return null;
		}
	}

	public Object transByteForInput(byte val) throws SQLException {
		byte[] barr = new byte[1];
		barr[0] = val;
		return this.transBytesForInput(barr);
	}

	public Object transBytesForInput(byte[] arg) throws java.sql.SQLException {
		if (arg == null) {
			return arg;
		}
		this.checkFieldLimits(arg.length);
		return arg;
	}

	public Object transSpecificForInput(Object obj) throws SQLException {
		Object result = null;
		
		String clazzName;

		if (obj instanceof byte[]) {
			result = this.transBytesForInput((byte[]) obj);
		} else if(obj instanceof Object[]) {
			result = this.transObjectArrayForInput((Object[])obj);
		} else if(obj instanceof Struct) {
			result = this.transObjectArrayForInput(((Struct)obj).getAttributes());
		} else if(obj instanceof SQLData) {
			result = this.transSQLDataForInput((SQLData)obj);
		}
		return result;
	}

	public Object transSQLDataForInput(SQLData sqlData) 
		throws SQLException {
		StructureSQLOutput sqlOutput = new StructureSQLOutput(this);
		sqlData.writeSQL(sqlOutput);
		return sqlOutput.getData();
	}

	public Object transObjectArrayForInput(Object[] objectArray) 
		throws SQLException {
        if(objectArray.length != structureConverter.length) {
			throw new SQLExceptionSapDB(MessageTranslator.translate(MessageKey.ERROR_STRUCTURE_ARRAYWRONGLENTGH, 
				new Integer(structureConverter.length), new Integer(objectArray.length)));
		}
		StructuredBytes sb = new StructuredBytes(this.physicalLength-1);
		for(int i=0; i<objectArray.length; ++i) {
			if(objectArray[i] == null) {
				throw new SQLExceptionSapDB(MessageTranslator.translate(MessageKey.ERROR_STRUCT_ELEMENT_NULL, new Integer(i+1)));
			}
			structureConverter[i].putObject(sb, objectArray[i]);
		}
		return sb.bytes();
	}

	public Object transCharacterStreamForInput(
		java.io.Reader stream,
		int length)
		throws SQLException {
		if (length <= 0) {
			return null;
		}
		try {
			char[] ba = new char[length];
			int r = stream.read(ba);
			if (r != length) {
				if (r == -1)
					r = 0;
				char[] ba2 = ba;
				ba = new char[r];
				System.arraycopy(ba2, 0, ba, 0, r);
			}
			return transStringForInput(new String(ba));
		} catch (IOException ioex) {
			throw new SQLExceptionSapDB(ioex.getMessage());
		}
	}

	public Object transBinaryStreamForInput(
		java.io.InputStream stream,
		int length)
		throws SQLException {
		if (length <= 0) {
			return null;
		}
		try {
			byte[] ba = new byte[length];
			int r = stream.read(ba);
			if (r != length) {
				if (r == -1)
					r = 0;
				byte[] ba2 = ba;
				ba = new byte[r];
				System.arraycopy(ba2, 0, ba, 0, r);
			}
			return transBytesForInput(ba);
		} catch (IOException ioex) {
			throw new SQLExceptionSapDB(ioex.getMessage());
		}
	}

	public InputStream getBinaryStream(
		SQLParamController controller,
		StructuredMem mem,
		StructuredMem longData)
		throws SQLException {
		byte[] asBytes = this.getBytes(controller, mem);
		InputStream result;
		if (asBytes == null) {
			return null;
		}
		result = new ByteArrayInputStream(asBytes);
		return result;
	}

	public void setProcParamInfo(DBProcParameterInfo info) 
		throws SQLException {
		this.parameterStructure = info;
		this.structureConverter = StructMemberTranslator
			.createStructMemberTranslators(info, this.unicode);
	}

}
