/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

/* 
   This file illustrates an example of how to create a proximity matrix from a 
   Spatial Temporal Element Set (STElementSet). 
   In this example, the distance strategy is used to construct the matrix.   

   Author: Karine Reis   
*/

#include <TeSTElementSet.h>
#include <TeSTEFunctionsDB.h>
#include <TeInitQuerierStrategy.h>
#include <TeGeneralizedProxMatrix.h>
#include <TeMySQL.h>

int main()
{	
 	// Datatabase server parameters
	string host = "localhost";
	string dbname = "TerraLib";
	string user = "root";
	string password = "ferreira";

	// Open a connection to the TerraTeste MySQL database 
	TeDatabase* db = new TeMySQL();
	if (!db->connect(host, user, password, dbname))
	{
		cout << "Error: " << db->errorMessage() << endl;
		cout << endl << "Press Enter\n";
		getchar();
		return 1;
	}
	cout << "Connection successful to the database \"" << dbname << "\" on MySQL server \"" << host << "\" !\n";;
		
	
	// Opens a connection to a TerraLib database 
    // Loads a layer named DistritosSP 
	TeLayer* DistritosSP =  new TeLayer("DistritosSP");
	if (!db->loadLayer(DistritosSP))
	{
	    cout << "Fail to load the layer!" << db->errorMessage() << endl;
		db->close();
		cout << endl << "Press Enter\n";
		getchar();
        return 1;
	}

	// Inits querier strategies
	TeInitQuerierStrategies();
    
	// Creates a STElementSet from DistritosSP layer 
	TeSTElementSet steSet(DistritosSP);

	// Builds the STElementSet with geometries
	if(!TeSTOSetBuildDB(&steSet, true, false))
	{
		cout << "Error! " << endl;
		cout << endl << "Press Enter\n";
		getchar();
		return 1;
	}
	
	// Shows how many elements the elementSet has
	cout << "Number of elements: " <<  steSet.numElements() << endl;

	
	// Builds the proximity matrix
	TeGeneralizedProxMatrix				proxMat;
	
	TeProxMatrixLocalDistanceStrategy   sc_dist (&steSet, TePOLYGONS, 12000.00);
	TeProxMatrixNoWeightsStrategy		sw_no;
	TeProxMatrixNoSlicingStrategy		ss_no;
	
	proxMat.setCurrentSlicingStrategy(&ss_no);
	proxMat.setCurrentWeightsStrategy(&sw_no);
	proxMat.setCurrentConstructionStrategy(&sc_dist);

	if(!proxMat.ConstructMatrix())
	{
		cout << "Error building proximity matrix! " << endl;
		getchar();
		db->close ();
		return 0;
	}

	// Shows the neighbours of each object from proxMat
	TeSTElementSet::iterator it = steSet.begin();
	while ( it != steSet.end())
	{
		cout<< " The neighbours of the element  "<< (*it).objectId() << " are: " << endl;
			
		// Gets the neighbours of an element
		TeNeighboursMap neighbors = proxMat.getMapNeighbours((*it).objectId());
		TeNeighboursMap::iterator itN = neighbors.begin();
		while (itN != neighbors.end())
		{
			cout<< "         "<<(*itN).first << endl;
			++itN;
		}
		cout << endl;
		++it;
	}

	
	db->close();
	cout << endl << "Press Enter\n";
	getchar();
	return 0;
}
