/*
TerraLib - a library for developing GIS applications.
Copyright  2001, 2002, 2003 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular
purpose. The library provided hereunder is on an "as is" basis, and the
authors have no obligation to provide maintenance, support, updates,
enhancements, or modifications.
In no event shall INPE be held liable to any party
for direct, indirect, special, incidental, or consequential damages arising
out of the use of this library and its documentation.
*/

#ifndef TEPDICONTRAST_HPP
  #define TEPDICONTRAST_HPP

  #include "TePDIAlgorithm.hpp"
  #include "TePDIRgbPalette.hpp"
  #include "TePDITypes.hpp"

  #include "TePDISharedPtr.cpp"

  /**
   *   This is the class for contrast image processing algorithms.
   */
  class TePDIContrast : public TePDIAlgorithm{
    public :
      typedef TePDISharedPtr< TePDIContrast > pointer;
      typedef const TePDISharedPtr< TePDIContrast > const_pointer;

      /**
       * Allowed contrast types.
       */
      enum TePDIContrastType{
        TePDIContrastMinMax = 1,
        TePDIContrastLinear = 2,
        TePDIContrastSquareRoot = 3,
        TePDIContrastSquare = 4,
        TePDIContrastLog = 5,
        TePDIContrastNegative = 6,
        TePDIContrastHistEqualizer = 7,
        TePDIContrastSimpleSlicer = 8
      };

      /**
       * Alternative Constructor.
       *
       * The general required parameters by any contrast type:
       *
       * contrast_type (TePDIContrastType),
       * input_image (TePDITypes::TePDIRasterPtrType),
       * output_image (TePDITypes::TePDIRasterPtrType),
       * histo_levels (int) - Number of interpolated histogram levels.
       *
       * The specific required parameters ( by parameter type ):
       *
       * channels (std::vector<int>) - Band(s) to process. For
       * TePDIContrastSimpleSlicer just one band is allowed.
       *
       * min_level (double) - Minimal level for Linear Luts
       * Required by: TePDIContrastLinear, TePDIContrastSquareRoot,
       * TePDIContrastSquare, TePDIContrastLog, TePDIContrastNegative,
       * TePDIContrastSimpleSlicer
       *
       * max_level (double) - Minimal level for Linear Luts
       * Required by: TePDIContrastLinear, TePDIContrastSquareRoot,
       * TePDIContrastSquare, TePDIContrastLog, TePDIContrastNegative,
       * TePDIContrastSimpleSlicer
       *
       * rgb_palette (TePDIRgbPalette::pointer) -
       * Reference RGB palette.
       * Required by: TePDIContrastSimpleSlicer
       *
       * @param params Algoritm parameters.
       */
      TePDIContrast( const TePDIParameters& params );

      /**
       * Default Constructor.
       *
       */
      TePDIContrast();

      /**
       * Default Destructor
       */
      ~TePDIContrast();

      /**
       * Checks if the supplied parameters fits the requirements of each
       * PDI algorithm implementation.
       *
       * Error log messages must be generated. No exceptions generated.
       *
       * @param parameters The parameters to be checked.
       * @return true if the parameters are OK. false if not.
       */
      bool CheckParameters( const TePDIParameters& parameters );

    protected :
      /**
       * A cache of level ordered histograms to avoid rebuilding
       * histograms all the times.
       */
      TePDITypes::TePDIHistogramsMapType histo_cache_;

      /**
       * Runs the current algorithm implementation.
       *
       * @return true if OK. false on error.
       */
      bool RunImplementation();

      /**
       * Reset the internal state to the initial state.
       */
      void ResetState();

      /**
       * Build level ordered histograms from the required raster channels.
       *
       * The result histograms will remain inside the protected
       * object histo_cache_
       *
       * @param inRaster The input Raster.
       * @param histo_levels Number of histogram interpolated levels.
       * @param channels Channels to use for histogram generation.
       * @param force Force histogram regeneratin even if already in cache.
       */
      void BuildHistograms(
        TePDITypes::TePDIRasterPtrType& inRaster,
        unsigned int histo_levels,
        std::vector< int >& channels,
        bool force = false );

      /**
       * Do level remapping from one raster to another.
       *
       * @param inRaster The input Raster.
       * @param outRaster The Output Raster.
       * @param lut The look-up table.
       * @param in_channel in_raster channel to remap.
       * @param out_channel out_raster channel to write on.
       */
      void RemapLevels(
        TePDITypes::TePDIRasterPtrType& inRaster,
        TePDITypes::TePDILutType& lut,
        int in_channel,
        int out_channel,
        TePDITypes::TePDIRasterPtrType& outRaster );

      /**
       * Get the maximum histogram level.
       *
       * Only for histograms with levels equal or greater than zero.
       *
       * @param hist The input LEVEL ORDERED histogram.
       * @return the maximum level.
       */
      double GetHistMax( TePDITypes::TePDIHistogramType& hist );

      /**
       * Get the minimum histogram level.
       *
       * Only for histograms with levels equal or greater than zero.
       *
       * @param hist The input LEVEL ORDERED histogram.
       * @return the minimum level.
       */
      double GetHistMin( TePDITypes::TePDIHistogramType& hist );

      /**
       * Build a min-max look-up table.
       *
       * @param hist Base histogram.
       * @return  The look-up table.
       */
      TePDITypes::TePDILutType GetMinMaxLut(
        TePDITypes::TePDIHistogramType& hist );

      /**
       * Build a Linear look-up table.
       *
       * @param min Minimal level value.
       * @param max Maximum level value.
       * @param hist Base histogram.
       * @return The look-up table.
       */
      TePDITypes::TePDILutType GetLinearLut(
        TePDITypes::TePDIHistogramType& hist,
        double min, double max );

      /**
       * Build a Square Root look-up table.
       *
       * @param min Minimal level value.
       * @param max Maximum level value.
       * @param hist Base histogram.
       * @return The look-up table.
       */
      TePDITypes::TePDILutType GetSquareRootLut(
        TePDITypes::TePDIHistogramType& hist,
        double min, double max );

      /**
       * Build a Square look-up table.
       *
       * @param min Minimal level value.
       * @param max Maximum level value.
       * @param hist Base histogram.
       * @return The look-up table.
       */
      TePDITypes::TePDILutType GetSquareLut(
        TePDITypes::TePDIHistogramType& hist,
        double min, double max );

      /**
       * Build a Log look-up table.
       *
       * @param min Minimal level value.
       * @param max Maximum level value.
       * @param hist Base histogram.
       * @return The look-up table.
       */
      TePDITypes::TePDILutType GetLogLut(
        TePDITypes::TePDIHistogramType& hist,
        double min, double max );

      /**
       * Build a Negative look-up table.
       *
       * @param min Minimal level value.
       * @param max Maximum level value.
       * @param hist Base histogram.
       * @return The look-up table.
       */
      TePDITypes::TePDILutType GetNegativeLut(
        TePDITypes::TePDIHistogramType& hist,
        double min, double max );

      /**
       * Build a Histogram equalizer look-up table.
       *
       * @param hist A LEVEL-ORDERED Base histogram.
       * @return The look-up table.
       */
      TePDITypes::TePDILutType GetHistEqualizerLut(
        TePDITypes::TePDIHistogramType& hist );

      /**
       * Builds a Simple slicer look-up table.
       *
       * @param hist A LEVEL-ORDERED Base histogram.
       * @param in_palette The reference RGB palette.
       * @param min Minimal level value.
       * @param max Maximum level value.
       * @param out_lut The look-up table.
       */
      void GetSimpleSlicerLut(
        TePDITypes::TePDIHistogramType& hist,
        TePDIRgbPalette::pointer in_palette,
        double min,
        double max,
        TePDITypes::TePDILutType& out_lut );
  };

#endif //TEPDICONTRAST_HPP
