/*
TerraLib - a library for developing GIS applications.
Copyright  2001, 2002, 2003 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular
purpose. The library provided hereunder is on an "as is" basis, and the
authors have no obligation to provide maintenance, support, updates,
enhancements, or modifications.
In no event shall INPE be held liable to any party
for direct, indirect, special, incidental, or consequential damages arising
out of the use of this library and its documentation.
*/

/**
 *   This file contains utilitary functions.
 */

#ifndef TEPDIUTILS_HPP
  #define TEPDIUTILS_HPP

  #include "TePDITypes.hpp"

  #include <TeRaster.h>

  #include <sstream>
  #include <string>
  #include <iostream>
  #include <map>

  namespace TePDIUtils{

    /**
     * Data convertion to string.
     *
     * @param data Data to be converted.
     * @return The converted string.
     */
     template< class T >
     std::string to_string( const T& data )
     {
        std::stringstream temp_ss;
        temp_ss << data;
        return temp_ss.str();
     }

    /**
     * Generates a discrete level-ordered histogram by interpolation, using
     * the supplied parameters.
     *
     * @param in_raster Input raster.
     * @param levels The number of interpolated levels, and if set to zero,
     * the number of levels will be automaticlly found by using
     * level step aproximation to one.
     * @param band Band number.
     * @param out_hist The generated histogram.
     * @param force_zero_reference If true, the histogram will always
     * start( or end ) with the zero level.
     * @return true if the histogram has been generated, false on error.
     */
    bool TeGenerateHistogram(
      TePDITypes::TePDIRasterPtrType& in_raster,
      unsigned int levels,
      unsigned int band,
      TePDITypes::TePDIHistogramType& out_hist,
      bool force_zero_reference = 0 );

    /**
     * Allocates one RAM raster using the info of another.
     *
     * @param template_raster Input raster.
     * @param RAMRaster Output raster.
     * @return true if OK. false on error.
     */
    bool TeAllocRAMRaster(
      TePDITypes::TePDIRasterPtrType& template_raster,
      TePDITypes::TePDIRasterPtrType& RAMRaster );

    /**
     * Save the raster to a GeoTIFF file.
     *
     * @param in_raster Input raster.
     * @param file_name Output file name.
     * @return true if OK. false on error.
     */
    bool TeRaster2Geotiff(
      TePDITypes::TePDIRasterPtrType& in_raster,
      const std::string& file_name );

    /**
     * Do pixel by pixel copy from on raster to another ( all bands ).
     *
     * @param source_raster Input raster.
     * @param target_raster Output raster.
     * @return true if OK. false on error.
     */
    bool TeCopyRasterPixels(
      TePDITypes::TePDIRasterPtrType& source_raster,
      TePDITypes::TePDIRasterPtrType& target_raster );

    /**
     * Reset the raster to the new parameters.
     *
     * If there are no decoder associated with the raster, a
     * default MEMORY DECODER will be created.
     *
     * @param raster Input raster.
     * @param bands The new number of bands.
     * @param lines The new number of lines.
     * @param columns The new number of columns.
     * @param palette A palette reference, for generation of a paletted
     * raster.
     * @return true if OK. false on error.
     */
    bool TeResetRaster(
      TePDITypes::TePDIRasterPtrType& raster,
      unsigned int bands, unsigned int lines, unsigned int columns,
      TePDIRgbPalette* palette = 0 );

  };

#endif //TEPDIUTILS_HPP
