/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

/*! \file TeImportExport.h
    This file contains functions to deal with the importing/exporting
	of geographical data to and from  a TerraLib database in usual
	formats
*/
#ifndef __TERRALIB_INTERNAL_IMPORTEXPORT_H
#define __TERRALIB_INTERNAL_IMPORTEXPORT_H

#include "TeTable.h"
#include "TeBox.h"
#include "TeDataTypes.h"

#include <string>
using namespace std;

class TeLayer;
class TeSPRFile;
class TeAsciiFile;
class TeProjection;
class TeDatabase;

/** @defgroup ImpExp Importing and Exporting Functions
 *  TerraLib importing and exporting functions
 *  @{
 */

/** @defgroup MIDMIF Functions to deal with MID/MIF format
 *  @ingroup ImpExp
 *  Functions related to MID MIF format
 *  @{
 */
 
/*!	\fn TeLayer* TeImportMIF(const string& mifFileName, TeDatabase* db, const string& layerName="")
	\brief Imports a geo data in MID/MIF format to a TerraLib database
	\param mifFileName MIF file name
	\param db pointer to a TerraLib database
	\param layerName name of the layer that will contain the data (optional)
	\return a pointer to created layer if the data was imported successfully and 0 otherwise
*/
TeLayer* TeImportMIF(const string& mifFileName, TeDatabase* db, const string& layerName="");

/*!	\fn bool TeImportMIF(TeLayer* layer, const string& mifFileName, string attrTableName="", string objectIdAttr="", int unsigned chunkSize = 60)
	\brief Imports a MID/MIF file to a specific layer TerraLib
	\param layer pointer to a layer
	\param mifFileName MIF file name
	\param objectIdAttr name of the attribute that has the identification of objects
	\param attrTableName the name that the attribute table will be saved
	\param chunkSize the number of objects in memory before save them in the layer
	\return TRUE if the data was imported successfully and FALSE otherwise
*/
bool TeImportMIF(TeLayer* layer, const string& mifFileName, string attrTableName="",  
				 string objectIdAttr="", int unsigned chunkSize = 60);

/*!	\fn void TeReadMIFAttributeList(const string& mifFileName, TeAttributeList& attList)
	\brief Reads the attributes definition of a data in MID/MIF format
*/
void TeReadMIFAttributeList(const string& mifFileName, TeAttributeList& attList);

/*!	\fn TeProjection* TeReadMIFProjection(const string& mifFileName)
	\brief Reads the geographical projection of a data in MID/MIF format
*/
TeProjection* TeReadMIFProjection(const string& mifFileName);

/*!\fn bool TeExportMIF (TeLayer* layer, const string& mifFileName, const string& tableName)
   \brief Exports a layer in a TerraLib database to a file in MID/MIF format
   \param layer pointer to the layer
   \param mifFileName name of the output MID/MIF file
   \param tableName name of the attribute table to be exported
   \return TRUE if the data was successfully exported and FALSE otherwise
*/
bool TeExportMIF (TeLayer* layer, const string& mifFileName, const string& tableName);

/** @} */ // End of subgroup MID/MIF format

/** @defgroup SPR Functions to deal with SPRING GEO/TAB format
 *  @ingroup ImpExp
 *  Functions related to SPRING format
 *  @{
 */ 
/*!	\fn TeLayer* TeImportGEO(const string& geoFileName, TeDatabase* db, const string& layerName="")
	\brief Imports a geo data in SPRING geo/tab format to a TerraLib database
	\param geoFileName GEO/TAB file name
	\param db pointer to a TerraLib database
	\param layerName name of the layer that will contain the data (optional)
	\return a pointer to created layer if the data was imported successfully and 0 otherwise
*/
TeLayer* TeImportGEO(const string& geoFileName, TeDatabase* db, const string& layerName="");

/*! \fn bool TeImportGEO(TeLayer* layer, const string& geoFileName, string attrTableName="", unsigned int chunkSize=60)
	\brief Import a GEO/TAB data to a layer TerraLib
    \param layer pointer to a layer
	\param geoFileName GEO/TAB file name
	\param attrTableName the name that the attribute table will be saved
	\param chunkSize the number of objects in memory before save them in the layer
*/
bool TeImportGEO(TeLayer* layer, const string& geoFileName, string attrTableName="", unsigned int chunkSize=60);

/*! \fn TeReadGeoAttributeList(const string& geoFileName,TeAttributeList& attList, string& labelName )
    \brief Reads the list of attributes  of GEO/TAB file
	\param geoFileName  DBF input file name
	\param attList returns the DBF list of attributes
	\param labelName returns the name of the column that is the link with geometries
	\return TRUE if the list of attributes were was successfully read and FALSE otherwise
*/
void TeReadGeoAttributeList(const string& geoFileName,TeAttributeList& attList, string& labelName );

/*! \fn TeExportSPR(TeLayer* layer, const string& sprFileName, const string& tableName, TeSpringModels cat, const string& name);
	\brief Exports a layer in a TerraLib database to a file in ASCII SPRING format
    \param layer pointer to the layer
    \param sprFileName name of the output SPRING file
    \param tableName name of the attribute table to be exported
    \param cat Spring category
	\param name name of the column that has label or class identifier
    \return TRUE if the data was successfully exported and FALSE otherwise
    \note Parameter 'name' is the Object name if exporting to categories Cadastral or Network .
   Is the name of the column that has the quote value if Numeric category or the name
   of the column that has the class information if category Thematic.
*/
bool TeExportSPR(TeLayer* layer, const string& sprFileName, const string& tableName, 
			 TeSpringModels cat, const string& name);
/** @} */ // End of subgroup SPRING format

/** @defgroup DBF Functions to deal with DBF tables
 *  @ingroup ImpExp
 *  Functions related to DBF table format
 *  @{
 */

/*! \fn bool TeReadDBFAttributeList(const string& dbfFileName, TeAttributeList& attList)
   \brief Reads the list of attributes specified in DBF file 
   \param dbfFileName  DBF input file name
   \param attList returns the DBF list of attributes
   \return TRUE if the list of attributes were was successfully read and FALSE otherwise
*/
bool TeReadDBFAttributeList(const string& dbfFileName, TeAttributeList& attList);

/*! \fn bool TeReadDBFAttributes (const string& dbfFileName, TeTable& table, int nrecords=-1, int rinitial=0, bool createAuto=false)
	\brief Reads a set of records from a shapelib to a structure in memory
	\param dbfFileName DBF file name
	\param table returns the set of records in memory
	\param nrecords number of records to be read
	\param rinitial number of the initial record
	\param createAuto  flag used to indicate if an auto number columns should be added to each record
*/
bool TeReadDBFAttributes (const string& dbfFileName, TeTable& table, 
						  int nrecords=-1, int rinitial=0, bool createAuto=false);

/*! \fn bool TeImportDBFTable(const string& dbfFileName, TeDatabase* db, TeLayer* layer=0,TeAttrTableType tableType=TeAttrExternal)
	\brief Imports a DBF table to a TerraLib database
	This function imports a DBF to a TerraLib database. An attribute table can be associated to a
	particular layer, or as an external table that will be linked to any layer when genereated a 
	theme. A column called object_id_ will be added with unique values automatically generated.
	\param dbfFileName DBF file name
	\param db pointer to a TerraLib database
	\param layer pointer to a layer to which the table will associated
	\param tableType type of the attribute table
*/
bool TeImportDBFTable(const string& dbfFileName, TeDatabase* db, TeLayer* layer=0,TeAttrTableType tableType=TeAttrExternal, string objectIdAtt = "object_id_");

/** @} */ // End of subgroup DBF format

/** @defgroup SHP Functions to deal with Shapefile format
 *  @ingroup ImpExp
 *  Functions related to Shapefile format. Uses some functions related to DBF attribute table format.
 *  @{
 */ 
/*!	\fn TeLayer* TeImportShape(const string& shpFileName, TeDatabase* db, const string& layerName="")
	\brief Imports a geo data in a shapefile format to a TerraLib database
	\param shpFileName Shapefile name
	\param db pointer to a TerraLib database
	\param layerName name of the layer that will contain the data (optional)
	\return a pointer to created layer if the data was imported successfully and 0 otherwise
*/
TeLayer* TeImportShape(const string& shpFileName, TeDatabase* db, const string& layerName="");

/*! \fn bool TeImportShape (TeLayer* layer, const string& shpFileName, string attrTableName="",string objectIdAttr="", unsigned int  chunkSize = 60)
	\brief Import a shapefile to a layer TerraLib
	\param layer pointer to a layer
	\param shpFileName shapefile file name
	\param attrTableName the name that the attribute table will be saved
	\param objectIdAttr name of the attribute that has the identification of objects
	\param chunkSize the number of objects in memory before save them in the layer
*/
bool TeImportShape (TeLayer* layer, const string& shpFileName, string attrTableName="",  
					string objectIdAttr="", unsigned int  chunkSize = 60);

/*! \fn bool TeExportShapefile(TeLayer* layer, const string& shpFileName, const string& tableName)
   \brief Exports a layer in a TerraLib database to a file in MID/MIF format
   \param layer pointer to the layer
   \param shpFileName name of the output shapefile
   \param tableName name of the attribute table to be exported
   \return TRUE if the data was successfully exported and FALSE otherwise
*/
bool TeExportShapefile(TeLayer* layer, const string& shpFileName, const string& tableName);
/** @} */ // End of subgroup SHAPEFILE format

/** @defgroup BNA Functions to deal with BNA format
 *  @ingroup ImpExp
 *  Functions related to BNA format (Atlas GIS BNA)
 *  @{
 */
 
/*!	\fn TeLayer* TeImportBNA(const string& bnaFileName, TeDatabase* db, const string& layerName="")
	\brief Imports a geo data in BNA format to a TerraLib database
	\param bnaFileName BNA file name
	\param db pointer to a TerraLib database
	\param layerName name of the layer that will contain the data (optional)
	\return a pointer to created layer if the data was imported successfully and 0 otherwise
*/
TeLayer* TeImportBNA(const string& bnaFileName, TeDatabase* db, const string& layerName="");

/*!	\fn bool TeImportBNA(TeLayer* layer, const string& bnaFileName, string attrTableName="", string objectIdAttr="", int unsigned chunkSize = 60)
	\brief Imports a BNA file to a specific layer TerraLib
	\param layer pointer to a layer
	\param bnaFileName BNA file name
	\param attrTableName the name that the attribute table will be saved
	\param attList List of attribute of the bna file, used only to name the fields
	\param chunkSize the number of objects in memory before save them in the layer
	\return TRUE if the data was imported successfully and FALSE otherwise
*/
bool TeImportBNA(TeLayer* layer, const string& bnaFileName, string attrTableName,  
				 TeAttributeList& attList, int unsigned chunkSize, const string& linkCol = "");

/*!	\fn bool TeReadBNAAttributeList(const string& bnaFileName, TeAttributeList& attList)
	\brief Reads the attributes definition of a data in BNA format if attList is empty or says if attList size is coorect
*/
bool TeReadBNAAttributeList(const string& bnaFileName, TeAttributeList& attList);

/** @} */ // End of subgroup BNA format

/** \example ImportExample1.cpp
 * This is an example of how to import geographical data in MID/MIF format
 */

/** \example ImportExample2.cpp
 * This is an example of how to import geographical data in shapefile format
 */

/** \example ImportExample3.cpp
 * This is an example of how to import a DBF table
 */

/** \example ImportExample4.cpp
 * This is an example of how to import geographical data in SPRING GEO/TAB format
 */


/** @} */ // End of group ImpExp format

#endif

